/*
  ut.cpp
		
  Q3Plug 1.1b5
		
  Unreal Tournament server query

  Author:   Markus Baumgartner
  Compiler: Microsoft Visual C++ 6.0
  Last Mod: 05/01/2000
  Tab size: 2
*/


#include <assert.h>
#include <string.h>
#include <windows.h>
#include <stdio.h>

#include "npapi.h"
#include "commctrl.h"

#ifndef _UT_H_
#include "ut.h"
#endif

#ifndef _Q3PLUG_H_
#include "q3plug.h"
#endif

// game-specific information
void UT_getInfo(char* text[]) {
  text[0] = "Unreal Tournament";
  text[1] = "Unreal Tournament server query code written by Markus Baumgartner\nmarkus.baumgartner@liwest.at";
  text[2] = "ut.exe";
  text[3] = "%s:%s";      // connect string used in command line (hostname:port)
}

// column names  
void UT_getColumns(char* cols[]) {
  cols[0] = "Ping";
  cols[1] = "Score";
  cols[2] = "Player";
}

// custom compare function to compare columns correctly
int UT_compareFunc(char *val1, char *val2, int sortCol, int sortDir) {
  switch (sortCol) {
	  case 0: return sortDir*(atoi(val1) - atoi(val2));
		case 1: return sortDir*(atoi(val1) - atoi(val2));
		case 2: return sortDir*stricmp(val1,val2);
    default: return 0;
  }
}


// extract sematics from some rules
void UT_checkRule(PluginInstance *This, structRule r) {
  assert(This);
	
	if (0==strcmp(r.name, "maxplayers")) 
    strncpy(This->maxclients, r.value, 7);

  if (0==strcmp(r.name, "hostname")) 
    strncpy(This->hostname, r.value, 64);

  if (0==strcmp(r.name, "numplayers")) 
    strncpy(This->numplayers, r.value, 5);
}


// parse the buffer
BOOL UT_initBuffer(PluginInstance *This) {

	assert(This->buffer);
  structPlayer player;
  structRule rule;
  int count=0;
  char *cur;
  
  if (!strstr(This->buffer,"queryid"))
    return FALSE;

  if (!strstr(This->buffer,"player_")) {     // it's an info packet
    cur = strtok(This->buffer, "\\");
    while (cur && !strstr(cur,"final")) {
      rule.name = cur;
      rule.value = strtok(NULL, "\\");
      UT_checkRule(This, rule);
      UI_insertRule(This, rule);
      cur = strtok(NULL, "\\");
    }
  } else {
    cur = This->buffer;         // it's a player packet
    while (cur ) {
      if (cur=strstr(cur, "player_")) {
        while (*cur && *cur!='\\') cur++;
		    cur++;
	    	player.name = cur;
    		cur = strstr(cur, "\\frags_");
    	  assert(cur);
	    	*cur = '\0';
        cur++;
     	  while (*cur && *cur!='\\') cur++;
    		cur++;
    		player.frags = cur;
    		while (*cur && *cur!='\\') cur++;
    		*cur=0;
	    	cur++;
		    while (*cur && *cur!='\\') cur++;
		    *cur=0;
	    	cur++;
	    	player.ping = cur;
	    	while (*cur && *cur!='\\') cur++;
	    	*cur=0;
	    	cur++;
	    	UI_insertPlayer(This, player);
      }
    }
  }
  return TRUE;	 	
}


// this procedure is called when the plugin window is notified of the arrival
// of a packet on its socket
BOOL UT_receivePacket(PluginInstance *p) {
 	LARGE_INTEGER counter, freq;
  
  assert(p);
  assert(p->text);
 
  p->buffer = (char*) malloc(BUFSIZE);
	assert(p->buffer);

	// clear buffer (don't remove this!)
	memset(p->buffer, 0, BUFSIZE);

	/* receive packet */
  if (recv(p->sock, p->buffer, BUFSIZE, 0) == SOCKET_ERROR) {
    wsprintf(p->text, "Receive error (%d)", WSAGetLastError());
    free(p->buffer);
		return FALSE;
  }

  // have we already got ping time? if no, get packet time
	if (strlen(p->ping) == 0) {
    if (QueryPerformanceFrequency(&freq) && QueryPerformanceCounter(&counter))
	    p->ticks = 1000*(counter.QuadPart - p->ticks) / freq.QuadPart;
	  else
      p->ticks = GetTickCount() - p->ticks; 
  }

  // parse buffer
	if (!UT_initBuffer(p)) {
		strcpy(p->text, "Invalid packet");
		free(p->buffer);
		return FALSE;
	}
    
	// set packet time and reset status message 
  strcpy(p->text, "");
  itoa((int) p->ticks, p->ping, 10);
  

  // now we can free the buffer 
	free(p->buffer);

	return TRUE;
}

// Sends a status-request packet to destination server */
BOOL UT_sendPacket(PluginInstance *p) {
  struct sockaddr_in sa;
  LARGE_INTEGER t;
  HOSTENT *host;
  int err1,err2;

  assert(p);

   // close socket if still open
  if (isOpen(p->sock))
    closesocket(p->sock);

  /* create new socket */
  p->sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
  if (p->sock == INVALID_SOCKET) {
    wsprintf(p->text, "Socket error (%d)", WSAGetLastError());
    return FALSE;
  }
    
  /* set parameters */
  sa.sin_family = AF_INET;
  sa.sin_port = htons(p->queryport);           // 
  sa.sin_addr.S_un.S_addr = inet_addr(p->server);

  if (sa.sin_addr.S_un.S_addr == -1) {
    host = gethostbyname(p->server);
    if (!host) {
      strcpy(p->text, "Invalid server name");
      return FALSE;
    }
    memcpy(&(sa.sin_addr.s_addr), host->h_addr, sizeof(int));
  }
    
  /* send it (finally) */
  err1= sendto(p->sock, UT_getStatus, sizeof(UT_getStatus), 0, (sockaddr*) &sa, sizeof(sa));
  err2= sendto(p->sock, UT_getStatus2, sizeof(UT_getStatus2), 0, (sockaddr*) &sa, sizeof(sa));
  if (err1 == SOCKET_ERROR || err2 == SOCKET_ERROR) {
    wsprintf(p->text, "Send error (%d)", WSAGetLastError());
    return FALSE;
  }

  // we (ab)use p->ping as a boolean flag to check if it is the first packet. so initialize it here
  strcpy(p->ping, "");
  
  if (QueryPerformanceCounter(&t))
		p->ticks = t.QuadPart;
	else
		p->ticks = GetTickCount();

  /* set socket to non-blocking mode */
  err1=WSAAsyncSelect(p->sock, p->fhWnd, WM_SERVER_RESPONSE, FD_READ);
  if (err1 == SOCKET_ERROR) {
    wsprintf(p->text, "Select error (%d)", WSAGetLastError());
    return FALSE;
  }
  
  return TRUE;
}


// custom draw procedure
// just return FALSE if you want default drawing
BOOL UT_drawName(PluginInstance *This, HDC hdc, RECT win, char *name) {
	return FALSE;
}
