// rcon.c

#include "syshdrs.h"

#include "TraySpy.h"
#include "qserver.h"
#include "Rcon.h"
#include "resource.h"
#include "Strn.h"
#include "wsock.h"
#include "prefs.h"
#include "rcon.h"

#ifdef STRICT
WNDPROC gOrigEditCommandProc;
#else
FARPROC gOrigEditCommandProc;
#endif

// We keep a circular line buffer with command lines and responses.
#define MAX_COMMANDS 80
#define COMMAND_SIZE 100

// Ugh.... 16k for this.
typedef char ConsoleBufLine[MAX_COMMANDS];
ConsoleBufLine *gCommandBuffer;
int gCurCommandBuffer = 0;			// Current pointer into the line buffer.
char *gWtextBuf;					// We need a large buffer big enough to
									//  hold everything for a SetWindowText().

extern HINSTANCE ghInstance;
extern HWND gMainWnd;
extern Preferences gPrefs;
extern HFONT gCourier10;



// Remote console command.  If you specify a dbuf, you can get the server's
// reply to it.
//
int Rcon(char *const dbuf, const size_t dbufsize, int timeout, const char *const fmt, ...)
{
	char buf[512];
	char msgbuf[1500];
	va_list ap;
	int rc;

	assert((fmt != NULL) && (fmt[0] != '\0'));
	ZeroMemory(buf, sizeof(buf));
	va_start(ap, fmt);
	wvsprintf(buf, fmt, ap);
	va_end(ap);
	buf[sizeof(buf) - 1] = '\0';
	assert(buf[0] != '\0');

	if ((dbuf != NULL) && (dbufsize > 0)) {
		ZeroMemory(dbuf, dbufsize);
	}
	if (buf[0] == '\0')
		return (0);

	rc = SendRconRequest(gPrefs.rconPassword, "%s", buf);
	if (rc >= 0) {
		rc = WaitForReplyOrClose(timeout);
		if (rc != 0) {
			Strncpy(dbuf, "no reply received from server.", dbufsize);
		} else {
			rc = ReadReply(msgbuf, sizeof(msgbuf));
			if (rc >= 0) {
				if (strnicmp(msgbuf + 4, "print\n", 6) != 0) {
					// Invalid reply received from server,
					// but do we care?
					//
					rc = -1;
					Strncpy(dbuf, "invalid reply received from server.", dbufsize);
				} else if ((dbuf != NULL) && (dbufsize > 0)) {
					Strncpy(dbuf, msgbuf + 4 + 6, dbufsize);
				}
			}
		}
	}
	return (rc);
}	// Rcon




// The next few functions are interfaces to Rcon(), above.
//
int RconSetMap(const char *const newMapName)
{
	int rc;
	char msg[256];
	
	assert((newMapName != NULL) && (newMapName[0] != '\0'));
	rc = Rcon(msg, sizeof(msg), 30, "map %s", newMapName);
	if (rc < 0) {
		ErrBox("Remote console command failed: %s", msg);
	}
	return (rc);
}	// RconSetMap




int RconSetFragLimit(const char *const newLimit)
{
	int rc;
	char msg[256];

	assert((newLimit != NULL) && (newLimit[0] != '\0'));
	rc = Rcon(NULL, 0, RCON_DEFAULT_TIMEOUT, "say Changing fraglimit to %s.", newLimit);
	if (rc >= 0) {
		rc = Rcon(msg, sizeof(msg), RCON_DEFAULT_TIMEOUT, "fraglimit %s", newLimit);
		if ((rc < 0) || (msg[0] != '\0')) {
			ErrBox("Remote console command failed: %s", msg);
		}
	}
	return (rc);
}	// RconSetFragLimit




int RconSetTimeLimit(const char *const newLimit)
{
	int rc;
	char msg[256];

	assert((newLimit != NULL) && (newLimit[0] != '\0'));
	rc = Rcon(NULL, 0, RCON_DEFAULT_TIMEOUT, "say Changing timelimit to %s.", newLimit);
	if (rc >= 0) {
		rc = Rcon(msg, sizeof(msg), RCON_DEFAULT_TIMEOUT, "timelimit %s", newLimit);
		if ((rc < 0) || (msg[0] != '\0')) {
			ErrBox("Remote console command failed: %s", msg);
		}
	}
	return (rc);
}	// RSetTimeLimit




int RconSay(const char *const toWhom, const char *const what)
{
	int rc;

	assert((what != NULL) && (what[0] != '\0'));
	if (toWhom == NULL) {
		rc = Rcon(NULL, 0, RCON_DEFAULT_TIMEOUT, "say %s", what);
	} else {
		rc = Rcon(NULL, 0, RCON_DEFAULT_TIMEOUT, "say %s: %s", toWhom, what);
	}
	return (rc);
}	// RconSay




int RconKickPlayerID(const int id)
{
	int rc;
	char msg[256];

	assert((id >= 0) && (id < 128));
	rc = Rcon(msg, sizeof(msg), RCON_DEFAULT_TIMEOUT, "kick %d", id);
	if ((rc < 0) || (strstr(msg, "was kicked") == NULL)) {
		ErrBox("Remote console command failed: %s", msg);
	}
	return (rc);
}	// RconKickPlayerID




// This thing re-sets the entire console window each time
// a new line is added.  Ugly, but it works pretty fast
// on today's machines.
//
static void UpdateConsoleOutput(HWND hDlg)
{			
	HWND ctrl;
	char *linebuf, *src;
	int i, j;
	char *dst, *dstlim, c;

	ctrl = GetDlgItem(hDlg, IDC_COMMANDS);
	if (ctrl != NULL) {
		gWtextBuf[0] = '\0';
		dst = gWtextBuf;
		dstlim = dst + (sizeof(char) * MAX_COMMANDS * COMMAND_SIZE);
		for (i=0, j=gCurCommandBuffer; i<MAX_COMMANDS; i++, j++) {
			if (j >= MAX_COMMANDS)
				j = 0;
			linebuf = gCommandBuffer[j];
			if (linebuf[0] == '\0')
				continue;
			for (src = linebuf; ;) {
				c = *src++;
				if (c == '\0')
					break;
				if (dst < dstlim)
					*dst++ = c;
			}
		}
		*dst = '\0';
		SetWindowText(ctrl, gWtextBuf);
		SendMessage(ctrl, EM_LINESCROLL, 9999, 9999);
	}
}	// UpdateConsoleOutput




// This function takes the current contents of the
// command entry edit box, and runs the command from it.
//
static void RconCommandOK(HWND hDlg)
{
	HWND ctrl;
	char *linebuf;
	char *src, *dst, *dlim, c;
	char msgbuf[1500];

	ctrl = GetDlgItem(hDlg, IDC_COMMAND);
	if (ctrl != NULL) {
		linebuf = gCommandBuffer[gCurCommandBuffer];
		ZeroMemory(linebuf, COMMAND_SIZE);
		GetWindowText(ctrl, linebuf + 2, COMMAND_SIZE - 1 - 2);
		if (linebuf[2] != '\0') {
			linebuf[0] = '>';	// Prefix this line in the output with "> "
			linebuf[1] = ' ';
			gCurCommandBuffer++;
			if (gCurCommandBuffer >= MAX_COMMANDS)
				gCurCommandBuffer = 0;

			// Run the remote command.
			if ((Rcon(msgbuf, sizeof(msgbuf), RCON_DEFAULT_TIMEOUT, "%s", linebuf + 2) < 0) && (msgbuf[0] == '\0')) {
				STRNCPY(msgbuf, "rcon failed; is your password correct?");
			} else if (msgbuf[0] == '\0') {
				STRNCPY(msgbuf, "ok");
			}

			// Each line in the circular line buffer must have a trailing \r\n\0,
			// because the edit box requires line breaks that use \r\n.
			//
			Strncat(linebuf, "\r\n", COMMAND_SIZE);
			
			// Prep the first line of response.
			dst = gCommandBuffer[gCurCommandBuffer];
			dlim = dst + COMMAND_SIZE;
			ZeroMemory(dst, COMMAND_SIZE);

			// The server may send back a multi-line reply in the same packet,
			// delimited by \n's  (note: not \r\n's), so for each line,
			// add it to the circular line buffer.
			//
			for (src = msgbuf; ; ) {
				c = *src++;
				if (c == '\0') {
					// End of last line.
					if (dst < dlim)
						*dst++ = '\r';
					if (dst < dlim)
						*dst++ = '\n';
					*dst = '\0';
					break;
				}
				if (c == '\r')
					continue;
				if (c == '\n') {
					// End of line...
					if (dst < dlim)
						*dst++ = '\r';
					if (dst < dlim)
						*dst++ = '\n';
					*dst = '\0';

					// ...Now start on a new one.
					gCurCommandBuffer++;
					if (gCurCommandBuffer >= MAX_COMMANDS)
						gCurCommandBuffer = 0;
					dst = gCommandBuffer[gCurCommandBuffer];
					dlim = dst + COMMAND_SIZE;
					ZeroMemory(dst, COMMAND_SIZE);
					continue;
				}
				if (dst < dlim)
					*dst++ = c;
			}

			// Refresh the console and draw the new entries.
			SetWindowText(ctrl, "");
			UpdateConsoleOutput(hDlg);
		}
	}
}	// RconCommandOK




static LRESULT CALLBACK
CommandEditProc(HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	int ch;
	
	// This never gets called, but it is here in case we decide
	// to do some special processing for key input.
	//
	if (iMsg == WM_CHAR) {
		ch = (int) wParam;
		if ((ch == '\n') || (ch == '\r')) {
			return 0;
		}
	}

	return CallWindowProc(gOrigEditCommandProc, hwnd, iMsg, wParam, lParam);
}	/* CommandEditProc */




static void InitRconDlg(HWND hDlg)
{
	HWND ctrl;

	gOrigEditCommandProc = NULL;
	ctrl = GetDlgItem(hDlg, IDC_COMMANDS);
	if (ctrl != NULL) {
		// We can save the console contents in between
		// views, so reload it if we have it.
		//
		UpdateConsoleOutput(hDlg);
	}

	ctrl = GetDlgItem(hDlg, IDC_COMMAND);
	if (ctrl != NULL) {
		SetWindowText(ctrl, "");

		// We don't need to use this proc yet (see above).
		//
#ifdef STRICT
		gOrigEditCommandProc = (WNDPROC) GetWindowLong((HWND) ctrl, GWL_WNDPROC);
#else
		gOrigEditCommandProc = (FARPROC) GetWindowLong((HWND) ctrl, GWL_WNDPROC);
#endif
		SetWindowLong((HWND) ctrl, GWL_WNDPROC, (LONG) CommandEditProc);
	}
}	// InitRconDlg




// So far, we don't need to do anything when the window
// closes.
//
static void RconDlgOK(HWND hDlg)
{
	HWND ctrl;

	ctrl = GetDlgItem(hDlg, IDC_COMMANDS);
	if (ctrl != NULL) {

	}

	ctrl = GetDlgItem(hDlg, IDC_COMMAND);
	if (ctrl != NULL) {

	}
}	// RconDlgOK




#pragma warning(disable : 4100)		// warning C4100: unreferenced formal parameter
static BOOL CALLBACK RconDlgProc(HWND hDlg, UINT iMsg, WPARAM w, LPARAM l_unused)
{
	WORD id;

	switch (iMsg) {
	case WM_INITDIALOG:
		InitRconDlg(hDlg);
		return TRUE;

	case WM_COMMAND:
		id = LOWORD(w);
		switch (id) {
			case IDOK:
				// They hit enter, so process the command.
				// Note that there is no "OK" button or
				// a cancel button.
				//
				RconCommandOK(hDlg);
				return TRUE;

			case IDCANCEL:			// Hit close button
				RconDlgOK(hDlg);
				EndDialog(hDlg, 0);
				return TRUE;

			//case EN_UPDATE:
				//idEditCtrl = (int) LOWORD(wParam);
				// hwndEditCtrl = (HWND) lParam;
				//RconCommandOK(hDlg);
				//break;
		}
		break;
	}
	return (FALSE);
}	// RconDlgProc
#pragma warning(default : 4100)		// warning C4100: unreferenced formal parameter




// Since our somewhat wasteful circular line buffer uses a considerable
// amount of memory (16k, well maybe not much these days...), don't
// allocate it until the user actually opens up the rcon window.
//
static BOOL AllocRConsole(void)
{
	if ((gCommandBuffer != NULL) && (gWtextBuf != NULL))
		return (TRUE);	// Did this already

	gCommandBuffer = malloc(sizeof(char) * (MAX_COMMANDS * COMMAND_SIZE + 1));
	if (gCommandBuffer == NULL)
		return (FALSE);
	ZeroMemory(gCommandBuffer, (DWORD) (sizeof(char) * (MAX_COMMANDS * COMMAND_SIZE + 1)));

	gWtextBuf = malloc(sizeof(char) * (MAX_COMMANDS * COMMAND_SIZE + 1));
	if (gWtextBuf == NULL) {
		free(gCommandBuffer);
		gCommandBuffer = NULL;
		return (FALSE);
	}
	ZeroMemory(gWtextBuf, (DWORD) (sizeof(char) * (MAX_COMMANDS * COMMAND_SIZE + 1)));
	return (TRUE);
}	// AllocRConsole




// This pops up the remote console window.  This probably should have
// been a modeless dialog box rather than a modal dialog.
//
void ShowConsole(void)
{
	if (AllocRConsole() == FALSE) {
		ErrBox("Could not allocate memory needed for console buffers.");
		return;
	}
	DialogBox(ghInstance, MAKEINTRESOURCE(IDD_RCON), gMainWnd, RconDlgProc);
}	// ShowConsole





void InitConsole(void)
{
	gCommandBuffer = NULL;
	gWtextBuf = NULL;
	gCurCommandBuffer = 0;
}	// InitConsole




void DisposeConsole(void)
{
	if (gCommandBuffer != NULL) {
		free(gCommandBuffer);
		gCommandBuffer = NULL;
	}
	if (gWtextBuf != NULL) {
		free(gWtextBuf);
		gWtextBuf = NULL;
	}
}	// DisposeConsole
