// TraySpy
//
// Copyright (C) 1998 by Mike Gleason.
// All Rights Reserved.
//
// mgleason@NcFTP.com
//
#include "syshdrs.h"

#include "TraySpy.h"
#include "wsock.h"
#include "qserver.h"
#include "prefs.h"
#include "Strn.h"
#include "resource.h"
#include "rcon.h"

static char szAppName[] = "TraySpy";

// Note:  despite the use of TCHAR in some places, this code isn't really
// ready for Unicode.
//
TCHAR gWndClass[32];

HINSTANCE ghInstance = 0;
HWND gMainWnd = 0;
HFONT gTimesNewRoman12, gTimesNewRomanBold12, gCourier10;
HMENU ghMapMenu, ghPlayerMenu, ghServerMenu, ghMainWndDefaultMenu;

// We use a "ticker" to trigger events.  When the program starts up we start the ticker,
// which increments by one each second.  Depending on what second it is we may do something,
// like hide the window or query the server.
//
time_t gStart = 0;
UINT gTimerID = 0;
int gIncrements = 0;
int gHideIncrement = 5;

// You can do a "Refresh Now" by setting this to >0.
int gRefreshOnNextIncrement = 0;

// This is the stuff we use for the system tray.
// We track whether we're even using it, and keep the structure for re-use.
//
BOOL gbTrayIconLoaded = FALSE;
NOTIFYICONDATA nid;

// The tray icon indicates how many players are on (we have an icon for
// each number from 0 to 33), and if any buddies are on (a duplicate set
// of number icons with the number in red to denote a buddy is also on).
//
HICON gCountIcons[34];
HICON gCountBIcons[34];

// If a player was right clicked, save it for use for the command chosen.
// We save the "ID" because some commands work better using the ID number,
// since those same commands get confused when players have weird characters
// in their name.
//
char gSelectedPlayerName[32];
int gSelectedPlayerID;

// When you change the map, we want to print a message to the console saying
// we are going to change the map.  Then we want to delay a few seconds
// before doing the actual change.
//
int gChangeMapIncrement = 0;
char gNewMapName[32];

// Max number of players that will be ranked on the game status window.
// I should probably just break down and use a scroll bar instead of
// simply truncating the ranks if the window is too small.
//
#define MAX_VIEWABLE_PLAYERS	64

// A bunch of rectangles, pre-computed for speed and so we can do our
// right-click hit-testing on these.
//
static RECT gRankHeaderRect, gNameHeaderRect, gScoreHeaderRect, gPingHeaderRect;
static RECT gServerNameRect, gMapNameRect;
static RECT gServerNameHeaderRect, gMapNameHeaderRect;
static RECT gMsgRect;
static RECT gMainWndRect;

static RECT gRankRect[MAX_VIEWABLE_PLAYERS];
static RECT gNameRect[MAX_VIEWABLE_PLAYERS];
static RECT gScoreRect[MAX_VIEWABLE_PLAYERS];
static RECT gPingRect[MAX_VIEWABLE_PLAYERS];

extern QuakeServerStatus gServerStatus;
extern Preferences gPrefs;



// Global utility routine to pop-up a dialog box.
void ErrBox(const char *const fmt, ...)
{
	char buf[512];
	va_list ap;

	ZeroMemory(buf, sizeof(buf));
	va_start(ap, fmt);
	wvsprintf(buf, fmt, ap);
	va_end(ap);

	MessageBox(gMainWnd, buf, szAppName, MB_OK | MB_ICONINFORMATION);
}	// ErrBox




void PlaySoundFile(LPCSTR fn)
{
	PlaySound(fn, ghInstance, SND_FILENAME|SND_ASYNC|SND_NOWAIT);
}	// PlaySoundFile




// Initializes our use of the system tray in the taskbar.
static void InitTray(void)
{
	if (gbTrayIconLoaded)
		return;				// Already loaded

	ZeroMemory(&nid, sizeof(nid));
	nid.cbSize = (DWORD) sizeof(nid);
	nid.hWnd = gMainWnd;							// Handle of the window that receives notification messages associated with the icon
	nid.uID = IDR_TRAY;								// Application-defined identifier of the taskbar icon. 
	nid.uFlags = (NIF_ICON | NIF_MESSAGE | NIF_TIP);
	nid.uCallbackMessage = WM_TRAY_TRAYSPY;
	nid.hIcon = gCountIcons[0];
	STRNCPY(nid.szTip, "TraySpy");

	if (Shell_NotifyIcon(NIM_ADD, &nid)) {
		gbTrayIconLoaded = TRUE;
	}
}	// InitTray




static void SetTrayTip(char *cp)
{
	if (!gbTrayIconLoaded)
		return;

	STRNCPY(nid.szTip, cp);
	nid.uFlags = (NIF_ICON | NIF_MESSAGE | NIF_TIP);
	(void) Shell_NotifyIcon(NIM_MODIFY, &nid);
}	// SetTrayTip




static void SetTrayIcon(HICON hIcon)
{
	if (!gbTrayIconLoaded)
		return;

	nid.hIcon = hIcon;
	nid.uFlags = (NIF_ICON | NIF_MESSAGE | NIF_TIP);
	(void) Shell_NotifyIcon(NIM_MODIFY, &nid);
}	// SetTrayIcon



// Here's where we determine which icon to use for the tray.
// We use an icon with a number stamped onto it in black ink
// if no buddies are logged on, and we use a set of icons
// that use red ink for the case when buddies are logged on.
//
static void SetTrayIconByStatus(void)
{
	int n;
	HICON hIcon;

	n = gServerStatus.nPlayers;
	if (n > (sizeof(gCountIcons) / sizeof(HICON)))
		n = (sizeof(gCountIcons) / sizeof(HICON));
	else if (n < 0)
		n = 0;
	if (gServerStatus.buddiesPlaying > 0) {
		hIcon = gCountBIcons[n];
	} else {
		hIcon = gCountIcons[n];
	}
	if (hIcon != (HICON) 0)
		SetTrayIcon(hIcon);
}	// SetTrayIconByStatus




// Don't forget to do this so the tray knows your application
// quit and can remove the icon.
//
static void DisposeTray(void)
{
	if (!gbTrayIconLoaded)
		return;

	if (Shell_NotifyIcon(NIM_DELETE, &nid)) {
		gbTrayIconLoaded = FALSE;
	}
}	// DisposeTray




static void LoadIcons(void)
{
	// Looks ugly, but it was easier to do this rather than to have
	// a for loop and re-edit the resource.h file.
	//
	ZeroMemory(gCountIcons, sizeof(gCountIcons));
	gCountIcons[0] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_0));
	gCountIcons[1] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_1));
	gCountIcons[2] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_2));
	gCountIcons[3] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_3));
	gCountIcons[4] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_4));
	gCountIcons[5] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_5));
	gCountIcons[6] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_6));
	gCountIcons[7] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_7));
	gCountIcons[8] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_8));
	gCountIcons[9] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_9));
	gCountIcons[10] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_10));
	gCountIcons[11] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_11));
	gCountIcons[12] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_12));
	gCountIcons[13] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_13));
	gCountIcons[14] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_14));
	gCountIcons[15] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_15));
	gCountIcons[16] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_16));
	gCountIcons[17] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_17));
	gCountIcons[18] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_18));
	gCountIcons[19] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_19));
	gCountIcons[20] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_20));
	gCountIcons[21] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_21));
	gCountIcons[22] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_22));
	gCountIcons[23] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_23));
	gCountIcons[24] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_24));
	gCountIcons[25] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_25));
	gCountIcons[26] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_26));
	gCountIcons[27] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_27));
	gCountIcons[28] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_28));
	gCountIcons[29] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_29));
	gCountIcons[30] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_30));
	gCountIcons[31] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_31));
	gCountIcons[32] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_32));
	gCountIcons[33] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_33));

	// Same, for the red icons (BIcons == Buddy Icons).
	//
	ZeroMemory(gCountBIcons, sizeof(gCountBIcons));
	gCountBIcons[0] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B0));
	gCountBIcons[1] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B1));
	gCountBIcons[2] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B2));
	gCountBIcons[3] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B3));
	gCountBIcons[4] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B4));
	gCountBIcons[5] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B5));
	gCountBIcons[6] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B6));
	gCountBIcons[7] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B7));
	gCountBIcons[8] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B8));
	gCountBIcons[9] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B9));
	gCountBIcons[10] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B10));
	gCountBIcons[11] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B11));
	gCountBIcons[12] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B12));
	gCountBIcons[13] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B13));
	gCountBIcons[14] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B14));
	gCountBIcons[15] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B15));
	gCountBIcons[16] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B16));
	gCountBIcons[17] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B17));
	gCountBIcons[18] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B18));
	gCountBIcons[19] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B19));
	gCountBIcons[20] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B20));
	gCountBIcons[21] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B21));
	gCountBIcons[22] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B22));
	gCountBIcons[23] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B23));
	gCountBIcons[24] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B24));
	gCountBIcons[25] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B25));
	gCountBIcons[26] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B26));
	gCountBIcons[27] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B27));
	gCountBIcons[28] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B28));
	gCountBIcons[29] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B29));
	gCountBIcons[30] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B30));
	gCountBIcons[31] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B31));
	gCountBIcons[32] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B32));
	gCountBIcons[33] = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_ICON_B33));
}	// gCountIcons




static void SetTrayTipByStatus(void)
{
	char shortServerName[32], *cp;
	char szTip[64];
	char buf[64];

	if (gServerStatus.name[0] != '\0') {
		STRNCPY(szTip, "TraySpy: ");
		
		// The tray tooltip seems to be limited in the amount of characters (64),
		// so abbreviate the server name if needed.
		//
		ZeroMemory(shortServerName, sizeof(shortServerName));
		STRNCPY(shortServerName, gServerStatus.name);
		cp = shortServerName + sizeof(shortServerName) - 2;
		if (*cp != '\0') {
			// The name seems to have filled up the buffer, and
			// was truncated at the sizeof(shortServerName)'th 
			// character.  Now try to make an effort to add the
			// ellipsis at a word break.
			//
			cp -= 3;
			while (cp > (shortServerName + sizeof(shortServerName) - 12)) {
				if (isspace(*cp)) {
					*cp++ = '.';
					*cp++ = '.';
					*cp++ = '.';
					*cp = '\0';
					break;
				}
				--cp;
			}
			if (*cp != '\0') {
				cp = shortServerName + sizeof(shortServerName) - 2;
				cp[0] = '.';
				cp[-1] = '.';
				cp[-2] = '.';
				cp[1] = '\0';
			}
		}
		
		STRNCAT(szTip, shortServerName);
		STRNCAT(szTip, " (map: ");
		STRNCAT(szTip, gServerStatus.mapname);
		STRNCAT(szTip, ", ");
		if (gServerStatus.nPlayers == 0) {
			STRNCAT(szTip, "no players)");
		} else if (gServerStatus.nPlayers == 1) {
			STRNCAT(szTip, "1 player)");
		} else {
			sprintf(buf, "%d", gServerStatus.nPlayers);
			STRNCAT(szTip, buf);
			STRNCAT(szTip, " players)");
		}
		SetTrayTip(szTip);
	} else {
		SetTrayTip(_T("TraySpy"));
	}
}	// SetTrayTipByStatus




// This function gets called every second since we set up a
// time to generate WM_TIMER messages and call this.
//
// This is handy for scheduling events, such as refreshing
// the server, changing the map, or hiding the window.
//
static void PeriodicTasks(void)
{
	++gIncrements;

	if (
		((gIncrements % gPrefs.refreshIntervalSecs) == (gPrefs.refreshIntervalSecs - 1)) ||
		(gRefreshOnNextIncrement != 0)
		)
	{
		PollQuakeServer();
		SetTrayTipByStatus();
		SetTrayIconByStatus();
		InvalidateRgn(gMainWnd, (HRGN) 0, TRUE);	// Tell game status window to update
		gRefreshOnNextIncrement = 0;
	}

	if (gIncrements == gChangeMapIncrement) {
		gChangeMapIncrement = 0;
		RconSetMap(gNewMapName);
	}

	if (gIncrements == gHideIncrement) {
		ShowWindow(gMainWnd, SW_HIDE);
		InitTray();
	}
}	/* PeriodicTasks */




// This really is a mis-nomer, but it is close enough since
// the refresh will be done on the next timer tick.
//
void RefreshNow(void)
{
	gRefreshOnNextIncrement = 1;
}	// RefreshNow




// Dispose of a few things before exiting.
//
static void CleanUp(void)
{
	if (gTimerID != 0)
		KillTimer(gMainWnd, gTimerID);
	SaveWindowPosition(gMainWnd);
	SavePreferences();
	DisposeWinsock();
	DisposeTray();
	DisposeConsole();
}	// CleanUp




// Utility interface to call TextOut with varargs.
//
void TextOutAt(HDC hdc, int x, int y, const char *const fmt, ...)
{
	char buf[512];
	va_list ap;
	int len;

	va_start(ap, fmt);
	len = wvsprintf(buf, fmt, ap);
	va_end(ap);

	(void) TextOut(hdc, x, y, buf, len);
}	// TextOutAt




// Precompute the window regions so we don't have
// to do this each time we draw the window.
//
// Actually we can even get away with not recomputing
// the rectangles when they resize the window, since
// out stuff is not sensitive to the window bottom
// or right, with the exception of the last refresh
// area.  That area is always recomputed anyway.
//
#define col1	10
#define col2	35
#define col2a	70
#define col3	250
#define col4	320
#define rowh	18

static void ComputeRects(void)
{
	int rightMargin, headerRow, row;
	int i;

	GetClientRect(gMainWnd, &gMainWndRect);
	rightMargin = gMainWndRect.right - 20;

	gMsgRect = gMainWndRect;
	gMsgRect.left = 20;
	gMsgRect.right -= 20;
	gMsgRect.top = 20;

	SetRect(&gServerNameHeaderRect,	col1,	10,			col2a - 2,		rowh + 10);
	SetRect(&gMapNameHeaderRect,	col1,	rowh + 10,	col2a - 2,		rowh * 2 + 10);
	SetRect(&gServerNameRect,		col2a,	10,			rightMargin,	rowh + 10);
	SetRect(&gMapNameRect,			col2a,	rowh + 10,	rightMargin,	rowh * 2 + 10);

	headerRow = rowh * 3 + 10;
	SetRect(&gRankHeaderRect,		col1,	headerRow,	col2 - 6,		headerRow + rowh);
	SetRect(&gNameHeaderRect,		col2,	headerRow,	col3 - 2,		headerRow + rowh);
	SetRect(&gScoreHeaderRect,		col3,	headerRow,	col4 - 2,		headerRow + rowh);
	SetRect(&gPingHeaderRect,		col4,	headerRow,	rightMargin,	headerRow + rowh);

	for (i=0; i<MAX_VIEWABLE_PLAYERS; i++) {
		row = headerRow + ((i + 1) * rowh);
		SetRect(&gRankRect[i],		col1,	row,		col2 - 6,		row + 16);
		SetRect(&gNameRect[i],		col2,	row,		col3 - 2,		row + 16);
		SetRect(&gScoreRect[i],		col3,	row,		col4 - 2,		row + 16);
		SetRect(&gPingRect[i],		col4,	row,		rightMargin,	row + 16);
	}
}	// ComputeRects




static void OnDraw(HWND gMainWnd, HDC hdc)
{
	int i;
	RECT rect;
	QuakeServerStatusPtr qssp = &gServerStatus;
	HFONT oldFont = (HFONT) 0;
	HBRUSH white;
	TCHAR tbuf[256];
	char timestr[64];

	GetClientRect(gMainWnd, &gMainWndRect);
	FillRect(hdc, &gMainWndRect, (white = GetStockObject(WHITE_BRUSH)));
	
	if (gTimesNewRomanBold12 != (HFONT) 0)
		oldFont = SelectObject(hdc, gTimesNewRomanBold12);

	if (gPrefs.serverAddrStr[0] == '\0') {
		DrawText(hdc,
			_T("Select a Quake2 server to poll by right clicking the tray icon menu and selecting \"Configure\"."),
			-1, &gMsgRect,
			DT_WORDBREAK
			);
		if (oldFont != (HFONT) 0)
			(void) SelectObject(hdc, oldFont);
		return;
	} else if (qssp->name[0] == '\0') {
		DrawText(hdc,
			_T("The server you have selected is not responding."),
			-1, &gMsgRect,
			DT_WORDBREAK
			);
		if (oldFont != (HFONT) 0)
			(void) SelectObject(hdc, oldFont);
		return;
	}

	// Server (header)
	DrawText(hdc,
		_T("Server:"),
		-1, &gServerNameHeaderRect,
		DT_LEFT | DT_SINGLELINE
		);

	// Map (header)
	DrawText(hdc,
		_T("Map:"),
		-1, &gMapNameHeaderRect,
		DT_LEFT | DT_SINGLELINE
		);

	if (qssp->nPlayers == 0) {
		TextOutAt(hdc, col1, 64, "(server has no active players)");
	} else {
		// Ranks (headers)
		for (i=0; i<MAX_VIEWABLE_PLAYERS; i++) {
			if (qssp->players[i].name[0] == '\0')
				break;
			if (gNameRect[i].bottom > (gMainWndRect.bottom - 20))
				break;
			_stprintf(tbuf, _T("%d."), (i + 1));
			DrawText(hdc,
				tbuf,
				-1, &gRankRect[i],
				DT_RIGHT | DT_SINGLELINE
				);	
		}
		
		// Name (header)
		DrawText(hdc,
			"Player",
			-1, &gNameHeaderRect,
			DT_LEFT | DT_SINGLELINE
			);
		
		// Score (header)
		DrawText(hdc,
			"Score",
			-1, &gScoreHeaderRect,
			DT_LEFT | DT_SINGLELINE
			);
		
		// Ping (header)
		DrawText(hdc,
			"Ping",
			-1, &gPingHeaderRect,
			DT_LEFT | DT_SINGLELINE
			);
	}


	if (gTimesNewRoman12 != (HFONT) 0)
		(void) SelectObject(hdc, gTimesNewRoman12);

	// Server
	DrawText(hdc,
		qssp->name,
		-1, &gServerNameRect,
		DT_LEFT | DT_SINGLELINE | DT_END_ELLIPSIS
		);

	// Map
	DrawText(hdc,
		qssp->mapname,
		-1, &gMapNameRect,
		DT_LEFT | DT_SINGLELINE | DT_END_ELLIPSIS
		);
	
	SetTextColor(hdc, RGB(0,0,0));					// Black text.
	for (i=0; i<MAX_VIEWABLE_PLAYERS; i++) {
		if (qssp->players[i].name[0] == '\0')
			break;
		if (gNameRect[i].bottom > (gMainWndRect.bottom - 20))
			break;
		// Players
		if (qssp->players[i].isBuddy) {
			SetTextColor(hdc, RGB(255,0,0));		// Red text.
			DrawText(hdc,
				qssp->players[i].name,
				-1, &gNameRect[i],
				DT_LEFT | DT_SINGLELINE | DT_NOCLIP
				);
			SetTextColor(hdc, RGB(0,0,0));			// Black text.
		} else {
			DrawText(hdc,
				qssp->players[i].name,
				-1, &gNameRect[i],
				DT_LEFT | DT_SINGLELINE | DT_NOCLIP
				);
		}

		_stprintf(tbuf, _T("%d"), qssp->players[i].score);
		DrawText(hdc,
			tbuf,
			-1, &gScoreRect[i],
			DT_LEFT | DT_SINGLELINE
			);	

		_stprintf(tbuf, _T("%d"), qssp->players[i].ping);
		DrawText(hdc,
			tbuf,
			-1, &gPingRect[i],
			DT_LEFT | DT_SINGLELINE
			);	
	}


	strftime(timestr, sizeof(timestr), "Last update was at %I:%M:%S %p.", localtime(&qssp->lastUpdate));
	rect = gMainWndRect;
	rect.top = rect.bottom - 20;
	rect.right -= 5;
	DrawText(hdc, timestr, (int) strlen(timestr), &rect, DT_SINGLELINE | DT_RIGHT | DT_VCENTER);

	if (oldFont != (HFONT) 0) {
		(void) SelectObject(hdc, oldFont);
	}
}	/* OnDraw */




// Display the incredibly detailed about box :-)
//
static void About(void)
{
	DialogBox(ghInstance, MAKEINTRESOURCE(IDD_ABOUT), gMainWnd, AboutDlgProc);
}	// About



// Display the incredibly detailed about box :-)
//
static void JoinGame(void)
{
	char prog[300], *cp;
	int winExecResult, rc;
	OPENFILENAME o;
	
	if ((gPrefs.gamePath[0] == '\0') || (strcmp(gPrefs.gamePath, "unknown") == 0)) {
		// Must get path first
		ErrBox("The location of Quake2 was not auto-detected.  Please select it for me once and I will save it for future reference.");
		
		ZeroMemory(prog, sizeof(prog));
		ZeroMemory(&o, sizeof(o));
		o.lStructSize = sizeof(o);
		o.Flags				= OFN_FILEMUSTEXIST  | OFN_PATHMUSTEXIST;
		o.hwndOwner         = gMainWnd;
		o.lpstrFile         = prog;
		o.lpstrFilter       = _T("Programs (*.exe)\0*.exe\0All Files (*.*)\0*.*\0\0";);
		o.lpstrDefExt		= _T("exe");
		o.nMaxFile          = sizeof(prog);
		if (GetOpenFileName(&o)) {
			STRNCPY(gPrefs.gamePath, prog);
		} else {
			return;
		}
	}

	// Try to "cd" to the same directory of the app
	// before we run it.
	//
	STRNCPY(prog, gPrefs.gamePath);
	if (prog[1] == ':') {
		// Have a drive letter.
		cp = strrchr(prog, '\\');
		if ((cp == NULL) || (cp == (prog + 3))) {
			prog[2] = '\\';
			prog[3] = '\0';
		} else {
			*cp = '\0';
		}
		rc = SetCurrentDirectory(prog);
	} else if ((prog[0] == '\\') && (prog[1] == '\\') && (prog[2] != '\0') && (prog[2] != '\\')) {
		// Have a UNC path, like \\servername\share\directory\app.exe
		cp = strrchr(prog + 2, '\\');
		if (cp != NULL) {
			cp = strrchr(cp + 1, '\\');
			if (cp != NULL) {
				*cp = '\0';
				rc = SetCurrentDirectory(prog);
			}
		}
	}
	
	STRNCPY(prog, gPrefs.gamePath);
	STRNCAT(prog, " +connect ");
	STRNCAT(prog, gPrefs.serverAddrStr);
	
	// Run the program.
	winExecResult = WinExec(prog, SW_SHOW);
	if (winExecResult <= 31) switch (winExecResult) {
		case ERROR_BAD_FORMAT:
			ErrBox("Could not run:  %s\nReason:  %s", prog, "The .EXE file is invalid (non-Win32 .EXE or error in .EXE image)");
			break;
		case ERROR_FILE_NOT_FOUND:
			ErrBox("Could not run:  %s\nReason:  %s", prog, "The specified file was not found.");
			break;
		case ERROR_PATH_NOT_FOUND:
			ErrBox("Could not run:  %s\nReason:  %s", prog, "The specified path was not found.");
			break;
		default:
			ErrBox("Could not run:  %s\nReason:  Unknown error #%d.", prog, winExecResult);
			break;
	}
}	// JoinGame





void SetMenuStates(void)
{
	HMENU hSysMenu;

	hSysMenu = GetSystemMenu(gMainWnd, FALSE);
	if (hSysMenu == NULL)
		return;

	// Don't let them select the console until they configure
	// a password to use with it.
	//
	if (gPrefs.rconPassword[0] == '\0') {
		EnableMenuItem(hSysMenu, ID_TRAY_CONSOLE, MF_GRAYED|MF_DISABLED|MF_BYCOMMAND);
	} else {
		EnableMenuItem(hSysMenu, ID_TRAY_CONSOLE, MF_ENABLED|MF_BYCOMMAND);
	}

	// Don't let them play the game until they configure a server.
	//
	if (gPrefs.serverAddrStr[0] == '\0') {
		EnableMenuItem(hSysMenu, ID_TRAY_JOIN_GAME, MF_GRAYED|MF_DISABLED|MF_BYCOMMAND);
	} else {
		EnableMenuItem(hSysMenu, ID_TRAY_JOIN_GAME, MF_ENABLED|MF_BYCOMMAND);
	}
}	// SetMenuStates




// This snippet was derived from some public domain MFC code by (I think) Chris Maunder.
//
static void OnTrayIcon(WPARAM wParam, LPARAM lParam)
{
	HMENU hMenu, hSubMenu;
	POINT pos;

	// Return quickly if its not for this tray icon
	if (wParam != nid.uID)
		return;

	// Clicking with right button brings up a context menu
	if (LOWORD(lParam) == WM_RBUTTONUP)
	{	
		hMenu = LoadMenu(ghInstance, MAKEINTRESOURCE(nid.uID));
		if (hMenu == NULL) {
			return;
		}

		hSubMenu = GetSubMenu(hMenu, 0);
		if (hMenu == NULL) {
			return;
		}

		// Make first menu item the default (bold font)
		SetMenuDefaultItem(hSubMenu, 0, TRUE);
		
		if (gPrefs.rconPassword[0] == '\0') {
			EnableMenuItem(hSubMenu, ID_TRAY_CONSOLE, MF_GRAYED|MF_DISABLED|MF_BYCOMMAND);
		} else {
			EnableMenuItem(hSubMenu, ID_TRAY_CONSOLE, MF_ENABLED|MF_BYCOMMAND);
		}
		
		// Don't let them play the game until they configure a server.
		//
		if (gPrefs.serverAddrStr[0] == '\0') {
			EnableMenuItem(hSubMenu, ID_TRAY_JOIN_GAME, MF_GRAYED|MF_DISABLED|MF_BYCOMMAND);
		} else {
			EnableMenuItem(hSubMenu, ID_TRAY_JOIN_GAME, MF_ENABLED|MF_BYCOMMAND);
		}
		
		//Display and track the popup menu
		GetCursorPos(&pos);

		SetForegroundWindow(nid.hWnd);  
		TrackPopupMenu(hSubMenu, 0, pos.x, pos.y, 0, nid.hWnd, NULL);

		// BUGFIX: See "PRB: Menus for Notification Icons Don't Work Correctly"
		PostMessage(nid.hWnd, WM_USER, 0, 0);

		DestroyMenu(hMenu);
	} else if (LOWORD(lParam) == WM_LBUTTONDBLCLK) {
		hMenu = LoadMenu(ghInstance, MAKEINTRESOURCE(nid.uID));
		if (hMenu == NULL) {
			return;
		}

		hSubMenu = GetSubMenu(hMenu, 0);
		if (hMenu == NULL) {
			return;
		}

		// double click received, the default action is to execute first menu item
		SetForegroundWindow(nid.hWnd);
		SendMessage(nid.hWnd, WM_COMMAND, GetMenuItemID(hSubMenu, 0), 0);
		DestroyMenu(hMenu);
	}
}	// OnTrayIcon




// The user clicked the right mouse button, so now see if that point
// corresponds to an area that can popup a context menu.
//
static void OnRightMouseButton(POINT p)
{
	int i;
	QuakeServerStatusPtr qssp = &gServerStatus;

	if ((PtInRect(&gServerNameRect, p) || PtInRect(&gServerNameHeaderRect, p)) && (gPrefs.rconPassword[0] != '\0')) {
		ClientToScreen(gMainWnd, &p);
		TrackPopupMenu(ghServerMenu, 0, p.x, p.y, 0, gMainWnd, NULL);
		return;
	}

	if ((PtInRect(&gMapNameRect, p) || PtInRect(&gMapNameHeaderRect, p)) && (gPrefs.rconPassword[0] != '\0')) {
		ClientToScreen(gMainWnd, &p);
		TrackPopupMenu(ghMapMenu, 0, p.x, p.y, 0, gMainWnd, NULL);
		return;
	}
	
	// Note that the advanced admin options don't appear until they
	// configure a password.
	//
	if (gPrefs.rconPassword[0] != '\0') {
		for (i=0; i<MAX_VIEWABLE_PLAYERS; i++) {
			if (qssp->players[i].name[0] == '\0')
				break;
			if (PtInRect(&gNameRect[i], p)) {
				STRNCPY(gSelectedPlayerName, qssp->players[i].name);
				gSelectedPlayerID = qssp->players[i].id;
				ClientToScreen(gMainWnd, &p);
				TrackPopupMenu(ghPlayerMenu, 0, p.x, p.y, 0, gMainWnd, NULL);
				return;
			}
		}
	}

	// Default
	ClientToScreen(gMainWnd, &p);
	TrackPopupMenu(ghMainWndDefaultMenu, 0, p.x, p.y, 0, gMainWnd, NULL);
}	// OnRightMouseButton




// We don't waste screen space with a menu bar, although the
// window has grown to a size where it probably should have
// one.
//
static void InitSystemMenu(void)
{
	HMENU hMenu;

	hMenu = GetSystemMenu(gMainWnd, FALSE);
	if (hMenu != NULL) {
		AppendMenu(hMenu, MF_SEPARATOR, 0, NULL);
		AppendMenu(hMenu, MF_STRING, ID_TRAY_REFRESH, "Re&fresh\tF5");
		AppendMenu(hMenu, MF_STRING, ID_TRAY_CONSOLE, "&Rconsole...");
		AppendMenu(hMenu, MF_STRING, ID_TRAY_CONFIGURE, "&Configure...");
		AppendMenu(hMenu, MF_STRING, ID_TRAY_ABOUT, "&About...");
	}
}	// InitSystemMenu





LRESULT CALLBACK WndProc (HWND gMainWnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	HDC         hdc;
	PAINTSTRUCT ps;
	POINT		p;
	char		buf[256];

	switch (iMsg) {
	case WM_CREATE:
		// Initialize the context menus.
		//
		ghMapMenu = LoadMenu(ghInstance, MAKEINTRESOURCE(IDR_MAP_CONTEXT_MENU));
		ghMapMenu = GetSubMenu(ghMapMenu, 0);
		ghPlayerMenu = LoadMenu(ghInstance, MAKEINTRESOURCE(IDR_PLAYER_CONTEXT_MENU));
		ghPlayerMenu = GetSubMenu(ghPlayerMenu, 0);
		ghServerMenu = LoadMenu(ghInstance, MAKEINTRESOURCE(IDR_SERVER_CONTEXT_MENU));
		ghServerMenu = GetSubMenu(ghServerMenu, 0);
		ghMainWndDefaultMenu = LoadMenu(ghInstance, MAKEINTRESOURCE(IDR_MAINWND_CONTEXT_MENU));
		ghMainWndDefaultMenu = GetSubMenu(ghMainWndDefaultMenu, 0);
		return 0 ;
	
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ID_TRAY_SHOW:
			ShowWindow(gMainWnd, SW_RESTORE);
			return 0;
		case ID_TRAY_REFRESH:
			RefreshNow();
			return 0;
		case ID_TRAY_CONSOLE:
			ShowConsole();
			return 0;
		case ID_TRAY_JOIN_GAME:
			JoinGame();
			break;
		case ID_TRAY_ABOUT:
			About();
			return 0;
		case ID_TRAY_CONFIGURE:
			ShowPreferencesDlg();
			return 0;
		case ID_TRAY_EXIT:
			SendMessage(gMainWnd, WM_CLOSE, 0, 0L);
			return 0;
		case ID_MAP_CHANGE:
		case ID_SERVER_CHANGEMAP:
			STRNCPY(buf, gServerStatus.mapname);
			if (InputDlg("Map name:", buf, sizeof(buf))) {
				(void) Rcon(NULL, 0, RCON_DEFAULT_TIMEOUT, "say Changing map to %s...", buf);
				gChangeMapIncrement = gIncrements + 5;		// Wait 5 seconds before doing it
				STRNCPY(gNewMapName, buf);
			}
			return 0;
		case ID_PLAYER_KICK:
			if (gSelectedPlayerName[0] != '\0') {
				RconKickPlayerID(gSelectedPlayerID);
				gRefreshOnNextIncrement = 1;
				gSelectedPlayerName[0] = '\0';
				gSelectedPlayerID = -1;
			}
			return 0;
		case ID_PLAYER_SAYTO:
			buf[0] = '\0';
			if ((gSelectedPlayerName[0] != '\0') && (InputDlg("Say:", buf, sizeof(buf)))) {
				(void) RconSay(gSelectedPlayerName, buf);
			}
			gSelectedPlayerName[0] = '\0';
			gSelectedPlayerID = -1;
			return 0;
		case ID_SERVER_SAY:
			buf[0] = '\0';
			if (InputDlg("Message to all:", buf, sizeof(buf))) {
				(void) RconSay(NULL, buf);
			}
			return 0;
		case ID_SERVER_FRAGLIMIT:
			buf[0] = '\0';
			if (InputDlg("New frag limit:", buf, sizeof(buf))) {
				(void) RconSetFragLimit(buf);
			}
			return 0;
		case ID_SERVER_TIMELIMIT:
			buf[0] = '\0';
			if (InputDlg("New time limit:", buf, sizeof(buf))) {
				(void) RconSetTimeLimit(buf);
			}
			return 0;
		}
		return 0;
	
	case WM_SYSCOMMAND:
		switch (LOWORD(wParam)) {
		case ID_TRAY_CONSOLE:
			ShowConsole();
			return 0;
		case ID_TRAY_JOIN_GAME:
			JoinGame();
			break;
		case ID_TRAY_REFRESH:
			RefreshNow();
			return 0;
		case ID_TRAY_ABOUT:
			About();
			return 0;
		case ID_TRAY_CONFIGURE:
			ShowPreferencesDlg();
			return 0;
		}
		break;

	case WM_PAINT:
		hdc = BeginPaint(gMainWnd, &ps) ;
		OnDraw(gMainWnd, hdc);
		EndPaint(gMainWnd, &ps) ;
		return 0 ;
		
	case WM_TIMER:
		PeriodicTasks();
		return 0;

	case WM_TRAY_TRAYSPY:
		// The system tray sent us our custom message
		// that we registered with it.
		//
		OnTrayIcon(wParam, lParam);
		return 0;

	case WM_SIZE:
		if (wParam == SIZE_MINIMIZED) {
			// Hide the window instead of minimizing it.
			//
			// Before hiding it, save the window position
			// so we can save it in the registry.
			//
			SaveWindowRect(gMainWnd);
			ShowWindow(gMainWnd, SW_HIDE);
			return 0;
		}
		break;

	case WM_MOVING:
		// Don't care about where they're moving it,
		// but we want to disable the auto-hide
		// so it doesn't disappear while they're moving
		// the window around.
		//
		gHideIncrement = 0;
		break;

	case WM_LBUTTONDOWN:
		gHideIncrement = 0;
		break;

	case WM_RBUTTONDOWN:
		gHideIncrement = 0;
		p.x = LOWORD(lParam);
		p.y = HIWORD(lParam);
		OnRightMouseButton(p);
		break;

	case WM_DESTROY:
		CleanUp();
		PostQuitMessage(0);
		return 0 ;
	}
	
	return DefWindowProc (gMainWnd, iMsg, wParam, lParam) ;
}	// WndProc



#pragma warning(disable : 4100)		// warning C4100: unreferenced formal parameter
int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance_unused, PSTR szCmdLine_unused, int iCmdShow)
{
	MSG         msg ;
	WNDCLASSEX  wndclass ;
	int wwidth = 400;
	int wheight = 550;
	int initialx = CW_USEDEFAULT, initialy = CW_USEDEFAULT;
	RECT rect;
	HACCEL hAccel;
	int i;

	ghInstance = hInstance;
	
	// Use the first available TraySpy "slot".
	// For most users that will only be running one copy of
	// TraySpy, the first available one will be simply
	// instance ID 0.
	//
	for (i=0; i<100; i++) {
		_stprintf(gWndClass, _T("TraySpy%02d"), i);
		if (FindWindow(gWndClass, NULL) == NULL)
			break;
	}
	if (i == 100) {
		// Did you really try to run 100 or more of these?
		return 0;
	}
	
	// Do early, so the registry key we use is set for the
	// RestoreWindowPosition() call a bit later.
	//
	InitPreferences();

	ZeroMemory(&rect, (DWORD) sizeof(rect));
	GetClientRect (GetDesktopWindow(), &rect);
	if (((rect.bottom - rect.top) > 100) && ((rect.right - rect.left) > 100)) {
		// Sanity check
		initialx = (rect.right - rect.left - wwidth) / 2;
		initialy = (rect.bottom - rect.top - wheight) / 3;
	}
	RestoreWindowPosition(&initialx, &initialy, &wwidth, &wheight);

	wndclass.cbSize        = sizeof (wndclass) ;
	wndclass.style         = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc   = WndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_MAINFRAME));
	wndclass.hCursor       = LoadCursor(NULL, IDC_ARROW) ;
	wndclass.hbrBackground = (HBRUSH) GetStockObject(WHITE_BRUSH);
//	wndclass.lpszMenuName  = MAKEINTRESOURCE(IDR_TRAY);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = gWndClass;
	wndclass.hIconSm       = LoadIcon(ghInstance, MAKEINTRESOURCE(IDI_MAINFRAME));
	
	RegisterClassEx (&wndclass);

	// Create the main window, which is the
	// "game status" window, that shows the
	// server name, map name, and active
	// players.
	//
	gMainWnd = CreateWindow (
		gWndClass,					// window class name
		_T("TraySpy"),				// window caption
		WS_OVERLAPPEDWINDOW,		// window style
		initialx,					// initial x position
		initialy,					// initial y position
		wwidth,						// initial x size
		wheight,					// initial y size
		NULL,						// parent window handle
		NULL,						// window menu handle
		hInstance,					// program instance handle
		NULL);						// creation parameters

	if (gMainWnd == NULL) {
		return 0;
	}

	// Start the ticker.
	time(&gStart);
	gTimerID = SetTimer(gMainWnd, 1, 1000, NULL);

	if (InitWinsock() < 0) {
		ErrBox("Winsock could not be initialized.");
		return 0;
	}
	InitSystemMenu();
	LoadIcons();
	InitTray();
	InitQuakeServerStatus();
	LoadPreferences();
	ComputeRects();
	InitConsole();
	ZeroMemory(gNewMapName, sizeof(gNewMapName));
	ZeroMemory(gSelectedPlayerName, sizeof(gSelectedPlayerName));

	if (gPrefs.serverAddrStr[0] != '\0') {
		// Do the first update now, before the window
		// is even shown.  Since an update takes only
		// a fraction of a second in most cases, this
		// is just fine.
		//
		PollQuakeServer();
		SetTrayIconByStatus();
		SetTrayTipByStatus();
	} else {
		// Do not auto-hide the window, since
		// it mentions instructions when no
		// server has been selected.
		//
		gHideIncrement = -1;
	}
	SetMenuStates();
	
	// Get font handles for some fonts that we use.
	// Note that all of these are installed on all
	// Windows 95/98/NT systems (although it's 
	// conceivable that a doofus user removed them).
	//
	gCourier10 = CreateFont(
		-((GetDeviceCaps(GetDC(gMainWnd), LOGPIXELSY) * 10) / 72),
		0, 0, 0, FW_NORMAL, FALSE, FALSE, 0, 
		ANSI_CHARSET, OUT_DEFAULT_PRECIS,
		CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, 
		DEFAULT_PITCH | FF_MODERN,
		"Courier New"
		);
	
	gTimesNewRoman12 = CreateFont(
		-((GetDeviceCaps(GetDC(gMainWnd), LOGPIXELSY) * 12) / 72),
		0, 0, 0, FW_NORMAL, FALSE, FALSE, 0, 
		ANSI_CHARSET, OUT_DEFAULT_PRECIS,
		CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, 
		DEFAULT_PITCH | FF_MODERN,
		"Times New Roman"
		);

	gTimesNewRomanBold12 = CreateFont(
		-((GetDeviceCaps(GetDC(gMainWnd), LOGPIXELSY) * 12) / 72),
		0, 0, 0, FW_BOLD, FALSE, FALSE, 0, 
		ANSI_CHARSET, OUT_DEFAULT_PRECIS,
		CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, 
		DEFAULT_PITCH | FF_MODERN,
		"Times New Roman"
		);
	
	// This is mostly so we can have an Explorer-style "F5" key
	// do a refresh.
	//
	hAccel = LoadAccelerators(ghInstance, MAKEINTRESOURCE(IDR_ACCELERATOR1));

	// Here we go!
	//
	ShowWindow (gMainWnd, iCmdShow);
	UpdateWindow (gMainWnd);
	while (GetMessage (&msg, NULL, 0, 0)) {
		if (!TranslateAccelerator(gMainWnd, hAccel, &msg)) {
			TranslateMessage (&msg);
			DispatchMessage (&msg);
		}
	}
	return msg.wParam ;
}	// WinMain
#pragma warning(default : 4100)		// warning C4100: unreferenced formal parameter




#pragma warning(disable : 4100)		// warning C4100: unreferenced formal parameter
BOOL CALLBACK AboutDlgProc(HWND hDlg, UINT iMsg, WPARAM w, LPARAM l_unused)
{

	switch (iMsg) {
	case WM_INITDIALOG:
		return TRUE;

	case WM_COMMAND:
		if ((LOWORD(w) == IDOK) || (LOWORD(w) == IDCANCEL)) {
			EndDialog(hDlg, 0);
			return TRUE;
		}
		break;
	}
	return (FALSE);
}	// AboutDlgProc
#pragma warning(default : 4100)		// warning C4100: unreferenced formal parameter

