/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#ifndef __MODELUNDO_H
#define __MODELUNDO_H

#include "undo.h"
#include "model.h"
#include "glmath.h"
#include "sorted_list.h"

#include <list>
#include <string>

using std::list;
using std::string;

class Model;

class ModelUndo : public Undo
{
   public:
      ModelUndo() { s_allocated++; };
      virtual ~ModelUndo() { s_allocated--; };
      
      virtual void undo( Model * ) = 0;
      virtual void redo( Model * ) = 0;

      static int s_allocated;
};

class MU_NoOp : public Undo
{
   public:
      MU_NoOp() {};
      virtual ~MU_NoOp() {};

      void undo( Model * ) {};
      void redo( Model * ) {};
      bool combine( Undo * );  // Actually has an implementation

      unsigned size() { return sizeof(MU_NoOp); };
};

class MU_TranslateSelectedVertices : public ModelUndo
{
   public:
      MU_TranslateSelectedVertices();
      virtual ~MU_TranslateSelectedVertices();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size() { return sizeof(MU_TranslateSelectedVertices); };

      void setMatrix( const Matrix & rhs );

      Matrix getMatrix() const { return m_matrix; };

   private:
      Matrix m_matrix;
};

class MU_RotateSelectedVerticesOnPoint : public ModelUndo
{
   public:
      MU_RotateSelectedVerticesOnPoint();
      virtual ~MU_RotateSelectedVerticesOnPoint();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size() { return sizeof(MU_RotateSelectedVerticesOnPoint); };

      void setMatrixPoint( const Matrix & rhs, double * point );
      Matrix getMatrix() const { return m_matrix; };

   private:

      Matrix m_matrix;
      double m_point[3];
};

class MU_ApplyMatrix : public ModelUndo
{
   public:
      MU_ApplyMatrix();
      virtual ~MU_ApplyMatrix();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size() { return sizeof(MU_ApplyMatrix); };

      void setMatrix( const Matrix & rhs );
      Matrix getMatrix() const { return m_matrix; };

   private:
      Matrix m_matrix;
};

class MU_SelectionMode : public ModelUndo
{
   public:
      MU_SelectionMode();
      virtual ~MU_SelectionMode();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setSelectionMode( Model::SelectionMode mode, Model::SelectionMode oldMode ) { m_mode = mode; m_oldMode = oldMode; };

   private:
      Model::SelectionMode m_mode;
      Model::SelectionMode m_oldMode;
};

class MU_Select : public ModelUndo
{
   public:
      MU_Select( Model::SelectionMode mode );
      virtual ~MU_Select();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      Model::SelectionMode getSelectionMode() const { return m_mode; };

      void setSelectionDifference( int number, bool selected, bool oldSelected );
      void toggle( int number );

      unsigned diffCount() { return m_diff.size(); };

   private:
      typedef struct SelectionDifference_t
      {
         int  number;
         bool selected;
         bool oldSelected;
         bool operator< ( const struct SelectionDifference_t & rhs ) const
         {
            return ( this->number < rhs.number );
         }
         bool operator== ( const struct SelectionDifference_t & rhs ) const
         {
            return ( this->number == rhs.number );
         }
      } SelectionDifference;

      typedef sorted_list< SelectionDifference > SelectionDifferenceList;

      Model::SelectionMode m_mode;
      SelectionDifferenceList m_diff;
      
};

class MU_Hide : public ModelUndo
{
   public:
      MU_Hide( Model::SelectionMode mode );
      virtual ~MU_Hide();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      Model::SelectionMode getSelectionMode() const { return m_mode; };

      void setHideDifference( int number, bool visible );

   private:
      typedef struct HideDifference_t
      {
         int  number;
         bool visible;
      } HideDifference;

      typedef list< HideDifference > HideDifferenceList;

      Model::SelectionMode m_mode;
      HideDifferenceList m_diff;
};

class MU_InvertNormal : public ModelUndo
{
   public:
      MU_InvertNormal();
      virtual ~MU_InvertNormal();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void addTriangle( int triangle );

   private:
      list<int> m_triangles;
};

class MU_MoveVertex : public ModelUndo
{
   public:
      MU_MoveVertex();
      virtual ~MU_MoveVertex();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void addVertex( int v, double x, double y, double z,
            double oldx, double oldy, double oldz );
      void addBoneJoint( int j, double x, double y, double z,
            double oldx, double oldy, double oldz );

   private:

      typedef struct MoveVertex_t
      {
         int number;
         double x;
         double y;
         double z;
         double oldx;
         double oldy;
         double oldz;
         bool   isJoint;

         bool operator< ( const struct MoveVertex_t & rhs ) const
         {
            return ( this->number < rhs.number || 
                  ( this->number == rhs.number && this->isJoint == false && rhs.isJoint == true ) 
                  );
         };
         bool operator== ( const struct MoveVertex_t & rhs ) const
         {
            return ( this->number == rhs.number && this->isJoint == rhs.isJoint );
         };
      } MoveVertex;
      typedef sorted_list<MoveVertex> MoveVertexList;

      MoveVertexList m_vertices;
};

class MU_SetTexture : public ModelUndo
{
   public:
      MU_SetTexture();
      virtual ~MU_SetTexture();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setTexture( unsigned groupNumber, int newTexture, int oldTexture );

   private:
      typedef struct SetTexture_t
      {
         unsigned groupNumber;
         int newTexture;
         int oldTexture;
      } SetTexture;
      typedef list<SetTexture> SetTextureList;

      SetTextureList m_list;
};

class MU_AddVertex : public ModelUndo
{
   public:
      MU_AddVertex();
      virtual ~MU_AddVertex();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void redoRelease();

      unsigned size();

      void addVertex( unsigned index, Model::Vertex * vertex );

   private:
      typedef struct AddVertex_t
      {
         unsigned index;
         Model::Vertex * vertex;
      } AddVertex;
      typedef list<AddVertex> AddVertexList;

      AddVertexList m_list;
};

class MU_AddTriangle : public ModelUndo
{
   public:
      MU_AddTriangle();
      virtual ~MU_AddTriangle();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void redoRelease();

      unsigned size();

      void addTriangle( unsigned index, Model::Triangle * );

   private:
      typedef struct AddTriangle_t
      {
         unsigned index;
         Model::Triangle * triangle;
      } AddTriangle;
      typedef list<AddTriangle> AddTriangleList;

      AddTriangleList m_list;
};

class MU_AddGroup : public ModelUndo
{
   public:
      MU_AddGroup();
      virtual ~MU_AddGroup();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void redoRelease();

      unsigned size();

      void addGroup( unsigned index, Model::Group * );

   private:
      typedef struct AddGroup_t
      {
         unsigned index;
         Model::Group * group;
      } AddGroup;
      typedef list<AddGroup> AddGroupList;

      AddGroupList m_list;
};

class MU_AddTexture : public ModelUndo
{
   public:
      MU_AddTexture();
      virtual ~MU_AddTexture();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void redoRelease();

      unsigned size();

      void addTexture( unsigned index, Model::Material * );

   private:
      typedef struct AddTexture_t
      {
         unsigned index;
         Model::Material * texture;
      } AddTexture;
      typedef list<AddTexture> AddTextureList;

      AddTextureList m_list;
};

class MU_SetTextureCoords : public ModelUndo
{
   public:
      MU_SetTextureCoords();
      virtual ~MU_SetTextureCoords();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void addTextureCoords( unsigned triangle, unsigned vertexIndex, float s, float t, float oldS, float oldT );

   private:
      typedef struct SetTextureCoords_t
      {
         unsigned triangle;
         unsigned vertexIndex;
         float t;
         float s;
         float oldT;
         float oldS;
         bool operator< ( const struct SetTextureCoords_t & rhs ) const
         {
            return ( this->triangle < rhs.triangle
                  || (this->triangle == rhs.triangle && this->vertexIndex < rhs.vertexIndex ) );
         };
         bool operator == ( const struct SetTextureCoords_t & rhs ) const
         {
            return ( this->triangle == rhs.triangle && this->vertexIndex == rhs.vertexIndex );
         };
      } SetTextureCoords;

      typedef sorted_list<SetTextureCoords> STCList;

      STCList m_list;
};

class MU_AddToGroup : public ModelUndo
{
   public:
      MU_AddToGroup();
      virtual ~MU_AddToGroup();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void addToGroup( unsigned groupNum, unsigned triangleNum );

   private:
      typedef struct _AddToGroup
      {
         unsigned groupNum;
         unsigned triangleNum;
      } AddToGroup;
      typedef list<AddToGroup> AddToGroupList;

      AddToGroupList m_list;
};

class MU_RemoveFromGroup : public ModelUndo
{
   public:
      MU_RemoveFromGroup();
      virtual ~MU_RemoveFromGroup();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void removeFromGroup( unsigned groupNum, unsigned triangleNum );

   private:
      typedef struct _RemoveFromGroup
      {
         unsigned groupNum;
         unsigned triangleNum;
      } RemoveFromGroup;
      typedef list<RemoveFromGroup> RemoveFromGroupList;

      RemoveFromGroupList m_list;
};

class MU_DeleteTriangle : public ModelUndo
{
   public:
      MU_DeleteTriangle();
      virtual ~MU_DeleteTriangle();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void undoRelease();

      unsigned size();

      void deleteTriangle( unsigned triangleNum, Model::Triangle * triangle );

   private:
      typedef struct DeleteTriangle_t
      {
         unsigned triangleNum;
         Model::Triangle * triangle;
      } DeleteTriangle;
      typedef list<DeleteTriangle> DeleteTriangleList;

      DeleteTriangleList m_list;
};

class MU_DeleteVertex : public ModelUndo
{
   public:
      MU_DeleteVertex();
      virtual ~MU_DeleteVertex();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void undoRelease();

      unsigned size();

      void deleteVertex( unsigned vertexNum, Model::Vertex * vertex );

   private:
      typedef struct DeleteVertex_t
      {
         unsigned vertexNum;
         Model::Vertex * vertex;
      } DeleteVertex;
      typedef list<DeleteVertex> DeleteVertexList;

      DeleteVertexList m_list;
};

class MU_DeleteBoneJoint : public ModelUndo
{
   public:
      MU_DeleteBoneJoint();
      virtual ~MU_DeleteBoneJoint();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void undoRelease();

      unsigned size();

      void deleteBoneJoint( unsigned jointNum, Model::Joint * joint );

   private:
      unsigned       m_jointNum;
      Model::Joint * m_joint;
};

class MU_AddBoneJoint : public ModelUndo
{
   public:
      MU_AddBoneJoint();
      virtual ~MU_AddBoneJoint();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void redoRelease();

      unsigned size();

      void addBoneJoint( unsigned jointNum, Model::Joint * joint );

   private:
      unsigned       m_jointNum;
      Model::Joint * m_joint;
};

class MU_DeleteGroup : public ModelUndo
{
   public:
      MU_DeleteGroup();
      virtual ~MU_DeleteGroup();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void undoRelease();

      unsigned size();

      void deleteGroup( unsigned groupNum, Model::Group * group );

   private:
      typedef struct DeleteGroup_t
      {
         unsigned groupNum;
         Model::Group * group;
      } DeleteGroup;
      typedef list<DeleteGroup> DeleteGroupList;

      DeleteGroupList m_list;
};

class MU_DeleteTexture : public ModelUndo
{
   public:
      MU_DeleteTexture();
      virtual ~MU_DeleteTexture();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void undoRelease();

      unsigned size();

      void deleteTexture( unsigned textureNum, Model::Material * texture );

   private:
      typedef struct DeleteTexture_t
      {
         unsigned textureNum;
         Model::Material * texture;
      } DeleteTexture;
      typedef list<DeleteTexture> DeleteTextureList;

      DeleteTextureList m_list;
};

class MU_SetLightProperties : public ModelUndo
{
   public:
      typedef enum
      {
         LightAmbient  = 0,
         LightDiffuse,
         LightSpecular,
         LightEmissive,
         LightTypeMax  
      } LightType;

      MU_SetLightProperties();
      virtual ~MU_SetLightProperties();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setLightProperties( unsigned textureNum, LightType type, const float * newLight, const float * oldLight );

   private:
      typedef struct LightProperties_t
      {
         unsigned textureNum;
         float newLight[ LightTypeMax ][4];
         float oldLight[ LightTypeMax ][4];
         bool  isSet[ LightTypeMax ];
      } LightProperties;

      typedef list<LightProperties> LightPropertiesList;

      list<LightProperties> m_list;
};

class MU_SetShininess : public ModelUndo
{
   public:

      MU_SetShininess();
      virtual ~MU_SetShininess();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setShininess( unsigned textureNum, const float & newValue, const float & oldValue );

   private:
      typedef struct Shininess_t
      {
         unsigned textureNum;
         float oldValue;
         float newValue;
      } Shininess;

      typedef list<Shininess> ShininessList;

      list<Shininess> m_list;
};

class MU_SetTextureName : public ModelUndo
{
   public:

      MU_SetTextureName();
      virtual ~MU_SetTextureName();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setTextureName( unsigned textureNum, const char * newName, const char * oldName );

   private:

      unsigned m_textureNum;
      string   m_newName;
      string   m_oldName;
};

class MU_SetTriangleVertices : public ModelUndo
{
   public:
      MU_SetTriangleVertices();
      virtual ~MU_SetTriangleVertices();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setTriangleVertices( unsigned triangleNum, 
            unsigned newV1, unsigned newV2, unsigned newV3,
            unsigned oldV1, unsigned oldV2, unsigned oldV3 );

   private:
      typedef struct TriangleVertices_t
      {
         unsigned triangleNum;
         unsigned newVertices[3];
         unsigned oldVertices[3];
      } TriangleVertices;

      typedef list<TriangleVertices> TriangleVerticesList;

      TriangleVerticesList m_list;
};

class MU_SubdivideSelected : public ModelUndo
{
   public:
      MU_SubdivideSelected();
      virtual ~MU_SubdivideSelected();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size() { return sizeof( MU_SubdivideSelected ); };

   private:
};

class MU_SubdivideTriangle : public ModelUndo
{
   public:
      MU_SubdivideTriangle();
      virtual ~MU_SubdivideTriangle();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void subdivide( unsigned a, unsigned b, unsigned c, unsigned d );
      void addVertex( unsigned v );

   private:

      typedef struct SubdivideTriangle_t
      {
         unsigned a;
         unsigned b;
         unsigned c;
         unsigned d;
      } SubdivideTriangle;

      typedef list<SubdivideTriangle> SubdivideTriangleList;

      SubdivideTriangleList m_list;
      list<unsigned> m_vlist;
};

class MU_ChangeAnimState : public ModelUndo
{
   public:
      MU_ChangeAnimState();
      virtual ~MU_ChangeAnimState();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setState( Model::AnimationMode newMode, Model::AnimationMode oldMode, unsigned anim, unsigned frame );

   protected:

      Model::AnimationMode m_newMode;
      Model::AnimationMode m_oldMode;
      unsigned m_anim;
      unsigned m_frame;
};

class MU_SetAnimName : public ModelUndo
{
   public:

      MU_SetAnimName();
      virtual ~MU_SetAnimName();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setName( Model::AnimationMode mode, unsigned animNum, const char * newName, const char * oldName );

   private:

      Model::AnimationMode m_mode;
      unsigned m_animNum;
      string   m_newName;
      string   m_oldName;
};

class MU_SetAnimFrameCount : public ModelUndo
{
   public:

      MU_SetAnimFrameCount();
      virtual ~MU_SetAnimFrameCount();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setAnimFrameCount( Model::AnimationMode m, unsigned animNum, unsigned newCount, unsigned oldCount );

   private:

      Model::AnimationMode m_mode;
      unsigned m_animNum;
      unsigned m_newCount;
      unsigned m_oldCount;
};

class MU_SetAnimFPS : public ModelUndo
{
   public:

      MU_SetAnimFPS();
      virtual ~MU_SetAnimFPS();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setFPS( Model::AnimationMode mode, unsigned animNum, double newFps, double oldFps );

   private:

      Model::AnimationMode m_mode;
      unsigned m_animNum;
      double   m_newFPS;
      double   m_oldFPS;
};

class MU_SetAnimKeyframe : public ModelUndo
{
   public:
      MU_SetAnimKeyframe();
      virtual ~MU_SetAnimKeyframe();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setAnimationData( const unsigned & anim, const unsigned & frame, const bool & isRotation );

      void addBoneJoint( int j, bool isNew, double x, double y, double z,
            double oldx, double oldy, double oldz );

   private:

      typedef struct SetKeyframe_t
      {
         int number;
         bool isNew;
         double x;
         double y;
         double z;
         double oldx;
         double oldy;
         double oldz;

         bool operator< ( const struct SetKeyframe_t & rhs ) const
         {
            return ( this->number < rhs.number );
         };
         bool operator== ( const struct SetKeyframe_t & rhs ) const
         {
            return ( this->number == rhs.number );
         };
      } SetKeyframe;
      typedef sorted_list<SetKeyframe> SetKeyframeList;

      SetKeyframeList m_keyframes;
      unsigned        m_anim;
      unsigned        m_frame;
      bool            m_isRotation;
};

class MU_DeleteKeyframe : public ModelUndo
{
   public:
      MU_DeleteKeyframe();
      virtual ~MU_DeleteKeyframe();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void undoRelease();

      unsigned size();

      void setAnimationData( const unsigned & anim );
      void deleteKeyframe( Model::Keyframe * keyframe );

   private:
      typedef list<Model::Keyframe *> DeleteKeyframeList;

      DeleteKeyframeList m_list;
      unsigned           m_anim;
};

class MU_SetJointName : public ModelUndo
{
   public:

      MU_SetJointName();
      virtual ~MU_SetJointName();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setName( unsigned joint, const char * newName, const char * oldName );

   private:

      unsigned m_joint;
      string   m_newName;
      string   m_oldName;
};

class MU_MoveFrameVertex : public ModelUndo
{
   public:
      MU_MoveFrameVertex();
      virtual ~MU_MoveFrameVertex();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setAnimationData( const unsigned & anim, const unsigned & frame  );

      void addVertex( int v, double x, double y, double z,
            double oldx, double oldy, double oldz );

   private:

      typedef struct MoveFrameVertex_t
      {
         int number;
         double x;
         double y;
         double z;
         double oldx;
         double oldy;
         double oldz;

         bool operator< ( const struct MoveFrameVertex_t & rhs ) const
         {
            return ( this->number < rhs.number );
         };
         bool operator== ( const struct MoveFrameVertex_t & rhs ) const
         {
            return ( this->number == rhs.number );
         };
      } MoveFrameVertex;
      typedef sorted_list<MoveFrameVertex> MoveFrameVertexList;

      MoveFrameVertexList m_vertices;
      unsigned            m_anim;
      unsigned            m_frame;
};

class MU_AddFrameAnimFrame : public ModelUndo
{
   public:
      MU_AddFrameAnimFrame();
      virtual ~MU_AddFrameAnimFrame();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void redoRelease();

      unsigned size();

      void setAnimationData( const unsigned & anim );
      void addFrame( const unsigned & frame, Model::FrameAnimVertexList * list );

   private:
      typedef struct AddFrame_t
      {
         unsigned frame;
         Model::FrameAnimVertexList * list;
      } AddFrame;
      typedef list<AddFrame> FrameVertexList;

      FrameVertexList m_list;
      unsigned        m_anim;
};

class MU_DeleteFrameAnimFrame : public ModelUndo
{
   public:
      MU_DeleteFrameAnimFrame();
      virtual ~MU_DeleteFrameAnimFrame();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void undoRelease();

      unsigned size();

      void setAnimationData( const unsigned & anim );
      void deleteFrame( const unsigned & frame, Model::FrameAnimVertexList * list );

   private:
      typedef struct DeleteFrame_t
      {
         unsigned frame;
         Model::FrameAnimVertexList * list;
      } DeleteFrame;
      typedef list<DeleteFrame> FrameVertexList;

      FrameVertexList m_list;
      unsigned         m_anim;
};

class MU_SetVertexBoneJoint : public ModelUndo
{
   public:
      MU_SetVertexBoneJoint();
      virtual ~MU_SetVertexBoneJoint();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setVertexBoneJoint( const unsigned & vertex, 
            const int & newBone, const int & oldBone );

   private:
      typedef struct SetJoint_t
      {
         unsigned vertex;
         int newBone;
         int oldBone;
      } SetJoint;
      typedef list<SetJoint> SetJointList;

      SetJointList m_list;
};

class MU_AddAnimation : public ModelUndo
{
   public:
      MU_AddAnimation();
      virtual ~MU_AddAnimation();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void redoRelease();

      unsigned size();

      void addAnimation( const unsigned & anim, Model::SkelAnim  * skelanim  );
      void addAnimation( const unsigned & anim, Model::FrameAnim * frameanim );

   private:
      
      unsigned m_anim;
      Model::SkelAnim  * m_skelAnim;
      Model::FrameAnim * m_frameAnim;
};

class MU_DeleteAnimation : public ModelUndo
{
   public:
      MU_DeleteAnimation();
      virtual ~MU_DeleteAnimation();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      void undoRelease();

      unsigned size();

      void deleteAnimation( const unsigned & anim, Model::SkelAnim  * skelanim  );
      void deleteAnimation( const unsigned & anim, Model::FrameAnim * frameanim );

   private:
      
      unsigned m_anim;
      Model::SkelAnim  * m_skelAnim;
      Model::FrameAnim * m_frameAnim;
};

class MU_SetJointParent : public ModelUndo
{
   public:
      MU_SetJointParent();
      virtual ~MU_SetJointParent();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * ) { return false; };

      unsigned size() { return sizeof(MU_SetJointParent); };

      void setJointParent( unsigned joint, int newParent, int oldParent );

   protected:
      unsigned m_joint;
      int m_newParent;
      int m_oldParent;
};

class MU_SetJointRotation : public ModelUndo
{
   public:
      MU_SetJointRotation();
      virtual ~MU_SetJointRotation();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * ) { return false; };

      unsigned size() { return sizeof(MU_SetJointRotation); };

      void setJointRotation( const unsigned & joint, const double * newRotation, const double * oldRotation );

   protected:
      unsigned m_joint;
      double   m_newRotation[3];
      double   m_oldRotation[3];
};

class MU_SetJointTranslation : public ModelUndo
{
   public:
      MU_SetJointTranslation();
      virtual ~MU_SetJointTranslation();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * ) { return false; };

      unsigned size() { return sizeof(MU_SetJointTranslation); };

      void setJointTranslation( const unsigned & joint, const double * newTranslation, const double * oldTranslation );

   protected:
      unsigned m_joint;
      double   m_newTranslation[3];
      double   m_oldTranslation[3];
};

class MU_SetGroupSmooth : public ModelUndo
{
   public:
      MU_SetGroupSmooth();
      virtual ~MU_SetGroupSmooth();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setGroupSmooth( const unsigned & group, 
            const uint8_t & newSmooth, const uint8_t & oldSmooth );

   private:
      typedef struct SetSmooth_t
      {
         unsigned group;
         uint8_t newSmooth;
         uint8_t oldSmooth;
      } SetSmooth;
      typedef list<SetSmooth> SetSmoothList;

      SetSmoothList m_list;
};

class MU_SetGroupAngle : public ModelUndo
{
   public:
      MU_SetGroupAngle();
      virtual ~MU_SetGroupAngle();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setGroupAngle( const unsigned & group, 
            const uint8_t & newAngle, const uint8_t & oldAngle );

   private:
      typedef struct SetAngle_t
      {
         unsigned group;
         uint8_t newAngle;
         uint8_t oldAngle;
      } SetAngle;
      typedef list<SetAngle> SetAngleList;

      SetAngleList m_list;
};

class MU_SetGroupName : public ModelUndo
{
   public:

      MU_SetGroupName();
      virtual ~MU_SetGroupName();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setGroupName( unsigned groupNum, const char * newName, const char * oldName );

   private:

      unsigned m_groupNum;
      string   m_newName;
      string   m_oldName;
};

class MU_MoveAnimation : public ModelUndo
{
   public:

      MU_MoveAnimation();
      virtual ~MU_MoveAnimation();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void moveAnimation( const Model::AnimationMode & mode, const unsigned & oldIndex, const unsigned & newIndex );

   private:

      Model::AnimationMode m_mode;
      unsigned m_oldIndex;
      unsigned m_newIndex;
};

class MU_SetFrameAnimVertexCount : public ModelUndo
{
   public:

      MU_SetFrameAnimVertexCount();
      virtual ~MU_SetFrameAnimVertexCount();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * ) { return false; };

      unsigned size();

      void setCount( const unsigned & newCount, const unsigned & oldCount );

   private:

      unsigned m_newCount;
      unsigned m_oldCount;
};

class MU_SetMaterialClamp : public ModelUndo
{
   public:
      MU_SetMaterialClamp();
      virtual ~MU_SetMaterialClamp();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setMaterialClamp( const unsigned & material, const bool & isS,
            const bool & newClamp, const bool & oldClamp );

   private:
      unsigned m_material;
      bool m_isS;
      bool m_oldClamp;
      bool m_newClamp;
};

class MU_SetBackgroundImage : public ModelUndo
{
   public:
      MU_SetBackgroundImage();
      virtual ~MU_SetBackgroundImage();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setBackgroundImage( const unsigned & index, 
            const char * newFilename, const char * oldFilename );

   private:
      unsigned m_index;
      string   m_newFilename;
      string   m_oldFilename;
};

class MU_SetBackgroundScale : public ModelUndo
{
   public:
      MU_SetBackgroundScale();
      virtual ~MU_SetBackgroundScale();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setBackgroundScale( const unsigned & index, 
            float newScale, float oldScale );

   private:
      unsigned m_index;
      float    m_newScale;
      float    m_oldScale;
};

class MU_SetBackgroundCenter : public ModelUndo
{
   public:
      MU_SetBackgroundCenter();
      virtual ~MU_SetBackgroundCenter();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void setBackgroundCenter( const unsigned & index, 
            float newX, float newY, float newY,
            float oldX, float oldY, float oldY );

   private:
      unsigned m_index;
      float    m_new[3];
      float    m_old[3];
};

class MU_AddMetaData : public ModelUndo
{
   public:
      MU_AddMetaData();
      virtual ~MU_AddMetaData();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void addMetaData( const std::string & key,
            const std::string & value );

   private:
      std::string m_key;
      std::string m_value;
};

class MU_ClearMetaData : public ModelUndo
{
   public:
      MU_ClearMetaData();
      virtual ~MU_ClearMetaData();

      void undo( Model * );
      void redo( Model * );
      bool combine( Undo * );

      unsigned size();

      void clearMetaData( const Model::MetaDataList & list );

   private:
      Model::MetaDataList m_list;
};

#endif //  __MODELUNDO_H
