/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#ifndef __MODEL_H
#define __MODEL_H

#include "glheaders.h"
#include "glmath.h"
#include "bsptree.h"
#include "drawcontext.h"
#include "sorted_list.h"

#ifdef MM3D_EDIT
#include "undomgr.h"
#endif // MM3D_EDIT

#include <stdint.h>

#include <list>
#include <vector>
#include <string>

using std::list;
using std::vector;

class Texture;

class Model
{
   public:
      class Vertex
      {
         public:
            static int flush();
            static void stats();
            static Vertex * get();
            void release();

            int    m_boneId;
            double m_coord[3];
            bool   m_selected;
            bool   m_visible;
            bool   m_marked;
            bool   m_marked2;

         protected:
            Vertex();
            virtual ~Vertex();
            void init();

            static list<Vertex *> s_recycle;
            static int s_allocated;
      };

      class Triangle
      {
         public:
            static int flush();
            static void stats();
            static Triangle * get();
            void release();

            unsigned m_vertexIndices[3];
            float m_s[3];
            float m_t[3];
            double m_finalNormals[3][3];
            double m_vertexNormals[3][3];
            double m_flatNormals[3][3];
            bool  m_selected;
            bool  m_visible;
            bool  m_marked;
            bool  m_marked2;
         protected:
            Triangle();
            virtual ~Triangle();
            void init();

            static list<Triangle *> s_recycle;
            static int s_allocated;
      };

      class Group
      {
         public:
            static int flush();
            static void stats();
            static Group * get();
            void release();

            std::string m_name;
            int         m_materialIndex;
            vector<int> m_triangleIndices;
            uint8_t     m_smooth;
            uint8_t     m_angle;
            bool        m_selected;
            bool        m_visible;
            bool        m_marked;
         protected:
            Group();
            virtual ~Group();
            void init();

            static list<Group *> s_recycle;
            static int s_allocated;
      };

      class Material
      {
         public:
            typedef enum 
            {
               MATTYPE_TEXTURE  =  0,
               MATTYPE_BLANK    =  1,
               MATTYPE_COLOR    =  2,
               MATTYPE_GRADIENT =  3,
               MATTYPE_MAX      =  4
            } MaterialType;

            static int flush();
            static void stats();
            static Material * get();
            void release();

            std::string   m_name;
            MaterialType  m_type;
            float         m_ambient[4];
            float         m_diffuse[4];
            float         m_specular[4];
            float         m_emissive[4];
            float         m_shininess;
            uint8_t       m_color[4][4];
            bool          m_sClamp;
            bool          m_tClamp;
            GLuint        m_texture;
            std::string   m_filename;
            std::string   m_alphaFilename;
            Texture     * m_textureData;
         protected:
            Material();
            virtual ~Material();
            void init();

            static list<Material *> s_recycle;
            static int s_allocated;
      };

      class Keyframe
      {
         public:
            static int flush();
            static void stats();
            static Keyframe * get();

            void release();

            int m_jointIndex;
            unsigned m_frame;
            double m_time;
            double m_parameter[3];
            bool   m_isRotation;

            bool operator<( const Keyframe & rhs )
            {
               return ( this->m_frame < rhs.m_frame || (this->m_frame == rhs.m_frame && !this->m_isRotation && rhs.m_isRotation) );
            };
            bool operator==( const Keyframe & rhs )
            {
               return ( this->m_frame == rhs.m_frame && this->m_isRotation == rhs.m_isRotation );
            };

         protected:
            Keyframe();
            virtual ~Keyframe();
            void init();

            static list<Keyframe *> s_recycle;
            static int s_allocated;
      };


      class Joint
      {
         public:
            static int flush();
            static void stats();
            static Joint * get();
            void release();

            std::string m_name;
            double m_localRotation[3];
            double m_localTranslation[3];
            Matrix m_absolute;
            Matrix m_relative;

            int m_currentRotationKeyframe;
            int m_currentTranslationKeyframe;
            Matrix m_final;

            int m_parent;

            bool m_selected;
            bool m_visible;
            bool m_marked;

         protected:
            Joint();
            virtual ~Joint();
            void init();

            static list<Joint *> s_recycle;
            static int s_allocated;
      };

      class Point
      {
         public:
            static int flush();
            static void stats();
            static Point * get();
            void release();

            std::string m_name;
            int m_boneId;
            double m_trans[3];
            double m_rot[3];
            double m_localTranslation[3];
            double m_localRotation[3];

            bool   m_selected;
            bool   m_visible;
            bool   m_marked;

         protected:
            Point();
            virtual ~Point();
            void init();

            static list<Point *> s_recycle;
            static int s_allocated;
      };

      typedef sorted_ptr_list<Keyframe *>        KeyframeList;
      typedef vector<KeyframeList>      JointKeyframeList;

      class SkelAnim
      {
         public:
            static int flush();
            static void stats();
            static SkelAnim * get();
            void release();

            std::string m_name;
            JointKeyframeList m_jointKeyframes;
            double   m_fps;
            double   m_spf;
            unsigned m_frameCount;
            bool     m_validNormals;

         protected:
            SkelAnim();
            virtual ~SkelAnim();
            void init();

            static list<SkelAnim *> s_recycle;
            static int s_allocated;
      };

      class FrameAnimVertex
      {
         public:
            static int flush();
            static void stats();
            static FrameAnimVertex * get();
            void release();

            double m_coord[3];
            double m_normal[3];

         protected:
            FrameAnimVertex();
            virtual ~FrameAnimVertex();
            void init();

            static list<FrameAnimVertex *> s_recycle;
            static int s_allocated;
      };

      typedef vector<FrameAnimVertex *> FrameAnimVertexList;

      class FrameAnim
      {
         public:
            static int flush();
            static void stats();
            static FrameAnim * get();
            void release();

            std::string m_name;
            vector<FrameAnimVertexList *> m_frameVertices;
            double m_fps;
            bool   m_validNormals;

         protected:
            FrameAnim();
            virtual ~FrameAnim();
            void init();

            static list<FrameAnim *> s_recycle;
            static int s_allocated;
      };

      class KeyframeVertex
      {
         public:
            double m_coord[3];
            double m_normal[3];
            double m_rotatedNormal[3];
      };

      class KeyframeTriangle
      {
         public:
            double m_normals[3][3];
      };

      class BackgroundImage
      {
         public:
            BackgroundImage();
            virtual ~BackgroundImage();

            std::string m_filename;
            float m_scale;
            float m_center[3];
      };

      typedef Model::Vertex * VertexPtr;
      typedef Model::Triangle * TrianglePtr;
      typedef Model::Group * GroupPtr;
      typedef Model::Material * MaterialPtr;
      //typedef Model::Joint * JointPtr;

      class FormatData
      {
         public:
            FormatData()
               : offsetType( 0 ),
                 format( "" ),
                 index( 0 ),
                 len( 0 ),
                 data( NULL ) { };
            virtual ~FormatData();

            uint16_t      offsetType;  // 0 = none, is valid
            std::string   format;      // Should not be empty
            uint32_t      index;       // for formats with multiple data sets
            uint32_t      len;         // length of data in 'data'
            uint8_t     * data;        // pointer to data

            virtual void serialize();
      };

      class MetaData
      {
         public:
            std::string key;
            std::string value;
      };
      typedef std::vector< MetaData > MetaDataList;

      enum CompareBits
      {
         CompareGeometry  = 0x01,   // Vertices and Faces match
         CompareFaces     = 0x02,   // Faces match, vertices may not
         CompareGroups    = 0x04,   // Groups match
         CompareSkeleton  = 0x08,   // Bone joints hierarchy matches
         CompareTextures  = 0x10,   // Textures and texture coordinates match
         CompareAnimSets  = 0x20,   // Number of animations, frame counts, and fps match
         CompareAnimData  = 0x40,   // Animation movements match
         CompareMeta      = 0x80,   // Names and other non-visible data match
         CompareAll       = 0xFF    // All of the above
      };

      enum ModelError
      {
         ERROR_NONE = 0,
         ERROR_CANCEL,
         ERROR_UNKNOWN_TYPE,
         ERROR_UNSUPPORTED_OPERATION,
         ERROR_BAD_ARGUMENT,
         ERROR_NO_FILE,
         ERROR_NO_ACCESS,
         ERROR_FILE_OPEN,
         ERROR_FILE_READ,
         ERROR_BAD_MAGIC,
         ERROR_UNSUPPORTED_VERSION,
         ERROR_BAD_DATA,
         ERROR_UNEXPECTED_EOF,
         ERROR_FILTER_SPECIFIC,
         ERROR_UNKNOWN
      };

      enum SelectionMode
      {
         SelectNone,
         SelectVertices,
         SelectTriangles,
         SelectConnected,
         SelectGroups,
         SelectJoints
      };

      enum ProjectionDirection
      {
         View3d = 0,
         ViewFront,
         ViewBack,
         ViewLeft,
         ViewRight,
         ViewTop,
         ViewBottom
      };

      enum
      {
         MAX_GROUP_NAME_LEN = 32,
         MAX_BACKGROUND_IMAGES = 6
      };

      enum
      {
         DO_NONE      = 0x00,
         DO_TEXTURE   = 0x01,  // powers of 2
         DO_SMOOTHING = 0x02,
         DO_WIREFRAME = 0x04,
         DO_BADTEX    = 0x08,
         DO_ALPHA     = 0x10
      };

      enum AnimationMode
      {
         ANIMMODE_NONE = 0,
         ANIMMODE_SKELETAL,
         ANIMMODE_FRAME,
         ANIMMODE_FRAMERELATIVE,
         ANIMMODE_MAX
      };

      enum DrawJointMode_e
      {
         JOINTMODE_NONE = 0,
         JOINTMODE_LINES,
         JOINTMODE_BONES,
         JOINTMODE_MAX
      };
      typedef enum DrawJointMode_e DrawJointMode;

      typedef enum AnimationMerge_e
      {
         AM_NONE = 0,
         AM_ADD,
         AM_MERGE
      } AnimationMerge;

      // Public methods
      Model();
      virtual ~Model();

      static const char * errorToString( Model::ModelError, Model * model = NULL );
      static bool operationFailed( Model::ModelError );

      // Returns mask of successful compares (see enum CompareBits)
      int equivalent( const Model * model, int compareMask = CompareGeometry, double tolerance = 0.00001 );

      const char * getFilename() { return m_filename.c_str(); };
      void setFilename( const char * filename ) { 
            if ( filename && filename[0] ) { m_filename = filename; } };

      void setFilterSpecificError( const char * str ) { s_lastFilterError = m_filterSpecificError = str; };
      const char * getFilterSpecificError() { return m_filterSpecificError.c_str(); };
      static const char * getLastFilterSpecificError() { return s_lastFilterError.c_str(); };

      void draw( unsigned drawOptions = DO_TEXTURE, unsigned context = 0, float *viewPoint = NULL );
      void drawLines();
      void drawPoints();
      bool loadTextures( unsigned context = 0 );
      void removeContext( unsigned context );

      // Use these functions to preserve data that Misfit doesn't support natively
      int  addFormatData( FormatData * fd );
      bool deleteFormatData( unsigned index );
      unsigned getFormatDataCount();
      FormatData * getFormatData( unsigned index );
      FormatData * getFormatDataByFormat( const char * format, unsigned index = 0 ); // not case sensitive

      bool setCurrentAnimation( const AnimationMode & m, const char * name );
      bool setCurrentAnimation( const AnimationMode & m, unsigned index );
      bool setCurrentAnimationFrame( unsigned frame );
      bool setCurrentAnimationTime( double time );

      unsigned getCurrentAnimation();
      unsigned getCurrentAnimationFrame();
      double   getCurrentAnimationTime();

      void setAnimationLooping( bool o );
      bool isAnimationLooping();

      void setNoAnimation();

      double   getAnimFPS( const AnimationMode & mode, const unsigned & anim );

      const char * getAnimName( const AnimationMode & mode, const unsigned & anim );

      unsigned getAnimFrameCount( const AnimationMode & mode, const unsigned & anim );

      AnimationMode getAnimationMode() { return m_animationMode; };
      bool inSkeletalMode()  { return (m_animationMode == ANIMMODE_SKELETAL); };

      bool getFrameAnimVertexCoords( const unsigned & anim, const unsigned & frame, const unsigned & vertex, 
            double & x, double & y, double & z );
      bool getFrameAnimVertexNormal( const unsigned & anim, const unsigned & frame, const unsigned & vertex, 
            double & x, double & y, double & z );

      bool hasSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation );
      bool getSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation,
            double & x, double & y, double & z );

      bool getNormal( unsigned triangleNum, unsigned vertexIndex, float *normal );

      void calculateNormals();
      void calculateSkelNormals();
      void calculateFrameNormals( const unsigned & anim );
      void invalidateNormals();

      void calculateBspTree();
      void invalidateBspTree();

      const char * getGroupName( unsigned groupNum );
      inline int getGroupCount() const { return m_groups.size(); };
      int getGroupByName( const char * groupName, bool ignoreCase = false );
      int getMaterialByName( const char * materialName, bool ignoreCase = false );
      Material::MaterialType getMaterialType( unsigned materialIndex );
      int getMaterialColor( unsigned materialIndex, unsigned c, unsigned v = 0 );

      uint8_t getGroupSmooth( const unsigned & groupNum );
      uint8_t getGroupAngle( const unsigned & groupNum );

      const char * getTextureName( unsigned textureId );
      const char * getTextureFilename( unsigned textureId );
      Texture * getTextureData( unsigned textureId );
      inline int getTextureCount() { return m_materials.size(); };

      bool getTextureAmbient(   unsigned textureId,       float * ambient   );
      bool getTextureDiffuse(   unsigned textureId,       float * diffuse   );
      bool getTextureEmissive(  unsigned textureId,       float * emissive  );
      bool getTextureSpecular(  unsigned textureId,       float * specular  );
      bool getTextureShininess( unsigned textureId,       float & shininess );

      bool getTextureSClamp( unsigned textureId );
      bool getTextureTClamp( unsigned textureId );

      list<int> getUngroupedTriangles();
      list<int> getGroupTriangles( unsigned groupNumber ) const;
      int       getGroupTextureId( unsigned groupNumber );

      list<int> getBoneJointVertices( const int & joint );
      const char * getBoneJointName( const unsigned & joint );

      int getBoneJointParent( const unsigned & joint );

      inline int getVertexCount()    const { return m_vertices.size(); }
      inline int getTriangleCount()  const { return m_triangles.size(); }
      inline int getBoneJointCount() const { return m_joints.size(); }
      inline int getPointCount()     const { return m_points.size(); }
      unsigned getAnimCount( const AnimationMode & m ) const;

      bool getVertexCoords( const unsigned & vertexNumber, double *coord );
      bool getVertexCoords2d( const unsigned & vertexNumber, const ProjectionDirection & dir, double *coord );
      int getVertexBoneJoint( const unsigned & vertexNumber );
      bool getTextureCoords( const unsigned & triangleNumber, const unsigned & vertexIndex, float & s, float & t );
      bool setTextureCoords( const unsigned & triangleNumber, const unsigned & vertexIndex, const float & s, const float & t );
      bool getBoneJointCoords( const unsigned & jointNumber, double * coord );

      bool getBoneJointFinalMatrix( const unsigned & jointNumber, Matrix & m );
      bool getPointFinalMatrix( const unsigned & pointNumber, Matrix & m );

      /*
      bool getRelativeBoneJointPosition( const unsigned & jointNumber,
            const double & newX, const double & newY, const double & newZ,
            double & diffX, double & diffY, double & diffZ
            );
      */

      int getTriangleVertex( unsigned triangleNumber, unsigned vertexIndex );
      int getTriangleGroup( unsigned triangleNumber );

      void setupJoints();

      void forceAddOrDelete( bool o ) { m_forceAddOrDelete = o; };
      int getNumFrames();

      void invalidateTextures();

      static int    s_glTextures;

#ifdef MM3D_EDIT
      void addMetaData( const char * key, const char * value );
      bool getMetaData( const char * key, char * value, size_t valueLen );
      bool getMetaData( unsigned int index, char * key, size_t keyLen, char * value, size_t valueLen );
      unsigned int getMetaDataCount();
      void clearMetaData();
      void removeLastMetaData(); // For undo only!

      void displayFrameAnimPrimitiveError();

      bool mergeAnimations( Model * model ); // For skeletal, skeletons must match
      bool mergeModels( Model * model, bool textures, AnimationMerge mergeMode, bool emptyGroups,
            double * trans = NULL, double * rot = NULL );

      void drawJoints();

      void setSaved( bool o ) { if ( o ) { m_undoMgr->setSaved(); }; };
      bool getSaved()         { return m_undoMgr->isSaved(); };

      void setCanvasDrawMode( int m ) { m_canvasDrawMode = m; };
      int  getCanvasDrawMode() { return m_canvasDrawMode; };

      void setDrawJoints( DrawJointMode m ) { m_drawJoints = m; };
      DrawJointMode getDrawJoints() { return m_drawJoints; };

      void setUndoSizeLimit( unsigned sizeLimit );
      void setUndoCountLimit( unsigned countLimit );

      int addVertex( double x, double y, double z );
      int addTriangle( unsigned vert1, unsigned vert2, unsigned vert3 );
      int addGroup( const char * name );
      int addBoneJoint( const char * name, const double & x, const double & y, const double & z, 
            const double & xrot, const double & yrot, const double & zrot,
            const int & parent = -1 );
      int addPoint( const char * name, const double & x, const double & y, const double & z, 
            const double & xrot, const double & yrot, const double & zrot,
            const int & boneId = -1 );
      int addTexture( Texture * tex );
      int addColorMaterial( const char * name );
      int addAnimation( const AnimationMode & mode, const char * name );

      void deleteAnimation( const AnimationMode & mode, const unsigned & index );

      int  copyAnimation( const AnimationMode & mode, const unsigned & anim, const char * newName );
      int  splitAnimation( const AnimationMode & mode, const unsigned & anim, const char * newName, const unsigned & frame );
      bool joinAnimations( const AnimationMode & mode, const unsigned & anim1, const unsigned & anim2 );
      bool mergeAnimations( const AnimationMode & mode, const unsigned & anim1, const unsigned & anim2 );
      int  convertAnimToFrame( const AnimationMode & mode, const unsigned & anim1, const char * newName, const unsigned & frameCount );

      bool moveAnimation( const AnimationMode & mode, const unsigned & oldIndex, const unsigned & newIndex );

      bool clearAnimFrame( const AnimationMode & mode, const unsigned & anim, const unsigned & frame );

      bool     setAnimFPS( const AnimationMode & mode, const unsigned & anim, const double & fps );
      bool setAnimName( const AnimationMode & mode, const unsigned & anim, const char * name );
      bool     setAnimFrameCount( const AnimationMode & mode, const unsigned & anim, const unsigned & count );

      void setFrameAnimVertexCount( const unsigned & vertexCount );

      bool setFrameAnimVertexCoords( const unsigned & anim, const unsigned & frame, const unsigned & vertex, 
            const double & x, const double & y, const double & z );

      int  setSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation, 
            const double & x, const double & y, const double & z );
      bool deleteSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation );

      bool insertSkelAnimKeyframe( const unsigned & anim, Keyframe * keyframe );
      bool removeSkelAnimKeyframe( const unsigned & anim, const unsigned & frame, const unsigned & joint, const bool & isRotation, bool release = false );

      bool setTriangleVertices( unsigned triangleNum, unsigned vert1, unsigned vert2, unsigned vert3 );

      void deleteVertex( unsigned vertex );
      void deleteTriangle( unsigned triangle );
      void deleteGroup( unsigned group );
      void deleteBoneJoint( unsigned joint );
      void deletePoint( unsigned point );
      void deleteTexture( unsigned texture );

      void deleteOrphanedVertices();
      void deleteFlattenedTriangles();
      void deleteSelected();

      void invertNormals( unsigned triangleNum );

      bool moveVertex( unsigned v, double x, double y, double z );
      bool moveBoneJoint( unsigned j, double x, double y, double z );

      // No undo on this one
      bool relocateBoneJoint( unsigned j, double x, double y, double z );

      bool selectVertex( unsigned v );
      bool unselectVertex( unsigned v );
      bool isVertexSelected( unsigned v );

      bool selectTriangle( unsigned t );
      bool unselectTriangle( unsigned t );
      bool isTriangleSelected( unsigned v );

      bool selectGroup( unsigned m );
      bool unselectGroup( unsigned m );
      bool isGroupSelected( unsigned v );

      bool selectBoneJoint( unsigned j );
      bool unselectBoneJoint( unsigned j );
      bool isBoneJointSelected( unsigned j );

      class SelectionTest
      {
         public:
            virtual ~SelectionTest() {};
            virtual bool shouldSelect( void * element ) = 0;
      };

      bool selectInVolumeXY( double x1, double y1, double x2, double y2, SelectionTest * test = NULL );
      bool selectInVolumeXZ( double x1, double z1, double x2, double z2, SelectionTest * test = NULL );
      bool selectInVolumeYZ( double y1, double z1, double y2, double z2, SelectionTest * test = NULL );

      bool unselectInVolumeXY( double x1, double y1, double x2, double y2, SelectionTest * test = NULL );
      bool unselectInVolumeXZ( double x1, double z1, double x2, double z2, SelectionTest * test = NULL );
      bool unselectInVolumeYZ( double y1, double z1, double y2, double z2, SelectionTest * test = NULL );

      bool unselectAll();

      bool invertSelection();

      // Don't call these directly... use selection/hide selection
      bool hideVertex( unsigned );
      bool unhideVertex( unsigned );
      bool hideTriangle( unsigned );
      bool unhideTriangle( unsigned );
      bool hideJoint( unsigned );
      bool unhideJoint( unsigned );

      bool hideSelected();
      bool hideUnselected();
      bool unhideAll();

      bool getBoundingRegion( double *x1, double *y1, double *z1, double *x2, double *y2, double *z2 );
      bool getSelectedBoundingRegion( double *x1, double *y1, double *z1, double *x2, double *y2, double *z2 );

      unsigned getSelectedVertexCount();
      unsigned getSelectedTriangleCount();
      unsigned getSelectedBoneJointCount();

      void translateSelectedVertices( const Matrix & m );
      void rotateSelectedVerticesOnPoint( const Matrix & m, double * point );
      void applyMatrix( const Matrix & m );

      void subdivideSelectedTriangles();
      void unsubdivideTriangles( unsigned t1, unsigned t2, unsigned t3, unsigned t4 );

      void setSelectionMode( SelectionMode m );
      inline SelectionMode getSelectionMode() { return m_selectionMode; };

      bool setGroupName( unsigned groupNum, const char * groupName );

      bool setGroupSmooth( const unsigned & groupNum, const uint8_t & smooth );
      bool setGroupAngle( const unsigned & groupNum, const uint8_t & angle );

      void setSelectedAsGroup( unsigned groupNum );
      void addSelectedToGroup( unsigned groupNum );

      void setTextureName( unsigned textureId, const char * name );

      bool setTextureAmbient(   unsigned textureId, const float * ambient   );
      bool setTextureDiffuse(   unsigned textureId, const float * diffuse   );
      bool setTextureEmissive(  unsigned textureId, const float * emissive  );
      bool setTextureSpecular(  unsigned textureId, const float * specular  );
      bool setTextureShininess( unsigned textureId, const float & shininess );

      bool setTextureSClamp( unsigned textureId, bool clamp );
      bool setTextureTClamp( unsigned textureId, bool clamp );

      list<int> getSelectedVertices();
      list<int> getSelectedTriangles();
      list<int> getSelectedGroups();
      list<int> getSelectedBoneJoints();

      bool      setGroupTextureId( unsigned groupNumber, int textureId );

      bool setVertexBoneJoint( const unsigned & vertex, const int & joint );
      bool setBoneJointName( const unsigned & joint, const char * name );
      bool setBoneJointParent( const unsigned & joint, const int & parent = -1 );
      bool setBoneJointRotation( const unsigned & j, const double * rot );
      bool setBoneJointTranslation( const unsigned & j, const double * trans );

      bool setUndoEnabled( bool o ) { bool old = m_undoEnabled; m_undoEnabled = o; return old; };

      void operationComplete( const char * opname = NULL );

      bool canUndo();
      bool canRedo();
      const char * getUndoOpName();
      const char * getRedoOpName();
      void undo();
      void redo();
      void undoCurrent();

      bool canAddOrDelete() { return (m_frameAnims.size() == 0 || m_forceAddOrDelete); };

      // These functions are for undo and features such as this.
      // Don't use them unless you are me.
      void insertVertex( unsigned index, Vertex * vertex );
      void removeVertex( unsigned index );

      void insertTriangle( unsigned index, Triangle * triangle );
      void removeTriangle( unsigned index );

      void insertGroup( unsigned index, Group * group );
      void removeGroup( unsigned index );

      void insertBoneJoint( unsigned index, Joint * joint );
      void removeBoneJoint( unsigned index );

      void insertPoint( unsigned index, Point * point );
      void removePoint( unsigned index );

      void insertTexture( unsigned index, Material * material );
      void removeTexture( unsigned index );

      void insertFrameAnim( const unsigned & index, FrameAnim * anim );
      void removeFrameAnim( const unsigned & index );

      void insertSkelAnim( const unsigned & anim, SkelAnim * fa );
      void removeSkelAnim( const unsigned & anim );

      void insertFrameAnimFrame( const unsigned & anim, const unsigned & frame,
            FrameAnimVertexList * list );
      void removeFrameAnimFrame( const unsigned & anim, const unsigned & frame );

      void addTriangleToGroup( unsigned groupNum, unsigned triangleNum );
      void removeTriangleFromGroup( unsigned groupNum, unsigned triangleNum );

      bool unselectAllVertices();
      bool unselectAllTriangles();
      bool unselectAllGroups();
      bool unselectAllBoneJoints();

      void setBackgroundImage( unsigned index, const char * str );
      void setBackgroundScale( unsigned index, float scale );
      void setBackgroundCenter( unsigned index, float x, float y, float z );

      const char * getBackgroundImage( unsigned index );
      float getBackgroundScale( unsigned index );
      void getBackgroundCenter( unsigned index, float & x, float & y, float & z );

#endif // MM3D_EDIT

      void setLocalMatrix( const Matrix & m ) { m_localMatrix = m; };

   protected:

      DrawingContext * getDrawingContext( unsigned context );

      void deleteGlTextures( unsigned context );

#ifdef MM3D_EDIT

      void adjustVertexIndices( unsigned index, int count );
      void adjustTriangleIndices( unsigned index, int count );

      void beginSelectionDifference();
      void endSelectionDifference();

      void beginHideDifference();
      void endHideDifference();

      void hideVerticesFromTriangles();
      void unhideVerticesFromTriangles();

      void hideTrianglesFromVertices();
      void unhideTrianglesFromVertices();

      bool selectVerticesInVolume( bool select, int c1, int c2, double a1, double b1, double a2, double b2, SelectionTest * test = NULL );
      bool selectTrianglesInVolume( bool select, int c1, int c2, double a1, double b1, double a2, double b2, bool connected, SelectionTest * test = NULL );
      bool selectGroupsInVolume( bool select, int c1, int c2, double a1, double b1, double a2, double b2, SelectionTest * test = NULL );
      bool selectBoneJointsInVolume( bool select, int c1, int c2, double a1, double b1, double a2, double b2, SelectionTest * test = NULL );

      void selectTrianglesFromGroups();
      void selectVerticesFromTriangles();

      void selectTrianglesFromVertices( bool all = true );
      void selectGroupsFromTriangles( bool all = true );

      bool parentJointSelected( int joint );
      bool directParentJointSelected( int joint );

      void sendUndo( Undo * undo );
      void noTexture( unsigned id );

#endif // MM3D_EDIT

      friend class ModelFilter;

      bool          m_initialized;
      std::string   m_filename;
      std::string   m_filterSpecificError;
      static std::string   s_lastFilterError;

      MetaDataList          m_metaData;
      vector<Vertex *>      m_vertices;
      vector<Triangle *>    m_triangles;
      vector<Group *>       m_groups;
      vector<Material *>    m_materials;
      vector<Joint *>       m_joints;
      vector<Point *>       m_points;
      vector<FrameAnim *>   m_frameAnims;
      vector<SkelAnim *>    m_skelAnims;

      DrawingContextList m_drawingContexts;

      BspTree     m_bspTree;
      bool        m_validBspTree;

      vector<KeyframeVertex>    m_keyframeVertices;
      vector<KeyframeTriangle>  m_keyframeTriangles;

      vector<FormatData *>  m_formatData;

      int  m_canvasDrawMode;
      DrawJointMode m_drawJoints;

      bool m_validNormals;
      bool m_validJoints;

      bool m_forceAddOrDelete;

      int    m_numFrames;

      AnimationMode m_animationMode;
      bool     m_animationLoop;
      unsigned m_currentFrame;
      unsigned m_currentAnim;
      double   m_currentTime;

#ifdef MM3D_EDIT

      BackgroundImage * m_background[6];

      bool          m_saved;
      SelectionMode m_selectionMode;

      UndoManager * m_undoMgr;
      UndoManager * m_animUndoMgr;
      bool m_undoEnabled;
#endif // MM3D_EDIT

      Matrix   m_localMatrix;
};

extern void model_show_alloc_stats();
extern int model_free_primitives();

#endif //__MODEL_H
