// md2 io
#include "stdafx.h"
#include "..\MedDLeGFX\MedDLeTypes.h"
#include "..\MedDLeGFX\MedDLe3D.h"
#include "IOPCX.h"
#include <vfw.h>
#include <fstream.h>

struct PCXHeader
{
        unsigned char   manufacturer;
        unsigned char   version;
        unsigned char   encoding;
        unsigned char   bits_per_pixel_per_plane;
        short           xmin;
        short           ymin;
        short           xmax;
        short           ymax;
        unsigned short  hresolution;
        unsigned short  vresolution;
        unsigned char   colormap[48];
        unsigned char   reserved;
        unsigned char   nplanes;
        unsigned short  bytes_per_line;
        short           palette_info;
        unsigned char   filler[58];   // Header is 128 bytes
};

PCXHeader hdr;

BOOL IOPCXRead(LPCSTR fname, CMedDLeTexture &mt )
{

    ifstream ifs(fname, ios::in | ios::binary);
    if(!ifs)
    {
        return 0;
    }

	// Read PCX header
    ifs.read((unsigned char*)&hdr, sizeof(PCXHeader));

	// Check if image file format is acceptable
    if(hdr.manufacturer != 0x0a) return 0;

	// We only handle 1, 4, 8, or 24-bit images
    int bits_per_pixel = hdr.nplanes *
                         hdr.bits_per_pixel_per_plane;

    if(bits_per_pixel != 1 &&
       bits_per_pixel != 4 &&
       bits_per_pixel != 8 &&
       bits_per_pixel != 24) return 0;

    unsigned short image_width = hdr.xmax - hdr.xmin + 1;
    unsigned short image_height = hdr.ymax - hdr.ymin + 1;

// Allocate space where the PCX image will be unpacked.
// Read in PCX image into this area.
    long pcx_image_size = (long) hdr.nplanes *
                          (long) image_height *
						  (long) hdr.bytes_per_line;

	BYTE *image = new BYTE [pcx_image_size]; 
    if(image== NULL) return 0;


// Decode run-length encoded image data

    int i, byte, count;
    unsigned long pos = 0L;
    
    while((byte = ifs.get()) != EOF)
    {
        if((byte & 0xc0) == 0xc0)
        {
            count = byte & 0x3f;
            if((byte = ifs.get()) != EOF)
            {
                for(i = 0; i < count; i++)
                {
                    if(pos >= (unsigned long) pcx_image_size) break;
                    image[pos] = byte;
                    pos++;
                }
            }
        }
        else
        {
            if(pos >= (unsigned long)pcx_image_size) break;
            image[pos]= byte;
            pos++;
        }
    }

// Allocate memory for the device independent bitmap (DIB)
// Note that the number of bytes in each line of a DIB image 
// must be a multiple of 4.

    unsigned short bytes_per_line_per_plane = (image_width * 
                       hdr.bits_per_pixel_per_plane + 7) / 8;

    unsigned short actual_bytes_per_line = (image_width * 
                                            hdr.nplanes *
                       hdr.bits_per_pixel_per_plane + 7) / 8;

    int bytes_per_line = actual_bytes_per_line;

    if(bytes_per_line % 4)
        bytes_per_line = 4 *
                      (bytes_per_line/4 + 1);

// Make room for a palette
    int palettesize = 16;
    if(bits_per_pixel == 1) palettesize = 2;
    if(hdr.version >= 5 && bits_per_pixel > 4)
    {
// Go back 769 bytes from the end of the file
        ifs.seekg(-769L, ios::end);
        if(ifs.get() == 12)
        {
// There is a 256-color palette following this byte
            palettesize = 256;
        }
    }
// If image has more than 256 colors then there is no palette
    if(bits_per_pixel > 8) palettesize = 0;



	if(mt.bitmap!=NULL) delete [] mt.bitmap;

    mt.bitmap = new BYTE[(long) bytes_per_line * (long) image_height];

	if(mt.bitmap==NULL)
	{
		delete [] image; 
		return FALSE;
	}
	
	if(mt.header) delete [] mt.header;
	
	long ssize=sizeof(BITMAPINFOHEADER);
	if(palettesize)
	{
		ssize+=palettesize*sizeof(RGBQUAD);
	}

	mt.header = new BYTE[ssize];
	if(mt.header==NULL)
	{
		delete [] mt.bitmap;
		mt.bitmap=NULL;
		delete [] image; 
		return FALSE;
	}
	mt.bmpheader=(LPBITMAPINFOHEADER)mt.header;

	if(palettesize==0) mt.colors=NULL;
	else 
	{
		mt.colors = (RGBQUAD*)(mt.header+sizeof(BITMAPINFOHEADER));
	}

// Set up bitmap info header
    LPBITMAPINFOHEADER p_bminfo = mt.bmpheader;
    p_bminfo->biSize = sizeof(BITMAPINFOHEADER);
    p_bminfo->biWidth = image_width;
    p_bminfo->biHeight = image_height;
    p_bminfo->biPlanes = 1;
    p_bminfo->biBitCount = hdr.bits_per_pixel_per_plane *
                           hdr.nplanes;
    p_bminfo->biCompression = BI_RGB;
    p_bminfo->biSizeImage = (long)image_height * 
                            (long)bytes_per_line;
    p_bminfo->biXPelsPerMeter = 0;
    p_bminfo->biYPelsPerMeter = 0;
    p_bminfo->biClrUsed = 256;
    p_bminfo->biClrImportant = 0;

// Set up the color palette
    if(palettesize > 0)
    {
        RGBQUAD *palette = mt.colors;
	
        int palindex;
        for(palindex = 0; palindex < palettesize; palindex++)
        {
            if(palettesize == 256)
            {
// Read palette from file
                palette[palindex].rgbRed       = ifs.get();
                palette[palindex].rgbGreen     = ifs.get();
                palette[palindex].rgbBlue      = ifs.get();
                palette[palindex].rgbReserved  = 0;
            }
            if(palettesize == 16)
            {
// 16-color palette from PCX header
                palette[palindex].rgbRed = 
                                  hdr.colormap[3*palindex];
                palette[palindex].rgbGreen = 
                                  hdr.colormap[3*palindex+1];
                palette[palindex].rgbBlue = 
                                  hdr.colormap[3*palindex+2];
                palette[palindex].rgbReserved  = 0;
            }
            if(palettesize == 2)
            {
// Set up palette for black and white images
                palette[palindex].rgbRed = 
                                  palindex * 255;
                palette[palindex].rgbGreen = 
                                  palindex * 255;
                palette[palindex].rgbBlue = 
                                  palindex * 255;
                palette[palindex].rgbReserved  = 0;
            }
        }
    }    

// Load image data into the DIB. Note the DIB image must be 
// stored "bottom to top" line order. That's why we position
// data at the end of the array so that the image can be
// stored backwards--from the last line to the first.
    unsigned char *data = mt.bitmap;
//	+                (unsigned long)(image_height - 1) * (unsigned long)(bytes_per_line);

// Define a macro to access bytes in the PCX image according
// to specified line and plane index.

    int lineindex, byteindex, planeindex;

#define bytepos(lineindex,planeindex,byteindex)  \
            ((long)(lineindex)*(long)hdr.bytes_per_line* \
             (long)hdr.nplanes + \
             (long)(planeindex)*(long)hdr.bytes_per_line + \
             (long)(byteindex))

// Construct packed pixels out of decoded PCX image.

    unsigned short onebyte, bits_copied, loc, few_bits, 
        k, bbpb = 8/hdr.bits_per_pixel_per_plane;

// Build a mask to pick out bits from each byte of the PCX image
    unsigned short himask = 0x80, mask;
    if(hdr.bits_per_pixel_per_plane > 1)
        for(i = 0; i < hdr.bits_per_pixel_per_plane - 1;
            i++) himask = 0x80 | (himask >> 1);

    for(lineindex = 0; lineindex < image_height; lineindex++,data += bytes_per_line)
    {
        if(actual_bytes_per_line < bytes_per_line)
            for(loc = actual_bytes_per_line;
                loc < bytes_per_line; loc++)
                                        data[loc] = 0;
        loc = 0;
        onebyte = 0;
        bits_copied = 0;
        for(byteindex = 0; 
            byteindex < bytes_per_line_per_plane;
            byteindex++)
        {
            for(k = 0, mask = himask; k < bbpb; 
                k++, mask >>= hdr.bits_per_pixel_per_plane)
            {
// Go through all scan line for all planes and copy bits into
// the data array
                for(planeindex = 0; planeindex < hdr.nplanes;
                    planeindex++)
                {
                    few_bits = image[bytepos(lineindex,
                               planeindex, byteindex)] & mask;

// Shift the selcted bits to the most significant position
                    if(k > 0) few_bits <<= 
                                (k*hdr.bits_per_pixel_per_plane);

// OR the bits with current pixel after shifting them right
                    if(bits_copied > 0)
                            few_bits >>= bits_copied;

                    onebyte |= few_bits;
                    bits_copied += hdr.bits_per_pixel_per_plane;

                    if(bits_copied >= 8)
                    {
                        data[loc] = (unsigned char)onebyte;
                        loc++;
                        bits_copied = 0;
                        onebyte = 0;
                    }
                }
            }
        }
    }
    delete image;

// Success!
    return TRUE;

}

typedef struct
{
    char	manufacturer;
    char	version;
    char	encoding;
    char	bits_per_pixel;
    unsigned short	xmin,ymin,xmax,ymax;
    unsigned short	hres,vres;
    unsigned char	palette[48];
    char	reserved;
    char	color_planes;
    unsigned short	bytes_per_line;
    unsigned short	palette_type;
    char	filler[58];
    unsigned char	data;			// unbounded
} pcx_t;

PCXHeader pcx;

BOOL IOPCXWrite(LPCSTR fname, CMedDLeTexture &mt )
{
	FILE *out =fopen(fname, "wb");
	if(out == NULL) return FALSE;

	int		i, j, length;
	BYTE *id;
	pcx_t	*pcx;
	BYTE	*pack;
	BYTE *data =mt.bitmap;
	  
    LPBITMAPINFOHEADER p_bminfo = mt.bmpheader;
	if(p_bminfo->biBitCount!=8) return FALSE;

	id = new BYTE[p_bminfo->biWidth*p_bminfo->biHeight*2+1000];
	pcx = (pcx_t*) id;
	memset (pcx, 0, p_bminfo->biWidth*p_bminfo->biHeight*2+1000);

/*    p_bminfo->biSize = sizeof(BITMAPINFOHEADER);
    p_bminfo->biWidth = image_width;
    p_bminfo->biHeight = image_height;
    p_bminfo->biPlanes = 1;
    p_bminfo->biBitCount = hdr.bits_per_pixel_per_plane *
                           hdr.nplanes;
    p_bminfo->biCompression = BI_RGB;
    p_bminfo->biSizeImage = (long)image_height * 
                            (long)bytes_per_line;
    p_bminfo->biXPelsPerMeter = 0;
    p_bminfo->biYPelsPerMeter = 0;
    p_bminfo->biClrUsed = 256;
    p_bminfo->biClrImportant = 0;
*/

	pcx->manufacturer = 0x0a;	// PCX id
	pcx->version = 5;			// 256 color
 	pcx->encoding = 1;		// uncompressed
	pcx->bits_per_pixel = 8;		// 256 color
	pcx->xmin = 0;
	pcx->ymin = 0;
	pcx->xmax = p_bminfo->biWidth-1;
	pcx->ymax = p_bminfo->biHeight-1;
	pcx->hres = p_bminfo->biWidth;
	pcx->vres = p_bminfo->biHeight;
	pcx->color_planes = 1;		// chunky image
	pcx->bytes_per_line = p_bminfo->biWidth;
	pcx->palette_type = 2;		// not a grey scale

	// pack the image
	pack = &pcx->data;
	
	for (i=0 ; i<p_bminfo->biHeight; i++)
	{
		for (j=0 ; j<p_bminfo->biWidth; j++)
		{
			if ( (*data & 0xc0) != 0xc0)
				*pack++ = *data++;
			else
			{
				*pack++ = 0xc1;
				*pack++ = *data++;
			}
		}
	}
			
	// write the palette
	*pack++ = 0x0c;	// palette ID byte
	for(i=0; i<256; i++)
	{
		*pack++=mt.colors[i].rgbRed;
		*pack++=mt.colors[i].rgbGreen;
		*pack++=mt.colors[i].rgbBlue;
	}
		
// write output file 
	length = pack - (byte *)pcx;

	fwrite(pcx,1,length,out);

	fclose(out);

	delete [] id;
	return TRUE;
} 



/*
==============
LoadPCX
==============

void LoadPCX (char *filename, byte **pic, byte **palette, int *width, int *height)
{
	byte	*raw;
	pcx_t	*pcx;
	int		x, y;
	int		len;
	int		dataByte, runLength;
	byte	*out, *pix;

	//
	// load the file
	//
	len = LoadFile (filename, (void **)&raw);

	//
	// parse the PCX file
	//
	pcx = (pcx_t *)raw;
	raw = &pcx->data;

	pcx->xmin = LittleShort(pcx->xmin);
	pcx->ymin = LittleShort(pcx->ymin);
	pcx->xmax = LittleShort(pcx->xmax);
	pcx->ymax = LittleShort(pcx->ymax);
	pcx->hres = LittleShort(pcx->hres);
	pcx->vres = LittleShort(pcx->vres);
	pcx->bytes_per_line = LittleShort(pcx->bytes_per_line);
	pcx->palette_type = LittleShort(pcx->palette_type);

	if (pcx->manufacturer != 0x0a
		|| pcx->version != 5
		|| pcx->encoding != 1
		|| pcx->bits_per_pixel != 8
		|| pcx->xmax >= 640
		|| pcx->ymax >= 480)
		Error ("Bad pcx file %s", filename);
	
	if (palette)
	{
		*palette = malloc(768);
		memcpy (*palette, (byte *)pcx + len - 768, 768);
	}

	if (width)
		*width = pcx->xmax+1;
	if (height)
		*height = pcx->ymax+1;

	if (!pic)
		return;

	out = malloc ( (pcx->ymax+1) * (pcx->xmax+1) );
	if (!out)
		Error ("Skin_Cache: couldn't allocate");

	*pic = out;

	pix = out;

	for (y=0 ; y<=pcx->ymax ; y++, pix += pcx->xmax+1)
	{
		for (x=0 ; x<=pcx->xmax ; )
		{
			dataByte = *raw++;

			if((dataByte & 0xC0) == 0xC0)
			{
				runLength = dataByte & 0x3F;
				dataByte = *raw++;
			}
			else
				runLength = 1;

			while(runLength-- > 0)
				pix[x++] = dataByte;
		}

	}

	if ( raw - (byte *)pcx > len)
		Error ("PCX file %s was malformed", filename);

	free (pcx);
}

/* 
============== 
WritePCXfile 
============== 
*/ 
