//
// Routines for performing CSG on polygonal objects (b-rep)
//
// Written by Alexander Enzmann, September 1996
//
//
//
#include "StdAfx.h"

#include <stdio.h>
#include <math.h>
#include <assert.h>
#include <string.h>
#include "Polycsg.h"

#ifdef _DEBUG
#if defined( DEBUG_NEW )
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif
#endif

#ifndef WIN32
#define OutputDebugString(x) printf(x)
#endif

static char sbuf[256];

#if !defined( EPSILON )
#define EPSILON 1.0e-6
#endif

#if !defined( INFINITY )
#define INFINITY 1.0e30f
#endif

#if defined(_DEBUG)
#define DEBUG_TREES 1
#define DEBUG_LEAVES 0
#else
#define DEBUG_TREES 0
#define DEBUG_LEAVES 0
#endif

#define TRUE  1
#define FALSE 0

TriVertex::
~TriVertex()
{
	int i = 0;
}

VertexSet::~VertexSet()
{
	// Clean up memory
	for (iterator vitr=begin();vitr!=end();vitr++) {
		TriVertexPtr v = *vitr;
		delete v.ptr;
	}
}

void PolyTri::compute_indices(int &i1, int &i2)
{
	MVector3 tN(plane.A, plane.B, plane.C);
	if (fabs(tN[0]) > fabs(tN[1]) &&
		fabs(tN[0]) > fabs(tN[2])) {
		i1 = 1;
		i2 = 2;
	} else if (fabs(tN[1]) > fabs(tN[2])) {
		i1 = 0;
		i2 = 2;
	} else {
		i1 = 0;
		i2 = 1;
	}
}

// Figure out the axis-aligned bounding box of a PolyTri.
void PolyTri::
compute_bbox(Range3D &bbox)
{
	int i, j;
	float t;

	TriVertex *v = vertices[0].ptr;
	for (j=0;j<3;j++)
		bbox[0][j] = bbox[1][j] = (float)v->W[j];
	for (i=1;i<3;i++) {
		v = vertices[i].ptr;
		for (j=0;j<3;j++) {
			t = (float)v->W[j];
			if (t < bbox[0][j])
				bbox[0][j] = t;
			if (t > bbox[1][j])
				bbox[1][j] = t;
		}
	}

	for (j=0;j<3;j++) {
		float t = 1.0e-5f * (float)fabs(bbox[0][j] + bbox[0][j]);
		bbox[0][j] -= t;
		bbox[1][j] += t;
		// Mask off the bottom 3 bits of the floating point
		// number to avoid rounding errors (I'm seeing one
		// bit difference at the bottom leading to errors).
		// *((int *)&bbox[0][j]) &= 0xFFFFFFF8;
	}
}

PolyTri::PolyTri(MVector3 *iW, MVector3 *iP, MVector3 *iU, MVector3 *iN, int i0, int i1, int i2,
				 void *iprim, void *idata)
{
	MVector3 W[3], P[3], N[3], U[3];
	W[0] = iW[i0]; W[1] = iW[i1]; W[2] = iW[i2];
	P[0] = iP[i0]; P[1] = iP[i1]; P[2] = iP[i2];
	if (iU != NULL) {
		U[0] = iU[i0]; U[1] = iU[i1]; U[2] = iU[i2];
	} else {
		U[0] = W[0]; U[1] = W[1]; U[2] = W[2];
	}
	MVector3 tN = W[2] - W[0];
	MVector3 B = W[1] - W[0];
	tN = tN.crossProduct(B);
	tN.normalize();
	plane.set(W[0], tN);
	// compute_indices(tN);
	if (iN != NULL) {
		N[0] = iN[i0]; N[1] = iN[i1]; N[2] = iN[i2];
	} else {
		N[0] = N[1] = N[2] = tN;
	}

	int i;
	for (i=0;i<3;i++) {
		vertices[i].ptr = new TriVertex(W[i], P[i], N[i], U[i]);
	}

#if 0
	PolyTriPtr tri = this;
	sprintf(sbuf, "W (%.3g,%.3g,%.3g) (%.3g,%.3g,%.3g) (%.3g,%.3g,%.3g)\n",
			tri->vertices[0].ptr->W[0], tri->vertices[0].ptr->W[1], tri->vertices[0].ptr->W[2],
			tri->vertices[1].ptr->W[0], tri->vertices[1].ptr->W[1], tri->vertices[1].ptr->W[2], 
			tri->vertices[2].ptr->W[0], tri->vertices[2].ptr->W[1], tri->vertices[2].ptr->W[2]);
	OutputDebugString(sbuf);
	sprintf(sbuf, "N (%.3g,%.3g,%.3g) (%.3g,%.3g,%.3g) (%.3g,%.3g,%.3g)\n",
			tri->vertices[0].ptr->N[0], tri->vertices[0].ptr->N[1], tri->vertices[0].ptr->N[2],
			tri->vertices[1].ptr->N[0], tri->vertices[1].ptr->N[1], tri->vertices[1].ptr->N[2], 
			tri->vertices[2].ptr->N[0], tri->vertices[2].ptr->N[1], tri->vertices[2].ptr->N[2]);
	OutputDebugString(sbuf);
#endif

	adjacent[0] = adjacent[1] = adjacent[2] = NULL;
	containment_value = unknown;
	prim = iprim;
	data = idata;
	// compute_bbox();
#if defined(_DEBUG) && 0
	// Ensure we don't have any degenerate PolyTris
	assert((fabs(W[0][0] - W[1][0]) > EPSILON ||
			fabs(W[0][1] - W[1][1]) > EPSILON ||
			fabs(W[0][2] - W[1][2]) > EPSILON) &&
		   (fabs(W[0][0] - W[2][0]) > EPSILON ||
			fabs(W[0][1] - W[2][1]) > EPSILON ||
			fabs(W[0][2] - W[2][2]) > EPSILON) &&
		   (fabs(W[1][0] - W[2][0]) > EPSILON ||
			fabs(W[1][1] - W[2][1]) > EPSILON ||
			fabs(W[1][2] - W[2][2]) > EPSILON));
#endif

#if defined(_DEBUG) && 0
sprintf(sbuf, "Tri: <%g,%g,%g> <%g,%g,%g> <%g,%g,%g>\n",
		W[0][0], W[0][1], W[0][2],
		W[1][0], W[1][1], W[1][2],
		W[2][0], W[2][1], W[2][2]);
OutputDebugString(sbuf);
#endif
}

PolyTri::PolyTri(MVector3 *iW, MVector3 *iP, MVector3 *iU, MVector3 *iN, void *iprim, void *idata)
{
	int i;
	MVector3 W[3], P[3], N[3], U[3];

	W[0] = iW[0]; W[1] = iW[1]; W[2] = iW[2];
	P[0] = iP[0]; P[1] = iP[1]; P[2] = iP[2];
	if (iU != NULL) {
		U[0] = iU[0]; U[1] = iU[1]; U[2] = iU[2];
	} else {
		U[0] = W[0]; U[1] = W[1]; U[2] = W[2];
	}
	MVector3 tN = W[2] - W[0];
	MVector3 B = W[1] - W[0];
	tN = tN.crossProduct(B);
	tN.normalize();
	plane.set(W[0], tN);
	// compute_indices(tN);
	if (iN != NULL) {
		N[0] = iN[0]; N[1] = iN[1]; N[2] = iN[2];
	} else {
		N[0] = N[1] = N[2] = tN;
	}

	for (i=0;i<3;i++) {
		vertices[i].ptr = new TriVertex(W[i], P[i], N[i], U[i]);
	}

#if 0
	PolyTriPtr tri = this;
	sprintf(sbuf, "   %p (%.3g,%.3g,%.3g) (%.3g,%.3g,%.3g) (%.3g,%.3g,%.3g)\n",
			tri,
			tri->vertices[0].ptr->W[0], tri->vertices[0].ptr->W[1], tri->vertices[0].ptr->W[2],
			tri->vertices[1].ptr->W[0], tri->vertices[1].ptr->W[1], tri->vertices[1].ptr->W[2], 
			tri->vertices[2].ptr->W[0], tri->vertices[2].ptr->W[1], tri->vertices[2].ptr->W[2]);
	OutputDebugString(sbuf);
#endif

	adjacent[0] = adjacent[1] = adjacent[2] = NULL;
	containment_value = unknown;
	prim = iprim;
	data = idata;
	// compute_bbox();
}

PolyTri::
PolyTri(TriVertex *v0, TriVertex *v1, TriVertex *v2, void *iprim, void *idata)
{
	vertices[0].ptr = new TriVertex(v0);
	vertices[1].ptr = new TriVertex(v1);
	vertices[2].ptr = new TriVertex(v2);

	MVector3 tN = v2->W - v0->W;
	MVector3 B = v1->W - v0->W;
	tN = tN.crossProduct(B);
	tN.normalize();
	plane.set(v0->W, tN);

	// The pointers to adjacent triangles must be rebuilt.  Otherwise, as we are creating
	// a clone of a surface, each triangle will end up pointing back to the original for
	// it's adjacency information.
	adjacent[0] = adjacent[1] = adjacent[2] = NULL;
	containment_value = unknown;

	prim = iprim;
	data = idata;
}

PolyTri::PolyTri(PolyTri *tri)
{
	for (int i=0;i<3;i++) {
		vertices[i].ptr = new TriVertex(tri->vertices[i].ptr);
	}
	plane = tri->plane;
	// i1 = tri->i1;
	// i2 = tri->i2;
	// The pointers to adjacent triangles must be rebuilt.  Otherwise, as we are creating
	// a clone of a surface, each triangle will end up pointing back to the original for
	// it's adjacency information.
	adjacent[0] = adjacent[1] = adjacent[2] = NULL;
	containment_value = unknown;
	prim = tri->prim;
	data = tri->data;
	// bbox = tri->bbox;
}

// compute the PolyTri normal based on clockwise vertex order
MVector3 PolyTri::Normal(void) const
{
	MVector3 N = vertices[2].ptr->W - vertices[0].ptr->W;
	MVector3 B = vertices[1].ptr->W - vertices[0].ptr->W;
	N = N.crossProduct(B);
	N.normalize();
	return N;
}

//	check polygon for containment of a coplanar point
//	based on code by Eric Haines from Graphics Gems IV
bool PolyTri::Contains(const MVector3 &pt) const
{
	int axis, x, y;
	float axisVal = plane.A;
	// Determine major axis
	axis = 0;
	if (fabs(plane.C) > fabs(plane.A)) {
		axis = 1;
		axisVal = plane.B;
	}
	if (fabs(plane.C) > fabs(axisVal)) {
		axis = 2;
		axisVal = plane.C;
	}
	
	switch (axis) {
	case 0:	x = 1; y = 2; break;
	case 1:	x = 2; y = 0; break;
	case 2:	x = 0; y = 1; break;
	}

	float tx = pt[x], ty = pt[y];
	MVector3 p1 = vertices[2].ptr->W;
	MVector3 p2 = vertices[0].ptr->W;

	//	check to see which side of the test point the first point is on
	int yflag0 = (p1[y] >= ty),
	inside = FALSE;
	//	start with inside false
	for (int i=1;i<3;i++) {
		int	yflag1 = (p2[y] >= ty);
		if (yflag0 != yflag1) {
			//	if the points aren't on the same side
			//	check which side of the test coordinate the start point is on
			int	xflag0 = (p1[x] >= tx);
			if (xflag0 == (p2[x] >= tx)) {
				//	if the edge is all the way to one side of the test point
				//	begin
				if (xflag0)
					//	if the start point is greater than the test point
					inside = !inside;
			} else {
				//	otherwise, the edge spans the test point on both axes
				//	if the intersection of the tx axis is on the right side of the test point
				if ((p2[x] - (p2[y] - ty) * (p1[x] - p2[x]) / (p1[y] - p2[y])) >= tx)
					//	increment the crossing count
					inside = !inside;
			}
		}
		yflag0 = yflag1;
		p1 = p2;
		p2 = vertices[i].ptr->W;
	}
	return (inside != 0);
}

// Reverse the orientation of the PolyTri.  This is accomplished by swapping
// the second and third vertices
void PolyTri::Invert(void)
{
	TriVertex *tv = vertices[1].ptr;
	vertices[1].ptr = vertices[2].ptr;
	vertices[2].ptr = tv;
	vertices[0].ptr->N *= -1.0f;
	vertices[1].ptr->N *= -1.0f;
	vertices[2].ptr->N *= -1.0f;

	plane.A *= -1.0;
	plane.B *= -1.0;
	plane.C *= -1.0;
}

void PolyTri::InvertNormals(void)
{
	vertices[0].ptr->N *= -1.0;
	vertices[1].ptr->N *= -1.0;
	vertices[2].ptr->N *= -1.0;
}

#define USE_EPSILON_TEST TRUE

/* this edge to edge test is based on Franlin Antonio's gem:
   "Faster Line Segment Intersection", in Graphics Gems III,
   pp. 199-202 */
#define EDGE_EDGE_TEST(V0,U0,U1)                      \
  Bx=U0[i0]-U1[i0];                                   \
  By=U0[i1]-U1[i1];                                   \
  Cx=V0[i0]-U0[i0];                                   \
  Cy=V0[i1]-U0[i1];                                   \
  f=Ay*Bx-Ax*By;                                      \
  d=By*Cx-Bx*Cy;                                      \
  if((f>0 && d>=0 && d<=f) || (f<0 && d<=0 && d>=f))  \
  {                                                   \
    e=Ax*Cy-Ay*Cx;                                    \
    if(f>0)                                           \
    {                                                 \
      if(e>=0 && e<=f) return 1;                      \
    }                                                 \
    else                                              \
    {                                                 \
      if(e<=0 && e>=f) return 1;                      \
    }                                                 \
  }

#define EDGE_AGAINST_TRI_EDGES(V0,V1,U0,U1,U2) \
{                                              \
  float Ax,Ay,Bx,By,Cx,Cy,e,d,f;               \
  Ax=V1[i0]-V0[i0];                            \
  Ay=V1[i1]-V0[i1];                            \
  /* test edge U0,U1 against V0,V1 */          \
  EDGE_EDGE_TEST(V0,U0,U1);                    \
  /* test edge U1,U2 against V0,V1 */          \
  EDGE_EDGE_TEST(V0,U1,U2);                    \
  /* test edge U2,U1 against V0,V1 */          \
  EDGE_EDGE_TEST(V0,U2,U0);                    \
}

#define POINT_IN_TRI(V0,U0,U1,U2)           \
{                                           \
  float a,b,c,d0,d1,d2;                     \
  /* is T1 completly inside T2? */          \
  /* check if V0 is inside tri(U0,U1,U2) */ \
  a=U1[i1]-U0[i1];                          \
  b=-(U1[i0]-U0[i0]);                       \
  c=-a*U0[i0]-b*U0[i1];                     \
  d0=a*V0[i0]+b*V0[i1]+c;                   \
                                            \
  a=U2[i1]-U1[i1];                          \
  b=-(U2[i0]-U1[i0]);                       \
  c=-a*U1[i0]-b*U1[i1];                     \
  d1=a*V0[i0]+b*V0[i1]+c;                   \
                                            \
  a=U0[i1]-U2[i1];                          \
  b=-(U0[i0]-U2[i0]);                       \
  c=-a*U2[i0]-b*U2[i1];                     \
  d2=a*V0[i0]+b*V0[i1]+c;                   \
  if(d0*d1>0.0)                             \
  {                                         \
    if(d0*d2>0.0) return 1;                 \
  }                                         \
}

static int
coplanar_tri_tri(MVector3 &N, MVector3 &V0, MVector3 &V1, MVector3 &V2, MVector3 &U0, MVector3 &U1, MVector3 &U2)
{
   MVector3 A;
   short i0,i1;
   /* first project onto an axis-aligned plane, that maximizes the area */
   /* of the triangles, compute indices: i0,i1. */
   A[0] = (float)fabs(N[0]);
   A[1] = (float)fabs(N[1]);
   A[2] = (float)fabs(N[2]);
   if(A[0]>A[1])
   {
      if(A[0]>A[2])
      {
          i0=1;      /* A[0] is greatest */
          i1=2;
      }
      else
      {
          i0=0;      /* A[2] is greatest */
          i1=1;
      }
   }
   else   /* A[0]<=A[1] */
   {
      if(A[2]>A[1])
      {
          i0=0;      /* A[2] is greatest */
          i1=1;
      }
      else
      {
          i0=0;      /* A[1] is greatest */
          i1=2;
      }
    }

    /* test all edges of triangle 1 against the edges of triangle 2 */
    EDGE_AGAINST_TRI_EDGES(V0,V1,U0,U1,U2);
    EDGE_AGAINST_TRI_EDGES(V1,V2,U0,U1,U2);
    EDGE_AGAINST_TRI_EDGES(V2,V0,U0,U1,U2);

    /* finally, test if tri1 is totally contained in tri2 or vice versa */
    POINT_IN_TRI(V0,U0,U1,U2);
    POINT_IN_TRI(U0,V0,V1,V2);

    return 0;
}

#define ISECT(VV0,VV1,VV2,D0,D1,D2,isect0,isect1) \
              isect0 = VV0+(VV1-VV0)*D0/(D0-D1);    \
              isect1 = VV0+(VV2-VV0)*D0/(D0-D2);

static bool
compute_intervals(float VV0, float VV1, float VV2, float D0, float D1, float D2,
				  float D0D1, float D0D2, float &isect0, float &isect1)
{
	if(D0D1>0.0f) {
		// here we know that D0D2<=0.0 */
		// that is D0, D1 are on the same side, D2 on the other or on the plane
		ISECT(VV2,VV0,VV1,D2,D0,D1,isect0,isect1);
	} else if(D0D2>0.0f) {
		// here we know that d0d1<=0.0
		ISECT(VV1,VV0,VV2,D1,D0,D2,isect0,isect1);
	} else if(D1*D2>0.0f || D0!=0.0f) {
		// here we know that d0d1<=0.0 or that D0!=0.0
		ISECT(VV0,VV1,VV2,D0,D1,D2,isect0,isect1);
	} else if(D1!=0.0f) {
		ISECT(VV1,VV0,VV2,D1,D0,D2,isect0,isect1);
	} else if(D2!=0.0f) {
		ISECT(VV2,VV0,VV1,D2,D0,D1,isect0,isect1);
	} else {
		// triangles are coplanar
		return false;
		// return coplanar_tri_tri(N1,V0,V1,V2,U0,U1,U2);
	}
	
	return true;
}

//	compute the distance at which the ray intersects the polygon
bool PolyTri::intersect(const PolyTri *tri)
{
	MVector3 E1, E2;
	MVector3 N1, N2, d1, d2;
	float du0,du1,du2,dv0,dv1,dv2;
	MVector3 D;
	float isect1[2], isect2[2];
	float du0du1,du0du2,dv0dv1,dv0dv2;
	short index;
	float vp0,vp1,vp2;
	float up0,up1,up2;
	float b,c,max;
	
	// compute signed distances from input triangle to the plane of this triangle
	du0 = tri->vertices[0].ptr->W * plane;
	du1 = tri->vertices[1].ptr->W * plane;
	du2 = tri->vertices[2].ptr->W * plane;
	
	// coplanarity robustness check
#if USE_EPSILON_TEST == TRUE
	if (fabs(du0) < EPSILON) du0=0.0f;
	if (fabs(du1) < EPSILON) du1=0.0f;
	if (fabs(du2) < EPSILON) du2=0.0f;
#endif
	du0du1 = du0 * du1;
	du0du2 = du0 * du2;
	
	if(du0du1 > 0.0f && du0du2 > 0.0f) // same sign on all of them + not equal 0 ?
		return false;                      // no intersection occurs */

	// compute signed distances from this triangle to the plane of the input triangle
	dv0 = vertices[0].ptr->W * tri->plane;
	dv1 = vertices[1].ptr->W * tri->plane;
	dv2 = vertices[2].ptr->W * tri->plane;

#if USE_EPSILON_TEST == TRUE
	if (fabs(dv0) < EPSILON) dv0 = 0.0f;
	if (fabs(dv1) < EPSILON) dv1 = 0.0f;
	if (fabs(dv2) < EPSILON) dv2 = 0.0f;
#endif

	dv0dv1=dv0*dv1;
	dv0dv2=dv0*dv2;

	if(dv0dv1>0.0f && dv0dv2>0.0f) // same sign on all of them + not equal 0 ?
		return false;                    // no intersection occurs

	// compute direction of intersection line
	N1.set(plane.A, plane.B, plane.C);
	N2.set(tri->plane.A, tri->plane.B, tri->plane.C);
	D = N1.crossProduct(N2);

	// compute and index to the largest component of D
	index = 0;
	max   = (float)fabs(D[0]);
	b     = (float)fabs(D[1]);
	c     = (float)fabs(D[2]);
	if (b > max) {
		max = b;
		index = 1;
	}
	if (c > max) {
		max = c;
		index = 2;
	}

	// this is the simplified projection onto L
	vp0 = vertices[0].ptr->W[index];
	vp1 = vertices[1].ptr->W[index];
	vp2 = vertices[2].ptr->W[index];

	up0 = tri->vertices[0].ptr->W[index];
	up1 = tri->vertices[1].ptr->W[index];
	up2 = tri->vertices[2].ptr->W[index];

	// compute interval for triangle 1
	if (!compute_intervals(vp0,vp1,vp2,dv0,dv1,dv2,dv0dv1,dv0dv2,isect1[0],isect1[1]))
		return false;

	// compute interval for triangle 2
	if (!compute_intervals(up0,up1,up2,du0,du1,du2,du0du1,du0du2,isect2[0],isect2[1]))
		return false;

	if (isect1[0] > isect1[1]) {
		b = isect1[0]; isect1[0] = isect1[1]; isect1[1] = b;
	}
	if (isect2[0] > isect2[1]) {
		b = isect2[0]; isect2[0] = isect2[1]; isect2[1] = b;
	}

	if (isect1[1] < isect2[0] ||
		isect2[1] < isect1[0])
		return false;
	else
		return true;
}

//	compute the distance at which the ray intersects the polygon
float PolyTri::intersect(const MRay &r) const
{
	float t, u, v;
	float det, inv_det;
	MVector3 edge1, edge2;
	MVector3 tvec, pvec, qvec;
	TriVertex *v0 = vertices[0].ptr;
	TriVertex *v1 = vertices[1].ptr;
	TriVertex *v2 = vertices[2].ptr;

	// find vectors for two edges sharing vert0
	edge1 = v1->W; edge1 -= v0->W;
	edge2 = v2->W; edge2 -= v1->W;

	// begin calculating determinant  also used to calculate U parameter
	pvec = r.Dir;
	pvec = pvec.crossProduct(edge2);

	// if determinant is near zero, ray lies in plane of triangle
	det = edge1 * pvec;

#ifdef TEST_CULL // define TEST_CULL if culling is desired
	if (det < EPSILON) 
		return -INFINITY; 

	// calculate distance from vert0 to ray origin
	tvec = orig; tvec -= vert0;
	// calculate U parameter and test bounds
	u = tvec * pvec;
	if (u < 0.0 || u > det)
		return -INFINITY;

	// prepare to test V parameter
	qvec = tvec ^ edge1;
	// calculate V parameter and test bounds
	v = dir * qvec;
	if (v < 0.0 || u * v > det)
		return -INFINITY;

	// calculate t, scale parameters, ray intersects triangle
	t = edge2 * qvec;
	inv_det = 1.0 / det; 
	t *= inv_det; 
	u *= inv_det; 
	v *= inv_det; 
#else // the nonculling branch
	if (det > -EPSILON && det < EPSILON)
		return -INFINITY; 
	inv_det = 1.0f / det; 
	// calculate distance from vert0 to ray origin
	tvec = r.Org;
	tvec -= v0->W;
	// calculate U parameter and test bounds
	u = (tvec * pvec) * inv_det;
	if (u < 0.0 || u > 1.0)
		return -INFINITY;
	// calculate V parameter and test bounds
	qvec = tvec.crossProduct(edge1);
	v = (r.Dir * qvec) * inv_det;
	if (v < 0.0 || u + v > 1.0)
		return -INFINITY;
	// calculate t, ray intersects triangle
	t = (edge2 * qvec) * inv_det;
#endif

	return t; 
}

void PolyTri::FreeVertices() {
	delete vertices[0].ptr;
	delete vertices[1].ptr;
	delete vertices[2].ptr;
}

PolyTriList::~PolyTriList(void)
{
	iterator itr;
	for (itr=begin();itr!=end();itr++)
		delete (*itr);
	clear();

	VertexSet::iterator vitr;
	for (vitr=vertices.begin();vitr!=vertices.end();vitr++) {
		delete (*vitr).ptr;
	}
	vertices.clear();
}

void PolyTriList::Push(PolyTriPtr tri)
{
	std::pair< VertexSetPtr, bool > insert_result;
	for (int i=0;i<3;i++) {
		TriVertex *v = tri->vertices[i].ptr;
		insert_result = vertices.insert(v);
		if (!insert_result.second) {
			// We already have this vertex, don't need to store a second copy.
			// Check to see if we are actually referencing the same vertex or
			// a distinct copy.  If it is really distinct (as determined by
			// comparison of pointer values), then free it's memory and adjust
			// the triangle to use the old 
			TriVertex *v2 = (*insert_result.first).ptr;
			if (v != v2) {
				delete v;
			}
			// Update tri to point to the stored vertex information
			tri->vertices[i].ptr = v2;
		}
	}

	// Check for degenerate triangle
	if ((*tri->vertices[0].ptr == *tri->vertices[1].ptr) ||
		(*tri->vertices[0].ptr == *tri->vertices[2].ptr) ||
		(*tri->vertices[1].ptr == *tri->vertices[2].ptr)) {
		delete tri;
	} else {
		// Add the triangle to the list
		push_front(tri);
	}
}

void PolyTriList::Invert()
{
	iterator itr;
	for (itr=begin();itr!=end();itr++)
		(*itr)->Invert();
}

//	pop the head element
PolyTriPtr PolyTriList::Pop(void)
{
	if (empty())
		return NULL;

	PolyTriPtr result = front();
	pop_front();

	// Create new storage for the vertices.  We don't know where this triangle
	// will end up, so we don't want this list to delete the storage associated
	// with the vertices while they are in use.
	for (int i=0;i<3;i++) {
		TriVertex *v = result->vertices[i].ptr;
		result->vertices[i].ptr = new TriVertex(v);
	}

	return result;
}

void PolyTriList::
ComputeAdjacencies()
{
	if (size() < 2) {
		// Can't have reused vertices unless there are two or more triangles that
		// use them.
		return;
	}

	int unique_vertices = vertices.size();
	sprintf(sbuf, "Start verts: %d, unique: %d\n", 3 * size(), unique_vertices);
	OutputDebugString(sbuf);
}

void PolyTriList::FreeVertices()
{
	VertexSet::iterator vitr;
	for (vitr=vertices.begin();vitr!=vertices.end();vitr++) {
		TriVertex *v = (*vitr).ptr;
		delete v;
	}
	vertices.clear();
}


// Split a PolyTri against a plane.  It is possible to have 0, 1, or 2 PolyTris
// come out of either side of the split (well, these are the possible combos:
//    (0,1), (1, 0), (1,1), (1,2), (2,1)
tri23tree::split_result tri23tree::
Split3D(const PolyTriPtr tri, const MPlane &plane,
		PolyTriPtr &in0, PolyTriPtr &in1, PolyTriPtr &out0, PolyTriPtr &out1)
{
	// Storage for the vertices of PolyTris outside the plane
	TriVertex outV[4];
	// Storage for the vertices of any PolyTris inside the plane
	TriVertex inV[4];
	// Number of vertices on each side of the plane
	int out_c = 0, in_c = 0;
	// Current vertices and their relationship to the plane
	MVector3  ptA, ptB, v;
	float sideA, sideB;
	// Start by assuming the PolyTri is coplanar with the plane. (This
	// is really just to provide a flag, since the coplanar case is
	// much less frequent than the others.)
	split_result poly_class = coplanar;

	// Grab the last point in the PolyTri so we can start with the
	// edge from v2 to v0.  This makes the for() loop below easier to
	// manage.  The first thing we do is see what side of the plane
	// this point is lying on.
	TriVertex *v0, *v1;
	v0 = tri->vertices[2].ptr;
	ptA = v0->W;
	sideA = ptA * plane;

	//	loop on the points
	int i, j;
	for (i=0,j=2;i<3;j=i,i++) {
		v1  = tri->vertices[i].ptr;
		ptB = v1->W;
		// See which side of the plane this point lies on. (Or possibly in.)
		sideB = ptB * plane;
		//	if the current point is on the positive side
		if (sideB > EPSILON) {
			if (poly_class == coplanar)
				// If the polygon classification is currently coplanar, then
				// we can classify the entire polygon as outside
				poly_class = outside;
			else if (poly_class != outside)
				// The PolyTri must have a vertex on the other side of the plane.
				// This means that the result needs to be split.
				poly_class = splitting;
			if (sideA < -EPSILON) {
				// If the previous vertex was on the opposite side of the plane
				// then we make a split on the current edge.
				// v is the the ratio of the split point along the edge.  It is
				// also used to linearly interpolate the normals and u/v coordinates.
				v = ptB - ptA;
				float d = -(ptA * plane) / (v * plane.normal());
				assert((d >= 0.0f) && (d <= 1.0f));
				// add the interpolated point to the partitions
				outV[out_c].lerp(d, v0, v1);
				inV[in_c] = outV[out_c];
				out_c++;
				in_c++;
				//	set the poly_class appropriately
				poly_class = splitting;
			}
			//	add the current point to the positive partition
			outV[out_c] = v1;  // Should this be a copy?
			out_c++;
		//	the current point is on the negative side
		} else if (sideB < -EPSILON) {
			//	if the polygon classification is on
			if (poly_class == coplanar)
				//	classify the polygon as in
				poly_class = inside;
			//	else if the polygon classification is not in
			else if (poly_class != inside)
				//	set the polygon classification to spanning
				poly_class = splitting;
			//	if the previous point was on the opposite side of the plane
			if (sideA > EPSILON) {
				// compute the ratio of the split point along the edge
				v = ptB - ptA;
				float d = -(ptA * plane) / (v * plane.normal());
				assert((d >= 0.0f) && (d <= 1.0f));
				// add the interpolated point to the partitions
				outV[out_c].lerp(d, v0, v1);
				inV[in_c] = outV[out_c];
				out_c++;
				in_c++;
				//	set the poly_class appropriately
				poly_class = splitting;
			}

			//	add the current point to the negative partition
			inV[in_c] = v1;
			in_c++;
		//	the current point is on the plane
		} else {
			//	add the current point to the partitions
			outV[out_c] = v1;
			inV[in_c]   = v1;
			out_c++;
			in_c++;
		}

		// Set point a to the same as point b (TBD: is it worth making
		// a check to see if we are done before doing this?  It would
		// save several assignment statements (a vector has 3) at the
		// cost of an if (i == 2) { } ).
		if (i < 2) {
			v0    = v1;
			ptA   = ptB;
			sideA = sideB;
		}
	}

	// Create the output PolyTris based on how the classification of the
	// input PolyTri and the number of vertices in the in/out vertex buckets.
	switch (poly_class) {
	case outside:
		// The input PolyTri is entirely outside the plane - just send it
		// back unchanged.
		out0 = tri;
		out1 = NULL;
		break;
	case inside:
		// The input PolyTri is entirely inside the plane.  Send it back.
		in0 = tri;
		in1 = NULL;
		break;
	case splitting:
		// The input PolyTri is chopped into pieces by the plane.  Create
		// PolyTris out of the vertices on the two sides.
		out0 = new PolyTri(&outV[0], &outV[1], &outV[2], tri->prim);
		if (out_c == 4) {
			out1 = new PolyTri(&outV[0], &outV[2], &outV[3], tri->prim);
		} else
			out1 = NULL;
		in0 = new PolyTri(&inV[0], &inV[1], &inV[2], tri->prim);
		if (in_c == 4) {
			in1 = new PolyTri(&inV[0], &inV[2], &inV[3], tri->prim);
		} else
			in1 = NULL;
		tri->FreeVertices();
		delete tri;
		break;
	case coplanar:
		// The PolyTri is coplanar with the plane.  Don't do anything.
		// TBD: whatever code is calling Split3D() should probably look
		// at this result and see if there are actions that need to be
		// performed in 2D.  For example, if this PolyTri is being
		// split against another one, then a splitting based on edges
		// could be performed.
		// assert(false);
		break;
	}

	//	return the classification
	return poly_class;
}

tri23tree::~tri23tree()
{
	state = deleting;
	shread_two_three(Root);
	Root = NULL;
}

void tri23tree::cleanup(void *data)
{
	if (state == deleting) {
		PolyTriPtr tri = (PolyTriPtr)data;
		delete tri->vertices[0].ptr;
		delete tri->vertices[1].ptr;
		delete tri->vertices[2].ptr;
		delete tri;
	} else if (state == extracting) {
	} else {
		assert(state == unlinking);
	}
}

void tri23tree::Insert(PolyTriPtr tri)
{
	Range3D bbox;

	state = inserting;
	tri->compute_bbox(bbox);
	Range_Tree::Insert(bbox, (void *)tri);
}

void tri23tree::Delete(PolyTriPtr tri)
{
	Range3D bbox;

	state = deleting;
	tri->compute_bbox(bbox);
	Range_Tree::Delete(bbox, (void *)tri);
}

void tri23tree::Insert(PolyTriList &tris)
{
	Range3D bbox;
	PolyTriList::iterator triitr;
	PolyTriPtr tri;

	state = inserting;
	while (!tris.empty()) {
		// tri = tris.front();
		// tris.pop_front();
		tri = tris.Pop();
		tri->compute_bbox(bbox);
		Range_Tree::Insert(bbox, (void *)tri);
	}
	tris.FreeVertices();
}

void tri23tree::Delete(PolyTriList &tris)
{
	Range3D bbox;
	PolyTriList::iterator triitr;
	PolyTriPtr tri;

	state = deleting;
	while ((tri = tris.Pop()) != NULL) {
		tri->compute_bbox(bbox);
		Range_Tree::Delete(bbox, (void *)tri);
	}
}

// By the time we get to this check, we have already verified that the bounding
// box of the PolyTri intersects the bounding box of any PolyTris at this
// point in the 2-3 tree.  Next step is to do a PolyTri-PolyTri intersection
// test.
void tri23tree::check_PolyTri_intersections(range_data *data_list, PolyTriPtr tri)
{
	while (data_list != NULL) {
		PolyTriPtr tri0 = (PolyTriPtr)data_list->data;
		if (tri->intersect(tri0)) {
			result_list.Push(tri0);
		}
		data_list = data_list->next;
	}
}

// data_list contains PolyTris in this tree that (potentially) intersect
// PolyTris in the input tree.  For each one of them, we will find PolyTris
// in the input tree that intersect them, chop those PolyTris and put the
// chopped PolyTris back into the input tree (removing the previously
// unchopped PolyTris).
void tri23tree::slice_tree(range_data *data_list, tri23tree *tree)
{
	PolyTriList chopped_tris;
	PolyTriPtr it0, it1, ot0, ot1;
	PolyTriList::iterator itr;

	for (;data_list != NULL;data_list=data_list->next) {
		PolyTriPtr tri0 = (PolyTriPtr)data_list->data;
		MPlane P = tri0->Plane();
		tree->FindSlicers(tri0);

		if (tree->result_list.empty())
			continue;

		// Remove the PolyTris that need chopping from the tree
		tree->Unlink(tree->result_list);

		// Slice them one at a time
		split_result sresult;
		while (!tree->result_list.empty()) {
			PolyTriPtr tri1 = tree->result_list.Pop();

			sresult = Split3D(tri1, P, it0, it1, ot0, ot1);
			switch (sresult) {
			case coplanar:
				chopped_tris.Push(tri1);
				break;
			case inside:
				chopped_tris.Push(it0);
				assert(it1 == NULL);
				break;
			case outside:
				chopped_tris.Push(ot0);
				assert(ot1 == NULL);
				// Neither of these should occur - it means that the PolyTri was
				// entirely to the side of the cutting plane, contrary to the notion
				// that tree->result_list only contains PolyTris that intersect tri0.
				// assert(false);
				break;
			case splitting:
				chopped_tris.Push(it0);
				if (it1 != NULL)
					chopped_tris.Push(it1);
				chopped_tris.Push(ot0);
				if (ot1 != NULL)
					chopped_tris.Push(ot1);
				break;
			}
		}

		// Step through all the chopped PolyTris and add them back to the input tree
		tree->Insert(chopped_tris);
	}
}

void tri23tree::collect_PolyTris(Range3D &range, range_data *data_list, PolyTriList *list)
{
	if (data_list->next == NULL) {
		// Only one PolyTri here - we don't have to do extra range checks
		list->Push((PolyTriPtr)data_list->data);
	} else {
		PolyTriPtr tri0;
		Range3D bbox;

		while (data_list != NULL) {
			tri0 = (PolyTriPtr)data_list->data;
			tri0->compute_bbox(bbox);
			if (range.Intersects(bbox))
				list->Push(tri0);
			data_list = data_list->next;
		}
	}
}

void tri23tree::draw_triangle(range_data *data_list)
{
#if 0
	// This routine really only makes sense for a standalone application.  When
	// this code is in Polyray, it is disabled
	extern void DrawTriangle(PolyTriPtr tri);

	while (data_list != NULL) {
		PolyTriPtr tri0 = (PolyTriPtr)data_list->data;
		DrawTriangle(tri0);
		data_list = data_list->next;
	}
#endif
}

#if 1
// Timing tests seem to indicate that this definition of ZIntersectList
// is comparable with the array based one when using the SGI version of
// STL.  It is a little slower when using the MS version of STL.
class ZIntersectList : public std::list< MVector3 > {
public:
	ZIntersectList() {
		count = 0;
		eps = 1.0e-7f;
	}

	float eps;
	int count;

	void AddPoint(MVector3 &V) {
		iterator itr;
		MVector3 p;
		for (itr=begin();itr!=end();itr++) {
			p = *itr;
			p -= V;
			if ((p * p) < eps) {
				return;
			}
		}
		push_back(V);
		count++;
	}

	int Count() {
		return count;
	}
};
#else
class ZIntersectList {
public:
	ZIntersectList() {
		count = 0;
		eps = 1.0e-7f;
	}

	MVector3 vals[16];
	float eps;
	int count;

	void AddPoint(MVector3 &V) {
		int i;;
		MVector3 p;
		for (i=0;i<count;i++) {
			p = vals[i];
			p -= V;
			if ((p * p) < eps) {
				return;
			}
		}
		vals[count] = V;
		count++;
	}

	int Count() {
		return count;
	}
};
#endif

// This is a specialized intersection routine for a ray that is
// known to be parallel to the z-axis.  This should be faster than
// the more general one in the PolyTri class.
bool tri23tree::zintersect(MVector3 &P, PolyTriPtr tri, MVector3 &W)
{
#if 0
	// For testing purposes, we can use the general intersection
	// routine for PolyTris.
	Ray r;
	r.P = P;
	r.D.Set(0, 0, 1.0);
	r.D.Normalize();
	float t = tri->intersect(r);
	if (t <= 0) // tri->intersect() will return -INFINITY if it fails
		return false;
	else
		return true;
#else
	float u0, u1, u2;
	float v0, v1, v2;
	int i1, i2;
	tri->compute_indices(i1, i2);

	// First find the point of intersection with the
	// plane of tri.
	float t0 = tri->Plane().C;
	if (fabs(t0) < EPSILON)
		// Ray is perpendicular to the PolyTri
		return false;

	// Find the distance from the point P to the plane the PolyTri
	// lies in.
	t0 = -(P * tri->Plane()) / t0;
	if (t0 <= EPSILON)
		// Point of intersection is behind the starting point
		return false;

	// Compute the point of intersection of the ray with the plane.
	W.set(P[0], P[1], P[2] + t0);

	// Now let's see if the intersection point is inside the PolyTri
	// (using a barycentric approach.
	u0 = W[i1] - tri->vertices[0].ptr->W[i1];
	v0 = W[i2] - tri->vertices[0].ptr->W[i2];
	u1 = tri->vertices[1].ptr->W[i1] - tri->vertices[0].ptr->W[i1];
	v1 = tri->vertices[1].ptr->W[i2] - tri->vertices[0].ptr->W[i2];
	u2 = tri->vertices[2].ptr->W[i1] - tri->vertices[0].ptr->W[i1];
	v2 = tri->vertices[2].ptr->W[i2] - tri->vertices[0].ptr->W[i2];

	float a, b;
	if (fabs(u1) < EPSILON) {
		b = u0 / u2;
		if ((b >= 0.0) && (b <= 1.0))
			a = (v0 - b * v2) / v1;
		else
			return false;
	} else {
		b = (v0 * u1 - u0 * v1) / (v2 * u1 - u2 * v1);
		if ((b >= 0.0) && (b <= 1.0))
			a = (u0 - b * u2) / u1;
		else
			return false;
	}

	if ((a >= 0.0) && (a + b <= 1.0))
		return true;
	else
		return false;
#endif
}

void tri23tree::process(Range3D &range, void *value, void *data)
{
	if (state == finding_slicers) {
		check_PolyTri_intersections((range_data *)value, (PolyTriPtr)data);
	} else if (state == slicing) {
		// We are slicing another tree of PolyTris with the current PolyTris
		slice_tree((range_data *)value, (tri23tree *)data);
	} else if (state == drawing) {
		draw_triangle((range_data *)value);
	} else if (state == collecting) {
		collect_PolyTris(range, (range_data *)value, (PolyTriList *)data);
	} else if (state == counting_crossings) {
		// MVector3 P(range[0][0], range[0][1], range[0][2]);
		MVector3 W;
		range_data *tris = (range_data *)value;
		while (tris != NULL) {
			PolyTriPtr tri1 = (PolyTriPtr)tris->data;
#if defined(_DEBUG) && 0
sprintf(sbuf, "Intersect (%.3g,%.3g,%.3g) with <%.3g,%.3g,%.3g> <%.3g,%.3g,%.3g> <%.3g,%.3g,%.3g>[%d,%d]:",
		P[0], P[1], P[2],
		tri1->W[0][0], tri1->W[0][1], tri1->W[0][2], 
		tri1->W[1][0], tri1->W[1][1], tri1->W[1][2], 
		tri1->W[2][0], tri1->W[2][1], tri1->W[2][2],
		tri1->i1, tri1->i2);
OutputDebugString(sbuf);
#endif
			if (zintersect(P, tri1, W)) {
				ZIntersectList *ilist = (ZIntersectList *)data;
				ilist->AddPoint(W);
			} else {
			}
			tris = tris->next;
		}
	}
}

void tri23tree::FindSlicers(PolyTriPtr tri)
{
	Range3D bbox;
	tri->compute_bbox(bbox);

	state = finding_slicers;
	result_list.clear(); // Ensure there's nothing laying around
	// result_list.FreeVertices();
	Process(bbox, (void *)tri);
}

void tri23tree::Slice(tri23tree *tree)
{
	if (tree == NULL || tree->Root == NULL)
		return;

	state = slicing;
	ProcessAll(tree);
}

void tri23tree::Tris(PolyTriList &tris)
{
	if (Root == NULL)
		return;
	state = collecting;
	Process(Root->range, (void *)&tris);
}

void tri23tree::
extractTris(nodeptr intree, PolyTriList &tris)
{
	rangedataptr tnode1, tnode2;

	if (intree == NULL)
		return;
	else if (intree->kind == node23::LEAF_NODE) {
		// Remove all entries
		for (tnode1=intree->leaf->data;tnode1!=NULL;) {
			tnode2 = tnode1;
			tnode1 = tnode1->next;
			PolyTriPtr tri = (PolyTriPtr)tnode2->data;
			tris.Push(tri);
#if 0
sprintf(sbuf, "   %p (%.3g,%.3g,%.3g) (%.3g,%.3g,%.3g) (%.3g,%.3g,%.3g)\n",
		tri,
		tri->vertices[0].ptr->W[0], tri->vertices[0].ptr->W[1], tri->vertices[0].ptr->W[2],
		tri->vertices[1].ptr->W[0], tri->vertices[1].ptr->W[1], tri->vertices[1].ptr->W[2], 
		tri->vertices[2].ptr->W[0], tri->vertices[2].ptr->W[1], tri->vertices[2].ptr->W[2]);
OutputDebugString(sbuf);
#endif
			// delete tri;
			delete tnode2;
		}
		delete intree->leaf;
	} else if (intree->kind == node23::TWO_NODE) {
		extractTris(intree->ptrs[0], tris);
		extractTris(intree->ptrs[1], tris);
	} else {
		extractTris(intree->ptrs[0], tris);
		extractTris(intree->ptrs[1], tris);
		extractTris(intree->ptrs[2], tris);
	}
	delete intree;
}

// Generate an exact clone of this tree (including making full copies
// of all PolyTris)
void tri23tree::ExtractTris(PolyTriList &tris)
{
	state = extracting;
	extractTris(Root, tris);
	Root = NULL;
}

void tri23tree::TrisHitting(Range3D &range, PolyTriList &tris)
{
	if (Root == NULL)
		return;
	state = collecting;
	Process(range, (void *)&tris);
}

bool tri23tree::Inside(const MVector3 &p)
{
	Range3D box;
	if (Root == NULL)
		return false;
	state = counting_crossings;

	// First a quick rejection test
	if (p[0] < Root->range[0][0] || p[0] > Root->range[1][0] ||
		p[1] < Root->range[0][1] || p[1] > Root->range[1][1] ||
		p[2] < Root->range[0][2] || p[2] > Root->range[1][2])
		return false;

	// We need a way of saving intersections of a ray with PolyTris.
	// This list is checked so we can throw away duplicate hits that
	// occur at the boundary of two PolyTris.
	ZIntersectList ilist;

	// Use a ray that has only a z component.  The reason for this is that
	// we store boxes in the 2-3 tree by x, y, then z.  This way we can
	// eliminate possibilities quickest using the coordinate index that
	// varies the fastest.
	P = p;
	box[0][0] = (float)(P[0] - EPSILON);
	box[0][1] = (float)(P[1] - EPSILON);
	box[0][2] = (float)(P[2]);
	box[1][0] = (float)(P[0] + EPSILON);
	box[1][1] = (float)(P[1] + EPSILON);
	box[1][2] = 1.0e10f;
	Process(box, &ilist);

	return (ilist.Count() & 1) != 0;
}

bool tri23tree::InsideAll(const MVector3 &p)
{
	int count = 0;
	if (Root == NULL)
		return false;
	state = counting_crossings;
	P = p;
	ProcessAll(&count);

	return (count & 1) != 0;
}

void tri23tree::Unlink(PolyTriList &tris)
{
	Range3D bbox;
	PolyTriList::iterator triitr;
	PolyTriPtr tri;

	if (Root == NULL)
		return;
	state = unlinking;
	for (triitr=tris.begin();triitr!=tris.end();triitr++) {
		tri = *triitr;
		tri->compute_bbox(bbox);
		Range_Tree::Delete(bbox, (void *)tri);
	}
}

void tri23tree::UnlinkAll()
{
	state = unlinking;
	shread_two_three(Root);
	Root = NULL;
}

void tri23tree::Draw()
{
	if (Root == NULL)
		return;
	state = drawing;
	Process(Root->range, NULL);
}

// Copy the underlying range data (support for public Clone() method)
void tri23tree::clone_PolyTri_tree(node23 *old_tree, node23 *new_tree)
{
	assert(old_tree != NULL);
	new_tree->range = old_tree->range;

	switch (old_tree->kind) {
	case node23::LEAF_NODE:
		{
			range_data *data_list, *root, *last;
			PolyTriPtr tri;

			// Copy the list of PolyTris
			data_list = old_tree->leaf->data;
			assert(data_list != NULL);
			tri  = new PolyTri((PolyTriPtr)data_list->data);
			last = root = new range_data(tri);
			for (data_list=data_list->next;
				 data_list != NULL;
				 data_list=data_list->next) {
				tri = new PolyTri((PolyTriPtr)data_list->data);
				last->next = new range_data(tri);
				last = last->next;
			}

			// Set the key and range information
			new_tree->kind       = node23::LEAF_NODE;
			new_tree->leaf       = new node23leaf;
			new_tree->leaf->data = root;
			new_tree->leaf->key  = old_tree->leaf->key;
			new_tree->link       = new_tree->leaf;
		}
		break;
	case node23::TWO_NODE:
		new_tree->kind = node23::TWO_NODE;
		new_tree->ptrs[0] = new node23(old_tree->ptrs[0]->kind);
		new_tree->ptrs[1] = new node23(old_tree->ptrs[1]->kind);
		clone_PolyTri_tree(old_tree->ptrs[0], new_tree->ptrs[0]);
		clone_PolyTri_tree(old_tree->ptrs[1], new_tree->ptrs[1]);
		new_tree->key[0] = least_key(new_tree->ptrs[1]);
		new_tree->link = new_tree->ptrs[0]->link;
		break;
	case node23::THREE_NODE:
		new_tree->kind = node23::THREE_NODE;
		new_tree->ptrs[0] = new node23(old_tree->ptrs[0]->kind);
		new_tree->ptrs[1] = new node23(old_tree->ptrs[1]->kind);
		new_tree->ptrs[2] = new node23(old_tree->ptrs[2]->kind);
		clone_PolyTri_tree(old_tree->ptrs[0], new_tree->ptrs[0]);
		clone_PolyTri_tree(old_tree->ptrs[1], new_tree->ptrs[1]);
		clone_PolyTri_tree(old_tree->ptrs[2], new_tree->ptrs[2]);
		new_tree->key[0] = least_key(new_tree->ptrs[1]);
		new_tree->key[1] = least_key(new_tree->ptrs[2]);
		new_tree->link   = new_tree->ptrs[0]->link;
		break;
	}

#if DEBUG_TREES
	switch (new_tree->kind) {
	case node23::LEAF_NODE:
#if DEBUG_LEAVES
		{
		Range3D box;
		range_data *trangedata;
		for (trangedata=new_tree->leaf->data;trangedata!=NULL;trangedata=trangedata->next) {
			((PolyTriPtr)trangedata->data)->compute_bbox(box);
			assert(new_tree->range.Contains(box));
		}
		}
		assert(new_tree->link == new_tree->leaf);
#endif
		break;
	case node23::TWO_NODE:
		assert(new_tree->range.Contains(new_tree->ptrs[0]->range));
		assert(new_tree->range.Contains(new_tree->ptrs[1]->range));
		assert(this->compare(new_tree->key[0], least_key(new_tree->ptrs[1])) == 0);
		assert(new_tree->link == new_tree->ptrs[0]->link);
		assert(new_tree->key[0] == new_tree->ptrs[1]->link->key);
		break;
	case node23::THREE_NODE:
		assert(new_tree->range.Contains(new_tree->ptrs[0]->range));
		assert(new_tree->range.Contains(new_tree->ptrs[1]->range));
		assert(new_tree->range.Contains(new_tree->ptrs[2]->range));
		assert(compare(new_tree->key[0], least_key(new_tree->ptrs[1])) == 0);
		assert(compare(new_tree->key[1], least_key(new_tree->ptrs[2])) == 0);
		assert(compare(new_tree->key[0], new_tree->key[1]) < 0);
		assert(new_tree->link == new_tree->ptrs[0]->link);
		assert(new_tree->key[0] == new_tree->ptrs[1]->link->key);
		assert(new_tree->key[1] == new_tree->ptrs[2]->link->key);
		break;
	}
#endif
}

// Generate an exact clone of this tree (including making full copies
// of all PolyTris)
void tri23tree::Clone(tri23tree &tree)
{
	if (Root == NULL)
		tree.Root = NULL;
	else {
		tree.Root = new node23(Root->kind);
		clone_PolyTri_tree(Root, tree.Root);
	}
}

int tri23tree::TriCount()
{
	return entry_count;
}

#define USE_FOUR_TREES 1

void
MeshCSG(int csg_op, PolyTriList &obj1, PolyTriList &obj2, PolyTriList &resultobj)
{
	// TBD: Add a couple of simple optimizations - if we don't need
	// to keep any triangles from one of the trees, then we also
	// don't need to go to the work of slicing it (this could be
	// used to performing clipping of an object with respect to
	// another object.
	tri23tree t1, t2;

	// Create range trees of the original objects and their copies.
	t1.Insert(obj1);
	t2.Insert(obj2);

#if USE_FOUR_TREES
	tri23tree t3, t4;
	t1.Clone(t3);
	t2.Clone(t4);

	// Dice the two objects
	t1.Slice(&t4);
	t2.Slice(&t3);

	// list1 - triangles of obj1 (sliced so that none penetrate triangles of obj2).
	// list2 - triangles of obj2 (they don't penetrate obj1)
	PolyTriList list1, list2;
	t3.ExtractTris(list1);
	t4.ExtractTris(list2);
#else
	t1.Slice(&t2);
	t2.Slice(&t1);

	PolyTriList list1, list2;
	t1.Tris(list1);
	t2.Tris(list2);
#endif

	// Determine what triangles share common edges.  This information can be
	// used to do continuation operations, rather than checking inside/outside
	// for every triangle. [TBD: Actually do something here.]
	list1.ComputeAdjacencies();
	list2.ComputeAdjacencies();

	// The two while loops below walk though each triangle of an object and determine
	// if it is inside or outside of the other object.  The result is four lists, each
	// holding one of the possible cases:
	//     list3 - triangles of obj1 that are outside obj2
	//     list4 - triangles of obj2 that are outside obj1
	//     list5 - triangles of obj1 that are inside obj2
	//     list6 - triangles of obj2 that are inside obj1
	// The Inside(...) function used here is brute force - it takes the center of
	// a triangle and shoots a ray.  The number of times that ray intersects the
	// other object determines if the point is inside or outside (odd crossings
	// means outside).
	//
	// TBD: Keep track of which triangles were actually sliced.  They will represent
	// the boundary of triangles that are either inside or outside an object.  Once
	// we find a triangle that is inside an object, we can mark as inside all triangles
	// that are next to it and weren't sliced.
	// 
	PolyTriList list3, list4, list5, list6;
	PolyTriPtr tri;
	MVector3 P;

	// Separate the list of triangles of obj1 into inside and outside triangles
	while ((tri = list1.Pop()) != NULL) {
		P = tri->Center();
		if (!t2.Inside(P)) {
			list3.Push(tri);
		} else {
			tri->Invert();
			list5.Push(tri);
		}
	}

	// Separate the list of triangles of obj2 into inside and outside triangles
	while ((tri = list2.Pop()) != NULL) {
		P = tri->Center();
		if (!t1.Inside(P)) {
			list4.Push(tri);
		} else {
			tri->Invert();
			list6.Push(tri);
		}
	}

#if !USE_FOUR_TREES
	// Don't need the triangle trees anymore
	t1.UnlinkAll();
	t2.UnlinkAll();
#endif

	// CSG is now just the process of choosing the proper pair of triangles
	// that are inside/outside of the two objects.
	if (csg_op & CSG_OP_OBJ1_INSIDE) {
		while ((tri = list5.Pop()) != NULL) {
			if (csg_op & CSG_OP_OBJ2_INSIDE) {
				TriVertex *temp = tri->vertices[1].ptr;
				tri->vertices[1].ptr = tri->vertices[2].ptr;
				tri->vertices[2].ptr = temp;
			}
			resultobj.Push(tri);
		}
	}
	if (csg_op & CSG_OP_OBJ1_OUTSIDE) {
		while ((tri = list3.Pop()) != NULL)
			resultobj.Push(tri);
	}
	if (csg_op & CSG_OP_OBJ2_INSIDE) {
		while ((tri = list6.Pop()) != NULL) {
			if (csg_op & CSG_OP_OBJ1_INSIDE) {
				TriVertex *temp = tri->vertices[1].ptr;
				tri->vertices[1].ptr = tri->vertices[2].ptr;
				tri->vertices[2].ptr = temp;
			}
			resultobj.Push(tri);
		}
	}
	if (csg_op & CSG_OP_OBJ2_OUTSIDE) {
		while ((tri = list4.Pop()) != NULL)
			resultobj.Push(tri);
	}
}
