#include "StdAfx.h"
#include "PovrayRenderer.h"
#include "PovrayTranslator.h"

#include <IniFile.h>
#include <MLight.h>
#include <params/MParameterFactory.h>
#include <MSceneRenderer.h>
#include <MSystemManager.h>

#include <assert.h>

PovrayRenderer::PovrayRenderer() {
}

PovrayRenderer::~PovrayRenderer() {
}


// MSceneRenderer methods
std::string PovrayRenderer::getName() {
  return "PovRay 3.5 Ray Tracer";
}

void PovrayRenderer::initialise() {
  currentState = Initialised;
}


bool PovrayRenderer::start(const Aztec::MScenePtr &scene,
                           const Aztec::MParameterObjectListPtr &options, 
                           const UpdateListenerPtr &listener) {

  tempDirectory;

  // TODO: replace with a single function call which does 
  // the platform specific thing.
#ifdef _WIN32
  char tempDir[MAX_PATH];

  ::GetTempPath(MAX_PATH, tempDir);

  // now ensure that the path exists and it is a directory
  DWORD attributes = ::GetFileAttributes(tempDir);
  if (attributes != 0xffffffff && (attributes & FILE_ATTRIBUTE_DIRECTORY)) {
    tempDirectory = tempDir;
  }

#else
  tempDirectory = "/tmp/";
#endif
  // if we don't have a temporary directory, just use the config path :(
  if (tempDirectory == "") {
    tempDirectory = MSystemManager::getInstance()->getUserSettingsPath();
  }

  tempFilename = tempDirectory + "tempPovrayScene";

  // first we create a POVRAY ini file that turns off io restrictions, so we 
  // aren't hassled by this annoying feature
  iniFile = tempDirectory + "tempPovrayScene.ini";

  int width = 320;
  int height= 240;
  bool antialiasing = true;

  options->getParameter("outputWidth")->getValueInteger(width);
  options->getParameter("outputHeight")->getValueInteger(height);
  options->getParameter("antialiasing")->getValueBoolean(antialiasing);

  {
    FILE *f = fopen(iniFile.c_str(), "w");
    if (f != NULL) {

      fprintf(f, "Input_File_Name=%s.pov\n", tempFilename.c_str());
      fprintf(f, "Output_File_Name=%s.bmp\n", tempFilename.c_str());
      fprintf(f, "Width=%i\n", width);
      fprintf(f, "Height=%i\n", height);
      fprintf(f, "Display=off\n");
      fprintf(f, "Output_To_File=on\n");
      fprintf(f, "Output_File_Type=S\n");
      fprintf(f, "Antialias=%s\n", antialiasing ? "yes" : "no");
      fprintf(f, "Sampling_Method=2\n");
      fprintf(f, "Antialias_Depth=4\n");
      fprintf(f, "Buffer_Output=on\n");
      fprintf(f, "Pause_When_Done=off\n");

      fclose(f);
    }

  }

  MStr str;
  options->getParameter("povrayExe")->getValueString(str);
  povrayExecutable = str.c_str();

  // export the file.
  {
    char buf[1024];
    sprintf(buf, "%s.pov", tempFilename.c_str());
    MRefCountedPtr<PovrayTranslator> trans = new PovrayTranslator();
    trans->exportFile(buf, scene, options);
  }

  this->listener = listener;

  return createRenderingThread();
}

void PovrayRenderer::stop() {

}

static int blah = 0;

Aztec::MImagePtr PovrayRenderer::getImage() {
#ifdef _WIN32
  HANDLE mutex = CreateMutex(NULL, TRUE, "PovrayMutex");
  ::WaitForSingleObject(mutex, INFINITE);
  assert(blah == 0);
  ++blah;
#endif

  char buf[1024];
  sprintf(buf, "%s.bmp", tempFilename.c_str());
  MImagePtr image = Aztec::MSystemManager::getInstance()->loadImage(buf,true);

#ifdef _WIN32
  --blah;
  ReleaseMutex(mutex);
#endif

  return image;
}

MParameterObjectListPtr PovrayRenderer::getOptions() {
  MParameterObjectListPtr options = new MParameterObjectList(NULL);

  options->addParam(Aztec::MParameterFactory::createFilename("povrayExe", "povrayExe", "Povray Executable") );
  options->getParameter("povrayExe")->setValueString("C:\\Program Files\\POV-Ray for Windows v3.5\\bin\\pvengine.exe");

  return options;
}


bool PovrayRenderer::createRenderingThread() {
  int hThread;

  renderThreadProc = new MRenderThreadProc(this);
  hThread = renderThreadProc->spawn();

  return (hThread != NULL);
}


void MRenderThreadProc::main()
{                
  std::string tempFilename = renderer->getTempFilename();

  char buf[2048];
  sprintf(buf, 
    "/RENDER %s /exit", renderer->iniFile.c_str());

#ifdef WIN32
  STARTUPINFO *startInfo = renderer->getStartUpInfo();

  memset((void *)startInfo, 0, sizeof(STARTUPINFO));
  startInfo->cb = sizeof(STARTUPINFO);
  startInfo->dwFlags = STARTF_USESHOWWINDOW;
  startInfo->wShowWindow = SW_HIDE;

  int result = ::CreateProcess(
                  renderer->povrayExecutable.c_str(),
                  buf,
                  NULL,
                  NULL,
                  FALSE,
                  IDLE_PRIORITY_CLASS,
                  NULL,
                  renderer->tempDirectory.c_str(),
                  startInfo,
                  renderer->getProcessInformation());

  // if the spawning failed, set the rendering state to finished
  if (result == 0) {
    MSystemManager::getInstance()->logOutput("Error starting renderer '%s'.", renderer->povrayExecutable.c_str());
    renderer->setCurrentState(Finished);
    return;
  }

#else
  char sysBuf[2048+strlen("povray")];
  sprintf(sysBuf,"povray %s",buf);
  system(sysBuf);
#endif

  renderer->setCurrentState(Started);

  int hThread;
  renderFinishedProc = new MRenderFinishedProc(renderer);
  hThread = renderFinishedProc->spawn();

  MSceneRenderer::UpdateListenerPtr listener = renderer->getCurrentListener();

  while (renderer->getCurrentState() == Started) {
	  sleep(1000);
    if ((renderer->getCurrentState() == Started) && listener != NULL) {
         listener->onUpdate(renderer, 0.5, renderer->getImage());
    }
  } 
}


void MRenderFinishedProc::main()
{
  // here we have to wait until the rendering thread is completed. However, 
  // we only want to wait if the rendered hasn't finished already.

  if (renderer->getCurrentState() == Finished) {
    return;
  }
#ifdef WIN32
  PROCESS_INFORMATION *info = renderer->getProcessInformation();
  
  DWORD w = ::WaitForSingleObject(info->hProcess, 60000);
  
  if (w == WAIT_FAILED) {
    w = ::GetLastError();
    int i = 0;
    int j = 2 / i;
  }
#else
  // TODO: Make a Linux equivalent
#endif
  renderer->setCurrentState(Finished);

  MSceneRenderer::UpdateListenerPtr listener = renderer->getCurrentListener();

  if (listener != NULL) {
    listener->onUpdate(renderer, 1.0, renderer->getImage());
  }

}

