// OpenGLWnd.cpp : implementation file
//

#include <AztecMainPCH.h>
#include "resource.h"
#include "OpenGLWnd.h"

#include <typeinfo.h>

#include <gl/gl.h>
#include <gl/glu.h>

#include "MDLGlobs.h"
#include "MLight.h"
#include "MDLMsgs.h"
#include "MainFunc.h"

#include "ModelGLConst.h"

#include "KeyFuncs.h"
#include "KeyFuncView.h"

#include "MBaseObject.h"
#include "MMesh.h"

#include "math.h"
#include "DlgGlobs.h"

#include "UndoGeneral.h"
#include "UvEditViewWnd.h"

#include "ToolClasses.h"
#include "..\..\include\views\openglwnd.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//--------------
//  COpenGLWnd
//--------------
COpenGLWnd::COpenGLWnd()
{
  m_Zoom = 0.5;
  m_Dist = 200;
  m_Persp = 1.0;
  
  m_ShadingMode = glWireframe;
  m_TexturingMode = glNone;
  m_LightingMode = glLit;
  setFlag((WORD)(GLVIEW_FLAGS_CULLBACKFACE | GLVIEW_FLAGS_DRAWGRID));
  
  m_TextureNames = NULL;
  m_NumTextureNames = 0;
  m_TextureMode = GL_LINEAR;
  
  m_GridBasisX = MVector3(1,0,0);
  m_GridBasisY = MVector3(0,1,0);
  
  unsetFlag(GLVIEW_FLAGS_FULLSCREEN);
  
  m_PopupRect.left = 0;
  m_PopupRect.right = 50;
  m_PopupRect.top = 0;
  m_PopupRect.bottom = 16;

#if USE_SPACEBALL
  m_DevHdl = 0;
#endif

  MBaseOpenGLWnd::MBaseOpenGLWnd();
}

COpenGLWnd::~COpenGLWnd()
{
}

MBaseObjectPtr COpenGLWnd::createNew()
{
  COpenGLWnd  *Wnd;
  
  Wnd = new COpenGLWnd;
  
  Wnd->SetViewName(m_ViewName);
  Wnd->m_Pan = m_Pan;
  Wnd->m_Rot = m_Rot;
  Wnd->m_Zoom = m_Zoom;
  Wnd->m_Dist = m_Dist;
  Wnd->m_Persp = m_Persp;
  Wnd->m_GridBasisX = m_GridBasisX;
  Wnd->m_GridBasisY = m_GridBasisY;
  
  Wnd->m_ShadingMode = m_ShadingMode;
  Wnd->m_TexturingMode = m_TexturingMode;
  Wnd->m_LightingMode = m_LightingMode;
  Wnd->m_TextureMode = m_TextureMode;
  
  return (MBaseObject*)Wnd;
}

BEGIN_MESSAGE_MAP(COpenGLWnd, CWnd)
//{{AFX_MSG_MAP(COpenGLWnd)
ON_WM_SIZE()
ON_WM_DESTROY()
ON_WM_PAINT()
ON_WM_KEYDOWN()
ON_WM_CLOSE()
	ON_WM_CREATE()
	//}}AFX_MSG_MAP
ON_WM_CREATE()
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// COpenGLWnd message handlers

void COpenGLWnd::AdjustProjectionMatrix(MSelectMethod PickMode, 
                                        int left, int top,
                                        int right, int bottom) {
  
  MBaseOpenGLWnd::AdjustProjectionMatrix(PickMode, left, top, right, bottom);
  
  float big, Small;
  
  if (m_WindowRect.right>m_WindowRect.bottom) {
    big = (float)m_WindowRect.right;
    Small = (float)m_WindowRect.bottom;
  } else {
    big = (float)m_WindowRect.bottom;
    Small = (float)m_WindowRect.right;
  }
  
  // set up the actual projection matrices
  if (m_Flags & GLVIEW_FLAGS_PERSPECTIVE)
  {
    float fov = 90;

    if (camera != NULL) {
      camera->getShapeObject()->findParameter("fov")->getValueFloat(fov);
    }

    float factor = (float)tan(DegToRad(fov / 2)) * m_Aspect / big;

    m_GateLeft = -factor * m_WindowRect.right;
    m_GateRight = factor * m_WindowRect.right;
    m_GateTop = factor * m_WindowRect.bottom;
    m_GateBottom = -factor * m_WindowRect.bottom;
    
    glFrustum( m_GateLeft, m_GateRight, m_GateBottom, m_GateTop, 3.0, 5024.0);
  } else {
    m_GateLeft = -32*m_WindowRect.right/big*m_Aspect;
    m_GateRight = 32*m_WindowRect.right/big*m_Aspect;
    m_GateTop = 32*m_WindowRect.bottom/big*m_Aspect;
    m_GateBottom = -32*m_WindowRect.bottom/big*m_Aspect;
    
    glOrtho(m_GateLeft, m_GateRight, m_GateBottom, m_GateTop, -512*m_Zoom,512*m_Zoom);
    glScalef(m_Zoom, m_Zoom, m_Zoom);
  }
  
  glMatrixMode(GL_MODELVIEW);
}

void COpenGLWnd::AdjustScaleFactor()
{
  if (isFlagged(GLVIEW_FLAGS_PERSPECTIVE))
  {
    double Factor;
    if (m_WindowRect.right > m_WindowRect.bottom)
      Factor = 4*m_Dist/m_Aspect*(m_GateRight-m_GateLeft)/1024.0;
    else
      Factor = 4*m_Dist/m_Aspect*(m_GateTop-m_GateBottom)/1024.0;
    
    
    glScalef((GLfloat)Factor, (GLfloat)Factor, (GLfloat)Factor);
  }
  else
  {
    glScalef(1/m_Zoom, 1/m_Zoom, 1/m_Zoom);
  }
}

float COpenGLWnd::GetScaleFactor(const MVector3 &Vec)
{
  float    Dist;
  MVector3 VecScreenSpace;
  MMatrix4 ModelViewMat;
  float    Mat[16];
  
  if (isFlagged(GLVIEW_FLAGS_PERSPECTIVE))
  {
    double Factor;
    
    glGetFloatv(GL_MODELVIEW_MATRIX, Mat);
    ModelViewMat = MMatrix4(Mat);
    ModelViewMat.transpose();
    VecScreenSpace = ModelViewMat * Vec;
    
    Dist = -VecScreenSpace.z;
    
    Factor = Dist / 100.0;
    
    /*      if (m_WindowRect.right > m_WindowRect.bottom)
    Factor = 4*m_Dist/m_Aspect*(m_GateRight-m_GateLeft)/1024.0;
    else
    Factor = 4*m_Dist/m_Aspect*(m_GateTop-m_GateBottom)/1024.0;*/
    
    return (float)Factor;
  }
  else
  {
    return 1.0f / m_Zoom;
  }
}


void COpenGLWnd::OnSize(UINT nType, int cx, int cy) 
{
  MBaseViewWnd::OnSize(nType, cx, cy);
  
  RECT clientRect;
  GetClientRect(&clientRect);
  if (glCanvas != NULL) {
    glCanvas->MoveWindow(&clientRect);
  }
  GLResizeToClient();
  
}

DWORD getMenuItemForMode(GLShadingMode shading, GLTextureMode texture, GLLightingMode light) {
  if (shading != glWireframe) {
    if (texture != glNone) {
      if ( light != glUnlit) {
        return ID_LITTEXTURED;
      } else {
        return ID_TEXTURED;
      }
    } else if (shading == glSmooth) {
      return ID_SMOOTHSHADED;
    } else {
      return ID_FLATSHADED;
    }
  } else {
    return ID_WIREFRAME;
  }
}

class ShadingMenu : public MMenuBar::PopupMenu {
public:
  COpenGLWnd *view;

  ShadingMenu(DWORD resource, COpenGLWnd *glView) 
    : PopupMenu(resource),
      view(glView)
  {
  }

  HMENU createMenu() {
    HMENU popup = PopupMenu::createMenu();

    if (view->isFlagged(GLVIEW_FLAGS_CULLBACKFACE)) {
      CheckMenuItem(popup, ID_BACKFACECULL, MF_CHECKED);
    } else {
      CheckMenuItem(popup, ID_BACKFACECULL, MF_UNCHECKED);
    }
    
    
    if (view->isFlagged(GLVIEW_FLAGS_DRAWGRID)) {
      CheckMenuItem(popup, ID_POPUP_GRID_VISIBLE, MF_CHECKED);
    } else {
      CheckMenuItem(popup, ID_POPUP_GRID_VISIBLE, MF_UNCHECKED);
    }
    
    {
      CheckMenuRadioItem(popup, ID_LITTEXTURED, ID_WIREFRAME, getMenuItemForMode(view->m_ShadingMode, view->m_TexturingMode, view->m_LightingMode), MF_BYCOMMAND);
    }
    
    {
      DWORD Target;
      if (view->m_TextureMode == GL_NEAREST)
        Target = ID_POPUP_TEXTUREMODE_NEAREST;
      if (view->m_TextureMode == GL_LINEAR)
        Target = ID_POPUP_TEXTUREMODE_LINEAR;
      CheckMenuRadioItem(popup, ID_POPUP_TEXTUREMODE_NEAREST, ID_POPUP_TEXTUREMODE_LINEAR, Target, MF_BYCOMMAND);
    }
    
    return popup;
  }
};


void COpenGLWnd::ViewCreate()
{
  MBaseOpenGLWnd::ViewCreate();
  
  int Error;
  
  Error = InitOpenGL(m_hWnd);
  
  if (Error!=0)	// Did OpenGL fail to intiialise
  {
    if (Error == WGL_CREATECONTEXT_FAILED)
      AfxMessageBox("wglCreateContext Failed");
    if (Error == WGL_MAKECURRENT_FAILED)
      AfxMessageBox("wglMakeCurrent Failed");
  }

  getMenuBar()->addMenu("Shading", new ShadingMenu(IDR_OPENGL_VIEWPORT_POPUP, this), 0);
}

void COpenGLWnd::OnDestroy() 
{
  CWnd::OnDestroy();
  KillOpenGL();
  
}

// Draws the manipulatrs. If Select is true, it draws using selection buffers, and returns 0 on none,
// x,y,z are 1,2,3 respectivley, 4 is parallel to the screen.
int COpenGLWnd::DrawManipulators(bool Select)
{
  MVector3    PivotPoint;
  UINT        *SelectBuf;
  
  if (Select)
  {
    SelectBuf = new UINT[16384];
    glSelectBuffer(16384, SelectBuf);
    glRenderMode(GL_SELECT);
    
    glInitNames();
    glPushName(-1);
  }
  
  g_ToolMan.GetTool()->DrawTool(Select, getShiftState(), this);
  if (g_ToolMan.IsTemporary()) {
    MToolTypePtr Tool;
    Tool = g_ToolMan.GetTool(1);
    if (Tool != NULL) {
      Tool->DrawTool(Select, getShiftState(), this);
    }
  }
  
  if (Select)     // Have we beein drawing in select mode
  {
    int      NumHits;
    int      ObjNum;
    int      i, Count, j;
    GLuint   *Ptr;
    
    ObjNum = 0;
    
    NumHits = glRenderMode(GL_RENDER);
    
    Ptr = SelectBuf;
    
    //      for (i=0;(i<NumHits && ObjNum == 0);i++)
    for (i=0;(i<NumHits);i++)
    {
      float    Depth1, Depth2;
      
      Count = *Ptr; 
      Ptr++;
      Depth1 = (float) *Ptr/0x7fffffff; Ptr++;
      Depth2 = (float) *Ptr/0x7fffffff; Ptr++;
      
      
      j = 0;
      while (j<Count)
      {
        if (*Ptr == -1)      // we are doing a new hit
        {
          
          Ptr++; j++;
          ObjNum = *Ptr;
        }
        Ptr++; j++;
        
      }
    }
    
    delete[] SelectBuf;
    
    return ObjNum;
  }
  
  
  return 0;
}

void COpenGLWnd::DrawViewExtras()
{
  MBaseOpenGLWnd::DrawViewExtras();
  
  glDisable(GL_LIGHTING);
  glDisable(GL_DEPTH_TEST);
  glEnable(GL_BLEND);
  
  //glClear(GL_DEPTH_BUFFER_BIT);
  // Draw the gizmos for the various tools
  
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  
  DrawManipulators(false);
  
  // Push the projection stack, so we can draw using screen coordinates
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  
  glViewport(0,0,m_WindowRect.right, m_WindowRect.bottom);
  
  glLoadIdentity();
  
  glOrtho(0,m_WindowRect.right,0,m_WindowRect.bottom,-20,20);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  
  
  glColor3f(1.0,1.0,1.0);
  glRasterPos2d(4,m_WindowRect.bottom-14);
  glListBase(GLVIEW_FONTSTART);

  if (camera == NULL) {
    glCallLists(m_ViewName.GetLength(), GL_UNSIGNED_BYTE, (LPCTSTR)m_ViewName);
  } else {
    glCallLists(camera->getName().GetLength(), GL_UNSIGNED_BYTE, (LPCTSTR)camera->getName().c_str());
  }
  
  glPushMatrix();
  
  glEnable(GL_DEPTH_TEST);
  glTranslatef(20,20,0);
  glRotateCamera();
  DrawAxisIcon(15, 1, 1);
  glDisable(GL_DEPTH_TEST);
  
  glPopMatrix();
  
  // Draw the maximize, minimize, and menu boxes
  {
    glPushMatrix();
    
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    glDisable(GL_CULL_FACE);
    glColor4f(0,0,0,0.2f);
    
    glTranslatef(m_WindowRect.right - 18, m_WindowRect.bottom - 18, 0);
    
    // Draw the backgound for the buttons
    glBegin(GL_QUADS);
    glVertex2f(0, 18);
    glVertex2f(18, 18);
    glVertex2f(18, 0);
    glVertex2f(0, 0);
    glEnd();
    
    glTranslatef(1,1,0);
    
    // draw the maximise button
    //
    // +----------
    // |  --------+
    //    |----   |
    //        / / |
    //       / /| |
    //      / / | |
    //      \/  |_|
    //
    //
    //
    glColor4f(1,1,1,0.4f);
    glBegin(GL_QUADS);
    glVertex2f(0, 16);
    glVertex2f(16, 16);
    glVertex2f(16, 0);
    glVertex2f(0, 0);
    glEnd();

    glColor4f(1,1,1,0.4f);
    if (isFlagged(GLVIEW_FLAGS_FULLSCREEN)) {
      glBegin(GL_QUADS);
      glVertex2f(16 - 2, 16 - 14); glVertex2f(16 - 14,16 -  14); 
      glVertex2f(16 - 10,16 -  12); glVertex2f(16 -  2,16 -  12);

      glVertex2f(16 - 12, 16 -  2); glVertex2f(16 - 12,16 -  10); 
      glVertex2f(16 - 14, 16 - 14); glVertex2f(16 - 14,16 -   2);

      glVertex2f(16 -  2, 16 -  4); glVertex2f(16 - 10,16 -  12); 
      glVertex2f(16 - 12, 16 - 10); glVertex2f(16 -  4,16 -   2);
      glEnd();

      glBegin(GL_TRIANGLES);
      glVertex2f(16 - 10, 16 - 12); 
      glVertex2f(16 - 14, 16 - 14); 
      glVertex2f(16 - 12, 16 - 10);
      glEnd();
    } else {
      glBegin(GL_QUADS);
      glVertex2f( 2, 14); glVertex2f(14, 14); 
      glVertex2f(10, 12); glVertex2f( 2, 12);

      glVertex2f(12,  2); glVertex2f(12, 10); 
      glVertex2f(14, 14); glVertex2f(14,  2);

      glVertex2f( 2,  4); glVertex2f(10, 12); 
      glVertex2f(12, 10); glVertex2f( 4,  2);
      glEnd();

      glBegin(GL_TRIANGLES);
      glVertex2f(10, 12); 
      glVertex2f(14, 14); 
      glVertex2f(12, 10);
      glEnd();
    }
    
    glPopMatrix();
  }
  
  // Clean up this mess matrix wise
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}

void COpenGLWnd::DoViewportTransform()
{
  MBaseOpenGLWnd::DoViewportTransform();

  if (camera == NULL) {
    glTranslatef(0.0, 0.0, -m_Dist);
  
    glRotateCamera();
  
    glTranslatef(-m_Pan.x, -m_Pan.y, m_Pan.z);
  } else {
    MVector3 trans = camera->getTranslateVector(g_Scene->getTime());

    glRotateCamera();
    glTranslatef(-trans.x, -trans.y, -trans.z);
  }

  glGetDoublev(GL_PROJECTION_MATRIX, m_ProjMat);
  glGetDoublev(GL_MODELVIEW_MATRIX, m_ModelMat);
}

void COpenGLWnd::DrawGrid(const MVector3 &BX, const MVector3 &BY)
{
  // Draw a grid using the given basis as the plane
  MVector3     Vec1,Vec2;
  
  // Draw the most origin line
  glDisable(GL_LIGHTING);
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_POLYGON_OFFSET_LINE);
  glDisable(GL_BLEND);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LESS);
  
  glColor3f(0.0,0.0,0.0);
  
  glBegin(GL_LINES);
  {
    Vec1 = (-250)*BX;
    Vec2 = (250)*BX;
    
    glVertex3fv((float*)&Vec1);
    glVertex3fv((float*)&Vec2);
    
    Vec1 = (-250)*BY;
    Vec2 = (250)*BY;
    
    glVertex3fv((float*)&Vec1);
    glVertex3fv((float*)&Vec2);
  }
  glEnd();
  
  glColor3f(0.3f,0.3f,0.3f);
  glBegin(GL_LINES);
  {
    int n;
    for (n = -250;n<250+10;n+=10)
    {
      Vec1 = (-250)*BX;
      Vec1 += (n)*BY;
      Vec2 = (250)*BX;
      Vec2 += (n)*BY;
      
      glVertex3fv((float*)&Vec1);
      glVertex3fv((float*)&Vec2);
    }
    
    for (n = -250;n<250+10;n+=10)
    {
      Vec1 = (-250)*BY;
      Vec1 += (n)*BX;
      Vec2 = (250)*BY;
      Vec2 += (n)*BX;
      
      glVertex3fv((float*)&Vec1);
      glVertex3fv((float*)&Vec2);
    }
  }
  glEnd();
  
}

void COpenGLWnd::glRotateCamera() {
  if (camera != NULL) {
    // we do an inital rotation so our camera is pointing in the x direction initially.
    glRotatef(-90, 1, 0, 0);

    // Get the world transformation matrix.
    float    mat[16];
    MMatrix4 matrix;
    g_Scene->getWorldTransformMatrix( g_Scene->getObjectList()->findObject(camera), matrix );

    // get rid of the translation parts
    matrix.m[3][0] = 0.0;
    matrix.m[3][1] = 0.0;
    matrix.m[3][2] = 0.0;
    matrix.inverse();

    // convert it to an open gl matrix.
    matrix.transpose();
    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
        mat[j*4+i] = matrix.m[i][j];
    }

    glMultMatrixf(mat);
    
  } else {
    glRotatef(m_Rot.y,0,1,0);
    glRotatef(m_Rot.x,1,0,0);
    glRotatef(m_Rot.z,0,0,1);
  }
}


void COpenGLWnd::DrawView() {
  MBaseOpenGLWnd::DrawView();
  drawView(smNone);
}

void COpenGLWnd::drawView(MSelectMethod method,
                          std::vector<MSelectionItem> *selectResultList, 
                          int left, int top, 
                          int right, int bottom) 
{
  MBaseOpenGLWnd::DrawView();
  // check to see if tehe viewport is 0 size, then don't draw it
  
  if (m_WindowRect.bottom == 0 || m_WindowRect.right == 0) {
    return;
  }
  
  GLMakeCurrent();
  
  glDisable(GL_POINT_SMOOTH);
  glDisable(GL_LINE_SMOOTH);
  //   glEnable(GL_LINE_SMOOTH);
  glEnable(GL_DEPTH_TEST);
  
  glEnable(GL_BLEND);
  //   glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPointSize(4);
  
  
  if (m_Flags & GLVIEW_FLAGS_CULLBACKFACE)
   	glEnable(GL_CULL_FACE);
  else
   	glDisable(GL_CULL_FACE);
  
  // set up the drawing modes
  
  if (m_ShadingMode == glWireframe) {
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  } else {
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  }
  
  glLineWidth(1);
  
  MSceneViewFlags   ViewFlags;
  
  ViewFlags.m_ShadingMode = m_ShadingMode;
  ViewFlags.m_LightingMode = m_LightingMode;
  ViewFlags.m_TexturingMode = m_TexturingMode;
  ViewFlags.m_CullBackface = m_Flags & GLVIEW_FLAGS_CULLBACKFACE;
  ViewFlags.m_TextureMode = m_TextureMode;
  ViewFlags.m_SelectMethod = method;
  
  AdjustProjectionMatrix(method, left, top, right, bottom);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  DoViewportTransform();
  
  
  if (method != smNone)
  {
    ViewFlags.m_SelectBuf = new UINT[16384];
    glSelectBuffer(16384, ViewFlags.m_SelectBuf);
    glRenderMode(GL_SELECT);
  }
  
  
  glClearColor(0.4f, 0.4f, 0.4f, 0.0f);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  
  if (g_Playing) {
	  ViewFlags.m_Playing = true;
  } else {
	  ViewFlags.m_Playing = false;
  }

  g_Scene->drawScene(ViewFlags, selectResultList);

  if (method == smNone)
  {
    if (isFlagged(GLVIEW_FLAGS_DRAWGRID))
      DrawGrid( m_GridBasisX, m_GridBasisY);
    
  }
  
  if (method == smNone) {
    DrawViewExtras();
  }
  
  if (method != smNone) {
    delete[] ViewFlags.m_SelectBuf;
    ViewFlags.m_SelectBuf = NULL;
  }


  glFlush();
  glFinish();
  
  GLSwapBuffers();
}

void COpenGLWnd::OnPaint() 
{
  CPaintDC PaintDc(this);
  
  DrawView();
  
}

DWORD COpenGLWnd::ViewPopupMenu(int x, int y)
{
  CMenu    Menu;
  CMenu    *Popup;
  DWORD    Choice;
  
  Menu.LoadMenu(IDR_OPENGL_VIEWPORT_POPUP);
  
  Popup = Menu.GetSubMenu(0);
  InitPopupMenu(Popup);
  
  
  // Check the carious menu items depending on the current flags etc.
  
  if (m_Flags & GLVIEW_FLAGS_CULLBACKFACE)
    Popup->CheckMenuItem(ID_BACKFACECULL, MF_CHECKED);
  else
    Popup->CheckMenuItem(ID_BACKFACECULL, MF_UNCHECKED);
  
  
  if (m_Flags & GLVIEW_FLAGS_DRAWGRID) {
    Popup->CheckMenuItem(ID_POPUP_GRID_VISIBLE, MF_CHECKED);
  } else {
    Popup->CheckMenuItem(ID_POPUP_GRID_VISIBLE, MF_UNCHECKED);
  }
  
  Menu.CheckMenuRadioItem(ID_LITTEXTURED, ID_WIREFRAME, getMenuItemForMode(m_ShadingMode, m_TexturingMode, m_LightingMode), MF_BYCOMMAND);
  
  {
    DWORD Target;
    if (m_TextureMode == GL_NEAREST)
      Target = ID_POPUP_TEXTUREMODE_NEAREST;
    if (m_TextureMode == GL_LINEAR)
      Target = ID_POPUP_TEXTUREMODE_LINEAR;
    Menu.CheckMenuRadioItem(ID_POPUP_TEXTUREMODE_NEAREST, ID_POPUP_TEXTUREMODE_LINEAR, Target, MF_BYCOMMAND);
  }
  
  
  
  Choice = Popup->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_NONOTIFY | TPM_RETURNCMD, x, y, this);
  
  Popup->DestroyMenu();
  
  bool     Update;
  int      Result;
  
  Result = HandlePopupCommand(Choice);
  
  // The command was handled, but the view was changed, so we must return.
  if (Result == -1)
    return 1;
  
  if (Result == 1)
    Update = true;
  else 
    Update = false;
  
  if (Update)
  {
    DrawView();
  }
  
  return Choice;
}

int COpenGLWnd::HandlePopupCommand(DWORD Cmd)
{
  int Result;
  
  Result = MBaseOpenGLWnd::HandlePopupCommand(Cmd);
  
  if (Result)
    return Result;
  
  switch (Cmd)
  {
  case ID_LITTEXTURED:
    m_ShadingMode = glSmooth;
    m_TexturingMode = glLinear;
    m_LightingMode = glLit;
    return 1;
  case ID_TEXTURED:
    m_ShadingMode = glSmooth;
    m_TexturingMode = glLinear;
    m_LightingMode = glUnlit;
    return 1;
  case ID_SMOOTHSHADED:
    m_ShadingMode = glSmooth;
    m_TexturingMode = glNone;
    m_LightingMode = glLit;
    return 1;
  case ID_FLATSHADED:
    m_ShadingMode = glFlat;
    m_TexturingMode = glNone;
    m_LightingMode = glLit;
    return 1;
  case ID_WIREFRAME:
    m_ShadingMode = glWireframe;
    m_TexturingMode = glNone;
    m_LightingMode = glUnlit;
    return 1;
  case ID_BACKFACECULL:
    m_Flags = m_Flags ^ GLVIEW_FLAGS_CULLBACKFACE;
    return 1;
  case ID_POPUP_GRID_VISIBLE:
    m_Flags ^= GLVIEW_FLAGS_DRAWGRID;
    return 1;
  case ID_POPUP_GRID_USEXYPLANE:
    m_GridBasisX = MVector3(1,0,0);
    m_GridBasisY = MVector3(0,1,0);
    return 1;
  case ID_POPUP_GRID_USEXZPLANE:
    m_GridBasisX = MVector3(1,0,0);
    m_GridBasisY = MVector3(0,0,1);
    return 1;
  case ID_POPUP_GRID_USEYZPLANE:
    m_GridBasisX = MVector3(0,1,0);
    m_GridBasisY = MVector3(0,0,1);
    return 1;
  case ID_POPUP_TEXTUREMODE_NEAREST:
    m_TextureMode = GL_NEAREST;
    return 1;
  case ID_POPUP_TEXTUREMODE_LINEAR:
    m_TextureMode = GL_LINEAR;
    return 1;
  }
  
  return 0;
}

BOOL COpenGLWnd::OnCommand(WPARAM wParam, LPARAM lParam) 
{
  return CWnd::OnCommand(wParam, lParam);
}

void COpenGLWnd::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
//  m_ShiftState.SetFromKeyState();
  
  CWnd::OnKeyDown(nChar, nRepCnt, nFlags);
}

BOOL COpenGLWnd::PreTranslateMessage(MSG* pMsg) 
{
  return MBaseOpenGLWnd::PreTranslateMessage(pMsg);
}

void COpenGLWnd::SetGridBasis(const MVector3 &X, const MVector3 &Y)
{
  m_GridBasisX = X;
  m_GridBasisY = Y;
}

void COpenGLWnd::getGridBasis(MVector3 &x, MVector3 &y) {
  x = m_GridBasisX;
  y = m_GridBasisY;
}

MPlane COpenGLWnd::getGridPlane() {
  MVector3 normal = m_GridBasisX / m_GridBasisY;
  normal.normalize();

  return MPlane(MVector3(0,0,0), normal);
}

void COpenGLWnd::performSelection(int x, int y, std::vector<MSelectionItem> *list) {
  drawView(smSelect, list, x,y);
}

void COpenGLWnd::performSelection(int left, int top, int right, int bottom, std::vector<MSelectionItem> *list) {
  drawView(smSelectBox, list, left, top, right, bottom);
}


MToolTypePtr COpenGLWnd::getSelectTool() {
  return new MSelectToolType();
}

bool COpenGLWnd::isAnythingSelected() {
  return g_Scene->getNumSelectedObjects() > 0;
}

void COpenGLWnd::zoomIn() {
  if (isFlagged(GLVIEW_FLAGS_PERSPECTIVE)) {
    m_Dist -= 30;
  } else {
    m_Zoom *= 1.4f;
  }
}

void COpenGLWnd::zoomOut() {
  if (isFlagged(GLVIEW_FLAGS_PERSPECTIVE)) {
    m_Dist += 30;
  } else {
    m_Zoom /= 1.4f;
  }
}

void COpenGLWnd::zoomToFitAll() {
  // TODO: Change it so it zooms to everything instead of the selection
  MVector3 Centre = g_Scene->getSelectionCentre();
  m_Pan = Centre;
  m_Pan.z = -Centre.z;
}

void COpenGLWnd::zoomToFitSelected() {
  // TODO: make it so it does actual zooming, instead of just moving the
  // focus of the camera to the centre of the selection.
  MVector3 Centre = g_Scene->getSelectionCentre();
  m_Pan = Centre;
  m_Pan.z = -Centre.z;
}

void COpenGLWnd::onMouseDown(int X, int Y, MShiftState Shift)
{
  MSystemManager::getInstance()->getUndoManager()->beginUndo("Action");
  SetFocus();
  UpdateXFormMatricies();
  
  MToolType      *PushedTool = NULL;
  
  if (Shift.m_Left || Shift.m_Mid)
  {
    g_PickedManip = 0;      // set it so no manipulator as been picked
    
    // If our current too requires a selection, and we have none, put our tool to at select
    if (g_ToolMan.GetTool()->RequiresSelection()) {
      if (!isAnythingSelected()) {
        g_ToolMan.setTemporary(getSelectTool());
      }
    }
    
    if (getShiftState() == g_ProgSet.m_PanOrbitKey) {
      g_ToolMan.setTemporary(new MPanOrbitToolType(X, Y));
    } else if (getShiftState() == g_ProgSet.m_PanTrackKey) {
      g_ToolMan.setTemporary(new MPanTrackToolType(X, Y));
    } else if (getShiftState() == g_ProgSet.m_PanZoomKey) {
      g_ToolMan.setTemporary(new MPanZoomToolType(X, Y));
    }
    
  }
  
  // Draw the current tool using the select buffer to find out what manipulator we're using.
  {
    int      Res;
    
	   glMatrixMode(GL_PROJECTION);
     glPushMatrix();
     
     m_UpX = m_DownX;
     m_UpY = m_DownY;
     g_ToolMan.GetTool()->m_UpPos.set(X,Y,0);
     AdjustProjectionMatrix(smSelect,X,Y);
     Res = DrawManipulators(true);
     
     glMatrixMode(GL_PROJECTION);
     glPopMatrix();
     
     glMatrixMode(GL_MODELVIEW);
     
     // check to see if the user is using the default manipulator
     if (getShiftState() == g_ProgSet.m_ManipGrabKey)
       Res = g_ToolMan.GetTool()->GetDefaultManip();
     
     g_ToolMan.GetTool()->m_PickedManip = Res;
     
     if (Res == 0 && g_ToolMan.GetTool()->GetDefaultManip() != -1)  // The user missed all the manipultors
     {
       g_ToolMan.setTemporary(getSelectTool());
     }
  }
  
  g_ToolMan.GetTool()->m_DownRay = GetViewRay(X, Y);
  g_ToolMan.GetTool()->m_CurRay = GetViewRay(X, Y);
  g_ToolMan.GetTool()->m_DownPos.set(X,Y,0);
  g_ToolMan.GetTool()->m_CurPos.set(X,Y,0);

  // after setting up the view do the tool's action
  g_ToolMan.GetTool()->onMouseDown(X, Y, getShiftState());
  
  g_MainUndoNode = NULL;
}

void COpenGLWnd::onMouseUp(int X, int Y, MShiftState Shift)
{
  glDisable(GL_CLIP_PLANE0);
  GLMakeCurrent();
  
  // set up the current coordinates and what not of the current tool.
  g_ToolMan.GetTool()->m_CurRay = GetViewRay(X, Y);
  
  g_ToolMan.GetTool()->m_CurPos.set(X,Y,0);
  g_ToolMan.GetTool()->m_UpPos.set(X,Y,0);
  
  g_ToolMan.GetTool()->onMouseUp(X, Y, getShiftState());
  
  // check to see if no mouse buttons are pressed.
  if (!Shift.m_Left && !Shift.m_Mid && !Shift.m_Right) {
    m_UpX = X;
    m_UpY = Y;
    
    g_ToolMan.GetTool()->m_Dragging = false;
    
    DrawView();
    
    if (g_MainUndoNode != NULL) {
      Aztec::getSystemManager()->getUndoManager()->addUndoNode(g_MainUndoNode);
      g_MainUndoNode = NULL;
    }
    
    g_ToolMan.UnsetTemporary();
  }
  g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);

  MSystemManager::getInstance()->getUndoManager()->endUndo();
  
}

void COpenGLWnd::UpdateXFormMatricies()
{
  MBaseOpenGLWnd::UpdateXFormMatricies();
  m_ViewXForm.makeRotationMatrix((float)(-m_Rot.z*M_PI/180), (-m_Rot.x-180)*M_PI/180, 0);
  m_ViewXFormInv.makeRotationMatrix((float)(-m_Rot.z*M_PI/180), (-m_Rot.x-180)*M_PI/180, 0);
  m_ViewXFormInv.inverse();
}

void COpenGLWnd::onMouseMove(int mouseX, int mouseY, MShiftState Shift) 
{
  GLMakeCurrent();
  ::SetCursor(LoadCursor(NULL, IDC_ARROW));

  static int     ox,oy;
  bool           Update;
  
  Update = false;
  
  m_UpX = mouseX;
  m_UpY = mouseY;
  
  // set up the current coordinates and what not of the current tool.
  g_ToolMan.GetTool()->m_CurRay = GetViewRay(mouseX, mouseY);
  g_ToolMan.GetTool()->m_CurPos.set(mouseX,mouseY,0);
  g_ToolMan.GetTool()->m_UpPos.set(mouseX,mouseY,0);
  
  // Check to see if we are dragging the tool
  if (g_ToolMan.GetTool()->m_Dragging)
  {
    // If so, check to see if we need to wrap it.
    if (g_ToolMan.GetTool()->WrapMouse()) {
      MToolTypePtr Tool;
      Tool = g_ToolMan.GetTool();
      int   tx,ty,dx,dy;
      tx = Tool->m_CurPos.x;
      ty = Tool->m_CurPos.y;
      dx = Tool->m_DownPos.x;
      dy = Tool->m_DownPos.y;
      
      WrapMouseToScreen(m_hWnd, tx,ty,dx,dy);
      Tool->m_CurPos.set(tx,ty,0);
      Tool->m_DownPos.set(dx,dy,0);
    }
  }
  
  if (getShiftState() == g_ProgSet.m_PanOrbitKey)
    g_ToolMan.setTemporary(new MPanOrbitToolType(ox, oy));
  else if (getShiftState() == g_ProgSet.m_PanTrackKey)
    g_ToolMan.setTemporary(new MPanTrackToolType(ox, oy));
  else if (getShiftState() == g_ProgSet.m_PanZoomKey)
    g_ToolMan.setTemporary(new MPanZoomToolType(ox, oy));
  
  int result;
  result = g_ToolMan.GetTool()->onMouseMove(mouseX, mouseY, getShiftState());

  handleToolResult(result);
  
  ox = mouseX;
  oy = mouseY;
  
}

#if USE_SPACEBALL
void
COpenGLWnd::processSpaceballMovement(SiSpwEvent *event, SiGetEventData *eData)
{
	long tx, ty, tz;
	long rx, ry, rz;
	long msec;

	tx = event->u.spwData.mData[0]; // Translation along X axis
	ty = event->u.spwData.mData[1]; // Translation along Y axis
	tz = event->u.spwData.mData[2]; // Translation along Z axis
	rx = event->u.spwData.mData[3]; // Rotation about X axis
	ry = event->u.spwData.mData[4]; // Rotation about Y axis
	rz = event->u.spwData.mData[5]; // Rotation about Z axis
	msec = event->u.spwData.period; // Time in milliseconds since the last event

	float scale = 0.1f;

#if 0
	// TBD: Ignore pan for now - see if we can get rotating to work
	m_Pan.x += scale * tx;
	m_Pan.y += scale * ty;
	m_Pan.z += scale * tz;
#endif

#if 0
	m_Rot.x += scale * rx;
	m_Rot.y += scale * rz;
	m_Rot.z += scale * ry;
#endif

#if 0
    //camera->getShapeObject()->findParameter("fov")->getValueFloat(fov);
	// Move the eyepoint the appropriate amount.
	char sbuf[512];
	sprintf(sbuf, "View: '%s', Tx: (%d, %d, %d), Rx: (%d, %d, %d), Time: %d\n",
		((camera == NULL) ? m_ViewName.c_str() : camera->getName().c_str()),
		tx, ty, tz, rx, ry, rz, msec);
	OutputDebugString(sbuf);
#endif

#if 0
	// This failed to do anything...
    g_ToolMan.setTemporary(new MPanOrbitToolType(0, 0));
	g_ToolMan.GetTool()->onMouseMove(scale * rx, scale * ry, getShiftState());
	handleToolResult(TOOLRESULT_DRAWCURRENT);
    g_ToolMan.UnsetTemporary();
#else
	if (((int)m_Rot.x+360)%360 >180)
		m_Rot.z += (float)0.5*(scale * ry);
	else
		m_Rot.z -= (float)0.5*(scale * ry);

	m_Rot.x += (float)0.5*(scale * rx);

	if (m_Rot.x<0)
		m_Rot.x += 360;
	if (m_Rot.z<0)
		m_Rot.z += 360;

	if (m_Rot.x>360)
		m_Rot.x -= 360;
	if (m_Rot.z>360)
		m_Rot.z -= 360;
	if ( isFlagged(GLVIEW_FLAGS_PERSPECTIVE) == false) {
		if (m_Rot == MVector3(-90,180,0))
			SetViewName("Top");
		else if (m_Rot == MVector3(-90,90,0))
			SetViewName("Front");
		else if (m_Rot == MVector3(0,90,0))
			SetViewName("Right");
		else
			SetViewName("User");
	}
    
    UpdateXFormMatricies();
    DrawView();
#endif

	}
#endif

LRESULT COpenGLWnd::WindowProc(UINT message, WPARAM wParam, LPARAM lParam) 
{
  if (message == WM_MOUSEWHEEL)
  {
    short zDelta;
    
    zDelta = (short) HIWORD(wParam);
    if (zDelta < 0)
    {
      //         SetCurrentView(this);
      KViewportZoomOut();
    }
    else if (zDelta > 0)
    {
      //         SetCurrentView(this);
      KViewportZoomIn();
    }
    return 1;
  }
  
#if USE_SPACEBALL
	SiSpwEvent event; /* 3DxWare Event */
	SiGetEventData eData; /* 3DxWare Event Data */

	if (m_DevHdl != 0) {
		// Check if this is a 3DxWare event
		SiGetEventWinInit(&eData, message, wParam, lParam);
		if (SiGetEvent(m_DevHdl, 0, &eData, &event) == SI_IS_EVENT) {
			switch (event.type) {
			case SI_MOTION_EVENT:
				processSpaceballMovement(&event, &eData);

				break;

			case SI_ZERO_EVENT:
				break;

			case SI_BUTTON_EVENT:
				break; 

			default:
				break;
			}

			return 1;
		}
	}
#endif
	
  return CWnd::WindowProc(message, wParam, lParam);
}

void COpenGLWnd::OnClose() {
  g_ViewList.DeleteView(this);
  
#if USE_SPACEBALL
  if (m_DevHdl != NULL) {
	  SiClose(m_DevHdl);
	  m_DevHdl = NULL;
  }
#endif

  //	CWnd::OnClose();
}

int COpenGLWnd::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CWnd::OnCreate(lpCreateStruct) == -1)
		return -1;
	
#if USE_SPACEBALL
	// Tell the spaceball we want input for this window
	SiOpenData oData; // OS Independent data to open device.
	SiOpenWinInit(&oData, m_hWnd); // Initialize Windows specific data.
	if ((m_DevHdl = SiOpen("Aztec", SI_ANY_DEVICE, SI_NO_MASK, SI_EVENT, &oData)) == NULL ) {
		MessageBox("No supported 3DxWare device available.\n", NULL, MB_OK);
	} else {
		// Turn the driver GUI on.
		SiSetUiMode(&m_DevHdl, SI_UI_ALL_CONTROLS);
	}
#endif

	
	return 0;
}
