// ImageButton.cpp : implementation file
//

#include <AztecMainPCH.h>
#include "ImageButton.h"

#include "MToolTipWnd.h"

#include <io.h>
#include <fcntl.h>

#include "MDLGlobs.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//----------------------------------------------------------------------------------------
//  MDIBImage
//----------------------------------------------------------------------------------------
MDIBImage::MDIBImage()
{
  m_Palette = NULL;
  m_Pixels = NULL;
  
  m_hBitmap = NULL;
  
  m_Width = NULL;
  m_Height = NULL;
  
  m_CompleteHeader = NULL;
}


MDIBImage::~MDIBImage()
{
  DeleteBitmap();
  if (m_Palette)
  {
    delete[] m_Palette;
    m_Palette = NULL;
  }
}

int MDIBImage::SetFromBitmap(const MDIBImage &Bmp, HDC hDC)
{
  if (CreateEmptyBitmap(hDC, Bmp.m_Width, Bmp.m_Height, Bmp.m_Depth, Bmp.m_Palette) == 0)
    return 0;
  
  memcpy(m_Pixels, Bmp.m_Pixels, m_Width*m_Height*m_Depth/8);
  
  return 1;
}

int MDIBImage::CreateEmptyBitmap(HDC hDC, int Width, int Height, int Depth, RGBQUAD *SrcPalette)
{
  if (Width <= 0 || Height <= 0  || Depth <= 0)
    return 0;
  
  // Delete any previous bitmap.
  DeleteBitmap();
  
  // Fill out the BITMAPINFOHEADER structure with the correct values.
  m_BitmapInfoHeader.biSize = sizeof(BITMAPINFOHEADER);
  m_BitmapInfoHeader.biWidth = Width;
  m_BitmapInfoHeader.biHeight = Height;
  m_BitmapInfoHeader.biBitCount = Depth;
  m_BitmapInfoHeader.biPlanes = 1;
  m_BitmapInfoHeader.biCompression = BI_RGB;
  m_BitmapInfoHeader.biSizeImage = Width*Height*Depth/8;
  m_BitmapInfoHeader.biXPelsPerMeter = 72;
  m_BitmapInfoHeader.biYPelsPerMeter = 72;
  m_BitmapInfoHeader.biClrUsed = 0;
  m_BitmapInfoHeader.biClrImportant = 0;
  
  
  // Create an array that contians the bitmap info header, and the palette if necessary in one contigous chunk
  int      NumCols = 0;
  
  if (Depth < 16)
  {
    NumCols = 1 << Depth;
    if (m_Palette)
      delete[] m_Palette;
    m_Palette = new RGBQUAD[NumCols];
    
    if (SrcPalette == NULL)
    {
      GetSystemPaletteEntries(hDC, 0, NumCols, (PALETTEENTRY*)m_Palette);
    }
    else
    {
      memcpy(m_Palette, SrcPalette, sizeof(RGBQUAD)*NumCols);
    }
    
  }     
  
  m_CompleteHeader = new BYTE[sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD)*NumCols];
  memcpy(m_CompleteHeader, &m_BitmapInfoHeader, sizeof(BITMAPINFOHEADER));
  memcpy(m_CompleteHeader + sizeof(BITMAPINFOHEADER), m_Palette, sizeof(RGBQUAD)*NumCols);
  
  m_hBitmap = CreateDIBSection(hDC, (BITMAPINFO*)m_CompleteHeader, DIB_RGB_COLORS, (void**)&m_Pixels, NULL,0);
  
  m_Width = Width;
  m_Height = Height;
  m_Depth = Depth;
  
  if (m_hBitmap)
    return 1;
  
  return 0;
}

void MDIBImage::DeleteBitmap()
{
  if (!m_hBitmap)
    return;
  
  DeleteObject(m_hBitmap);
  m_hBitmap = NULL;
  m_Width = 0;
  m_Height = 0;
  m_Depth = 0;
  
  if (m_Palette)
    delete[] m_Palette;
  m_Palette = NULL;
  
  if (m_CompleteHeader)
  {
    delete[] m_CompleteHeader;
    m_CompleteHeader = NULL;
  }
}

int MDIBImage::Draw(HDC hDC, int x, int y, int w, int h)
{
  if (!m_hBitmap)
    return 0;
  
  if (w == -1)
    w = m_Width;
  if (h == -1)
    h = m_Height;
  
  return StretchDIBits(hDC, x, y, w, h, 0, 0, w, h, m_Pixels, (BITMAPINFO*)m_CompleteHeader, DIB_RGB_COLORS, SRCCOPY);
  //   return SetDIBitsToDevice(hDC, x, y, w, h, 0,m_Height-h,0, h, m_Pixels, (BITMAPINFO*)m_CompleteHeader, DIB_RGB_COLORS);
}

int MDIBImage::Draw(HDC hDC, int destx, int desty, int srcx, int srcy, int w, int h)
{
  if (!m_hBitmap)
    return 0;
  
  if (w > m_Width - srcx)
    w = m_Width - srcx;
  if (h > m_Height - srcy)
    h = m_Height - srcy;
  
  return StretchDIBits(hDC, destx, desty, w, h, srcx, srcy, w, h, m_Pixels, (BITMAPINFO*)m_CompleteHeader, DIB_RGB_COLORS, SRCCOPY);
  //   return SetDIBitsToDevice(hDC, destx, desty, w, h, srcx, srcy, srcy, h, m_Pixels, (BITMAPINFO*)m_CompleteHeader, DIB_RGB_COLORS);
}


int MDIBImage::LoadFromBMP(const char *Filename, HDC hDC)
{
  MImagePtr image = g_SysMan->loadImage(Filename);

  if (image != NULL) {
    setFromImage(image, hDC);
    return 1;
  }

  return 0;
}

void MDIBImage::setFromImage(const MImagePtr &newImage, HDC hDC) {
  if (newImage == NULL) {
    DeleteBitmap();
    return;
  }
  if (CreateEmptyBitmap(hDC, newImage->getWidth(), newImage->getHeight(), 24))
  {
    
    // the unfortunate part there is that RGB needs to be conveted to BGR, and
    // the image also needs to be turned upside down
    newImage->flipVertical();
    int imageSize = newImage->getWidth()*newImage->getHeight()*3;
    memcpy(m_Pixels, newImage->getPixelData(), imageSize);
    
    for (int i = 0; i < imageSize; i += 3) {
      BYTE r = m_Pixels[i];
      BYTE b = m_Pixels[i+2];
      
      m_Pixels[i+2] = r;
      m_Pixels[i] = b;
    }

    // now flip it back to normal
    newImage->flipVertical();
  }
}

class ButtonToolListener : public MToolChangeListener {
public:
  ButtonToolListener(CImageButton *but) {
    m_But = but;
  }

  void onToolChange(const std::string &group, MToolTypePtr oldTool) {
    if (group != m_But->getToolGroup().c_str()) {
      return;
    }

    std::string toolName = g_ToolMan.GetTool(m_But->getToolGroup().c_str())->getName().c_str();
    std::string oldToolName;
    std::string actionName = m_But->GetActionName().c_str();

    if (oldTool != NULL) {
      oldToolName = oldTool->getName().c_str();
    }

    if (toolName == actionName) {
      m_But->setChecked(true);
    } else if (oldTool != NULL && oldToolName == m_But->GetActionName().c_str()) {
      m_But->setChecked(false);
    }
  }

protected:
  CImageButton *m_But;
};

//----------------------------------------------------------------------------------------
//  CImageButton
//----------------------------------------------------------------------------------------

CImageButton::CImageButton(const MStr &actionName, const MStr &toolGroup)
{
  m_Timed = 0;
  m_FirstTimer = 0;
  m_TimerID = 0;
  m_Executing = false;
  m_ToolTipTimed = false;
  m_Checked = false;
  g_ToolMan.addListener(listener = new ButtonToolListener(this));

  m_ActionName = actionName;
  m_ToolGroup = toolGroup;

  m_Repeatable = false;
}

CImageButton::~CImageButton() {
  g_ToolMan.removeListener(listener);
  listener = NULL;
}

BEGIN_MESSAGE_MAP(CImageButton, CButton)
//{{AFX_MSG_MAP(CImageButton)
ON_WM_SETFOCUS()
//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CImageButton message handlers

void CImageButton::DrawItem(LPDRAWITEMSTRUCT lpDrawItemStruct) 
{
  // do stuff
  //   HBRUSH   hBrush;
  RECT     Rect;
  HDC      DC;
  DWORD topLeftCol, botRightCol;

  if (lpDrawItemStruct->itemState & ODS_SELECTED) {
    botRightCol = ::GetSysColor(COLOR_3DHILIGHT);
    topLeftCol = ::GetSysColor(COLOR_3DSHADOW);
  } else {
    topLeftCol = ::GetSysColor(COLOR_3DHILIGHT);
    botRightCol = ::GetSysColor(COLOR_3DSHADOW);
  }
  
  DC = ::GetDC(m_hWnd);
  
  GetWindowRect(&Rect);
  ScreenToClient(&Rect);
  
  HPEN     hPen, hOldPen;

  
  hPen = ::CreatePen(PS_SOLID, 1, topLeftCol);
  hOldPen = (HPEN)::SelectObject(lpDrawItemStruct->hDC, hPen);

  Rect = lpDrawItemStruct->rcItem;
  Rect.right--;
  Rect.bottom--;
  
/*  ::MoveToEx(lpDrawItemStruct->hDC, Rect.left,Rect.bottom, NULL);
  ::LineTo(lpDrawItemStruct->hDC, Rect.left, Rect.top);
  ::LineTo(lpDrawItemStruct->hDC, Rect.right, Rect.top);

  ::SelectObject(lpDrawItemStruct->hDC, hOldPen);
  ::DeleteObject(hPen);
  hPen = ::CreatePen(PS_SOLID, 1, botRightCol);

  ::LineTo(lpDrawItemStruct->hDC, Rect.right, Rect.bottom);
  ::LineTo(lpDrawItemStruct->hDC, Rect.left,Rect.bottom);
  
  ::SelectObject(lpDrawItemStruct->hDC, hOldPen);
  ::DeleteObject(hPen);
*/  
  
  // We draw a green border around the button if this is
  // our current tool.
  if ( (g_ToolMan.GetTool(m_ToolGroup.c_str()) != NULL && 
        g_ToolMan.GetTool(m_ToolGroup.c_str())->Is(m_ActionName)) ||
       (isChecked())) {
    hPen = ::CreatePen(PS_SOLID, 1, 0x007C00);
    hOldPen = (HPEN)::SelectObject(lpDrawItemStruct->hDC, hPen);
    
    Rect.left++;
    Rect.right--;
    Rect.top++;
    Rect.bottom--;
    
    ::MoveToEx(lpDrawItemStruct->hDC, Rect.left,Rect.top, NULL);
    ::LineTo(lpDrawItemStruct->hDC, Rect.right, Rect.top);
    ::LineTo(lpDrawItemStruct->hDC, Rect.right, Rect.bottom);
    ::LineTo(lpDrawItemStruct->hDC, Rect.left, Rect.bottom);
    ::LineTo(lpDrawItemStruct->hDC, Rect.left,Rect.top);
    
    Rect.left++;
    Rect.right--;
    Rect.top++;
    Rect.bottom--;
    
    ::MoveToEx(lpDrawItemStruct->hDC, Rect.left,Rect.top, NULL);
    ::LineTo(lpDrawItemStruct->hDC, Rect.right, Rect.top);
    ::LineTo(lpDrawItemStruct->hDC, Rect.right, Rect.bottom);
    ::LineTo(lpDrawItemStruct->hDC, Rect.left, Rect.bottom);
    ::LineTo(lpDrawItemStruct->hDC, Rect.left,Rect.top);
    
    ::SelectObject(lpDrawItemStruct->hDC, hOldPen);
    ::DeleteObject(hPen);
    
    if (lpDrawItemStruct->itemState & ODS_SELECTED) {
      m_Image.Draw(DC, 3,3, 2, 4, m_Image.m_Width - 6, m_Image.m_Height - 6);
    } else {
      m_Image.Draw(DC, 3,3, 3, 3, m_Image.m_Width - 6, m_Image.m_Height - 6);
    }

  // otherwise we draw the image as per normal.
  } else {
    if (lpDrawItemStruct->itemState & ODS_SELECTED) {
      m_Image.Draw(DC, 1,1, 0, 2, m_Image.m_Width - 2, m_Image.m_Height - 2);
    } else {
      m_Image.Draw(DC, 1,1, 1, 1, m_Image.m_Width - 2, m_Image.m_Height - 2);
    }
  }
  
  ::ReleaseDC(m_hWnd, DC);
}

int CImageButton::SetFrom(const CImageButton &But, bool ResizeCtrl)
{
  m_Image.DeleteBitmap();
  m_Image.SetFromBitmap(But.m_Image);
  
  m_ActionName = But.m_ActionName;
  m_ImageFilename = But.m_ImageFilename;
  
  if (ResizeCtrl)
  {
    RECT     Rect;
    
    GetWindowRect(&Rect);
    GetParent()->ScreenToClient(&Rect);
    MoveWindow(Rect.left,Rect.top,m_Image.m_Width, m_Image.m_Height);
  }
  
  return 1;
}


int CImageButton::SetImage(const MStr &Filename, bool ResizeCtrl)
{
  return SetImage((LPCTSTR)Filename, ResizeCtrl);
}

int CImageButton::SetImage(const char *Filename, bool ResizeCtrl)
{
  int result = 0;
  HDC hDC = ::GetDC(m_hWnd);

  if (m_Image.LoadFromBMP(Filename, hDC))
  {
#if 0
    bool doAlpha = false;
    MDIBImage multiply, alpha;

    // if we have a 32x32 button, do some rgoovy alpha blending on it.
    if (m_Image.m_Width == 32 && m_Image.m_Height == 32) {
      multiply.LoadFromBMP(g_ProgSet.m_PrefsPath + "\\Buttons\\Shadowing32.bmp");
      alpha.LoadFromBMP(g_ProgSet.m_PrefsPath + "\\Buttons\\buttonAlpha32.bmp");
      doAlpha = true;

    } else if (m_Image.m_Width == 24 && m_Image.m_Height == 24) {
      multiply.LoadFromBMP(g_ProgSet.m_PrefsPath + "\\Buttons\\Shadowing24.bmp");
      alpha.LoadFromBMP(g_ProgSet.m_PrefsPath + "\\Buttons\\buttonAlpha24.bmp");
      doAlpha = true;

    } else if (m_Image.m_Width == 20 && m_Image.m_Height == 20) {
      multiply.LoadFromBMP(g_ProgSet.m_PrefsPath + "\\Buttons\\Shadowing20.bmp");
      alpha.LoadFromBMP(g_ProgSet.m_PrefsPath + "\\Buttons\\buttonAlpha20.bmp");
      doAlpha = true;
    }


    if (doAlpha) {
      DWORD btnCol = ::GetSysColor(COLOR_3DFACE);
      int c[3];

      c[0] = btnCol & 0x0000FF;
      c[1] = (btnCol & 0x00FF00) >> 8;
      c[2] = (btnCol & 0xFF0000) >> 16;

      // go through the image, and do some math
      for (int i = m_Image.m_Width*m_Image.m_Height*3-1; i >= 0; --i) {
        float newVal, imgSrc, mulSrc, alphaSrc, colSrc;

        imgSrc = (float)m_Image.m_Pixels[i]/255.0;
        mulSrc = (float)multiply.m_Pixels[i]/255.0;
        alphaSrc = (float)alpha.m_Pixels[i]/255.0;
        colSrc = (float)c[i%3]/255.0;

        newVal = imgSrc * (mulSrc * 2);
        newVal = newVal * alphaSrc + colSrc * (1.0 - alphaSrc);
        if (newVal < 0) newVal = 0;
        if (newVal > 1) newVal = 1;
        m_Image.m_Pixels[i] = 255 * newVal;
      }
    }
#endif
    
    RECT     Rect;
    
    m_ImageFilename = Filename;
    
    if (ResizeCtrl)
    {
      GetWindowRect(&Rect);
      GetParent()->ScreenToClient(&Rect);
      MoveWindow(Rect.left,Rect.top,m_Image.m_Width, m_Image.m_Height);
    }

    ModifyStyle(0, BS_BITMAP);
    SetBitmap(m_Image.m_hBitmap);
    HBITMAP hb = GetBitmap();

    result = 1;
  }
  
  ::ReleaseDC(m_hWnd, hDC);

  Invalidate();
  return result;
}                   

void CImageButton::SetActionName(const MStr &ActionName)
{
  m_ActionName = ActionName;
  Invalidate();
}

void CImageButton::setChecked(bool checked) { 
  if ((m_Checked != checked) || (GetCheck() != 0) != checked) {
    m_Checked = checked; 
    SetCheck(checked);
    Invalidate(FALSE);
  }
}

bool CImageButton::isChecked() {
  return m_Checked;
}

void CImageButton::setPosition(int left, int top) {
  SetWindowPos(NULL, left, top, -1, -1, SWP_NOSIZE | SWP_NOZORDER);
}

void CImageButton::setLeft(int left) {
  setPosition(left, getTop());
}
void CImageButton::setTop(int top) {
  setPosition(getLeft(), top);
}

void CImageButton::setSize(int width, int height) {
  SetWindowPos(NULL, -1, -1, width, height, SWP_NOMOVE | SWP_NOZORDER);
}

int CImageButton::getWidth() {
  RECT rect;
  getPosition(&rect);
  return rect.right - rect.left;
}

int CImageButton::getHeight() {
  RECT rect;
  getPosition(&rect);
  return rect.bottom - rect.top;
}

int CImageButton::getLeft() {
  RECT rect;
  getPosition(&rect);
  return rect.left;
}

int CImageButton::getTop() {
  RECT rect;
  getPosition(&rect);
  return rect.top;
}

void CImageButton::getPosition(RECT *rect) {
  GetWindowRect(rect);

  CWnd *parent = GetParent();

  if (parent != NULL) {
    parent->ScreenToClient(rect);
  }

}

void CImageButton::setRepeatable(bool repeatable) {
  m_Repeatable = repeatable;
}

bool CImageButton::isRepeatable() {
  return m_Repeatable;
}


LRESULT CImageButton::WindowProc(UINT message, WPARAM wParam, LPARAM lParam) 
{
  if (message == BM_SETCHECK)
  {
    CButton::WindowProc(message, wParam, lParam);
    Invalidate();
    return 1;
  }
  if (message == WM_TIMER)
  {
    if (isRepeatable() && wParam == m_TimerID) {
      if (m_FirstTimer)
      {
        ::KillTimer(m_hWnd, m_TimerID);
        ::SetTimer(m_hWnd, m_TimerID, 100, NULL);
        m_FirstTimer = 0;
      }
      m_Timed = 1;
      SendMessage(BM_CLICK, 0,0);
      m_Timed = 0;
    
    }
    if (wParam == 2000)
    {
      //         g_ToolTipWnd.CreateToolTip(m_hWnd, m_TipX + 10, m_TipY + 10, 2000, 2000, "Test");
      m_ToolTipTimed = false;
    }
  }
  if (message == WM_LBUTTONDOWN || message == WM_LBUTTONDBLCLK)
  {
    // first we check to see if we are in a view
    setCurrentViewForControl(this);

    MShiftState    ShiftState;
    ShiftState.SetFromKeyState();
    
    if (ShiftState.m_Ctrl)
    {
      g_ProgSet.ShowToolParameterEditWindow(m_ActionName);
    }
    else
    {
      // set up a timer to control repeated clicks, use the control id as a timer id
      if (m_TimerID == 0)
      {
        // Set this window to capture mouse input so we can slways detect a mouse up
        ::SetCapture(m_hWnd);
        
        if (isRepeatable() && !m_Timed) {
          m_FirstTimer = 1;
          m_TimerID = GetDlgCtrlID();
          ::SetTimer(m_hWnd, m_TimerID, 500, NULL);
        }
      }
      if (m_ActionName.GetLength() > 0)
      {
        if (!m_Executing)
        {
          m_Executing = true;
          g_KeyList.ExecuteAction(m_ActionName);
          m_Executing = false;
        }
      }
    }
    return 0;
  }
  if (message == WM_LBUTTONUP)
  {
    if (m_TimerID != 0) {
      if (!m_Timed) {
        // Kill the timer to control repreated clicks
        ::KillTimer(m_hWnd, m_TimerID);
        m_TimerID = 0;
      }
    }
    if (!isRepeatable() || !m_Timed)
    {
      ::ReleaseCapture();
    }
    return 0;
  }

  if (message == WM_MOUSEMOVE)
  {
    if (!m_ToolTipTimed || g_ToolTipWnd.GetMouseOverWnd() != m_hWnd)
      m_TipX = LOWORD(lParam);
    m_TipY = HIWORD(lParam); 
    if (g_ToolTipWnd.GetParentWnd() != m_hWnd)
    {
      MKeystroke  *Key;
      MAction     *Act;
      MStr        ToolDisplay;
      
      Key = g_KeyList.FindKey(m_ActionName, "");
      
      if (Key)
      {
        ToolDisplay.Format("%s (%s)", (LPCTSTR)Key->getAction()->m_DisplayName, (LPCTSTR)Key->GetKeystrokeString());
      }
      else
      {
        Act = g_KeyList.FindAction(m_ActionName, "");
        if (Act)
        {
          ToolDisplay = Act->m_DisplayName;
        }
        else
        {
          ToolDisplay = m_ActionName;
        }
      }
      
      
      g_ToolTipWnd.CreateToolTip(m_hWnd, m_TipX + 10, m_TipY + 10, 500, 2000, ToolDisplay);
      SetTimer(2000, 2000, NULL);
      m_ToolTipTimed = true;
    }
    
    
    return 1;
  }
  
  
  return CButton::WindowProc(message, wParam, lParam);
}

void CImageButton::OnSetFocus( CWnd* pOldWnd ) {
  // do not allow image buttons to ever recieve focus.
  ::SetFocus(NULL);
}