#include "StdAfx.h"

#include "MMesh.h"
#include "MMeshShape.h"
#include "MShapeObject.h"
#include <params/MArrayParameter.h>

#include "scripting/ScriptSceneObject.h"
#include "scripting/ScriptVectorParam.h"
#include "scripting/MScriptInterp.h"
#include "scripting/ScriptUtils.h"
#include "scripting/MJavaScriptSupport.h"

#include "scripting/ScriptVector3.h"
#include "scripting/ScriptMesh.h"
#include "scripting/ScriptArrayParam.h"

#include "jsstddef.h"
#include "jscntxt.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {  
	
JSObject *
getScriptObject(JSContext *cx, JSObject *obj, MNamedObject *Obj)
{
	// Get the generic scripting support wrapper.
	MScriptingSupportPtr supportPtr = Obj->getScriptObject();

	// Attempt to cast to Javascript support wrapper
	MJavaScriptSupport *jscriptPtr = static_cast<MJavaScriptSupport *>(supportPtr.m_Ptr);

	// jobj will hold the Javascript unique object that reflects an Aztec objec.
	JSObject *jobj = NULL;

	// If the cast is unsuccessful, then drop down to the lowest level of support
	// (Reflection of a base MNamedObject).
	if (jscriptPtr == NULL) {
		MJavaScriptSupport jSupport;
		jobj = jSupport.getJavaScriptObject(cx, obj, Obj);
	} else {
		// If the cast is successful, then there is direct support for scripting.
		// Ask the object how it wants to reflect itself.
		jobj = jscriptPtr->getJavaScriptObject(cx, obj, Obj);
	}

	return jobj;
}

static void
updateAztecParam(MScriptInterpreter *interp, const MNamedObjectPtr &obj)
{
	updateParamObject(obj);
}

static JSBool
getMesh(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MNamedObject *AztecObj = (MNamedObject *)JS_GetInstancePrivate(cx, obj, &aztecSceneObject_class, NULL);
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	if (AztecObj == NULL || interp == NULL || argc != 0) {
		return JS_TRUE;
	}

	MMeshShapePtr shape = AZTEC_CAST(MMeshShape, AztecObj);
	if (shape == NULL) {
		// Try treating this as a scene object
		MSceneObject *sceneObj = AZTEC_CAST(MSceneObject, AztecObj);
		if (sceneObj != NULL) {
			MShapeObjectPtr shapeObj = sceneObj->getShapeObject();
			shape = AZTEC_CAST(MMeshShape, shapeObj);
		}
	}

	if (shape != NULL) {
		MMeshPtr mesh = shape->getMeshObject();
		JSObject *mobj = newMesh(cx, mesh);
		*rval = OBJECT_TO_JSVAL(mobj);
		return JS_TRUE;
	} else {
		JS_ReportError(cx, "%s doesn't have an editable mesh", AztecObj->getName().c_str());
	}


	// Don't know how to convert the arguments into the components of
	// a vector.
	return JS_FALSE;
}

static JSFunctionSpec aztecSceneObject_methods[] = {
	{"getMesh",		getMesh,		1},
	{0}
};

// See if a particular property exists in the Aztec object.  
static JSBool
aztec_lookupProperty(JSContext *cx, JSObject *obj, jsid id,
                     JSObject **objp, JSProperty **propp)
{
	MNamedObject *AztecObj = (MNamedObject *)JS_GetPrivate(cx, obj);
	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
	if (AztecObj == NULL || interp == NULL) {
		return JS_TRUE;
	}

    // OutputDebugString("Aztec lookupProperty\n");
    
    jsval idval;
	if (!JS_IdToValue(cx, id, &idval)) {
		// Failed to resolve the property name
        *objp = NULL;
        *propp = NULL;
		return JS_TRUE;
	}

    if (JSVAL_IS_STRING(idval)) {
		// Look for a property by name
		char *prop_name = JS_GetStringBytes(JSVAL_TO_STRING(idval));
		// See if any of the properties on the list match prop_name
		MStr val;
		MParameterObjectPtr param = AztecObj->findParameter(prop_name);
		*objp = obj;
		if (param != NULL) {
			*propp = (JSProperty *)1;
		} else {
			*propp = NULL;
		}
	} else if (JSVAL_IS_INT(idval)) {
		// Look for a property by index
		int i = JSVAL_TO_INT(idval);
		MParameterObjectPtr param = AztecObj->getParamList()->getParameter(i);
        *objp = obj;
		if (param != NULL) {
	        *propp = (JSProperty *)1;
		} else {
			*propp = NULL;
		}
	} else {
		// Failed to resolve the property name
        *objp = NULL;
        *propp = NULL;
	}

    return JS_TRUE;
}

static JSBool
aztec_defineProperty(JSContext *cx, JSObject *obj, jsid id, jsval value,
                         JSPropertyOp getter, JSPropertyOp setter,
                         uintN attrs, JSProperty **propp)
{
	// OutputDebugString("Attempt to define aztec property\n");

    return JS_TRUE;
}

static JSBool
aztec_getPropertyById(JSContext *cx, JSObject *obj, jsid id, jsval *vp)
{
	MNamedObject *AztecObj = (MNamedObject *)JS_GetPrivate(cx, obj);
	if (AztecObj == NULL) {
		return JS_TRUE;
	}

	jsval paramVal;
	if (!JS_IdToValue(cx, id, &paramVal)) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(paramVal)) {
		char *paramStr = JS_GetStringBytes(JS_ValueToString(cx, paramVal));
		MStr val;
		MParameterObjectPtr param = AztecObj->findParameter(paramStr);
		if (param != NULL) {
			*vp = convertParam(cx, obj, AztecObj, param);
			return JS_TRUE;
		}

		// May be a built-in function.
		for (int m=0;aztecSceneObject_methods[m].name != NULL;m++) {
			if (!strcmp(paramStr, aztecSceneObject_methods[m].name)) {
				// Yup - is a built-in function.
				JSFunction *jfunc =
					JS_NewFunction(cx, aztecSceneObject_methods[m].call,
									aztecSceneObject_methods[m].nargs,
									0, obj, paramStr);
				JSObject *jfobj = JS_GetFunctionObject(jfunc);
				*vp = OBJECT_TO_JSVAL(jfobj);
				return JS_TRUE;
			}
		}

		JS_ReportError(cx, "%s is not defined", paramStr);
		return JS_FALSE;
    } else if (JSVAL_IS_INT(paramVal)) {
		int paramNum = AztecObj->getParamList()->getNumParams();
		int paramId = JSVAL_TO_INT(paramVal);

		if (paramId < 0 || paramId > paramNum) {
			JS_ReportError(cx, "%s[%d] is not defined",
				AztecObj->getName().c_str(), paramId);
			return JS_FALSE;
		}

		MParameterObjectPtr param = AztecObj->getParamList()->getParameter(paramId);
		if (param == NULL) {
			JS_ReportError(cx, "%s[%s] is not defined",
				AztecObj->getName().c_str(), JS_GetStringBytes(JSVAL_TO_STRING(paramVal)));
			return JS_FALSE;
		}
		*vp = convertParam(cx, obj, AztecObj, param);
		return JS_TRUE;
	}

	JS_ReportError(cx, "Scene object property not found: %s",
		JS_GetStringBytes(JSVAL_TO_STRING(paramVal)));
	return JS_FALSE;
}

static JSBool
aztec_setPropertyById(JSContext *cx, JSObject *obj, jsid id, jsval *vp)
{
	MNamedObject *AztecObj = (MNamedObject *)JS_GetPrivate(cx, obj);
	if (AztecObj == NULL) {
		return JS_TRUE;
	}

	jsval paramVal;
	if (!JS_IdToValue(cx, id, &paramVal)) {
		return JS_FALSE;
	}

	char *propStr = JS_GetStringBytes(JS_ValueToString(cx, paramVal));
	char *valStr = JS_GetStringBytes(JS_ValueToString(cx, *vp));
#if 0
	OutputDebugString("Aztec setPropertyById: ");
	OutputDebugString(propStr);
	OutputDebugString(" : ");
	OutputDebugString(valStr);
	OutputDebugString("\n");
#endif

	MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);

	if (JSVAL_IS_STRING(paramVal)) {
		char *paramStr = JS_GetStringBytes(JS_ValueToString(cx, paramVal));
		MStr val;
		MParameterObjectPtr param = AztecObj->findParameter(paramStr);
		if (param != NULL) {
			if (JSVAL_IS_OBJECT(*vp)) {
				// This is an object param, so it is ok to hand it an object.
				int dataType = param->getDataType();
				JSObject *jsobj = JSVAL_TO_OBJECT(*vp);
				void *priv;
				
				// See if we assigning an Aztec object to an object parameter
				priv = JS_GetInstancePrivate(cx, jsobj, &aztecSceneObject_class, NULL);
				if (dataType == MParameterObject::TYPE_OBJECT && priv != NULL) {
					MObjectParameterPtr oparam = AZTEC_CAST(MObjectParameter, param);
					if (oparam != NULL) {
						MNamedObject *RHSObj = (MNamedObject *)priv;
						oparam->setValue(RHSObj);
						updateAztecParam(interp, AztecObj);
						return JS_TRUE;
					}
				}

				// See if we are assigning a vector to a vector parameter.
				priv = JS_GetInstancePrivate(cx, jsobj, &aztecVector3_class, NULL);
				if (dataType == MParameterObject::TYPE_VECTOR && priv != NULL) {
          MVector3 *RHSVec = (MVector3 *)priv;
          if (param->setValueVector(*RHSVec)) {
            updateAztecParam(interp, AztecObj);
            return JS_TRUE;
          }
				}
			}

			// Not assigning an object to an object parameter.  Therefore just set
			// the value based on the string representation of the RHS.
			param->setValueString(valStr);
			updateAztecParam(interp, AztecObj);
			return JS_TRUE;
		}

    } else if (JSVAL_IS_INT(id)) {
		int paramId = JSVAL_TO_INT(id);
		MParameterObjectPtr param = AztecObj->getParamList()->getParameter(paramId);
		if (param != NULL) {
			param->setValueString(valStr);
			updateAztecParam(interp, AztecObj);
		}
	} else {
		// What does this mean?  Perhaps a property outside the list of
		// parameters for this object...
	}

	return JS_TRUE;
}

static JSBool
aztec_getAttributes(JSContext *cx, JSObject *obj, jsid id,
                    JSProperty *prop, uintN *attrsp)
{
	OutputDebugString("Aztec getAttributes\n");

    // We don't maintain JS property attributes for Aztec objects
    *attrsp = JSPROP_PERMANENT | JSPROP_ENUMERATE;
    return JS_FALSE;
}

static JSBool
aztec_setAttributes(JSContext *cx, JSObject *obj, jsid id,
                    JSProperty *prop, uintN *attrsp)
{
	OutputDebugString("Aztec setAttributes\n");

    // We don't maintain JS property attributes for Aztec objects
    if (*attrsp != (JSPROP_PERMANENT | JSPROP_ENUMERATE)) {
        return JS_FALSE;
    }

    // Silently ignore all setAttribute attempts
    return JS_TRUE;
}

static JSBool
aztec_deleteProperty(JSContext *cx, JSObject *obj, jsid id, jsval *vp)
{
	OutputDebugString("Aztec deleteProperty\n");

	return JS_FALSE;
}

static JSBool
aztec_defaultValue(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	// OutputDebugString("Aztec defaultValue\n");
    switch (typ) {
    case JSTYPE_OBJECT:
        *vp = OBJECT_TO_JSVAL(obj);
        return JS_TRUE;

    case JSTYPE_FUNCTION:
        return JS_FALSE;

    case JSTYPE_VOID:
    case JSTYPE_STRING:
		{
		MNamedObject *AztecObj = (MNamedObject *)JS_GetPrivate(cx, obj);
		if (AztecObj == NULL) {
			JSString *str = JS_NewStringCopyZ(cx, "[Aztec Object]");
			*vp = STRING_TO_JSVAL(str);
		} else {
			MStr valStr;
			valStr = "[Aztec Object: ";
			valStr += AztecObj->getName();
			valStr += "]";
			JSString *str = JS_NewStringCopyZ(cx, valStr);
			*vp = STRING_TO_JSVAL(str);
		}
		return JS_TRUE;
		}

    case JSTYPE_NUMBER:
		return JS_FALSE;

    case JSTYPE_BOOLEAN:
	    return JS_FALSE;

    default:
        return JS_FALSE;
    }
}

static JSBool
aztec_newEnumerate(JSContext *cx, JSObject *obj, JSIterateOp enum_op,
                   jsval *statep, jsid *idp)
{
	MNamedObject *AztecObj = (MNamedObject *)JS_GetPrivate(cx, obj);

    // Check for native object
    if (AztecObj == NULL) {
		// No native object - say we don't have any properties.
        *statep = JSVAL_NULL;
        if (idp != NULL) {
            *idp = INT_TO_JSVAL(0);
		}
        return JS_TRUE;
    }

    switch(enum_op) {
    case JSENUMERATE_INIT:
		{
		MParameterObjectListPtr params = AztecObj->getParamList();
		if (params != NULL) {
			int paramCount = params->getNumParams();

			AztecEnumState *enumState = new AztecEnumState;
			enumState->numParams = paramCount;
			enumState->index = 0;
			*statep = PRIVATE_TO_JSVAL(enumState);
			if (idp != NULL) {
				*idp = INT_TO_JSVAL(paramCount);
			}
		} else {
			*statep = JSVAL_NULL;
		}
        return JS_TRUE;
		}
        
    case JSENUMERATE_NEXT:
		{
		MParameterObjectListPtr params = AztecObj->getParamList();
		if (params != NULL) {
			AztecEnumState *enumState = (AztecEnumState *)JSVAL_TO_PRIVATE(*statep);
			if (enumState->index < enumState->numParams) {
				MParameterObjectPtr param = params->getParameter(enumState->index);
				MStr paramStr = param->getShortName();
				JSString *str = JS_NewStringCopyZ(cx, paramStr.c_str());
	            JS_ValueToId(cx, STRING_TO_JSVAL(str), idp);
				enumState->index++;
				return JS_TRUE;
			}
		}
		// Drop through to destroy
		}

    case JSENUMERATE_DESTROY:
		{
		AztecEnumState *enumState = (AztecEnumState *)JSVAL_TO_PRIVATE(*statep);
		delete enumState;
        *statep = JSVAL_NULL;
        return JS_TRUE;
		}

    default:
		OutputDebugString("Bad enumeration state\n");
        return JS_FALSE;
    }
}

static JSBool
aztec_checkAccess(JSContext *cx, JSObject *obj, jsid id,
                      JSAccessMode mode, jsval *vp, uintN *attrsp)
{
	OutputDebugString("Aztec checkAccess\n");

    switch (mode) {
    case JSACC_WATCH:
        return JS_FALSE;

    case JSACC_IMPORT:
        return JS_FALSE;

    default:
        return JS_TRUE;
    }
}

// Specialized object operations - these allow us to dynamically reflect
// Aztec properties into the script interpreter.
JSObjectOps aztec_ops = {
    // Mandatory non-null function pointer members.
    js_ObjectOps.newObjectMap,
    js_ObjectOps.destroyObjectMap,
    aztec_lookupProperty,
    aztec_defineProperty,
    aztec_getPropertyById,  // getProperty
    aztec_setPropertyById,  // setProperty
    aztec_getAttributes,
    aztec_setAttributes,
    aztec_deleteProperty,
    aztec_defaultValue,
    aztec_newEnumerate,
    aztec_checkAccess,

    // Optionally non-null members start here.
    NULL,                       /* thisObject */
    NULL,                       /* dropProperty */
    NULL,                       /* call */
    NULL,                       /* construct */
    NULL,                       /* xdrObject */
    NULL,                       /* hasInstance */
    NULL,                       /* setProto */
    NULL,                       /* setParent */
    NULL,                       /* mark */
    NULL,                       /* clear */
    NULL,                       /* getRequiredSlot */
    NULL                        /* setRequiredSlot */
};

static JSObjectOps *
aztec_getObjectOps(JSContext *cx, JSClass *clazz)
{
    return &aztec_ops;
}

static JSBool
aztec_convert(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
#if 0
    OutputDebugString("converting it to ");
	OutputDebugString(JS_GetTypeName(cx, typ));
	OutputDebugString(" type\n");
#endif

    switch (typ) {
    case JSTYPE_OBJECT:
        *vp = OBJECT_TO_JSVAL(obj);
        return JS_TRUE;

    case JSTYPE_FUNCTION:
        return JS_FALSE;

    case JSTYPE_VOID:
    case JSTYPE_STRING:
		{
		JSString *str = JS_NewStringCopyZ(cx, "[Aztec Object]");
		*vp = STRING_TO_JSVAL(str);
		return JS_TRUE;
		}

    case JSTYPE_NUMBER:
		return JS_FALSE;

    case JSTYPE_BOOLEAN:
	    return JS_FALSE;

    default:
        return JS_FALSE;
    }

    return JS_TRUE;
}

static void
aztec_finalize(JSContext *cx, JSObject *obj)
{
	// OutputDebugString("Aztec finalize\n");
}

JSClass aztecSceneObject_class = {
    "AztecSceneObject", JSCLASS_HAS_PRIVATE,
		JS_PropertyStub, JS_PropertyStub,
		JS_PropertyStub, JS_PropertyStub, 
		JS_EnumerateStub, JS_ResolveStub,

		aztec_convert,      aztec_finalize,
		aztec_getObjectOps
};

JSObject *newSceneObject(JSContext *cx, JSObject *obj,
						 MNamedObject *aztec_obj)
{
	JSObject *jsobj = JS_NewObject(cx, &aztecSceneObject_class, NULL, obj);
	JS_SetPrivate(cx, jsobj, aztec_obj);

	return jsobj;
}

static JSBool aztecSceneObject_constructor(JSContext *cx, JSObject *obj, uintN argc,
                             jsval *argv, jsval *rval)
{
	return JS_SetPrivate(cx, obj, NULL);
}

void addSceneObjectClass(JSContext *cx)
{
	// Add a class to reflect Aztec scene objects
	JSObject *iobj =
    JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
		&aztecSceneObject_class, aztecSceneObject_constructor, 0,
		NULL, aztecSceneObject_methods,
		0, 0);

}

}
