#include "StdAfx.h"

#include "scripting/ScriptMatrix4.h"
#include "scripting/ScriptVector3.h"
#include "scripting/ScriptFuncs.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {  

static JSBool
mul(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat == NULL) {
		return JS_FALSE;
	}

	// Two useful possibilities: matrix4 * matrix4 and matrix4 * vector4.
	if (argc == 1 && JSVAL_IS_OBJECT(*argv)) {
		JSObject *obj2 = JSVAL_TO_OBJECT(*argv);

		MMatrix4 *mat2 = (MMatrix4 *)JS_GetInstancePrivate(cx, obj2, &aztecMatrix4_class, NULL);
		if (mat2 != NULL) {
			// Transform the matrix by this matrix.  Put the result back into
			// this matrix.
			*mat *= *mat2;
			*rval = JSVAL_VOID;
			return JS_TRUE;
		}
	}


	return JS_NewDoubleValue(cx, 42.0, rval);
}

static JSBool
rotX(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat == NULL || argc != 1 || !JSVAL_IS_NUMBER(*argv)) {
		return JS_FALSE;
	}

	double t;
	if (getDoubleVal(cx, *argv, t)) {
		MVector3 xaxis(1.0f, 0.0f, 0.0f);
		mat->identity();
		MMatrix4 omat = mat->rotateWithVector(xaxis, (float)t);
		*rval = JSVAL_VOID;
		*mat = omat;
		return JS_TRUE;
	}

	return JS_FALSE;
}

static JSBool
rotY(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat == NULL || argc != 1 || !JSVAL_IS_NUMBER(*argv)) {
		return JS_FALSE;
	}

	double t;
	if (getDoubleVal(cx, *argv, t)) {
		MVector3 yaxis(0.0f, 1.0f, 0.0f);
		mat->identity();
		MMatrix4 omat = mat->rotateWithVector(yaxis, (float)t);
		*mat = omat;
		*rval = JSVAL_VOID;
		return JS_TRUE;
	}

	return JS_FALSE;
}

static JSBool
rotZ(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat == NULL || argc != 1 || !JSVAL_IS_NUMBER(*argv)) {
		return JS_FALSE;
	}

	double t;
	if (getDoubleVal(cx, *argv, t)) {
		MVector3 zaxis(0.0f, 0.0f, 1.0f);
		mat->identity();
		MMatrix4 omat = mat->rotateWithVector(zaxis, (float)t);
		*mat = omat;
		*rval = JSVAL_VOID;
		return JS_TRUE;
	}

	return JS_FALSE;
}

static JSBool
setIdentity(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat == NULL || argc != 0) {
		return JS_FALSE;
	}

	mat->identity();
	*rval = JSVAL_VOID;
	return JS_TRUE;
}

static JSBool
transform(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat == NULL || argc == 0 || argc > 2) {
		return JS_FALSE;
	}

	if (argc == 1 && JSVAL_IS_OBJECT(*argv)) {
		JSObject *obj2 = JSVAL_TO_OBJECT(*argv);

		MVector3 *vec = (MVector3 *)JS_GetInstancePrivate(cx, obj2, &aztecVector3_class, NULL);
		if (vec != NULL) {
			// Transform the vector by this matrix.  Put the result back into
			// the vector.
			MVector3 rvec = *mat * *vec;
			*vec = rvec;
			*rval = JSVAL_VOID;
			return JS_TRUE;
		}
	}

	if (argc == 2 && JSVAL_IS_OBJECT(*argv) && JSVAL_IS_OBJECT(*(argv+1))) {
		JSObject *obj2 = JSVAL_TO_OBJECT(*argv);
		JSObject *obj3 = JSVAL_TO_OBJECT(*(argv+1));

		MVector3 *invec = (MVector3 *)JS_GetInstancePrivate(cx, obj2, &aztecVector3_class, NULL);
		MVector3 *outvec = (MVector3 *)JS_GetInstancePrivate(cx, obj3, &aztecVector3_class, NULL);

		if (invec != NULL && outvec != NULL) {
			// Transform the vector by this matrix.  Put the result back into
			// the vector.
			*outvec = *mat * *invec;
			*rval = JSVAL_VOID;
			return JS_TRUE;
		}
	}

	// Two useful possibilities: matrix4 * matrix4 and matrix4 * vector4.

	return JS_NewDoubleValue(cx, 42.0, rval);
}

static JSPropertySpec aztecMatrix4_props[] = {
	{"m00",	0,	JSPROP_ENUMERATE},
	{"m01",	1,	JSPROP_ENUMERATE},
	{"m02",	2,	JSPROP_ENUMERATE},
	{"m03",	3,	JSPROP_ENUMERATE},
	{"m10",	4,	JSPROP_ENUMERATE},
	{"m11",	5,	JSPROP_ENUMERATE},
	{"m12",	6,	JSPROP_ENUMERATE},
	{"m13",	7,	JSPROP_ENUMERATE},
	{"m20",	8,	JSPROP_ENUMERATE},
	{"m21",	9,	JSPROP_ENUMERATE},
	{"m22",	10,	JSPROP_ENUMERATE},
	{"m23",	11,	JSPROP_ENUMERATE},
	{"m30",	12,	JSPROP_ENUMERATE},
	{"m31",	13,	JSPROP_ENUMERATE},
	{"m32",	14,	JSPROP_ENUMERATE},
	{"m33",	15,	JSPROP_ENUMERATE},
	{0}
};

static JSFunctionSpec aztecMatrix4_methods[] = {
	{"mul",			mul,		0},
	{"rotX",		rotX,		0},
	{"rotY",		rotY,		0},
	{"rotZ",		rotZ,		0},
	{"setIdentity",	setIdentity,	0},
	{"transform",	transform,	0},
	{0}
};

static JSBool
aztecMatrix4_getProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat == NULL) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		if (strlen(propStr) == 3 && (propStr[0] = 'm' || propStr[0] == 'M') &&
			propStr[1] >= '0' && propStr[1] <= '3' &&
			propStr[2] >= '0' && propStr[2] <= '3') {
			int i = propStr[2] - '0';
			int j = propStr[1] - '0';

			return JS_NewDoubleValue(cx, mat->m[j][i], vp);
		} else {
			return JS_TRUE;
		}
	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
		if (index >= 0 && index <= 15) {
			int i = index % 4;
			int j = index / 4;

			return JS_NewDoubleValue(cx, mat->m[j][i], vp);
		} else {
			return JS_TRUE;
		}
	} else {
#if 0
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		OutputDebugString("Vec prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
#endif
		return JS_FALSE;
	}
}

static JSBool
aztecMatrix4_setProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat == NULL) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		if (strlen(propStr) == 3 && (propStr[0] = 'm' || propStr[0] == 'M') &&
			propStr[1] >= '0' && propStr[1] <= '3' &&
			propStr[2] >= '0' && propStr[2] <= '3') {
			int i = propStr[2] - '0';
			int j = propStr[1] - '0';

			double t;
			if (getDoubleVal(cx, *vp, t)) {
				mat->m[j][i] = (float)t;
				return JS_TRUE;
			} else {
				return JS_FALSE;
			}
		} else {
			return JS_TRUE;
		}
	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
		if (index >= 0 && index <= 15) {
			int i = index % 4;
			int j = index / 4;

			double t;
			if (getDoubleVal(cx, *vp, t)) {
				mat->m[j][i] = (float)t;
				return JS_TRUE;
			} else {
				return JS_FALSE;
			}
		} else {
			return JS_TRUE;
		}
	} else {
		return JS_TRUE;
	}
}

static JSBool
aztecMatrix4_convert(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat == NULL) {
		return JS_FALSE;
	}
	
    switch (typ) {
    case JSTYPE_OBJECT:
        *vp = OBJECT_TO_JSVAL(obj);
        return JS_TRUE;
		
    case JSTYPE_VOID:
    case JSTYPE_STRING:
		{
			char buf[512];
			sprintf(buf, "%f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f",
				mat->m[0][0], mat->m[0][1], mat->m[0][2], mat->m[0][3],
				mat->m[1][0], mat->m[1][1], mat->m[1][2], mat->m[1][3],
				mat->m[2][0], mat->m[2][1], mat->m[2][2], mat->m[2][3],
				mat->m[3][0], mat->m[3][1], mat->m[3][2], mat->m[3][3]);
			JSString *str = JS_NewStringCopyZ(cx, buf);
			*vp = STRING_TO_JSVAL(str);
			return JS_TRUE;
		}
		
	case JSTYPE_FUNCTION:
	case JSTYPE_NUMBER:
    case JSTYPE_BOOLEAN:
    default:
        return JS_FALSE;
    }
}

static void
aztecMatrix4_finalize(JSContext *cx, JSObject *obj)
{
	MMatrix4 *mat = (MMatrix4 *)JS_GetInstancePrivate(cx, obj, &aztecMatrix4_class, NULL);
	if (mat != NULL) {
		delete mat;
	}
}

JSClass aztecMatrix4_class = { 
	"AztecMatrix4", JSCLASS_HAS_PRIVATE, 
		JS_PropertyStub, JS_PropertyStub,					// add/del property
		aztecMatrix4_getProperty, aztecMatrix4_setProperty,	// get/set property
		JS_EnumerateStub, JS_ResolveStub,
		aztecMatrix4_convert, aztecMatrix4_finalize
};

static JSBool aztecMatrix4_constructor(JSContext *cx, JSObject *obj, uintN argc,
                             jsval *argv, jsval *rval)
{
	MMatrix4 *mat = new MMatrix4();
	if (mat == NULL) {
		return NULL;
	}

	// Aztec doesn't, by default, set any values in a new Matrix4().  So,
	// lets make it an identity array before going back to JavaScript.
	mat->identity();

	return JS_SetPrivate(cx, obj, mat);
}

JSObject *newMatrix4(JSContext *cx)
{
	JSObject *jsmat = JS_NewObject(cx, &aztecMatrix4_class, NULL, JS_GetGlobalObject(cx));

	if (jsmat == NULL) {
		return NULL;
	}

	MMatrix4 *pmat = new MMatrix4();
	JS_SetPrivate(cx, jsmat, pmat);

	return jsmat;
}

void addMatrix4Class(JSContext *cx)
{
	// Add a Matrix4 class (reflection of MMatrix43) to the global namespace
    JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
		&aztecMatrix4_class, aztecMatrix4_constructor, 0,
		aztecMatrix4_props, aztecMatrix4_methods, 0, 0);

}

}
