#include "StdAfx.h"

#include <stdlib.h>

#include "MSceneObject.h"
#include "MSystemManager.h"

#include <params/MParameterFactory.h>
#include <MEditableMesh.h>

#include <params/MParentObjectParameter.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#include <mesh/MEditMeshModifier.h>

namespace Aztec {
  
  //-------------------
  //  MSceneObject
  //-------------------

  // Construction/Destruction
  MSceneObject::MSceneObject(MShapeObjectPtr shape) {
    setObjectType(OBJECTFLAG_TYPE_MESH);

    // add in the mesh override parameter
    MChoiceParameterPtr drawWireParam = MParameterFactory::createChoice("drawWire", "drawWire", "Wireframe Only");
    drawWireParam->addChoice("no");
    drawWireParam->addChoice("yes");
    drawWireParam->setValueString("no");

    addParameter(drawWireParam);

    m_ShapeObjParam = MParameterFactory::createObject("shape", "shapeObj", "Shape Object");
    m_ShapeObjParam->unsetFlag(MParameterObject::FLAG_VISIBLE);
    addParameter(m_ShapeObjParam);

    m_MaterialParam = MParameterFactory::createObject("mat", "material", "Material");
    m_MaterialParam->unsetFlag(MParameterObject::FLAG_VISIBLE);
    addParameter(m_MaterialParam);

    m_ParentParam = new MParentObjectParameter(this, "parent", "parent", "Parent");
    m_ParentParam->unsetFlag(MParameterObject::FLAG_VISIBLE);
    addParameter(m_ParentParam);

    addParameter(MParameterFactory::createColour("viewCol", "viewCol", "View Colour"));
    setParamByName("viewCol", "1 1 1");

    setShapeObject(shape);

    addParameter(MParameterFactory::createBoolean("drawAxis", "drawAxis", "Draw Axis"));
    setParamByName("drawAxis", "false");
    findParameter("drawAxis")->setVisible(false);
  }
  
  MSceneObject::~MSceneObject() {
    setTextureMaterial(NULL);
  }
  
  
  MBaseObjectPtr MSceneObject::createNew() {
    MSceneObjectPtr NewObj;
    
    NewObj = new MSceneObject;
    NewObj->setFrom(this);
    
    return NewObj;
  }
  
  void MSceneObject::setFrom(MBaseObjectPtr SrcObj) {
    if (SrcObj == NULL) {
      return;
    }
    
    MTransformObject::setFrom(SrcObj);
  }
 
  MShapeObjectPtr MSceneObject::getShapeObject() {
    MNamedObjectPtr obj = m_ShapeObjParam->getValue();
    MShapeObjectPtr result = AZTEC_CAST(MShapeObject, obj);
    return result;
  }
  
  void MSceneObject::setShapeObject(MShapeObjectPtr shapeObj) {
    if (m_ShapeObjParam->isConnected()) {
      m_ShapeObjParam->setInputParameter(NULL);
    }
    m_ShapeObjParam->setValue(shapeObj);
  }

  MComponentisedObjectPtr MSceneObject::getComponentObject() {
    MComponentisedObjectPtr result;

    result = AZTEC_CAST(MComponentisedObject, this);

    if (result != NULL) {
      return result;
    }

    if (getShapeObject() != NULL) {
      result = getShapeObject()->getComponentObject();

      if (result != NULL) {
        return result;
      }
    }

    return NULL;
  }
  MEditableComponentisedObjectPtr MSceneObject::getEditableComponentObject() {
    MEditableComponentisedObjectPtr result;

    result = AZTEC_CAST(MEditableComponentisedObject, this);

    if (result != NULL) {
      return result;
    }

    if (getShapeObject() != NULL) {
      result = getShapeObject()->getEditableComponentObject();

      if (result == NULL) {
        // if we have gotten here, then we do not have an editable component 
        // object. If we have a mesh shape, then try adding an editable mesh 
        // modifier to see what happens.
        //
        // TODO: Make it so the shape object provides the modifier to add on.
        // That way, any shape object can provide a modifier, rather than 
        // having to add them in by hand for each one.
        if (AZTEC_CAST(MMeshShape, getShapeObject()) != NULL) {
          MEditMeshModifierPtr editMeshMod = new MEditMeshModifier();
          editMeshMod->setName("editMeshModifier");
          insertModifier(editMeshMod);
        }

        result = getShapeObject()->getEditableComponentObject();
      }

      if (result != NULL) {
        return result;
      }
    }

    return NULL;
  }

  MSceneObjectPtr MSceneObject::getParent() {
    return AZTEC_CAST(MSceneObject, m_ParentParam->getValue());
  }

  void MSceneObject::setParent(const MSceneObjectPtr &parent) {
    // if we are trying to parent to ourselves, do nothing, since we can't do that.
    if (parent == this) {
      return;
    }
    m_ParentParam->setValue(parent);
  }

  MTransformObjectPtr MSceneObject::getTransformObject() {
    return this;
  }
  
  void MSceneObject::getWorldTransformMatrix(MMatrix4 &matrix) {
    MSceneObjectPtr obj = this;
    MMatrix4 temp;

    matrix.identity();
    temp.identity();

    while (obj != NULL) {
      obj->getTransformMatrix(temp);
      matrix *= temp;


      obj = obj->getParent();
    }
  }
  
  void MSceneObject::getWorldInverseTransformMatrix(MMatrix4 &matrix) {
    getWorldTransformMatrix(matrix);
    matrix.inverse();
  }

  void MSceneObject::setTextureMaterial(MMaterialPtr material) {
    m_MaterialParam->setValue(material);
  }
  
  MMaterialPtr MSceneObject::getTextureMaterial() {
    return AZTEC_CAST(MMaterial, m_MaterialParam->getValue());
  }

  MStr MSceneObject::getTextureMaterialName() {
    MNamedObjectPtr material = getTextureMaterial();
    if (material != NULL) {
      return material->getName();
    }

    return "";
  }


  // Time related
  const MVector3 MSceneObject::getPivotPoint() {
    static   MVector3    ZeroVec;
    
    if (getTransformObject() != NULL) {
      return getTransformObject()->getPivotPoint();
    }
    
    return ZeroVec;
  }
  
  const MVector3 MSceneObject::centrePivot() {
    static   MVector3    ZeroVec;
    MVector3             Centre;
    
    if (getTransformObject() == NULL) {
      return ZeroVec;
    }

    MComponentisedObjectPtr compObj = getComponentObject();
    MShapeObjectPtr shape = getShapeObject();
    
    // Go through the scene object list, and add those pivots who are children to 
    // this object
    {
      MSystemManagerPtr SysMan;
      MTreeObjectNodePtr ThisNode, Node;
      int               Count;
      MBaseObjectPtr    BaseObj;
      
      Count = 0;
      
      SysMan = Aztec::getSystemManager();
      
      ThisNode = SysMan->getScene()->getObjectList()->findObject(this);
      
      if (ThisNode != NULL) {
        SysMan->getScene()->getObjectList()->beginIteration();
        
        while ((BaseObj = SysMan->getScene()->getObjectList()->getNext()) != NULL) {
          Node = SysMan->getScene()->getObjectList()->getCurrentNode();
          if (Node == NULL) continue;

          if (Node->isChildOf(ThisNode) || Node == ThisNode) {
            MSceneObjectPtr SceneObj;
            
            SceneObj = AZTEC_CAST(MSceneObject, Node->getObject());
            if (SceneObj != NULL) {
              MComponentisedObjectPtr nodeCompObj = SceneObj->getComponentObject();

              if (compObj != NULL) {
                MVector3    MeshCentre;
                
                MeshCentre = nodeCompObj->getFlaggedCentre(MComponentisedObject::OBJECT_TYPE, 0);
                MeshCentre = SysMan->getScene()->objectToWorldSpace(MBaseObjectPtr(SceneObj), MeshCentre);
                MeshCentre = SysMan->getScene()->worldToObjectSpace(this, MeshCentre);
                
                Centre += MeshCentre;
                Count++;
              }
            }
          }
        }
        SysMan->getScene()->getObjectList()->endIteration();

        if (Count) {
          Centre /= (float)Count;
        }
      } else if (compObj != NULL) {
        Centre = compObj->getFlaggedCentre(MComponentisedObject::OBJECT_TYPE, 0);
      }
    }
    
    getTransformObject()->setPivotPoint(Centre);
    
    return getTransformObject()->getPivotPoint();
  }
  
  
  int MSceneObject::updateObject()
  {
    return 1;
  }

  bool MSceneObject::insertModifier(const MMeshModifierPtr &modifier) {
    // now, if our scene object has a shape parameter, then we need to insert
    // this modifier into the stack. To do this, we first check to see if it is
    // connected. If it is, we switch the the connection that is currently there
    // to connect to our new modifier. If there is no current connection, then
    // we just use the shape object itself.
    
    if (m_ShapeObjParam->isConnected()) {
      // This changes
      //
      //              SceneObj
      //  modifier      /|\
      //                 |
      //               ShapeInput
      //
      //  to 
      //       SceneObj
      //         /|\
      //          |
      //       modifier
      //         /|\
      //          |
      //      ShapeInput
      
      MParameterObjectPtr inputParam = m_ShapeObjParam->getInputParameter();
      m_ShapeObjParam->setInputParameter(modifier->getOutputParameter());
      modifier->getInputParameter()->setInputParameter(inputParam);
    } else {
      MShapeObjectPtr shape = getShapeObject();
      m_ShapeObjParam->setInputParameter(modifier->getOutputParameter());
      modifier->getInputParameter()->setValue(shape);
    }

    return true;
  }
}
