#include "StdAfx.h"

#include "MFloatKeyList.h"
#include "MFloatKey.h"

#include <assert.h>
#include <algorithm>

namespace Aztec {
  class MFloatKeyImpl : public MFloatKey {
  public:
    MFloatKeyImpl(MFloatKeyListPtr list, int index) 
      : keyList(list), keyIndex(index)
    {
    }

    // MFloatKey methods
    float getValue() {
      return keyList->keys[keyIndex].value;
    }

    void setValue(float value) {
      keyList->keys[keyIndex].value = value;
      keyList->keys[keyIndex].flags.setFlag(MFloatKeyList::FloatKey::IN_INVALID);
      keyList->keys[keyIndex].flags.setFlag(MFloatKeyList::FloatKey::OUT_INVALID);
      if (keyIndex > 0) {
        keyList->keys[keyIndex-1].flags.setFlag(MFloatKeyList::FloatKey::IN_INVALID);
        keyList->keys[keyIndex-1].flags.setFlag(MFloatKeyList::FloatKey::OUT_INVALID);
      }
      if (keyIndex < (int)keyList->keys.size()-1) {
        keyList->keys[keyIndex+1].flags.setFlag(MFloatKeyList::FloatKey::IN_INVALID);
        keyList->keys[keyIndex+1].flags.setFlag(MFloatKeyList::FloatKey::OUT_INVALID);
      }
      keyList->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
    }

    float getInTangent() {
      return keyList->getInTangent(keyIndex);
    }

    float getOutTangent() {
      return keyList->getOutTangent(keyIndex);
    }

    void setInTangent(float in) {
      keyList->setInTangent(keyIndex, in);
    }

    void setOutTangent(float out) {
      keyList->setOutTangent(keyIndex, out);
    }

    // MKey methods
    void setKeyTime(int time) {
      if (keyList->keys[keyIndex].time != time) {
        // see if a key already takes up this spot
        int origKeyIndex = keyList->getKeyIndexAtTime(time);

        keyList->keys[keyIndex].time = time;
        keyList->keys[keyIndex].flags.setFlag(MFloatKeyList::FloatKey::IN_INVALID);
        keyList->keys[keyIndex].flags.setFlag(MFloatKeyList::FloatKey::OUT_INVALID);

        // if it is already taken up, then get rid of the old one.
        if (origKeyIndex != -1) {
          keyList->keys.erase(keyList->keys.begin() + origKeyIndex);
        }
        std::sort(keyList->keys.begin(), keyList->keys.end());
      }
      keyList->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
    }

    long getKeyTime() {
      return keyList->keys[keyIndex].time;
    }

    int getIndex() {
      return keyIndex;
    }

    MKeyListPtr getKeyList() {
      return keyList;
    }

    bool isSelected() {
      return keyList->keys[keyIndex].selected;
    }

    void setSelected(bool selected) {
      keyList->keys[keyIndex].selected = selected;
    }

  private:
    MFloatKeyListPtr keyList;

    int keyIndex;

  };


  MFloatKeyList::MFloatKeyList() {
    granularity = 120;
    initialValue = 0.0f;
  }

  MFloatKeyList::~MFloatKeyList() {
  }
  
  float MFloatKeyList::getValueAtIndex(int index) {
    return keys[index].value;
  }

  long MFloatKeyList::getTimeAtIndex(int index) {
    return keys[index].time;
  }

  // MFloatKeyableValue methods
  void MFloatKeyList::setInitialValue(float value) {
    if (value != initialValue) {
      ensureUndo();

      initialValue = value;

      // we only need an update if the initial value has an impact, and the 
      // initial value only changes things when we have no animation keys.
      if (keys.size() == 0) {
        flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
      }
    }
  }

  void MFloatKeyList::setKey(float value, long time) {
    ensureUndo();

    int index = getNearestKeyIndex(time);

    // if we couldn't find a key >= that this one, then we must place it at the end
    FloatKey *key;

    if (index == -1) {
      keys.push_back(FloatKey());
      invalidateKey(keys.size()-1);

      key = &keys[keys.size()-1];
    } else {
      if (time == keys[index].time) {
        key = &keys[index];
      } else {
        keys.insert(keys.begin() + index, FloatKey());
        key = &keys[index];
      }

      invalidateKey(index);
    }
    key->value = value;
    key->time = time;
    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  // MKeyableValue methods
  class FloatKeyTimeComparator {
  public:
    bool operator()(const MFloatKeyList::FloatKey &lhs, 
                    const MFloatKeyList::FloatKey &rhs) const 
    {
      return lhs.time < rhs.time;
    }
    
    bool operator()(const MFloatKeyList::FloatKey &lhs, 
                    const long &rhsTime) const 
    {
      return lhs.time < rhsTime;
    }

    bool operator()(const long &lhsTime, 
                    const MFloatKeyList::FloatKey &rhs) const 
    {
      return lhsTime < rhs.time;
    }
  };

  bool MFloatKeyList::hasKeySet(long time) {
    return std::binary_search(keys.begin(), keys.end(), time, FloatKeyTimeComparator());
  }

  void MFloatKeyList::shiftKeys(long onOrAfterTime, long beforeTime, long amount) {
    ensureUndo();

    // first we loop over the keys that are outside our time range.
    // Then for each key outside our range, we check to see if the shift
    // of the keys will cause a potential overwriting of keys.
    // if it does, then we should remove that key 

    // we can't do anything if we don't have any keys
    if (keys.size() == 0 || amount == 0) {
      return;
    }

    for (unsigned int index = 0; index < keys.size(); ++index) {
      if (keys[index].time >= onOrAfterTime &&
        keys[index].time < beforeTime) {
        continue;
      }

      // now get the key index of the key that will be moved to this position
      int origKeyIndex = getKeyIndexAtTime(keys[index].time - amount);

      // if we have a valid key, and that key is in the range given to this
      // function, that this key will need to be erased.
      if (origKeyIndex != -1 && 
          keys[origKeyIndex].time >= onOrAfterTime &&
          keys[origKeyIndex].time < beforeTime) {

        keys.erase(keys.begin() + index);
        // decrement the index so we start at the new place after this for
        // loop goes around once more
        index--;
      }
    }

    // now taht we have removed any possible duplicate keys, we now shift them.
    for (unsigned int index = 0; index < keys.size(); ++index) {
      if (keys[index].time >= onOrAfterTime &&
        keys[index].time < beforeTime) {

        keys[index].time += amount;
      }
    }

    // now sort the keys so they are in a good order.
    std::sort(keys.begin(), keys.end());

    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  void MFloatKeyList::shiftSelectedKeys(long amount) {
    ensureUndo();

    // first we loop over the keys that are outside our time range.
    // Then for each key outside our range, we check to see if the shift
    // of the keys will cause a potential overwriting of keys.
    // if it does, then we should remove that key 

    // we can't do anything if we don't have any keys
    if (keys.size() == 0 || amount == 0) {
      return;
    }

    for (unsigned int index = 0; index < keys.size(); ++index) {
      if (keys[index].selected) {
        continue;
      }

      // now get the key index of the key that will be moved to this position
      int origKeyIndex = getKeyIndexAtTime(keys[index].time - amount);

      // if we have a valid key, and that key is in the range given to this
      // function, that this key will need to be erased.
      if (origKeyIndex != -1 && keys[origKeyIndex].selected) {

        keys.erase(keys.begin() + index);
        // decrement the index so we start at the new place after this for
        // loop goes around once more
        index--;
      }
    }

    // now taht we have removed any possible duplicate keys, we now shift them.
    for (unsigned int index = 0; index < keys.size(); ++index) {
      if (keys[index].selected) {
        keys[index].time += amount;
      }
    }

    // now sort the keys so they are in a good order.
    std::sort(keys.begin(), keys.end());

    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  void MFloatKeyList::getKeyTimes(std::vector<int> &keyTimes) {
    keyTimes.resize( keys.size() );

    for (unsigned int i = 0; i < keys.size(); ++i) {
      keyTimes[i] = keys[i].time;
    }
  }

  void MFloatKeyList::getSelectedKeyTimes(std::vector<int> &keyTimes) {
    keyTimes.clear();

    for (unsigned int i = 0; i < keys.size(); ++i) {
      if (keys[i].selected) {
        keyTimes.push_back(keys[i].time);
      }
    }
  }

  void MFloatKeyList::deleteKey(long time) {
    ensureUndo();

    int keyIndex = getNearestKeyIndex(time);

    // if we don't have a key that lies on our time, do nothing
    if (keys[keyIndex].time != time) {
      return;
    }

    keys.erase(keys.begin() + keyIndex);
  }

  void MFloatKeyList::deleteSelectedKeys() {
    ensureUndo();

    // loop over the keys to see if we should delete any of them
    for (int index = keys.size() - 1; index >= 0; --index) {
      if (keys[index].selected) {
        keys.erase(keys.begin() + index);
      }
    }
  }

  bool MFloatKeyList::hasKeys() {
    return (keys.size() > 0);
  }


  // MFloatValue methods
  float MFloatKeyList::getValueAtTime(long time) {
    if (keys.size() == 0) {
      return initialValue;
    }

    int index = getNearestKeyIndex(time);

    // if the time exceeds our last key, just use that value
    if (index == -1) {
      return keys[keys.size()-1].value;
    }
    // if index was 0 then use the first key
    if (index == 0) {
      return keys[0].value;
    }

    // if we have an exact time, return the exact value
    if (keys[index].time == time) {
      return keys[index].value;
    }

    float s = (float)(time - keys[index-1].time) / (float)(keys[index].time - keys[index-1].time);
    float h1, h2,h3,h4;
    float s2 = s*s,s3 = s*s*s;
    float p1(keys[index-1].value);
    float p2(keys[index].value);
    float t1(getOutTangent(index-1));
    float t2(getInTangent(index));

    h1 =  2*s3 - 3*s2     + 1;
    h2 = -2*s3 + 3*s2        ;
    h3 =    s3 - 2*s2 + s    ;
    h4 =    s3 -   s2        ;

    float result;

    result = p1 * h1 + p2 * h2 + t1 * h3 + t2 * h4;

    return result;
  }

  // MKeyList methods
  int MFloatKeyList::getKeyCount() {
    return keys.size();
  }

  MKeyPtr MFloatKeyList::getKeyAtIndex(int index) {
    return new MFloatKeyImpl(this, index);
  }

  MKeyPtr MFloatKeyList::getKeyAtTime(long time) {
    // TODO: improve with a binary search
    for (unsigned int i = 0; i < keys.size(); ++i) {
      if (keys[i].time == time) {
        return new MFloatKeyImpl(this, i);
      } else if (keys[i].time > time) {
        if (i > 0) {
          return new MFloatKeyImpl(this, i-1);
        } else {
          return NULL;
        }
      }
    }
    return NULL;
  }

  void MFloatKeyList::setKey(MKeyPtr key, long time) {
    ensureUndo();

    MFloatKeyPtr float3Key = AZTEC_CAST(MFloatKey, key);

    if (float3Key != NULL) {
      setKey(float3Key->getValue(), time);
    }

    assert(0);
  }

  void MFloatKeyList::copyFromKeyList(MKeyListPtr source) {
    ensureUndo();

    MFloatKeyListPtr floatKeyList = AZTEC_CAST(MFloatKeyList, source);

    if (floatKeyList != NULL) {
      initialValue = floatKeyList->initialValue;
      keys = floatKeyList->keys;
      granularity = floatKeyList->granularity;
    }

    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  int MFloatKeyList::getGranularity() {
    return granularity;
  }

  void MFloatKeyList::setGranularity(int granularity) {
    ensureUndo();

    this->granularity = granularity;
  }

  // MValue methods
  bool MFloatKeyList::copyFromValue(MValuePtr source) {
    ensureUndo();

    MFloatKeyListPtr list = AZTEC_CAST(MFloatKeyList, source);

    // we can only copy from our own type.
    if (list == NULL) {
      return false;
    }

    initialValue = list->initialValue;
    keys = list->keys;
    granularity = list->granularity;

    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
    return true;
  }

  void MFloatKeyList::storeValue() {
    storedKeys = keys;
    storedGranularity = granularity;
    storedInitialValue = initialValue;
  }

  void MFloatKeyList::restoreValue() {
    ensureUndo();
    keys = storedKeys;
    granularity = storedGranularity;
    initialValue = storedInitialValue;
    flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  // MBaseObject methods
  MBaseObjectPtr MFloatKeyList::createNew() {
    MFloatKeyListPtr result = new MFloatKeyList;
    result->initialValue = initialValue;
    result->keys = keys;
    result->granularity = granularity;
    return result;
  }

  int MFloatKeyList::getNearestKeyIndex(long time) {
    // TODO: do a binary search to improve performance

    int index = -1;
    for (unsigned int i = 0; i < keys.size(); ++i) {
      if (keys[i].time >= time) {
        index = i;
        break;
      }
    }

    return index;
  }

  // MUndoableObject methods
  void MFloatKeyList::finishWithUndoNode() {
    undoNode = NULL;
  }

  // MFloatKeyList methods
  int MFloatKeyList::getKeyIndexAtTime(long time) {
    // TODO: do a binary search to improve performance
    std::vector<FloatKey>::iterator it = std::lower_bound(keys.begin(), keys.end(), time, FloatKeyTimeComparator());

    // if we couldn't find it, just return -1
    if (it == keys.end() || it->time != time) {
      return -1;
    }

    return std::distance(keys.begin(), it);
  }


  float MFloatKeyList::getInTangent(int keyIndex) {
    FloatKey *key = &keys[keyIndex];
    
    // if our tangent needs updating, then update it
    if (key->flags.isFlagged(FloatKey::IN_INVALID)) {
      if (keys.size() == 1) {
        key->in = 0;
      } else if (keyIndex == 0) { 
        key->in = (keys[1].value - keys[0].value);
      } else if (keyIndex == keys.size()-1) {
        key->in = (keys[keyIndex].value - keys[keyIndex - 1].value);
      } else {
        key->in = 0.5f * (keys[keyIndex + 1].value - keys[keyIndex - 1].value);
      }
      key->out = key->in;
      key->flags.unsetFlag(FloatKey::IN_INVALID);
      key->flags.unsetFlag(FloatKey::OUT_INVALID);
    }
    
    return key->in;
  }

  float MFloatKeyList::getOutTangent(int keyIndex) {
    getInTangent(keyIndex);
    return keys[keyIndex].out;
  }

  void MFloatKeyList::setInTangent(int keyIndex, float tangent) {
    // do nothing for now
  }

  void MFloatKeyList::setOutTangent(int keyIndex, float tangent) {
    // do nothing for now
  }

  void MFloatKeyList::invalidateKey(int index) {
    if (index > 0) {
      keys[index-1].flags.setFlag((FloatKey::IN_INVALID | FloatKey::OUT_INVALID));
    }
    
    keys[index].flags.setFlag((FloatKey::IN_INVALID | FloatKey::OUT_INVALID));

    if (index < (int)keys.size()-1) {
      keys[index+1].flags.setFlag((FloatKey::IN_INVALID | FloatKey::OUT_INVALID));
    }
  }

  void MFloatKeyList::ensureUndo() {
    if (undoNode == NULL && MUndoManager::isEnabled()) {
      undoNode = new FloatKeyListUndoNode(this);
      MUndoManager::getInstance()->addUndoNode(undoNode);
    }
  }

  // FloatKey constants
  const AztecFlags MFloatKeyList::FloatKey::IN_SMOOTH = 0x01;
  const AztecFlags MFloatKeyList::FloatKey::IN_FREE = 0x02;
  const AztecFlags MFloatKeyList::FloatKey::IN_LINEAR = 0x04;
  const AztecFlags MFloatKeyList::FloatKey::IN_INVALID = 0x08;
  const AztecFlags MFloatKeyList::FloatKey::OUT_SMOOTH = 0x10;
  const AztecFlags MFloatKeyList::FloatKey::OUT_FREE = 0x20;
  const AztecFlags MFloatKeyList::FloatKey::OUT_LINEAR = 0x40;
  const AztecFlags MFloatKeyList::FloatKey::OUT_INVALID = 0x80;

  MFloatKeyList::FloatKeyListUndoNode::FloatKeyListUndoNode(const MFloatKeyListPtr &keyList) 
  {
    list = keyList;
    keys = list->keys;
    granularity = list->granularity;
    initialValue = list->initialValue;
  }

  // MBaseUndoNode Methods
  MUndoableObject* MFloatKeyList::FloatKeyListUndoNode::getObject() {
    return &*list;
  }

  void MFloatKeyList::FloatKeyListUndoNode::undo() {
    std::swap(keys, list->keys);
    std::swap(granularity, list->granularity);
    std::swap(initialValue, list->initialValue);
    list->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }

  void MFloatKeyList::FloatKeyListUndoNode::redo() {
    std::swap(keys, list->keys);
    std::swap(granularity, list->granularity);
    std::swap(initialValue, list->initialValue);
    list->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
  }


}

