#include "StdAfx.h"

#include "MAnimMesh.h"

#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

  //----------------------------------------------------------------------------------------
  //  MAnimatedVertex
  //----------------------------------------------------------------------------------------
  MAnimatedVertex::MAnimatedVertex() {
    m_Pos = new MVector3KeyList;
  }

  MAnimatedVertex::~MAnimatedVertex() {
    m_Pos = NULL;
  }

  void MAnimatedVertex::set(int Time, float x, float y, float z) {
    m_Pos->setKey(MVector3(x,y,z), Time);
  }

  void MAnimatedVertex::set(int Time, const MVector3 &V) {
    m_Pos->setKey(V, Time);
  }

  void MAnimatedVertex::get(int Time, MVector3 &V) {
    V = m_Pos->getValueAtTime(Time);
  }

  //----------------------------------------------------------------------------------------
  //  MAnimMesh
  //----------------------------------------------------------------------------------------

  MAnimMesh::MAnimMesh()
  {
  }

  MAnimMesh::~MAnimMesh() {
    for (unsigned n = 0;n < m_AnimVerts.size();n++) {
      if (m_AnimVerts[n] != NULL) {
        delete m_AnimVerts[n];
      }
    }
  }

  MBaseObjectPtr MAnimMesh::createNew() {
    // !!! Todo: Finish this off !!!
    MAnimMesh *NewMesh;
  
    NewMesh = new MAnimMesh();
  
    NewMesh->setFrom(this);
  
    return NewMesh;
  }

  void MAnimMesh::setFrom(MBaseObjectPtr SrcObj) {
    if (SrcObj == NULL)
      return;
  
    MMesh::setFrom(SrcObj);
  
    MAnimMeshPtr Obj;
    MMeshPtr SrcMesh;
    SrcMesh = AZTEC_CAST(MMesh, SrcObj);
  
    // If the source isn't even a mesh object, we can't work with it.
    if (SrcMesh == NULL) {
      return;
    }
  
    Obj = AZTEC_CAST(MAnimMesh, SrcObj);

    
    if (SrcMesh->getTextureMesh() != NULL) {
      setTextureMesh(AZTEC_CAST(MMesh, SrcMesh->getTextureMesh()->createNew()));
    } else {
      setTextureMesh(NULL);
    }

    // If we haven't got an animated mesh as a source
    // we need to make some unkeyd animated vertices.
    if (Obj == NULL) {
    
      // Create the animated vertices
      if (getNumVerts() > 0) {
        // if we will have fewer animated vertices, delete the old ones.
        if (getNumVerts() < m_AnimVerts.size()) {
          for (int n = getNumVerts(); n < m_AnimVerts.size(); ++n) {
            if (m_AnimVerts[n] != NULL) {
              delete m_AnimVerts[n];
            }
          }
        }
        m_AnimVerts.resize(getNumVerts(), NULL);
        for (int i=getNumVerts()-1; i>=0 ; i--) {
          m_AnimVerts[i] = new MAnimatedVertex;
          m_AnimVerts[i]->m_Pos->setInitialValue(*m_Verts[i]);
        }
      } else {
        m_AnimVerts.clear();
      }
    
      return;
    }
  
    // Create the animated vertices
    if (getNumVerts() > 0) {
      if (getNumVerts() < m_AnimVerts.size()) {
        for (int n = getNumVerts(); n < m_AnimVerts.size(); ++n) {
          if (m_AnimVerts[n] != NULL) {
            delete m_AnimVerts[n];
          }
        }
      }
      m_AnimVerts.resize(getNumVerts());
      for (int i=getNumVerts()-1; i>=0 ; i--) {
        m_AnimVerts[i] = new MAnimatedVertex();
        m_AnimVerts[i]->m_Pos = AZTEC_CAST(MVector3KeyList, Obj->m_AnimVerts[i]->m_Pos->createNew());
      }
    } else {
      m_AnimVerts.clear();
    }
  
  }


  int MAnimMesh::addVertex(float x, float y, float z) {
  
    return addVertexArray(&MVector3(x,y,z), 1);
  }

  int MAnimMesh::addVertexArray(const MVector3 Vert[], int Num)
  {
    // Add the animated vertices first
    for (int i = 0; i < Num; i++) {
      m_AnimVerts.push_back(new MAnimatedVertex());
      m_AnimVerts.back()->m_Pos->setInitialValue(Vert[i]);
    }
  
    // now add the normal vertices
    return MEditableMesh::addVertexArray(Vert, Num);
  }

  void MAnimMesh::setVertexArray(MVector3 *Vert, int Num, int StartNum) {
    setVertexArray(getTime(), Vert, Num, StartNum);
  }

  void MAnimMesh::setVertexArray(long time, MVector3 *Vert, int Num, int StartNum) {
    for (int i=0;i<Num;i++) {
      m_AnimVerts[StartNum+i]->set(time, Vert[i]);
    }
  }

  void MAnimMesh::deleteVertex(int Num)
  {
    deleteVertexArray(&Num, 1);
  }

  void MAnimMesh::deleteVertexArray(const int *VertNums, int Num)
  {
    if (Num > getNumVerts())
      return;
  
    for (int i = 0; i < Num; ++i) {
      if (m_AnimVerts[VertNums[i]] != NULL) {
        delete m_AnimVerts[VertNums[i]];
        m_AnimVerts.erase(m_AnimVerts.begin() + VertNums[i]);
      }
    }
  
    MEditableMesh::deleteVertexArray(VertNums, Num);
  }

  bool MAnimMesh::doUpdateObject() {
    MMesh::doUpdateObject();

    long curTime = getTime();
  
    for (int n = 0; n < getNumVerts(); ++n) {
      m_AnimVerts[n]->get(curTime, *m_Verts[n]);
    }
    calculateNormals();

    return true;
  }

  void MAnimMesh::updateKey(long Time, bool Animate) {
    updateComponentAnimation(POINT_TYPE, 0, Time, Animate);
  }

  MAnimatedVertex* MAnimMesh::getAnimVert(int Num) {
    if (Num < 0 || Num >= getNumVerts()) {
      return NULL;
    }
  
    return m_AnimVerts[Num];
  }


  int MAnimMesh::collapseVertices(AztecFlags Flags) {
    return MEditableMesh::collapseVertices(Flags);

  }

  void MAnimMesh::updateComponentAnimation(AztecFlags type, AztecFlags flag, long time, bool createKey) {
    bool changed = false;
    bool *doVertex = NULL;

    switch (type) {
    case POINT_TYPE:
      {
        break;
      }
    case FACET_TYPE:
      {
        doVertex = new bool[m_Verts.size()];
        memset(doVertex, 0, sizeof(bool) * m_Verts.size());

        for (int i = 0; i < m_NumTris; ++i) {
          if (m_Tris[i] != NULL) {
            if (flag == 0 || m_Tris[i]->isFlagged(flag)) {
              doVertex[m_Tris[i]->getVertex(0)] = true;
              doVertex[m_Tris[i]->getVertex(1)] = true;
              doVertex[m_Tris[i]->getVertex(2)] = true;
            }
          }
        }

        break;
      }
    case EDGE_TYPE:
      {
        doVertex = new bool[m_Verts.size()];
        memset(doVertex, 0, sizeof(bool) * m_Verts.size());

        for (int i = 0; i < m_NumTris; ++i) {
          if (m_Tris[i] != NULL) {
            for (int edge = 0; edge < 3; ++edge) {
              if (flag == 0 || m_Tris[i]->isEdgeFlagged(edge, flag)) {
                doVertex[m_Tris[i]->getVertex(edge)] = true;
                doVertex[m_Tris[i]->getVertex((edge+1)%3)] = true;
              }
            }
          }
        }

        break;
      }
    }

    if (type == POINT_TYPE || doVertex != NULL) {
      for (unsigned int i = 0; i < m_Verts.size(); ++i) {
        if ((doVertex != NULL && doVertex[i]) || 
            (flag == 0 || m_Verts[i]->isFlagged(flag)) ) {
          updatePointAnimation(i, time, createKey);
          changed = true;
        }
      }
    }
    if (doVertex != NULL) {
      delete[] doVertex;
    }

    if (changed) {
      calculateNormals();
      flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
    }
  }

  void MAnimMesh::updateSingleComponentAnimation(AztecFlags type, int index, long time, bool createKey) {
    bool changed = false;

    switch (type) {
    case POINT_TYPE:
      {
        updatePointAnimation(index, time, createKey);
        changed = true;
        break;
      }
    case FACET_TYPE:
      {
        updatePointAnimation(m_Tris[index]->getVertex(0), time, createKey);
        updatePointAnimation(m_Tris[index]->getVertex(1), time, createKey);
        updatePointAnimation(m_Tris[index]->getVertex(2), time, createKey);
        changed = true;
        break;
      }
    case EDGE_TYPE:
      {
        int faceIndex = index / 3;
        int vertIndex = index % 3;

        updatePointAnimation(m_Tris[faceIndex]->getVertex(vertIndex), time, createKey);
        updatePointAnimation(m_Tris[faceIndex]->getVertex((vertIndex+1)%3), time, createKey);
        changed = true;
        break;
      }
    }

    if (changed) {
      calculateNormals();
      flagOutputs(OBJECTFLAG_NEEDS_UPDATE);
    }
  }

  void MAnimMesh::updatePointAnimation(int index, long time, bool createKey) {
    if (createKey) {
      m_AnimVerts[index]->set(time, *m_Verts[index]);
    } else {
      // we need to update the default type key
      m_AnimVerts[index]->m_Pos->setInitialValue(*m_Verts[index]);
    }
  }
}