#include "StdAfx.h"
#include "IniFile.h"

#include <stdio.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

  // default constructor
  CIniFile::CIniFile()
  {
  }

  // constructor based on a specific path
  CIniFile::CIniFile(const CIniString &inipath)
  {
	  path = inipath;
  }

  // default destructor
  CIniFile::~CIniFile()
  {

  }

  /////////////////////////////////////////////////////////////////////
  // Public Functions
  /////////////////////////////////////////////////////////////////////

  //sets path of ini file to read and write from
  void CIniFile::SetPath(const CIniString &newpath)
  {
	  path = newpath;
  }

  //reads ini file specified using CIniFile::SetPath()
  //returns true if successful, false otherwise
  bool CIniFile::ReadFile()
  {
	  std::ifstream inifile;
	  CIniString readinfo;
	  inifile.open(path.c_str());
	  int curkey = -1, curval = -1;
	  if (inifile.fail())
	  {
		  error = "Unable to open ini file.";
		  return false;
	  }
	  CIniString keyname, valuename, value;
	  CIniString temp;
	  while (!inifile.eof()) {
		  getline(inifile,readinfo);
		  if (readinfo != "")
		  {
			  if (readinfo[0] == '[' && readinfo[readinfo.length()-1] == ']') //if a section heading
			  {
				  keyname = readinfo.substr(1, readinfo.length() - 2);
			  }
			  else //if a value
			  {
				  int npos = readinfo.find('=');
				  valuename = readinfo.substr(0, npos);
				  value = readinfo.substr(npos+1);
				  SetValue(keyname, valuename, value);
			  }
		  }
	  }
	  inifile.close();
	  inifile.clear();
	  return true;
  }

  //writes data stored in class to ini file
  void CIniFile::WriteFile()
  {
	  std::ofstream inifile;
	  inifile.open(path.c_str());

	  CIniSections::iterator sections;
	  for (sections=iniSections.begin(); sections!=iniSections.end();sections++) {
		  const CIniString &sectionName = (*sections).first;
		  CSection &section = (*sections).second;
		  if (section.size() > 0) {
			  inifile << '[' << sectionName << ']' << std::endl;
			  CSection::iterator keys;
			  for (keys=section.begin();keys!=section.end();keys++) {
				  const CIniString &keyName = (*keys).first;
				  CIniString &keyVal  = (*keys).second;
				  inifile << keyName << "=" << keyVal << std::endl;
			  }
			  inifile << std::endl;
		  }
	  }
	  inifile.close();
  }

  //deletes all stored ini data
  void CIniFile::Reset()
  {
	  iniSections.clear();
  }

  //returns number of keys currently in the ini
  int CIniFile::GetNumKeys()
  {
	  return iniSections.size();
  }

  //returns number of values stored for specified key, or -1 if key found
  int CIniFile::GetNumValues(const CIniString &keyname)
  {
	  CIniSections::iterator section;
	  section = iniSections.find(keyname);
	  if (section != iniSections.end()) {
		  // Got a section with the given name
		  return (*section).second.size();
	  } else {
		  return -1;
	  }
  }

  bool CIniFile::ValueExists(const CIniString &keyname, const CIniString &valuename) {
	  CIniSections::iterator section;
	  section = iniSections.find(keyname);

	  if (section == iniSections.end()) {
      return false;
	  }

	  CSection &keys = (*section).second;
	  CSection::iterator key;
	  key = keys.find(valuename);

	  if (key == keys.end()) {
      return false;
	  }

    return true;

  }

  //  Get the value for a specific key, within a given section.
  CIniString CIniFile::GetValue(const CIniString &keyname, const CIniString &valuename)
  {
	  CIniSections::iterator section;
	  section = iniSections.find(keyname);

	  if (section == iniSections.end()) {
		  error = "Unable to locate specified key.";
		  return "";
	  }

	  CSection &keys = (*section).second;
	  CSection::iterator key;
	  key = keys.find(valuename);

	  if (key == keys.end()) {
		  error = "Unable to locate specified value.";
		  return "";
	  }

	  return (*key).second;
  }

  //  Get the value for a specific key, within a given section.
  int CIniFile::GetValueI(const CIniString &keyname, const CIniString &valuename)
  {
	  return atoi(GetValue(keyname, valuename).c_str());
  }

  //  Get the value for a specific key, within a given section.
  double CIniFile::GetValueF(const CIniString &keyname, const CIniString &valuename)
  {
	  return atof(GetValue(keyname, valuename).c_str());
  }

  // Set a key/value pair within the given section.  If create == false, then the
  // key/value pair will NOT be inserted if it doesn't already exist.  Result is
  // true if the key/value pair is inserted.
  bool CIniFile::SetValue(const CIniString &keyname, const CIniString &valuename,
						  const CIniString &value, bool create)
  {
	  CIniSections::iterator section;
	  section = iniSections.find(keyname);

	  // If we don't have the section, then add it
	  if (section == iniSections.end()) {
		  if (!create) {
			  // No key and no desire to create one
			  return false;
		  }
		  // Make a key/value pair in a new section
		  iniSections[keyname][valuename] = value;

      return true;
	  }

	  CSection &keys = (*section).second;
	  CSection::iterator key = keys.find(valuename);

	  if (key == keys.end()) {
		  if (!create) {
			  // No key and no desire to create one
			  return false;
		  }

    }
    // Set the new value, overwriting any previous value.
    keys[valuename] = value;

	  return true;
  }

  // Set a key/value pair within the given section.  If create == false, then the
  // key/value pair will NOT be inserted if it doesn't already exist.  Result is
  // true if the key/value pair is inserted.
  bool CIniFile::SetValueI(const CIniString &keyname, const CIniString &valuename, int value, bool create)
  {
	  char buf[64];
	  sprintf(buf, "%d", value);
	  return SetValue(keyname, valuename, CIniString(buf), create);
  }

  // Set a key/value pair within the given section.  If create == false, then the
  // key/value pair will NOT be inserted if it doesn't already exist.  Result is
  // true if the key/value pair is inserted.
  bool CIniFile::SetValueF(const CIniString &keyname, const CIniString &valuename, double value, bool create)
  {
	  char buf[64];
	  sprintf(buf, "%e", value);
	  return SetValue(keyname, valuename, CIniString(buf), create);
  }

  // Delete a specific key/value pair from the given section
  bool CIniFile::DeleteValue(const CIniString &keyname, const CIniString &valuename)
  {
	  CIniSections::iterator section;
	  section = iniSections.find(keyname);
	  if (section == iniSections.end()) {
		  return false;
	  }
	  CSection &keys = (*section).second;

	  return (keys.erase(valuename) > 0);
  }

  // Delete an entire section, including all key/value pairs within it
  bool CIniFile::DeleteKey(const CIniString &keyname)
  {
	  return (iniSections.erase(keyname) > 0);
  }

  // Overloaded getline(...) to allow use of CIniString
  std::ifstream &CIniFile:: getline(std::ifstream &is, CIniString &str)
  {
      char buf[2048];
      is.getline(buf,2048);
      str = buf;
      return is;
  }

  void CIniFile::startSectionIteration(const CIniString &key) {
    currentSectionIt = iniSections.find(key);

    // if we dont' have a valid key to iterate over, can't do anything.
    if (currentSectionIt == iniSections.end()) {
      return;
    }

    sectionIt = currentSectionIt->second.begin();
  }

  bool CIniFile::getNextSectionValue(CIniString &valueName, CIniString &value) {
    if (currentSectionIt == iniSections.end()) {
      return false;
    }

    if (sectionIt == currentSectionIt->second.end()) {
      return false;
    }

    valueName = sectionIt->first;
    value = sectionIt->second;
    ++sectionIt;
    return true;
  }

}
