#ifndef MDAGNode_Header
#define MDAGNode_Header

#include "ModelGeneric.h"

namespace Aztec {
  
  class MDAGNode;
  
  typedef MRefCountedPtr<MDAGNode> MDAGNodePtr;
}

#include "MBaseObject.h"

#include <set>
#include <vector>

#include <MBaseUndoNode.h>

namespace Aztec {

  /** 
   * This is a DAG (Directed A-Cyclic Graph) Node. It is an object which 
   * contains a number of inputs and a number of outputs. When created, 
   * it places a pointer of itself with.
   */
  class MGENEXPORT MDAGNode : public MBaseObject {
  public:
    // Construction/Destruction
    MDAGNode();
    ~MDAGNode();
    
    /**
     * This adds the given node as an input to the object.
     * It also adds this node as an output of the other node.
     */
    bool addInput(MDAGNode *node);

    /**
     * Adds the given node as an output from this object.
     * It also adds this node as an input of the other node.
     */
    bool addOutput(MDAGNode *node);

    /**
     * This removes the given node as an input.
     * It also removes this node as an output of the other node.
     */
    void removeInput(MDAGNode *node);

    /**
     * This removes the given node as an output.
     * It also removes this node as an input of the other node.
     */
    void removeOutput(MDAGNode *node);

    /**
     * Gets the number of inputs into this node
     */
    int getInputCount();

    /**
     * This gets all the inputs of this object, and places them
     * into the given set.
     */
    void getInputs(MDAGNode **inps);

    /**
     * Gets the number of outputs going out from this node.
     */
    int getOutputCount();

    /**
     * This gets all the outputs of this object, and places them
     * into the given set.
     */
    void getOutputs(MDAGNode **outs);

    /**
     * This flags this object, and traverses all of this nodes outputs
     * and flags them as well.
     */
    void flagOutputs(AztecFlags flag);

    // MBaseObject Methods
    MStr getClassName() {return MStr("MDAGNode");};
    MStr getParentClassName() {return MStr("MNamedObect");};
    MBaseObjectPtr createNew();
    void setFrom(MBaseObjectPtr srcObj);
    virtual bool doUpdateObject();
    
    typedef std::vector<MDAGNode*> NodeList;
  protected:


    /**
     * Our bunch of input nodes
     */ 
    NodeList inputNodes;

    /**
     * Our bunch of output nodes.
     */
    NodeList outputNodes;

    /**
     * This adds the given node as an input to the object.
     */
    void doAddInput(MDAGNode *node);

    /**
     * Adds the given node as an output from this object.
     */
    void doAddOutput(MDAGNode *node);

    /**
     * This removes the given node as an input.
     */
    void doRemoveInput(MDAGNode *node);

    /**
     * This removes the given node as an output.
     */
    void doRemoveOutput(MDAGNode *node);

    // MUndoableObject methods
    void finishWithUndoNode();

  private:
    /**
     * A unique ID assigned to each node.
     */
    unsigned long dagNodeID;

    class Undo : public MBaseUndoNode {
    public:
      Undo(MDAGNode *node);

      MUndoableObject* getObject();
      void undo();
      void redo();

    private:
      std::vector<MDAGNodePtr> storedInputs;
      std::vector<MDAGNodePtr> storedOutputs;
      MRefCountedPtr<MDAGNode> object;
    };

    MRefCountedPtr<Undo> nodeUndo;

    friend class Undo;

    friend class MDAGraph;
    friend class MDAGraphDownstreamIterator;
    friend class MDAGraphUpstreamIterator;
  };
  
}


#endif
