
#ifndef MComponentisedObject_Header
#define MComponentisedObject_Header

#include <ModelGeneric.h>

namespace Aztec {
  class MComponentisedObject;
  typedef MRefCountedPtr<MComponentisedObject> MComponentisedObjectPtr;
}

#include "ModelTypes.h"
#include "MMath.h"
#include <vector>

namespace Aztec {
  //--------------
  //  MComponentisedObject
  //--------------
  class MGENEXPORT MComponentisedObject : public virtual MRefCountedObject {
  public:
    // This is just a quick hack to allow the values to be defined in 
    // the header file. In VC7 and gcc, we can do static const ints
    // and they can be compile time constants.
    typedef enum { OBJECT_TYPE = 0x01, 
                   POINT_TYPE = 0x02, 
                   EDGE_TYPE = 0x04, 
                   FACET_TYPE = 0x08 } ComponentType;

    static const AztecFlags COMPONENT_SELECTED;

    /**
     * Gets the total number of component types the system understands. This
     * is useful when you have a bitmask of component types, and you want to
     * separate out each one.
     */
    static int getTypeCount();
    /**
     * This gets the value referring to the nth known type.
     */
    static ComponentType getType(int index);

    typedef enum MAction {atSet, atUnset, atToggle};

    virtual bool isInComponentMode(AztecFlags type = POINT_TYPE | EDGE_TYPE | FACET_TYPE) = 0;
    virtual void setComponentMode(AztecFlags type) = 0;
    virtual AztecFlags getComponentMode() = 0;

    /**
     * Does the object have the given component type.
     */
    virtual bool hasComponentType(AztecFlags type) = 0;
    /**
     * Gets the number of components of the given type
     */
    virtual int getComponentCount(AztecFlags type) = 0;
    /**
     * Indicates that the given component at the given index of the 
     * given type should be selected, deselected, or toggled.
     */
    virtual bool flagComponent(AztecFlags type, int index, AztecFlags flag, MAction action = atSet) = 0;
    /**
     * Is the component at the given index of the type selected?
     */
    virtual bool isComponentFlagged(AztecFlags type, int index, AztecFlags flag = COMPONENT_SELECTED) = 0;
    /**
     * Are there any components of the guven type flagged?
     */
    virtual bool hasComponentsFlagged(AztecFlags type, AztecFlags flag = COMPONENT_SELECTED) = 0;
    /**
     * Sets the colour of the given component if possible. It
     * is a temporary colour and not a permanent one
     */
    virtual bool setComponentFeedbackColour(AztecFlags type, int index, const MRGBAFloat &colour) = 0;

    /**
     * This disables component feedback colouring for a particular component.
     */
    virtual bool unsetComponentFeedbackColour(AztecFlags type, int index) = 0;

    /**
     * Gets the feedback colour of a given component.
     */
    virtual MRGBAFloat getComponentFeedbackColour(AztecFlags type, int index) = 0;

    /**
     * This selects all the components connected to the current selection
     */
    virtual bool flagConnected(ComponentType type, AztecFlags flag = COMPONENT_SELECTED) = 0;
    /**
     * This expands the selection of the components. If any component is 
     * touching any selected component, it too is selected.
     */
    virtual bool growFlagged(ComponentType type, AztecFlags flag = COMPONENT_SELECTED) = 0;
    /**
     * This contracts the given the selection. Any component that
     * is touching an unselected component is deselected.
     */
    virtual bool shrinkFlagged(ComponentType type, AztecFlags flag = COMPONENT_SELECTED) = 0;

    /**
     * This gets the centre of the component in object space.
     *
     * @param type The type of the component to retrieve
     * @param index The index of the component to retrieve.
     * @return The centre of the component
     */
    virtual MVector3 getComponentCentre(AztecFlags type, int index) = 0;

    /**
     * This gets the centre of the flagged component in object space.
     *
     * @param type The type of the component to retrieve
     * @param flag The flags that ach component must have to be counted towards the centre.
     * @return The centre of the component
     */
    virtual MVector3 getFlaggedCentre(AztecFlags type, AztecFlags flag = COMPONENT_SELECTED) = 0;

    /**
     * This gets the normal of the component in object space.
     *
     * @param type The type of the component to retrieve
     * @param index The index of the component to retrieve.
     * @return The normal of the given component.
     */
    virtual MVector3 getComponentNormal(AztecFlags type, int index) = 0;

  };
}
#endif
