#include <AztecGUICommonPCH.h>
#include <gui/MTreeControl.h>

#include <gui/win32/MAppImpl.h>
#include <gui/win32/MTreeControlImpl.h>

#include <algorithm>

#include <assert.h>

namespace Aztec {


  MTreeControl::MTreeControl() {
    INITCOMMONCONTROLSEX init;
    init.dwSize = sizeof(init);
    init.dwICC = ICC_TREEVIEW_CLASSES;
    InitCommonControlsEx(&init);

    clickedItem = NULL;
    m_hFirstSelectedItem = NULL;
    m_bSelectPending = FALSE;
    m_bSelectionComplete = TRUE;
  }


  MTreeControl::~MTreeControl() {
  }

  MTreeItemPtr MTreeControl::getItem(HTREEITEM hItem) {
    return new MTreeItem(m_hWnd, hItem);
  }

  bool MTreeControl::createImpl() {
    MApp *app;
    HWND parentHWND = NULL;

    if (m_Parent != NULL) {
      parentHWND = m_Parent->getHWnd();
    }

    app = MApp::getInstance();

    m_hWnd = ::CreateWindowEx(0, WC_TREEVIEW, "graphTreeView", 
      WS_CHILD | TVS_DISABLEDRAGDROP | TVS_HASLINES | TVS_HASBUTTONS | TVS_LINESATROOT | TVS_SHOWSELALWAYS,
      0, 0, 20, 20, parentHWND, NULL, app->getHInstance(), NULL);
//    m_hWnd = ::CreateWindow("WC_TREEVIEW", NULL, WS_CHILD | WS_CLIPCHILDREN,
//                            0, 10, 0, 10, parentHWND, NULL, app->getHInstance(), NULL);

    if (m_hWnd != 0) {
      app->registerComponent(m_hWnd, this);

      onCreate();

      setVisible(true);
      ::UpdateWindow(m_hWnd);

      return true;
    }

    return false;
  }

  bool MTreeControl::preTranslateMessage(MSG *msg) {
    bool handled = false;

    if (msg->message == WM_LBUTTONDOWN) {
      WPARAM nFlags = msg->wParam;        // key flags 
      POINT point = { LOWORD(msg->lParam), HIWORD(msg->lParam) };  // vertical position of cursor   
      
      UINT flags = 0;
      HTREEITEM item = hitTest(point.x, point.y, &flags);

      if( flags & TVHT_ONITEM )
      {
        ::SetFocus(m_hWnd);
        
        clickedItem = item;
        
        // Is the clicked item already selected ?
        BOOL clickedItemSelected = getItemState( item, TVIS_SELECTED ) & TVIS_SELECTED;
        
        if ( clickedItemSelected ) {
          // Maybe user wants to drag/drop multiple items!
          // So, wait until OnLButtonUp() to do the selection stuff. 
          m_bSelectPending = TRUE;
        } else {
          SelectMultiple( item, nFlags, point );
          m_bSelectPending = FALSE;
        }
        
        m_ptClick = point;

        handled = true;
      } 

      notifyChangedSelections();

    } else if (msg->message == WM_LBUTTONUP) {
      if ( m_bSelectPending )
      {
        WPARAM nFlags = msg->wParam;        // key flags 
        POINT point = { LOWORD(msg->lParam), HIWORD(msg->lParam) };  // vertical position of cursor   

        // A select has been waiting to be performed here
        SelectMultiple( clickedItem, nFlags, point );
        m_bSelectPending = FALSE;
      }
      
      clickedItem = NULL;

      notifyChangedSelections();
    }

    if (!handled) {
      return MComponent::preTranslateMessage(msg);
    } else {
      return true;
    }
  }

  bool MTreeControl::handleWMCommandNotify(int notifyCode, int id) {

    return MComponent::handleWMCommandNotify(notifyCode, id);
  }

  void MTreeControl::selectItem(const MTreeItemPtr &item, bool selected) {
    TVITEM tvItem = { TVIF_STATE, 
                    item != NULL ? item->item : NULL,
                    selected ? TVIS_SELECTED : 0,
                    TVIS_SELECTED,
                    NULL, 
                    0,
                    0,
                    0,
                    0 };

    TreeView_SetItem(m_hWnd, &tvItem);

//    TreeView_SelectItem(m_hWnd, item != NULL ? item->item : NULL);
  }

  void MTreeControl::clear() {
    TreeView_DeleteAllItems(m_hWnd);
  }

  MTreeItemPtr MTreeControl::addItem(const std::string &text) {
    return addItem(text, NULL);;
  }

  MTreeItemPtr MTreeControl::addItem(const std::string &text, const MTreeItemPtr &parent) {
    TVINSERTSTRUCT tv;

    tv.hParent = parent != NULL ? parent->item : NULL;
    tv.hInsertAfter = TVI_LAST;

    // TODO: make this handle the difference versions of the com control library. I.e before and after version 4.71
    tv.item.mask = TVIF_TEXT;
    tv.item.hItem = NULL;
    tv.item.pszText = const_cast<char*>(text.c_str());
    tv.item.cchTextMax = text.length() + 1;

    return getItem(TreeView_InsertItem(m_hWnd, &tv));
  }

  void MTreeControl::expandItem(const MTreeItemPtr &item, bool expand) {
  }

  bool MTreeControl::onItemSelected(const MTreeItemPtr &item) {
    return false;
  }

  HTREEITEM MTreeControl::hitTest(int x, int y, UINT *flags) {
    TVHITTESTINFO test;
    test.pt.x = x;
    test.pt.y = y;
    test.flags = 0;
    test.hItem = NULL;

    TreeView_HitTest(m_hWnd, &test);

    *flags = test.flags;

    return test.hItem;

  }

  UINT MTreeControl::getItemState(HTREEITEM item, UINT mask) {
    TVITEM data = { TVIF_STATE, item, 0, mask } ;

    TreeView_GetItem(m_hWnd, &data);

    return data.state & mask;
  }

  void MTreeControl::setItemState(HTREEITEM item, UINT value, UINT mask) {
    TVITEM data = { TVIF_STATE, 
                    item,
                    value,
                    mask,
                    NULL, 
                    0,
                    0,
                    0,
                    0 };

    TreeView_SetItem(m_hWnd, &data);

    assert(getItemState(item, mask) == value);


  }

  DWORD MTreeControl::getItemData(HTREEITEM item) {
    TVITEM data = { TVIF_PARAM, item, 0, 0, 0, 0, 0, 0, 0, 0} ;

    TreeView_GetItem(m_hWnd, &data);

    return data.state;
  }

  
  void MTreeControl::SelectMultiple( HTREEITEM hClickedItem, UINT nFlags, POINT point )
  {
    // Start preparing an NM_TREEVIEW struct to send a notification after selection is done
    NM_TREEVIEW tv;
    memset(&tv.itemOld, 0, sizeof(tv.itemOld));
    
    HWND pWnd = ::GetParent(m_hWnd);
    
    HTREEITEM hOldItem = TreeView_GetSelection(m_hWnd);
    
    if ( hOldItem )
    {
      tv.itemOld.hItem = hOldItem;
      tv.itemOld.state = getItemState( hOldItem, 0xffffffff );
      tv.itemOld.lParam = getItemData( hOldItem );
      tv.itemOld.mask = TVIF_HANDLE|TVIF_STATE|TVIF_PARAM;
    }
    
    // Flag signaling that selection process is NOT complete.
    // (Will prohibit TVN_SELCHANGED from being sent to parent)
    m_bSelectionComplete = FALSE;
    
    // Action depends on whether the user holds down the Shift or Ctrl key
    if ( nFlags & MK_SHIFT )
    {
      // Select from first selected item to the clicked item
      if ( !m_hFirstSelectedItem ) {
        m_hFirstSelectedItem = TreeView_GetSelection(m_hWnd);
      }
      
      SelectItems( m_hFirstSelectedItem, hClickedItem );
    }
    else if ( nFlags & MK_CONTROL )
    {
      // Find which item is currently selected
      HTREEITEM hSelectedItem = TreeView_GetSelection(m_hWnd);
      
      // Is the clicked item already selected ?
      BOOL bIsClickedItemSelected = getItemState( hClickedItem, TVIS_SELECTED ) & TVIS_SELECTED;
      BOOL bIsSelectedItemSelected = FALSE;
      if ( hSelectedItem )
        bIsSelectedItemSelected = getItemState( hSelectedItem, TVIS_SELECTED ) & TVIS_SELECTED;
      
      // Must synthesize a TVN_SELCHANGING notification
      if ( pWnd )
      {
        tv.hdr.hwndFrom = m_hWnd;
        tv.hdr.idFrom = GetWindowLong( m_hWnd, GWL_ID );
        tv.hdr.code = TVN_SELCHANGING;
        
        tv.itemNew.hItem = hClickedItem;
        tv.itemNew.state = getItemState( hClickedItem, 0xffffffff );
        tv.itemNew.lParam = getItemData( hClickedItem );
        
        tv.itemOld.hItem = NULL;
        tv.itemOld.mask = 0;
        
        tv.action = TVC_BYMOUSE;
        
        tv.ptDrag.x = point.x;
        tv.ptDrag.y = point.y;
        
        ::SendMessage( pWnd, WM_NOTIFY, tv.hdr.idFrom, (LPARAM)&tv );
      }
      
      // If the previously selected item was selected, re-select it
      if ( bIsSelectedItemSelected ) {
        selectItem(hSelectedItem, true);
      }
      
      // We want the newly selected item to toggle its selected state,
      // so unselect now if it was already selected before
      if ( bIsClickedItemSelected ) {
//        setItemState( hClickedItem, 0, TVIS_SELECTED );
        selectItem(hClickedItem, false);
      } else {
//        TreeView_SelectItem(m_hWnd, hClickedItem);
//        setItemState( hClickedItem, TVIS_SELECTED, TVIS_SELECTED );
        selectItem(hClickedItem, true);
      }
      
      // If the previously selected item was selected, re-select it
      if ( bIsSelectedItemSelected && hSelectedItem != hClickedItem ) {
//        setItemState( hSelectedItem, TVIS_SELECTED, TVIS_SELECTED );
        selectItem(hSelectedItem, true);
      }
      
      // Store as first selected item (if not already stored)
      if ( m_hFirstSelectedItem==NULL )
        m_hFirstSelectedItem = hClickedItem;
    }
    else
    {
      // Clear selection of all "multiple selected" items first
      ClearSelection();
      
      // Then select the clicked item
//      TreeView_SelectItem( m_hWnd, hClickedItem );
//      setItemState( hClickedItem, TVIS_SELECTED, TVIS_SELECTED );
      selectItem(hClickedItem, true);
      
      // Store as first selected item
      m_hFirstSelectedItem = hClickedItem;
    }
    
    // Selection process is now complete. Since we have 'eaten' the TVN_SELCHANGED 
    // notification provided by Windows' treectrl, we must now produce one ourselves,
    // so that our parent gets to know about the change of selection.
    m_bSelectionComplete = TRUE;

    if ( pWnd )
    {
      tv.hdr.hwndFrom = m_hWnd;
      tv.hdr.idFrom = GetWindowLong( m_hWnd, GWL_ID );
      tv.hdr.code = TVN_SELCHANGED;
      
      tv.itemNew.hItem = clickedItem;
      tv.itemNew.state = getItemState( clickedItem, 0xffffffff );
      tv.itemNew.lParam = getItemData( clickedItem );
      tv.itemNew.mask = TVIF_HANDLE|TVIF_STATE|TVIF_PARAM;
      
      tv.action = TVC_UNKNOWN;
      
      ::SendMessage( pWnd, WM_NOTIFY, tv.hdr.idFrom, (LPARAM)&tv );
    }
  }

  void MTreeControl::ClearSelection()
  {
    //	if ( !bMultiOnly )
    //		SelectItem( NULL );
    
    for ( HTREEITEM hItem=TreeView_GetRoot(m_hWnd); hItem!=NULL; hItem=TreeView_GetNextVisible( m_hWnd, hItem ) )
      if ( getItemState( hItem, TVIS_SELECTED ) & TVIS_SELECTED ) {
        selectItem(hItem, false);
      }
  }

  BOOL MTreeControl::SelectItems( HTREEITEM hFromItem, HTREEITEM hToItem )
  {
    // Determine direction of selection 
    // (see what item comes first in the tree)
    HTREEITEM hItem = TreeView_GetRoot(m_hWnd);
    
    while ( hItem && hItem!=hFromItem && hItem!=hToItem )
      hItem = TreeView_GetNextVisible( m_hWnd, hItem );
    
    if ( !hItem )
      return FALSE;	// Items not visible in tree
    
    BOOL bReverse = hItem==hToItem;
    
    // "Really" select the 'to' item (which will deselect 
    // the previously selected item)
    
    TreeView_SelectItem( m_hWnd, hToItem );
    
    // Go through all visible items again and select/unselect
    
    hItem = TreeView_GetRoot(m_hWnd);
    BOOL bSelect = FALSE;
    
    while ( hItem )
    {
      if ( hItem == ( bReverse ? hToItem : hFromItem ) )
        bSelect = TRUE;
      
      if ( bSelect )
      {
        if ( !( getItemState( hItem, TVIS_SELECTED ) & TVIS_SELECTED ) )
          selectItem(hItem, true);
//          setItemState( hItem, TVIS_SELECTED, TVIS_SELECTED );
      }
      else
      {
        if ( getItemState( hItem, TVIS_SELECTED ) & TVIS_SELECTED )
          selectItem(hItem, false);
//          setItemState( hItem, 0, TVIS_SELECTED );
      }
      
      if ( hItem == ( bReverse ? hFromItem : hToItem ) )
        bSelect = FALSE;
      
      hItem = TreeView_GetNextVisible( m_hWnd, hItem );
    }
    
    return TRUE;
  }
  
  void MTreeControl::sendTVSelChanged(HTREEITEM item, const POINT &p) {
  }

  bool MTreeControl::handleWMNotify(WPARAM wParam, LPARAM lParam) {
    // see if it is a tree control notfication
    NMHDR *notify = (NMHDR*)lParam;
    if (notify->code == TVN_SELCHANGED && m_bSelectionComplete) {

      NMTREEVIEW *nm = (NMTREEVIEW*)lParam;
      if (nm->hdr.code == TVN_SELCHANGED) {
        return onItemSelected(getItem(nm->itemNew.hItem));
      }
    }
    
    return false;
  }

  void MTreeControl::selectItem(HTREEITEM item, bool selected) {
    UINT newValue = selected ? TVIS_SELECTED : 0;
    if (getItemState(item, TVIS_SELECTED) != newValue) {
      setItemState( item, newValue, TVIS_SELECTED );
      changedItems.insert(item);
    }
  }

  void MTreeControl::notifyChangedSelections() {
    HWND pWnd = ::GetParent(m_hWnd);
    if ( pWnd ) {
      NM_TREEVIEW tv;
      memset(&tv.itemOld, 0, sizeof(tv.itemOld));

      tv.hdr.hwndFrom = m_hWnd;
      tv.hdr.idFrom = GetWindowLong( m_hWnd, GWL_ID );
      tv.hdr.code = TVN_SELCHANGED;
      
      tv.itemNew.hItem = clickedItem;
      tv.itemNew.state = getItemState( clickedItem, 0xffffffff );
      tv.itemNew.lParam = getItemData( clickedItem );
      tv.itemNew.mask = TVIF_HANDLE|TVIF_STATE|TVIF_PARAM;
      
      tv.action = TVC_UNKNOWN;

      for (std::set<HTREEITEM>::iterator it = changedItems.begin(); it != changedItems.end(); ++it) {
        HTREEITEM item = *it;
        tv.itemNew.hItem = item;
        tv.itemNew.state = getItemState( item, 0xffffffff );
        tv.itemNew.lParam = getItemData( item);
        ::SendMessage( pWnd, WM_NOTIFY, tv.hdr.idFrom, (LPARAM)&tv );
      }

      changedItems.clear();
    }
  }

  //-------------

  MTreeItem::MTreeItem() {
    tree = NULL;
    item = NULL;
  }

  MTreeItem::MTreeItem(const MTreeItem &src) {
    tree = src.tree;
    item = src.item;
  }

  MTreeItem::MTreeItem(HWND treeWnd, HTREEITEM treeitem) {
    tree = treeWnd;
    item = treeitem;
  }

  // IMTreeItem methods.
  std::string MTreeItem::getText() const {
    char textBuf[1024];
    TVITEM info;
    info.mask = TVIF_TEXT;
    info.hItem = item;
    info.state = 0;
    info.stateMask = 0;
    info.pszText = textBuf;
    info.cchTextMax = 1024;
    info.iImage = 0;
    info.iSelectedImage = 0;
    info.cChildren = 0;
    info.lParam = 0;

    TreeView_GetItem(tree, &info);

    return textBuf;
  }

  int MTreeItem::compare(const MTreeItemPtr &rhs) {

    if (std::make_pair(tree, item) < std::make_pair(rhs->tree, rhs->item)) {
      return -1;
    } else if (std::make_pair(rhs->tree, rhs->item) < std::make_pair(tree, item)) {
      return 1;
    } else {
      return 0;
    }
  }

  MTreeItemPtr MTreeItem::getParent() const {
    HTREEITEM parent = TreeView_GetParent(tree, item);

    if (parent != NULL) {
      return new MTreeItem(tree, parent);
    } else {
      return NULL;
    }
  }

  bool MTreeItem::getSelected() const {
    TVITEM data = { TVIF_STATE, item, 0 , TVIS_SELECTED } ;

    TreeView_GetItem(tree, &data);

    return (data.state & TVIS_SELECTED) != 0;
  }



}

