#include <Aztec3DPCH.h>
#include <views/AztecViewManager.h>

#include <controls/ChannelBar.h>
#include <controls/TimeSlider.h>
#include <MScene.h>

#include <set>
#include <map>
#include <algorithm>

namespace AztecGUI {

  static AztecView* currentView = NULL;
  static AztecView* mouseOverView = NULL;
  static std::set<AztecView*> allViews;
  static std::map<std::string, AztecViewPtr> viewTypes;
  static int viewUpdateCounter = 0;
  static int pendingDrawCount = 0;
  static std::set<AztecView*> pendingDrawViews;

  void AztecViewManager::cleanup() {
    currentView = NULL;
    allViews.clear();
    viewTypes.clear();
  }

  AztecViewPtr AztecViewManager::getCurrentView() {
    return currentView;
  }

  void AztecViewManager::setCurrentView(const AztecViewPtr &view) {
    if (view != currentView) {
      AztecViewPtr oldView = currentView;
      currentView = &*view;

      if (oldView != NULL) {
        oldView->refresh();
      }
      if (currentView != NULL) {
        currentView->refresh();
      }
    }
  }
  
  void AztecViewManager::setLastMouseOverView(AztecViewPtr &view) {
    if (view != mouseOverView) {
      AztecViewPtr oldView = mouseOverView;
      mouseOverView = &*view;
    }
  }

  AztecViewPtr AztecViewManager::getLastMouseOverView() {
    return mouseOverView;
  }

  inline void doRefresh(AztecView *view) {
    view->refresh();
  }

  inline bool canUpdateViewports() {
    return viewUpdateCounter >= 0;
  }

  void AztecViewManager::redrawAllViews() {
    // Check to see if we can update the viewports
    if (canUpdateViewports()) {
      // if we can, reset the pending draw counts, and do the actual redrawing.
      pendingDrawViews.clear();
      pendingDrawCount = 0;

      // refresh the views
      std::for_each(allViews.begin(), allViews.end(), doRefresh);

      // refresh the channel bar
      Aztec::MScenePtr s = Aztec::MScene::getGlobalScene();
      Aztec::MNamedObjectPtr obj = AZTEC_CAST(Aztec::MNamedObject, s->getSelectedObjectList()->getHead());

      ChannelBar::getInstance()->showParametersFor(obj);
      MTimeSlider::getInstance()->refresh();
    } else {
      // otherwise, keep track of the redraw all command.
      
      pendingDrawCount++;
      pendingDrawViews.clear();
    }
  }

  void AztecViewManager::redrawCurrentView() {
    // check to see if we can update the viewport.
    if (canUpdateViewports()) {
      if (currentView != NULL) {
        currentView->refresh();
      }
    } else if (pendingDrawCount == 0) {
      // if we can't update, and if a redrawAllViews() hasn't been called, 
      // keep track of which views need to be updated.
      pendingDrawViews.insert(currentView);
    }
  }

  void AztecViewManager::registerView(AztecView *view) {
    allViews.insert(view);
  }

  void AztecViewManager::unregisterView(AztecView *view) {
    if (currentView == view) {
      currentView = NULL;
    }
    allViews.erase(view);
    pendingDrawViews.erase(view);
  }

  void AztecViewManager::registerViewType(const std::string &viewType, const AztecViewPtr &view) {
    viewTypes[viewType] = view;

    // make sure this view doesn't appear in the allViews set.
    allViews.erase(&*view);
  }

  AztecViewPtr AztecViewManager::createView(const std::string &viewType) {
    // return a copy of the view if we have one taht matches our type.
    AztecViewPtr view = viewTypes[viewType];
    if (view != NULL) {
      return view->createCopy();
    } else {
      return NULL;
    }
  }

  void AztecViewManager::enableViewUpdating() {
    ++viewUpdateCounter;
    // check to see if we can update the viewports.
    if (canUpdateViewports()) {
      // if we can, refresh any viewports that were attempted to be refreshed while we were disabled.
      if (pendingDrawCount > 0) {
        redrawAllViews();
      } else {
        std::for_each(pendingDrawViews.begin(), pendingDrawViews.end(), doRefresh);
        pendingDrawViews.clear();
      }
    }
  }

  void AztecViewManager::disableViewUpdating() {
    --viewUpdateCounter;
  }

}
