#include <Aztec3DPCH.h>
#include <tools/QuickSliceTool.h>


// Aztec2 includes
#include <views/Aztec3DView.h>
#include <utils/AztecGLUtils.h>
#include <config/UIConfig.h>
#include <views/AztecViewManager.h>
#include <utils/SceneFunctions.h>
#include <functions/mesh/MeshFunctions.h>

// AztecLib includes
#include <MUIManager.h>
#include <MScene.h>
#include <MEditableMesh.h>
#include <misc/MSceneHelper.h>
#include <MSystemManager.h>

namespace AztecGUI {
  
  QuickSliceTool::QuickSliceTool()
  {
    doingFirstPoint = true;
  }

  std::string QuickSliceTool::getName() {
    return "toolQuickSlice";
  }

  int QuickSliceTool::drawTool(bool select, const Aztec::MComponentPtr &component)
  {
    if (AztecViewManager::getCurrentView() == AztecView::getViewForComponent(component) && 
      !doingFirstPoint) {
      glColor3f(1.0, 0.95, 0.1);

      // draw a line      
      glDisable(GL_DEPTH_TEST);
      glBegin(GL_LINES);

      glVertex3fv((float*)&firstPoint);
      glVertex3fv((float*)&secondPoint);

      glEnd();
      glEnable(GL_DEPTH_TEST);
    }
    return 0;
  }

  void QuickSliceTool::initialise() {
    doingFirstPoint = true;
  }

  bool QuickSliceTool::finish() {
    // finishing has no meaning for this tool, so return false;
    return false;
  }

  bool QuickSliceTool::cancel() {
  
    // if we are only on the first point, then we are done with this tool
    if (doingFirstPoint) {
      return true;
    } else {
      // if we are on the second point, the cancel will bring us back to the first point.
      // so we return false to say that we are NOT done with the tool.
      doingFirstPoint = true;
      return false;
    }
  }

  bool QuickSliceTool::inProgress() {
    return !doingFirstPoint;
  }


  int QuickSliceTool::onMouseDown(const Aztec::MMouseEvent &event)
  {
    MXYZToolType::onMouseDown(event);
  
    if (doingFirstPoint) {
      firstPoint = getPositionFromMouse(event);
      secondPoint = getPositionFromMouse(event);
      doingFirstPoint = false;
    } else {
      secondPoint = getPositionFromMouse(event);
      doingFirstPoint = true;

      // work out the plane we clicked on.
      AztecGLCanvasPtr glView = AztecGLView::getGLCanvasFor(event.getComponent());
      Aztec::MRay ray = glView->getRay(event.getX(), event.getY());

      Aztec::MVector3 normal = (secondPoint - ray.Org) / (firstPoint - ray.Org);

      sliceSelectedMeshes(firstPoint, normal);
    }
    
    return TOOLRESULT_DRAWALL;
  }

  int QuickSliceTool::onMouseMove(const Aztec::MMouseEvent &event) {
    if (doingFirstPoint) {
      firstPoint = getPositionFromMouse(event);
    } else {
      secondPoint = getPositionFromMouse(event);
    }

    return TOOLRESULT_DRAWCURRENT;
  }

  int QuickSliceTool::onMouseUp(const Aztec::MMouseEvent &event) {
    return TOOLRESULT_DRAWNONE;
  }

  int QuickSliceTool::getDefaultManip() {
    return -1;
  }

  Aztec::MVector3 QuickSliceTool::getPositionFromMouse(const Aztec::MMouseEvent &event) {
    AztecGLCanvasPtr glView = AztecGLView::getGLCanvasFor(event.getComponent());

    Aztec::MVector3 normal = glView->getViewNormal();

    // add the new point in.
    Aztec::MRay ray = glView->getRay(event.getX(), event.getY());
    Aztec::MPlane plane(ray.Org + 10 * normal, normal);

    return ray.intersectWithPlane(plane);
  }

  class QuickSlice {
  public:
    QuickSlice(const Aztec::MVector3 &o, const Aztec::MVector3 &n)
      : planeOrigin(o), planeNormal(n)
    {
    }

    void operator()(const Aztec::MScenePtr &scene, 
                    const Aztec::MTreeObjectNodePtr &node,
                    const Aztec::MSceneObjectPtr &sceneObj, 
                    const Aztec::MEditableMeshPtr &mesh) {

      // transform the plane into object space.
      Aztec::MPlane plane;
      Aztec::MMatrix4 matrix;
      sceneObj->getWorldInverseTransformMatrix(matrix);

      // transform the origin.
      plane.set(matrix * planeOrigin, matrix * planeNormal - matrix * Aztec::MVector3(0,0,0));

      mesh->unsetTriangleFlags(TRIANGLE_FLAGFORCHANGE);

      std::map<Edge, int> edges;
      typedef std::vector< std::map<Edge, int>::iterator > ConnectArray;
      ConnectArray connects;

      for (int i = mesh->getNumTris() - 1; i >= 0; --i) {

        // skip over triangles we've already done.
        if (mesh->isTriangleFlagged(i, TRIANGLE_FLAGFORCHANGE)) {
          continue;
        }
        
        // get all the triangles in this polygon.
        std::set<int> polygon;
        mesh->getPolygonFromTriangle(i, polygon);


        // see if any of the visible edges are split by the plane
        Edge edgeA, edgeB;

        // loop over these triangles and get any edges that need to be divided.
        for (std::set<int>::iterator t = polygon.begin(); t != polygon.end(); ++t) {

          mesh->setTriangleFlag(*t, TRIANGLE_FLAGFORCHANGE);

          float d[3];
          int tri[3];
          for (int j = 0; j < 3; ++j) {
            tri[j] = mesh->getTriangleVertex(*t, j);
            d[j] = mesh->getVertexPosition(tri[j]) * plane;
          }

          for (int j = 0; j < 3; ++j) {

            if (mesh->isTriangleEdgeFlagged(*t, j, EDGE_VISIBLE)) {
          
              if (d[j] * d[(j+1)%3] < -0.0001) {

                if (edgeA.a == -1) {
                  edgeA.a = tri[j];
                  edgeA.b = tri[(j+1)%3];
                  edgeA.n = -1;
                } else if (edgeB.a == -1) {
                  edgeB.a = tri[j];
                  edgeB.b = tri[(j+1)%3];
                  edgeB.n = -1;
                } else {
                  // too many edge to split, break out of this one.
                  break;
                }
              } else if (fabs(d[j]) < 0.0001) {
                if (edgeA.a == -1) {
                  edgeA.a = tri[j];
                  edgeA.b = tri[(j+1)%3];
                  edgeA.n = tri[j];
                } else if (edgeB.a == -1) {
                  edgeB.a = tri[j];
                  edgeB.b = tri[(j+1)%3];
                  edgeB.n = tri[j];
                } else {
                  break;
                }

              }

            }
          }

        }

        if (edgeA.a != -1 && edgeB.b != -1) {
          std::map<Edge, int>::iterator a = edges.insert(std::make_pair(edgeA, edgeA.n)).first;
          std::map<Edge, int>::iterator b = edges.insert(std::make_pair(edgeB, edgeB.n)).first;

          connects.push_back(a);
          connects.push_back(b);
        }
          
      }

      // now loop over the edges, divide them and connect them up
      for (std::map<Edge, int>::iterator i = edges.begin(); i != edges.end(); ++i) {
        if (i->second == -1) {
          Aztec::MVector3 a = mesh->getVertexPosition(i->first.a);
          Aztec::MVector3 b = mesh->getVertexPosition(i->first.b);

          float da = a * plane;
          float db = b * plane;
          float total = db - da;
          float frac = db/total;
          if (frac > 0) frac = -1.0 + frac;
          Aztec::MVector3 newPos = a - frac*(b - a);

          i->second = mesh->divideEdge(i->first.a, i->first.b, newPos);
        }
        if (i->second != -1) {
          mesh->setVertexFlag(i->second, VERTEX_SELECTED);
        }
      }

      // now go and connect up all the new points
      for (ConnectArray::iterator i = connects.begin(); i != connects.end(); ++i) {
        int a = (*i)->second;
        ++i;
        int b = (*i)->second;

        pointConnectMesh(mesh, a, b);
        mesh->setEdgeFlag(a, b, EDGE_SELECTED);
      }

    }

    const Aztec::MVector3 planeOrigin;
    const Aztec::MVector3 planeNormal;
  };

  void QuickSliceTool::sliceSelectedMeshes(const Aztec::MVector3 &origin, const Aztec::MVector3 &normal) {
    applyToEditableMeshes(Aztec::MScene::getGlobalScene(), Aztec::MScene::getGlobalScene()->getObjectList(), ifSelectedCriteria, QuickSlice(origin, normal));
  }


}