#include <Aztec3DPCH.h>
#include <tools/MSelectTool.h>
#include <utils/AztecKeyUtils.h>

#include <views/AztecViewManager.h>
//#include <ActionManager.h>

#include <views/AztecGLView.h>
#include <views/Aztec3DView.h>

// AztecLib includes
#include <MUIManager.h>
#include <MScene.h>
#include <scripting/MScriptInterp.h>

namespace AztecGUI {

  using namespace Aztec;

  //-------------------
  //  MSelectToolType
  //-------------------
  MSelectToolType::MSelectToolType() {
    justCancelled = false;
    drawingBox = false;
  }

  std::string MSelectToolType::getName() {
    return "toolSelect";
  }

  void MSelectToolType::initialise() {
    justCancelled = false;
  }

  bool MSelectToolType::cancel() {
    // if we are cancelling while drawing a box, we stop the selection
    if (m_BoxSelect) {
      m_Selecting = false;
      m_BoxSelect = false;
      drawingBox = false;
      return false;
    }

    // if we have already cancelled the tool, then we go back to object mode.
    if (justCancelled && Aztec::MUIManager::getComponentMode() != Aztec::MComponentisedObject::OBJECT_TYPE) {
      Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.componentModeSet('object')");

      justCancelled = false;
    } else {
      Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.selectNone();");
      justCancelled = true;
    }

    // return false because we are not finished with this tool.
    return false;
  }

  bool MSelectToolType::inProgress() {
    return m_BoxSelect;
  }

  int MSelectToolType::drawTool(bool Select, const Aztec::MComponentPtr &View)
  {
    // If we are in select mode, no need to draw anything
    if (Select) {
      return 1;
    }
  
    if (View != AztecViewManager::getCurrentView()) {
      return 1;
    }
  
    AztecGLViewPtr viewGL = AZTEC_CAST(AztecGLView, View);

    if (viewGL == NULL) {
      return 0;
    }
    
    AztecGLCanvasPtr GLWnd = viewGL->getCanvas();
  
    // Check to see if the mouse buttons are down.
    if (drawingBox)
    {
      MSize2D WindowRect;
    
      WindowRect = GLWnd->getClientSize();
    
      glPushAttrib(GL_ENABLE_BIT);
      glDisable(GL_DEPTH_TEST);
    
      glDisable(GL_LIGHTING);
      glDisable(GL_DEPTH_TEST);
    
      // Push the projection stack, so we can draw using screen coordinates
	     glMatrixMode(GL_PROJECTION);
       glPushMatrix();
     
       glViewport(0,0,WindowRect.getWidth(), WindowRect.getHeight());
     
       glLoadIdentity();
     
       glOrtho(0,WindowRect.getWidth(),0,WindowRect.getHeight(),-20,20);
     
       glMatrixMode(GL_MODELVIEW);
       glPushMatrix();
     
       glLoadIdentity();
     
     
       // Draw the actual selection box
       glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
       glLineWidth(1);
       glColor3f(0.0f, 0.8f, 0.6f);
     
       glBegin(GL_LINE_LOOP);
     
       glVertex2f(m_DownPos.x,WindowRect.getHeight()-m_DownPos.y);
       glVertex2f(m_DownPos.x,WindowRect.getHeight()-m_CurPos.y);
       glVertex2f(m_CurPos.x,WindowRect.getHeight()-m_CurPos.y);
       glVertex2f(m_CurPos.x,WindowRect.getHeight()-m_DownPos.y);
     
       glEnd();
     
       // Clean up this mess matrix wise
       glMatrixMode(GL_PROJECTION);
       glPopMatrix();
     
       glMatrixMode(GL_MODELVIEW);
       glPopMatrix();
     
       glPopAttrib();
    }
  
    return 1;
  }

  int MSelectToolType::onMouseDown(const Aztec::MMouseEvent &event)
  {
    MToolType::onMouseDown(event);
  
    justCancelled = false;

    m_Selecting = false;
    m_Deselecting = false;
  
  
    m_BoxSelect = false;
    m_Deselecting = false;
  
    if (event.isLeftButtonDown()) {
      if (event.getShiftState().shiftPressed) {
        m_Deselecting = true;
      } else {
        m_Selecting = true;
      }
    }
  
    return TOOLRESULT_DRAWALL;
  }

  int MSelectToolType::onMouseUp(const Aztec::MMouseEvent &event)
  {
    MToolType::onMouseUp(event);

    // if the left mouse button isn't down, we are no longer drawing a box.
    if (!event.isLeftButtonDown()) {
      drawingBox = false;
    } else {
      // otherwise we must still be doing things, so don't redraw the screen, 
      // and wait for the user to let the left mouse button go.
      return TOOLRESULT_DRAWNONE;
    }

    justCancelled = false;
  
    // Check to see if the mouse buttons are down.
    if (m_Selecting == false && m_Deselecting == false) {
      return TOOLRESULT_DRAWNONE;
    }
  
 
    AztecGLViewPtr viewGL = AZTEC_CAST(AztecGLView, AztecViewManager::getCurrentView());

    if (viewGL == NULL) {
      return TOOLRESULT_DRAWNONE;
    }
    
    AztecGLCanvasPtr GLWnd = viewGL->getCanvas();


    bool              TargetSel, ToggleSel;
  
    // check to see if all the mouse buttons are up
    if (event.isLeftButtonDown() || event.isMiddleButtonDown() || event.isRightButtonDown()) {
      return TOOLRESULT_DRAWNONE;
    }
  
    if (Keyboard::isToggleSelectionKey(event) && !m_BoxSelect) {
      ToggleSel = true;
    } else {
      ToggleSel = false;
    }
  
    // If we are not holding ctrl, clear the selection
    if (!Keyboard::isToggleSelectionKey(event) &&
        !Keyboard::isAddSelectionKey(event) &&
        !Keyboard::isSubtractSelectionKey(event))
    {
      Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.selectNone()");
    }
  
    MSelectMethod method;
    // select all the objects in the selection
    if (m_Deselecting) {
      TargetSel = false;
      if (m_BoxSelect) {
        method = smDeselectBox;
      } else {
        method = smDeselect;
      }
    } else {
      TargetSel = true;
      if (m_BoxSelect) {
        method = smSelectBox;
      } else {
        method = smSelect;
      }
    }
  
    Aztec::MRay selectionRay;

    if (m_BoxSelect) {
      GLWnd->getSelection(m_DownPos.x, m_DownPos.y, event.getX(), event.getY(), m_InputList);
      selectionRay = GLWnd->getRay(0.5*(m_DownPos.x + event.getX()), 0.5*(m_DownPos.y + event.getY()));
    } else {
      GLWnd->getSelection(event.getX(), event.getY(), m_InputList);
      selectionRay = GLWnd->getRay(event.getX(), event.getY());
    }
  
    // Go through and select the required objects
    MBaseObjectPtr Obj;
  
    /// Number of items taht have been effected by selectino.
    int selectionCount = 0;

    // first only check for component selections. If there are none, move onto object selections.
    if (Aztec::MUIManager::getComponentMode() != Aztec::MComponentisedObject::OBJECT_TYPE) {
      for (int i = 0; i < m_InputList.size(); ++i) {
        if (m_InputList[i].getNumComponents() != 0) {
          selectionCount += processSelectionItem(m_InputList[i], TargetSel, ToggleSel, selectionRay, event);
        }
      }
    }

    // now check to see if there are any object level selections to take care of.
    if (selectionCount == 0) {
      for (int i = 0; i < m_InputList.size(); ++i) {
        if (m_InputList[i].getNumComponents() == 0) {
          selectionCount += processSelectionItem(m_InputList[i], TargetSel, ToggleSel, selectionRay, event);
        }
      }
    }
    
    return TOOLRESULT_DRAWALL;
  }

  int MSelectToolType::onMouseMove(const Aztec::MMouseEvent &event)
  {
    MToolType::onMouseMove(event);

    MUIManager::unsetCoordinates();
  
    if (event.isLeftButtonDown()) {
      drawingBox = true;
      m_BoxSelect = true;
      return TOOLRESULT_DRAWCURRENT;
    }
  
    return TOOLRESULT_DRAWNONE;
  }

  int MSelectToolType::processSelectionItem(const MSelectionItem& item, bool targetSelection, bool toggleSelection, const Aztec::MRay &viewer, const Aztec::MMouseEvent &event) {
    if (item.getNumComponents() == 0) {
      // if we have missed all the components, deselect our currently selected objects if we arne't in component mode.
      if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
        if (toggleSelection) {
          MScene::getGlobalScene()->selectObject(item.getObject(), !item.getObject()->isFlagged(OBJECTFLAG_SELECTED));
        } else if (item.getObject()->isFlagged(OBJECTFLAG_SELECTED) != targetSelection) {
          MScene::getGlobalScene()->selectObject(item.getObject(), targetSelection);
        }
        return 1;
      } else {
        return 0;
      }
    } else {
      return processComponentSelectionItem(item, targetSelection, toggleSelection, viewer, event);
    }
  }

  typedef std::pair<Aztec::MMeshPtr, Aztec::MMeshPtr> MeshPair;
  typedef std::set<MeshPair> TextureMeshes;

  static int selectTextureComponent(const MSceneObjectPtr &SelObj, MComponentisedObject::ComponentType mode, int componentIndex, MComponentisedObject::MAction action, TextureMeshes &textureMeshes) { 
    int selectionCount = 0;

    Aztec::MMeshPtr mesh;
    if (SelObj->getShapeObject() != NULL) {
      mesh = AZTEC_CAST(MMesh, SelObj->getShapeObject()->convertToMesh());
    }
    if (mesh != NULL) {
      MMeshPtr textureMesh = mesh->getTextureMesh();
      if (textureMesh != NULL) {
        textureMesh->flagComponent(mode, 
          componentIndex, 
          MComponentisedObject::COMPONENT_SELECTED,
          action);
        
        textureMeshes.insert(TextureMeshes::value_type(mesh, textureMesh));

        ++selectionCount;
      }
    }
    
    return selectionCount;
  }

  int MSelectToolType::processComponentSelectionItem(const MSelectionItem& item, bool targetSelection, bool toggleSelection, const Aztec::MRay &viewer, const Aztec::MMouseEvent &event) {
    int selectionCount = 0;

    TextureMeshes textureMeshes;
    
    MSceneObjectPtr SelObj;
    MComponentisedObjectPtr compObj;
    MComponentisedObject::MAction action;
    
    bool backfaceCull = true;

    Aztec3DSceneCanvasPtr GLWnd = AZTEC_CAST(Aztec3DSceneCanvas, AztecGLView::getGLCanvasFor(event.getComponent()));

    if (GLWnd != NULL) {
      backfaceCull = GLWnd->getBackfaceCulling();
    }

    if (toggleSelection) {
      action = MComponentisedObject::atToggle;
    } else {
      action = targetSelection ? MComponentisedObject::atSet : MComponentisedObject::atUnset;
    }
    
    SelObj = AZTEC_CAST(MSceneObject, item.getObject());
    
    if (SelObj != NULL) {
      compObj = SelObj->getComponentObject();
      
      for (int n=0; n<item.getNumComponents(); n++) {
        int Num;
        Aztec::AztecFlags Mode;
        Aztec::MComponentisedObject::ComponentType CompMode = Aztec::MComponentisedObject::OBJECT_TYPE;
        
        item.getComponent(n, Mode, Num);
        
        if (Mode == SELECTITEM_VERTEX) {
          CompMode = MComponentisedObject::POINT_TYPE;
        } else if (Mode == SELECTITEM_FACE) {
          CompMode = MComponentisedObject::FACET_TYPE;

        } else if (Mode == SELECTITEM_EDGE) {
          CompMode = MComponentisedObject::EDGE_TYPE;

          if (backfaceCull) {

            Aztec::MMeshPtr mesh = AZTEC_CAST(MMesh, compObj);
            // check to see if edge is facing the wrong way
            if (mesh != NULL) {
              // get the vertices of the edge.
              int a, b;
              mesh->getVertsFromEdge(Num, &a, &b);

              // now get the tris from this edge
              std::vector<int> tris;
              mesh->getTrianglesWithEdge(a, b, tris);

              // now f all the triangles have a normal facing away from us, then the selection doesn't count.
              bool fail = true;
              for (int i = 0; i < tris.size(); ++i) {
                if (mesh->getTriangleNormal(tris[i]) * viewer.Dir <= 0) {
                  fail = false;
                  break;
                }
              }

              if (fail) {
                continue;
              }
            } else if (compObj != NULL) {
              if (compObj->getComponentNormal(Mode, Num) * viewer.Dir >= 0.0) {
                continue;
              }
            }
          }
        }
        
        if (compObj != NULL && CompMode != Aztec::MComponentisedObject::OBJECT_TYPE) {
          compObj->flagComponent(CompMode, 
            Num, 
            MComponentisedObject::COMPONENT_SELECTED,
            action);

          ++selectionCount;

        } else if (Mode == SELECTITEM_UV_POINT) {
          selectTextureComponent(SelObj, MComponentisedObject::POINT_TYPE, Num, action, textureMeshes); 
        } else if (Mode == SELECTITEM_UV_FACE) {
          selectTextureComponent(SelObj, MComponentisedObject::FACET_TYPE, Num, action, textureMeshes); 
        } else if (Mode == SELECTITEM_UV_EDGE) {
          selectTextureComponent(SelObj, MComponentisedObject::EDGE_TYPE, Num, action, textureMeshes); 
        } else {
          // unknown selection type.
          MSystemManager::getInstance()->logOutput("Warning: MSelectToolType - Uknown selection mode '%i'", Mode);
        }

      }
    }
    return selectionCount;
  }


}
