#include <Aztec3DPCH.h>
#include <tools/MGLRotateTool.h>

// Aztec2 includes
#include <views/AztecGLView.h>
#include <views/UVEditView.h>
#include <utils/AztecGLUtils.h>
#include <utils/SceneFunctions.h>
#include <config/UIConfig.h>

// AztecLib includes
#include <MUIManager.h>
#include <MSystemManager.h>

// standard includes
#include <math.h>

namespace AztecGUI {

  MGLRotateToolType::MGLRotateToolType() {
    m_RequiresSel = true;
    m_RotAng = 0;
  }

  std::string MGLRotateToolType::getName() {
    return "toolRotate";
  }

  int MGLRotateToolType::drawTool(bool Select, const Aztec::MComponentPtr &comp) {
    bool           DrawManips, PopMatrix;
  
    AztecGLViewPtr viewGL = AZTEC_CAST(AztecGLView, comp);

    if (viewGL == NULL) {
      return 0;
    }
  
    AztecGLCanvasPtr GLWnd = viewGL->getCanvas();
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();

    Aztec::MVector3 pivot = getSelectionCentre(comp);
  
    if (!m_Dragging) {
      getAxisMatrix(GLWnd);
    }
  
    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_DEPTH_TEST);
    glEnable(GL_BLEND);
  
  
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
  
    // Perform the viewport transformation
  
    GLWnd->doCameraTransform();
    glMatrixMode(GL_MODELVIEW);
  
    PopMatrix = true;
    glPushMatrix();
  
    float ScaleFact = GLWnd->getScalingFactor(pivot) * 0.75;
    glTranslatef(pivot.x, pivot.y, pivot.z);
    glMultMatrixf((float*)m_AxisTransform.m);
    glScalef(ScaleFact, ScaleFact, ScaleFact);
  
    if (RequiresSelection()) {
      DrawManips = anythingSelected(comp);
    } else {
      DrawManips = true;
    }
  
    if (DrawManips) {
      glDrawRotateIcon(UIConfig::get3DWidgetSize(),0.5f,3, Select);
    
      if (m_Dragging) {
        float Rot;
        Aztec::MVector3 CurPos, Res;
      
        glPopMatrix();
        PopMatrix = false;
      
        glTranslatef(m_Orig.x, m_Orig.y, m_Orig.z);
        glScalef(ScaleFact, ScaleFact, ScaleFact);
        glScalef(UIConfig::get3DWidgetSize(), UIConfig::get3DWidgetSize(), UIConfig::get3DWidgetSize());
      
        glColor4f(0,1,1,0.1f);
      
        // have to draw a filled arc going from vecb to veca. Which lasts m_RotAng degrees
      
        if (m_RotAng < 0)
          Rot = -m_RotAng;
        else 
          Rot = m_RotAng;
      
        // draw it using a fan
        glBegin(GL_TRIANGLE_FAN);
      
        glVertex3f(0,0,0);
        glColor4f(0,1,1,0.3f);
      
        CurPos = m_VecB;
        while (Rot > 0)
        {
          glVertex3fv((float*)&CurPos);
          Aztec::rotateWithVector(CurPos, m_VecC, (float)(-10.0*3.14159 / 180.0), Res);
          CurPos = Res;
          Rot -= 10;
        }
      
        glVertex3fv((float*)&m_VecA);
        glEnd();
      }
    }
  
    if (PopMatrix) {
      glPopMatrix();
    }
  
    glPopMatrix();
  
    glPopAttrib();
  
    return 1;
  }

  int MGLRotateToolType::onMouseDown(const Aztec::MMouseEvent &event)
  {
    Aztec::MVector3    Orig, Dir, ViewNorm;
    Aztec::MMatrix4    XFormMat;
  
    getPlaneParams(AZTEC_CAST(AztecGLCanvas, event.getComponent()), Orig, Dir, ViewNorm ,XFormMat);
    m_ViewNorm = ViewNorm;
    m_PivotPoint = getSelectionCentre(event.getComponent());
  
    // Override the direction stuff to suit the rotation gizmo's needs
    if (m_PickedManip == 1) {
      Dir.set(1,0,0);
    } else if (m_PickedManip == 2) {
      Dir.set(0,1,0);
    } else if (m_PickedManip == 3) {
      Dir.set(0,0,1);
    } else if (m_PickedManip == 4) {
      Dir = ViewNorm;
    }
  
    if (m_PickedManip != 4) {
      Dir = m_AxisTransform * Dir;
    }

    Dir.normalize();
  
    m_Constraint.setConstraint(Aztec::MPlane(Orig, Dir));

    // do this work after we have set up our constrain plane 
    // so it works properly
    MXYZToolType::onMouseDown(event);

    return TOOLRESULT_DRAWALL;
  }

  int MGLRotateToolType::onMouseUp(const Aztec::MMouseEvent &event)
  {
    MXYZToolType::onMouseUp(event);
  
    UpdateKeys("Rotate", TRANSFORM_CHAN_ROTATE);
  
    return TOOLRESULT_DRAWALL;
  }

  class Rotate3D {
  public:
    Rotate3D(const Aztec::MVector3 &pivotPoint, const Aztec::MVector3 &rotateAxis, float angleToRotate) {
      pivot = pivotPoint;
      angle = angleToRotate;
      axis = rotateAxis;
    }

    bool operator()(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
      Aztec::MSceneObjectPtr sceneObj = AZTEC_CAST(Aztec::MSceneObject, node->getObject());

      if (sceneObj == NULL) {
        return true;
      }
    
      Aztec::MComponentisedObjectPtr compObj = sceneObj->getComponentObject();
      if (compObj != NULL && compObj->isInComponentMode()) {
        Aztec::MEditableComponentisedObjectPtr editableCompObj = sceneObj->getEditableComponentObject();
        if (editableCompObj != NULL && editableCompObj->isInComponentMode()) {

          editableCompObj->restoreComponentPositions(
            editableCompObj->getComponentMode(),
            Aztec::MComponentisedObject::COMPONENT_SELECTED);

          Aztec::MMatrix4 objToWorld;
          Aztec::MMatrix4 worldToObj;
          scene->getWorldTransformMatrix(node, objToWorld);
          worldToObj = objToWorld;
          worldToObj.inverse();

          Aztec::MMatrix4 transform;
          transform.identity();

          Aztec::MVector3 localPivot = worldToObj * pivot;

          worldToObj.m[3][0] = 0;
          worldToObj.m[3][1] = 0;
          worldToObj.m[3][2] = 0;

          transform.m[3][0] -= localPivot.x;
          transform.m[3][1] -= localPivot.y;
          transform.m[3][2] -= localPivot.z;

          transform.rotateWithVector(worldToObj * axis, angle*M_PI/180.0);

          transform.m[3][0] += localPivot.x;
          transform.m[3][1] += localPivot.y;
          transform.m[3][2] += localPivot.z;

          editableCompObj->transformComponents(
            editableCompObj->getComponentMode(),
            transform,
            Aztec::MComponentisedObject::COMPONENT_SELECTED,
            false);
        }

      } else if (sceneObj->isFlagged(OBJECTFLAG_SELECTED)) {
        Aztec::MTransformObjectPtr XForm;
      
        XForm = sceneObj->getTransformObject();
      
        if (XForm != NULL) {
          Aztec::MVector3 Vec;
          Aztec::MMatrix4 mat;
          Aztec::MVector3 x,y,z,nx,ny,nz;

          XForm->fetchValues();
          Vec = XForm->getRotateVector(scene->getTime());
          Vec *= (float)(M_PI / 180.0);
          mat.convertFromEuler(Vec);

          x.set(mat.m[0][0], mat.m[0][1], mat.m[0][2]);
          y.set(mat.m[1][0], mat.m[1][1], mat.m[1][2]);
          z.set(mat.m[2][0], mat.m[2][1], mat.m[2][2]);

          axis.normalize();
          Aztec::rotateWithVector(x, axis, angle*M_PI/ 180.0, nx);
          Aztec::rotateWithVector(y, axis, angle*M_PI/ 180.0, ny);
          Aztec::rotateWithVector(z, axis, angle*M_PI/ 180.0, nz);

          mat.m[0][0] = nx.x; mat.m[0][1] = nx.y; mat.m[0][2] = nx.z;
          mat.m[1][0] = ny.x; mat.m[1][1] = ny.y; mat.m[1][2] = ny.z;
          mat.m[2][0] = nz.x; mat.m[2][1] = nz.y; mat.m[2][2] = nz.z;

          mat.convertToEuler(Vec);
          Vec *= (float)(180.0 / M_PI);
          XForm->setRotateVector(Vec);
          XForm->setParamByName("Rot", Vec.convertToString(), false);

        }
      }         

      return true;
    }

  private:
    Aztec::MVector3 axis, pivot;
    float angle;
  };


  class RotateUV {
  public:
    RotateUV(const Aztec::MVector3 &pivotPoint, const Aztec::MVector3 &rotateAxis, float angleToRotate) {
      pivot = pivotPoint;
      angle = angleToRotate;
      axis = rotateAxis;
    }

    bool operator()(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
      Aztec::MSceneObjectPtr sceneObj = AZTEC_CAST(Aztec::MSceneObject, node->getObject());
    
      if (sceneObj == NULL) {
        return true;

      }

      Aztec::MMeshPtr Mesh;
      if (sceneObj->getShapeObject() != NULL) {
        Mesh = AZTEC_CAST(Aztec::MMesh, sceneObj->getShapeObject()->convertToMesh());
      }
      
      if (Mesh != NULL) {
        
        Aztec::MMeshPtr TextureMesh = Mesh->getTextureMesh();
        Aztec::MEditableComponentisedObjectPtr editableCompObj = AZTEC_CAST(Aztec::MEditableComponentisedObject, TextureMesh);
        
        if (editableCompObj != NULL) {

          editableCompObj->restoreComponentPositions(
            Aztec::MUIManager::getComponentMode(),
            Aztec::MComponentisedObject::COMPONENT_SELECTED);


          Aztec::MMatrix4 transform;
          transform.identity();

          Aztec::MVector3 localPivot = pivot;

          transform.m[3][0] -= localPivot.x;
          transform.m[3][1] -= localPivot.y;
          transform.m[3][2] -= localPivot.z;

          transform.rotateWithVector(axis, angle*M_PI/180.0);

          transform.m[3][0] += localPivot.x;
          transform.m[3][1] += localPivot.y;
          transform.m[3][2] += localPivot.z;

          editableCompObj->transformComponents(
            Aztec::MUIManager::getComponentMode(),
            transform,
            Aztec::MComponentisedObject::COMPONENT_SELECTED,
            false);
        }

      }         

      return true;
    }

  private:
    Aztec::MVector3 axis, pivot;
    float angle;
  };


  int MGLRotateToolType::onMouseMove(const Aztec::MMouseEvent &event)
  {
    MXYZToolType::onMouseMove(event);
  
    if (!m_Dragging) {
      return TOOLRESULT_DRAWNONE;
    }

    // figure out the point, axis and angle of rotation.
    m_RotAng = calculateRotations(m_PivotPoint, m_Constraint.getConstraintNormal(), m_DownVec, m_CurVec);

    // Go through the currently selected objects, and move their components.
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();

    if (is3DView(event.getComponent())) {
      applyToObjects(scene, scene->getObjectList(), allObjectsCriteria, Rotate3D(m_PivotPoint, m_Constraint.getConstraintNormal(), m_RotAng));
    } else {
      applyToObjects(scene, scene->getObjectList(), allObjectsCriteria, RotateUV(m_PivotPoint, m_Constraint.getConstraintNormal(), m_RotAng));
    }

    UpdateKeys("Rotate", TRANSFORM_CHAN_ROTATE, false);

    return TOOLRESULT_DRAWALL;
  }


  float MGLRotateToolType::calculateRotations(const Aztec::MVector3 &pivot, const Aztec::MVector3 &constraintNormal, const Aztec::MVector3 &downVector, const Aztec::MVector3 &currentVector) 
  {
    Aztec::MVector3    A,B, C, Cross;
    float       dot, A1;
    
    A = currentVector - pivot;
    B = downVector - pivot;

    A = A - constraintNormal * (constraintNormal * A);
    B = B - constraintNormal * (constraintNormal * B);
    
    A.normalize();
    B.normalize();
    
    dot = A*B;
    if (dot < -1) {
      dot = -1;
    } else if (dot > 1) {
      dot = 1;
    }
    A1 = (float)(acos(dot)* 180 / 3.14159); 
    
    Cross = A/B;
    Cross.normalize();
    dot = Cross * constraintNormal;
    if (dot > 0) {
      A1 = -A1;
    }
    
    m_Orig = m_PivotPoint;
    m_VecA = A;
    m_VecB = B;
    m_VecC = Cross;  
    m_VecC.normalize();
    m_VecD = constraintNormal;

    return A1;
  }

}