#include <Aztec3DPCH.h>
#include <controls/FourWaySplitterMethod.h>

#include <assert.h>

namespace AztecGUI {

  FourWaySplitter::FourWaySplitter(int borderWidth, int edgeWidth, SplitStyle style, float splitX, float splitY) {
    splitStyle = style;
    this->borderWidth = borderWidth;
    
    if (edgeWidth == 0) {
      this->edgeWidth = borderWidth/2;
    } else {
      this->edgeWidth = edgeWidth;
    }
    
    this->splitX = splitX;
    this->splitY = splitY;

    maximisedPane = -1;
  }

  int FourWaySplitter::getPaneCount() {
    if (splitStyle == FourWay) {
      return 4;
    } else if (splitStyle == TwoVertical) {
      return 2;
    } else {
      return 3;
    }
  }

  FourWaySplitter::HitTestResultEnum FourWaySplitter::hitTest(MultiSplitter *splitterWnd, int x, int y) {
    // first check to see if we have a pane that is maximised. If we do, hit tests never do anything.
    if (maximisedPane >= 0 && maximisedPane < getPaneCount()) {
      return NOTHING;
    }
    
    Aztec::MSize2D clientSize = splitterWnd->getClientSize();
    bool onVertical = false;
    bool onHorizontal = false;

    if (x > clientSize.getWidth() * splitX - borderWidth / 2 && 
        x < clientSize.getWidth() * splitX + borderWidth / 2) {

      switch (splitStyle) {
      case TwoAtBottom:
        if (y > clientSize.getHeight() * splitY - borderWidth / 2) {
          onVertical = true;
        }
        break;
      case TwoAtTop:
        if (y < clientSize.getHeight() * splitY + borderWidth / 2) {
          onVertical = true;
        }
        break;
      case TwoAtLeft:
      case TwoAtRight:
      case FourWay:
      case TwoVertical:
        onVertical = true;
        break;
      default:
        assert(0);

      }
    }
    if (y > clientSize.getHeight() * splitY - borderWidth / 2 && 
        y < clientSize.getHeight() * splitY + borderWidth / 2) {

      switch (splitStyle) {
      case TwoAtRight:
        if (x > clientSize.getWidth() * splitX - borderWidth / 2) {
          onHorizontal = true;
        }
        break;
      case TwoAtLeft:
        if (x < clientSize.getWidth() * splitX + borderWidth / 2) {
          onHorizontal = true;
        }
        break;
      case TwoAtTop:
      case TwoAtBottom:
      case FourWay:
        onHorizontal = true;
        break;
      case TwoVertical:
        onHorizontal = false;
        break;
      default:
        assert(0);

      }
    }

    if (onHorizontal && onVertical) {
      return Both;
    } else if (onHorizontal) {
      return Horizontal;
    } else if (onVertical) {
      return Vertical;
    }

    return NOTHING;
  }

  Aztec::MRect2D FourWaySplitter::getRectFor(MultiSplitter *splitterWnd, int index) {
    Aztec::MSize2D size = splitterWnd->getClientSize();
    

    // first check to see if we have a pane that is maximised.
    if (maximisedPane >= 0 && maximisedPane < getPaneCount()) {
      if (index == maximisedPane) {
        return size;
      } else {
        return Aztec::MRect2D(0,0,0,0);
      }
    }

    int middleX = size.getWidth() * splitX;
    int middleY = size.getHeight() * splitY;

    Aztec::MPoint2D topLeft;
    Aztec::MPoint2D bottomRight;

    switch(index) {
    case 0:
      switch (splitStyle) {
      case FourWay:
      case TwoAtTop:
      case TwoAtLeft:
        topLeft.x = edgeWidth;
        topLeft.y = edgeWidth;
        bottomRight.x = middleX - borderWidth/2;
        bottomRight.y = middleY - borderWidth/2;
        break;
      case TwoAtBottom:
        topLeft.x = edgeWidth;
        topLeft.y = edgeWidth;
        bottomRight.x = size.getWidth() - edgeWidth;
        bottomRight.y = middleY - borderWidth / 2;
        break;
      case TwoAtRight:
      case TwoVertical:
        topLeft.x = edgeWidth;
        topLeft.y = edgeWidth;
        bottomRight.x = middleX - borderWidth / 2;
        bottomRight.y = size.getHeight() - edgeWidth;
        break;
      default:
        assert(0);
      }


      break;
    case 1:
      switch (splitStyle) {
      case FourWay:
      case TwoAtTop:
      case TwoAtRight:
        topLeft.x = middleX + borderWidth/2;
        topLeft.y = edgeWidth;
        bottomRight.x = size.getWidth() - edgeWidth;
        bottomRight.y = middleY - borderWidth / 2;
        break;
      case TwoAtBottom:
        topLeft.x = edgeWidth;
        topLeft.y = middleY + borderWidth / 2;
        bottomRight.x = middleX - borderWidth / 2;
        bottomRight.y = size.getHeight() - edgeWidth;
        break;
      case TwoAtLeft:
      case TwoVertical:
        topLeft.x = middleX + borderWidth/2;
        topLeft.y = edgeWidth;
        bottomRight.x = size.getWidth() - edgeWidth;
        bottomRight.y = size.getHeight() - edgeWidth;
        break;
      default:
        assert(0);
      }

      break;
    case 2:
      switch (splitStyle) {
      case FourWay:
      case TwoAtLeft:
        topLeft.x = edgeWidth;
        topLeft.y = middleY + borderWidth / 2;
        bottomRight.x = middleX - borderWidth / 2;
        bottomRight.y = size.getHeight() - edgeWidth;
        break;
      case TwoAtBottom:
      case TwoAtRight:
      topLeft.x = middleX + borderWidth/2;
      topLeft.y = middleY + borderWidth/2;
      bottomRight.x = size.getWidth() - edgeWidth;
      bottomRight.y = size.getHeight() - edgeWidth;
        break;
      case TwoAtTop:
        topLeft.x = edgeWidth;
        topLeft.y = middleY + borderWidth / 2;
        bottomRight.x = size.getWidth() - edgeWidth;
        bottomRight.y = size.getHeight() - edgeWidth;
        break;
      default:
        assert(0);
      }

      break;
    case 3:
      switch (splitStyle) {
      case FourWay:
        topLeft.x = middleX + borderWidth/2;
        topLeft.y = middleY + borderWidth/2;
        bottomRight.x = size.getWidth() - edgeWidth;
        bottomRight.y = size.getHeight() - edgeWidth;
        break;
      default:
        assert(0);
      }

      break;
    default:
      // we should never get here because there should only ever be 4 panes
      // being used, so put an assert in to inform us of this.
      assert(0);
      return Aztec::MRect2D(0,0,0,0);
      break;
    }

    return Aztec::MRect2D(topLeft, bottomRight);
  }

  Aztec::MMouseCursor FourWaySplitter::getCursorForHit(HitTestResultEnum hitResult) {
    switch (hitResult) {
    case Vertical:
      return Aztec::Mouse_SizeEastWest;
      break;
    case Horizontal:
      return Aztec::Mouse_SizeNorthSouth;
      break;
    case Both:
      return Aztec::Mouse_SizeAll;
      break;
    case NOTHING:
    default:
      return Aztec::Mouse_Normal; 
    }
  }

  void FourWaySplitter::beginDrag(MultiSplitter *splitterWnd, HitTestResultEnum hitResult, int x, int y) {
    startX = x;
    startY = y;
    startSplitX = splitX;
    startSplitY = splitY;
  }

  void FourWaySplitter::doDrag(MultiSplitter *splitterWnd, HitTestResultEnum hitResult, int x, int y) {
    int height;
    int width;
    Aztec::MSize2D clientSize = splitterWnd->getClientSize();

    width = clientSize.getWidth();
    height = clientSize.getHeight();

    int xPos;
    int yPos;

    xPos = width * startSplitX + x - startX;
    yPos = height * startSplitY + y - startY;
    
    // check to make sure xPos and yPos are within a valid range.
    if (xPos < edgeWidth + borderWidth / 2) {
      xPos = edgeWidth + borderWidth / 2;
    } else if (xPos > width - edgeWidth - borderWidth / 2) {
      xPos = width - edgeWidth - borderWidth / 2;
    }

    if (yPos < edgeWidth + borderWidth / 2) {
      yPos = edgeWidth + borderWidth / 2;
    } else if (yPos > height - edgeWidth - borderWidth / 2) {
      yPos = height - edgeWidth - borderWidth / 2;
    }

    // depending on our hit result, only update certain parts of it.
    switch (hitResult) {
    case Both:
      splitX = (float)xPos / (float)width;
      splitY = (float)yPos / (float)height;
      break;
    case Vertical:
      splitX = (float)xPos / (float)width;
      break;
    case Horizontal:
      splitY = (float)yPos / (float)height;
      break;
    }
  }

  void FourWaySplitter::endDrag(MultiSplitter *splitterWnd, HitTestResultEnum hitResult, int x, int y) {
  }

  void FourWaySplitter::maximisePane(int pane) {
    maximisedPane = pane;
  }


}
