/*
Copyright (C) 2010 Ronie Salgado

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#include "stdafx.h"
#include "QPropSheet.h"
#include "QMainFrame.h"
#include "QooleId.h"

BEGIN_EVENT_TABLE(QPropSheet, wxFrame)
	EVT_AUINOTEBOOK_PAGE_CHANGED(QID_PROPSHEET_NOTEBOOK, QPropSheet::OnNotebookChanged)
	EVT_AUINOTEBOOK_PAGE_CHANGING(QID_PROPSHEET_NOTEBOOK, QPropSheet::OnNotebookChanging)
	EVT_CLOSE(QPropSheet::OnCloseWindow)
END_EVENT_TABLE()

QPropSheet::QPropSheet(wxWindow *parent, wxRect *position)
{
	Create(parent, wxID_ANY, _("Properties"),
			position ? position->GetPosition() : wxDefaultPosition,
			position ? position->GetSize() : wxSize(400, 300),
			wxDEFAULT_FRAME_STYLE & ~(wxMINIMIZE_BOX | wxMAXIMIZE_BOX) | wxFRAME_FLOAT_ON_PARENT);

	// Create the sizer.
	wxBoxSizer *sizer = new wxBoxSizer(wxHORIZONTAL);

	// Create the notebook.
	notebook = new wxAuiNotebook(this, QID_PROPSHEET_NOTEBOOK, wxDefaultPosition,
			wxDefaultSize, wxAUI_NB_SCROLL_BUTTONS | wxAUI_NB_TAB_MOVE);
	sizer->Add(notebook, 1, wxEXPAND);

	// Create the other panels.
	propFile = new QPropFile(notebook);
	notebook->AddPage(propFile, _("File"));

	propObject = new QPropObject(notebook);
	notebook->AddPage(propObject, _("Object"));

	propEntity = new QPropEntity(notebook);
	notebook->AddPage(propEntity, _("Entity"));

	propTexture = new QPropTexture(notebook);
	notebook->AddPage(propTexture, _("Texture"));
}

QPropSheet::~QPropSheet()
{
}

void QPropSheet::SetPage(SheetID id)
{
	switch(id)
	{
	case SID_FILE:
		notebook->SetSelection(notebook->GetPageIndex(propFile));
		break;
	case SID_OBJECT:
		notebook->SetSelection(notebook->GetPageIndex(propObject));
		break;
	case SID_ENTITY:
		notebook->SetSelection(notebook->GetPageIndex(propEntity));
		break;
	case SID_TEXTURE:
		notebook->SetSelection(notebook->GetPageIndex(propTexture));
		break;
	default:
		// Shouldn't reach here.
		ASSERT(NULL);
		break;
	}
}

void QPropSheet::RegisterDocument(QooleDoc *doc)
{
	if(!doc)
	{
		int selId = notebook->GetSelection();
		wxWindow *page = NULL;
		if(selId >= 0)
			page = notebook->GetPage(selId);

		// Send a deactivate event to the current page.
		if(page)
		{
			wxActivateEvent ev(wxEVT_ACTIVATE, false);
			QPropSheetPage *propPage = static_cast<QPropSheetPage*> (page);
			propPage->OnActivate(ev);
		}
	}

	// Register the document and send an update message.
	propFile->RegisterDocument(doc);
	propObject->RegisterDocument(doc);
	propEntity->RegisterDocument(doc);
	propTexture->RegisterDocument(doc);

	if(doc)
	{
		propFile->OnUpdate(NULL, NULL);
		propObject->OnUpdate(NULL, NULL);
		propEntity->OnUpdate(NULL, NULL);
		propTexture->OnUpdate(NULL, NULL);
	}
}

void QPropSheet::OnCloseWindow(wxCloseEvent &event)
{
	RegisterDocument(NULL);
	GetMainFrame()->OnDestroyPropSheet(GetRect());
	Destroy();
}

void QPropSheet::OnNotebookChanged(wxAuiNotebookEvent &event)
{
	wxWindow *oldPage = NULL;
	int oldId = event.GetOldSelection();
	if(oldId >= 0)
		oldPage = notebook->GetPage(oldId);

	wxWindow *newPage = NULL;
	int newId = event.GetSelection();
	if(newId >= 0)
		newPage = notebook->GetPage(newId);

	if(oldPage)
	{
		wxActivateEvent deact(wxEVT_ACTIVATE, false);
		QPropSheetPage *propPage = static_cast<QPropSheetPage*> (oldPage);
		propPage->OnActivate(deact);
	}

	if(newPage)
	{
		wxActivateEvent act(wxEVT_ACTIVATE, true);
		QPropSheetPage *propPage = static_cast<QPropSheetPage*> (newPage);
		propPage->OnActivate(act);
	}

	event.StopPropagation();
}

void QPropSheet::OnNotebookChanging(wxAuiNotebookEvent &event)
{
	event.StopPropagation();
}
