/*
Copyright (C) 1996-1997 GX Media, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

#ifndef __LCOMMON_H
#define __LCOMMON_H

#ifndef ASSERT
#include <assert.h>
#define ASSERT assert
#endif

#ifndef TRACE
#include <stdio.h>
#define TRACE(msg, ...) LTrace(msg, __VA_ARGS__)
#endif

#include "List.h"

// ========== LCommon ==========

void LError(const char *fmt, ...);
void LFatal(const char *fmt = 0, ...);
void LWarning(const char *fmt, ...);
void LTrace(const char *fmt, ...);

// ========== LFile ==========

#include <stdio.h>
#include <malloc.h>
#include <string.h>

#ifdef _WIN32
#include <io.h>
#elif defined(linux) || defined(_POSIX_SOURCE)
#include <sys/types.h>
#include <dirent.h>
#include <glob.h>
#include <unistd.h>
#endif

#define LFILE_READ 1
#define LFILE_WRITE 2
#define LFILE_APPEND 4
#define LFILE_FROMPAK 8

class LPak;

class LFile {
public:
	LFile(void);
	~LFile(void);

	static bool Exist(const wxString &filename);
	static bool ExistDir(const wxString &dirname);

	bool Open(const wxString &filename, int mode = LFILE_READ);
	void Close(void);

	int Size(void);
	int Tell(void);
	int Seek(int offset);
	int SeekCur(int offset);
	int Read(void *buf, int size, int count = 1);
	int Write(const void *buf, int size, int count = 1);

	bool EndOfFile(void);
	char *GetLine(void);
	char *GetNextLine(void);
	void ResetLineNumber(void) { lineNum = 0; }
	int GetLineNumber(void);
	char *Search(const char *pattern);

	static wxString GetModifyTime(const wxString &fileName);

	static bool Extract(const wxString &filename, const wxString &dst);
	static void MakeDir(const wxString &dirname);

	static void UseDir(const wxString &dirname);
	static LPak *UsePak(const wxString &filename);

	static const wxString &GetInitDir(void) { return initDir; }

	static void Init(void);
	static void Exit(void);

private:
	FILE *file;
	int fileSize;

	LPak *pak;
	int fileOffset;
	int pakOffset;

	int lineNum;
	char data[1024];

	static const int MaxSearchDirs = 32;
	static const int MaxSearchPaks = 32;

	static wxString dirs[MaxSearchDirs];
	static int numdirs;
	static LPak *paks[MaxSearchPaks];
	static int numpaks;

	static wxString initDir;
};


// ========== LPak ==========

struct pak_header_t {
	char magic[4];
	int offset;
	int size;
};

struct pak_entry_t {
	char filename[56];
	int offset;
	int size;
};

class LPak : public LFile {
public:
	LPak(const wxString &filename);
	~LPak(void);

	wxString filename;
	pak_header_t header;
	pak_entry_t *entry;
	int entries;
};


// ========== LFindFiles ==========

class LFindFiles {
public:
	LFindFiles(const wxString &rootdir, const wxString &filemask = wxT("*.*"));
	~LFindFiles();
	wxString Next(void);

private:
	wxString name;
	wxString rootdir;
#if defined(_WIN32)
	struct _finddata_t fileinfo;
#else
	glob_t globResult;
	int nextPos;
#endif
	int rc;
};


// ========== LFindDirs ==========

class LFindDirs {
public:
	LFindDirs(const wxString &dirname);
	wxString Next(void);
private:
	wxString name;
#if defined(_WIN32)
	struct _finddata_t fileinfo;
#else
	DIR *dir;
	struct dirent *dirent;
#endif
	int rc;
};


// ========== LConfig ==========

#define LVAR_STR 1
#define LVAR_INT 2
#define LVAR_FLOAT 3
#define LVAR_WXSTR 4

struct LVar {
	wxString name;
	void *ptr;
	int type;
};

class LConfig {
public:
	LConfig(const wxString &name);
	~LConfig(void);

	void RegisterVar(const wxString &name, void *ptr, int type);
	void SaveVars(void);

	static void Init(int curVer, int reqVer);
	static void Exit(void);

private:
	wxString sectionName;
	std::vector<LVar*> lvars;

	static int currentVer, requiredVer, registryVer;
};


// ========== LPalette ==========

#ifndef PALETTE_T
#define PALETTE_T
typedef struct {
    unsigned char   blue;
    unsigned char   green;
    unsigned char   red;
    unsigned char   alpha;
	} palette_t;
#pragma pack()
#endif

class LPalette {
public:
	LPalette();
	~LPalette();

	void Load(const wxString &filename, float gamma = 1.0f);
	static void GammaCorrect(int &r, int &g, int &b, float gamma);

	palette_t pal[256];
};


// ========== LLibrary ==========

class LLibrary : public ElmNode {
public:
	LLibrary(const wxString &libName);
	~LLibrary();

	void *GetSymbol(const wxString &sym);

	static LLibrary *FindLibrary(const wxString &libName);

private:
	wxString name;
	void *module;

	typedef std::list<LLibrary*> Libraries;
	static Libraries libs;
};


// ========== Time Code ==========

void StartTime(void);
int GetTime(void);


// ========== Print Formatting ==========

extern wxString FormatFloat(float number);

// ========== Conversion stuff ==========
extern bool Str2Int(const wxString &str, unsigned int &val);


// ========== Old VOS Stuff ==========
typedef char substr_t[160];
int String_Split(substr_t substr[], char *org_string, int max);
void String_Crop(char *str);
void String_Chop(char *str);
wxString String_Chop(const wxString &s);
wxString String_Crop(const wxString &s);
void String_CutAt(char *str, const char *at);
int String_Count(const char *str, char ch);
int strstri(const char *str1, const char *str2);
#ifndef _WIN32
void strlwr(char *buf);
void strupr(char *buf);
#endif
typedef struct sort_t {
	sort_t *next;
	int key;
} sort_t;

sort_t *lsort(sort_t *p);

struct lpal_t {
	unsigned char r, g, b;
};

void SavePCX(const wxString &name, unsigned char *surface, int width, int height, lpal_t *pal);

int NearestGreaterPower2(int x);


#endif // __LCOMMON_H

