#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

#define MAX_LINE_LENGTH 1024

// Function to normalize a vector
void normalize(float *x, float *y, float *z) {
    float length = sqrt((*x) * (*x) + (*y) * (*y) + (*z) * (*z));
    if (length > 0.00001f) {
        *x /= length;
        *y /= length;
        *z /= length;
    }
}

// Function to compute the cross product of two vectors
void cross_product(float ax, float ay, float az, float bx, float by, float bz,
                   float *rx, float *ry, float *rz) {
    *rx = ay * bz - az * by;
    *ry = az * bx - ax * bz;
    *rz = ax * by - ay * bx;
}

// Function to format numbers cleanly (integers when possible)
void format_number(char *buffer, float value) {
    if (fabs(value - roundf(value)) < 0.0001f)
        sprintf(buffer, "%d", (int)roundf(value));
    else
        sprintf(buffer, "%.3f", value);
}

// Function to convert an old texture format line to the new format
void convert_texture_format(const char *input, char *output) {
    float x1, y1, z1, x2, y2, z2, x3, y3, z3;
    float Uoffset, Voffset, rotation, Uscale, Vscale;
    char texture[128], extra[128];

    // ( x1 y1 z1 ) ( x2 y2 z2 ) ( x3 y3 z3 ) path/texturename Uoffset Voffset rotation Uscale Vscale SurfaceFlag ContentsFlag Value
    int parsed = sscanf(input,
        "( %f %f %f ) ( %f %f %f ) ( %f %f %f ) %s %f %f %f %f %f %[^\n]",
        &x1, &y1, &z1, &x2, &y2, &z2, &x3, &y3, &z3,
        texture, &Uoffset, &Voffset, &rotation, &Uscale, &Vscale, extra);

    if (parsed < 15) {
        strcpy(output, input);
        return;
    }

    float ax = x2 - x1, ay = y2 - y1, az = z2 - z1;
    float bx = x3 - x1, by = y3 - y1, bz = z3 - z1;
    float Nx, Ny, Nz;
    cross_product(ax, ay, az, bx, by, bz, &Nx, &Ny, &Nz);
    normalize(&Nx, &Ny, &Nz);

    float Ux = ax, Uy = ay, Uz = az;
    normalize(&Ux, &Uy, &Uz);

    float Vx, Vy, Vz;
    cross_product(Nx, Ny, Nz, Ux, Uy, Uz, &Vx, &Vy, &Vz);
    normalize(&Vx, &Vy, &Vz);

    char fx1[16], fy1[16], fz1[16], fx2[16], fy2[16], fz2[16], fx3[16], fy3[16], fz3[16];
    char fUx[16], fUy[16], fUz[16], fUoffset[16];
    char fVx[16], fVy[16], fVz[16], fVoffset[16];
    char fRotation[16], fUscale[16], fVscale[16];

    format_number(fx1, x1); format_number(fy1, y1); format_number(fz1, z1);
    format_number(fx2, x2); format_number(fy2, y2); format_number(fz2, z2);
    format_number(fx3, x3); format_number(fy3, y3); format_number(fz3, z3);

    format_number(fUx, Ux); format_number(fUy, Uy); format_number(fUz, Uz); format_number(fUoffset, Uoffset);
    format_number(fVx, Vx); format_number(fVy, Vy); format_number(fVz, Vz); format_number(fVoffset, Voffset);

    format_number(fRotation, rotation); format_number(fUscale, Uscale); format_number(fVscale, Vscale);

    // ( x1 y1 z1 ) ( x2 y2 z2 ) ( x3 y3 z3 ) path/texturename [ Ux Uy Uz Uoffset ] [ Vx Vy Vz Voffset ] rotation Uscale Vscale SurfaceFlag ContentsFlag Value
    snprintf(output, MAX_LINE_LENGTH,
        "( %s %s %s ) ( %s %s %s ) ( %s %s %s ) %s "
        "[ %s %s %s %s ] [ %s %s %s %s ] %s %s %s %s",
        fx1, fy1, fz1, fx2, fy2, fz2, fx3, fy3, fz3,
        texture, fUx, fUy, fUz, fUoffset, fVx, fVy, fVz, fVoffset,
        fRotation, fUscale, fVscale, extra);
}

// Function to check if a texture line is in the new format
int is_new_format(const char *line) {
    return strstr(line, "[") != NULL; // Check for square brackets in texture lines
}

// Function to convert a new format line back to the old format
void convert_to_old_format(const char *input, char *output) {
    float x1, y1, z1, x2, y2, z2, x3, y3, z3;
    float Ux, Uy, Uz, Uoffset, Vx, Vy, Vz, Voffset, rotation, Uscale, Vscale;
    char texture[128], extra[128];

    // Parse new format
    int parsed = sscanf(input,
        "( %f %f %f ) ( %f %f %f ) ( %f %f %f ) %s "
        "[ %f %f %f %f ] [ %f %f %f %f ] %f %f %f %[^\n]",
        &x1, &y1, &z1, &x2, &y2, &z2, &x3, &y3, &z3,
        texture, &Ux, &Uy, &Uz, &Uoffset, &Vx, &Vy, &Vz, &Voffset,
        &rotation, &Uscale, &Vscale, extra);

    if (parsed < 17) {
        strcpy(output, input);
        return;
    }

    char fx1[16], fy1[16], fz1[16], fx2[16], fy2[16], fz2[16], fx3[16], fy3[16], fz3[16];
    char fUoffset[16], fVoffset[16], fRotation[16], fUscale[16], fVscale[16];

    format_number(fx1, x1); format_number(fy1, y1); format_number(fz1, z1);
    format_number(fx2, x2); format_number(fy2, y2); format_number(fz2, z2);
    format_number(fx3, x3); format_number(fy3, y3); format_number(fz3, z3);

    format_number(fUoffset, Uoffset);
    format_number(fVoffset, Voffset);
    format_number(fRotation, rotation);
    format_number(fUscale, Uscale);
    format_number(fVscale, Vscale);

    // Old format output
    snprintf(output, MAX_LINE_LENGTH,
        "( %s %s %s ) ( %s %s %s ) ( %s %s %s ) %s %s %s %s %s %s %s",
        fx1, fy1, fz1, fx2, fy2, fz2, fx3, fy3, fz3, texture,
        fUoffset, fVoffset, fRotation, fUscale, fVscale, extra);
}

// Returns 1 if a texture line in the file indicates the new (Valve) format,
// otherwise returns 0 (old format).
int detect_input_format(const char *filename) {
    FILE *file = fopen(filename, "r");
    if (!file) return 0;  // assume old if file couldnt be opened
    char line[MAX_LINE_LENGTH];
    int is_new = 0;
    while (fgets(line, MAX_LINE_LENGTH, file)) {
        // Remove leading whitespace:
        char *trimmed = line;
        while (*trimmed && isspace((unsigned char)*trimmed))
            trimmed++;
        if (*trimmed == '(') { // texture line candidates
            if (strstr(trimmed, "[") != NULL) {
                is_new = 1;
                break;
            } else {
                // Found a texture line in the old format.
                break;
            }
        }
    }
    fclose(file);
    return is_new;
}

// Modified function to process map files with both formats
void process_map_file(const char *input_filename, const char *output_filename) {
    FILE *input_file = fopen(input_filename, "r");
    FILE *output_file = fopen(output_filename, "w");

    if (!input_file || !output_file) {
        fprintf(stderr, "Error opening files!\n");
        return;
    }

    // Determine format: input_is_new == 1 means input is in Valve (new) format.
    int input_is_new = detect_input_format(input_filename);
    // If input is old, we are converting to Valve; if input is new, we are converting to old.
    int output_is_valve = input_is_new ? 0 : 1;
    int inside_worldspawn = 0;
    int mapversion_added = 0;
    char line[MAX_LINE_LENGTH];
    char converted_line[MAX_LINE_LENGTH];

    while (fgets(line, MAX_LINE_LENGTH, input_file)) {
        // When inside worldspawn, we want to handle the mapversion key differently...
        if (inside_worldspawn) {
            // For conversion to the old format (output_is_valve == 0), skip any mapversion lines.
            if (strstr(line, "\"mapversion\"") != NULL && !output_is_valve) {
                continue;
            }
        }

        // Detect start of worldspawn
        if (strstr(line, "\"classname\" \"worldspawn\"")) {
            inside_worldspawn = 1;
            fprintf(output_file, "%s", line);
            // For output Valve maps, add mapversion immediately after worldspawn if it isnt present
            if (output_is_valve && !mapversion_added) {
                fprintf(output_file, "\"mapversion\" \"220\"\n");
                mapversion_added = 1;
            }
            continue;
        }

        // Optionally, you can also check special handling of the "// Format:" comment line
        if (strncmp(line, "// Format:", 10) == 0) {
            if (output_is_valve) {
                fprintf(output_file, "// Format: Quake2 (Valve)\n");
            } else {
                fprintf(output_file, "// Format: Quake2\n");
            }
            continue;
        }

        // If line looks like a texture line (might be preceded by whitespace), process it:
        char *trimmed = line;
        while (*trimmed && isspace((unsigned char)*trimmed))
            trimmed++;
        if (*trimmed == '(') {
            if (is_new_format(trimmed)) {
                // Input texture line is in new format, so convert to old.
                convert_to_old_format(trimmed, converted_line);
            } else {
                // Input texture line is in old format, so convert to new (Valve) format.
                convert_texture_format(trimmed, converted_line);
            }
            fprintf(output_file, "%s\n", converted_line);
        } else {
            fprintf(output_file, "%s", line);
        }
    }

    fclose(input_file);
    fclose(output_file);
}

int main(int argc, char *argv[]) {
    if (argc < 3) {
        printf("FREDZ convert Kingpin map to Kingpin valve map file.\nOr backwards\n");
        printf("Usage: %s <input.map> <output.map>\n", argv[0]);
        return 1;
    }

    process_map_file(argv[1], argv[2]);

    printf("Conversion complete! Output saved to %s\n", argv[2]);
    return 0;
}
