﻿#region ================= Namespaces

using System;
using System.Collections.Generic;
using System.IO;
using mxd.SQL2.Data;
using mxd.SQL2.DataReaders;
using mxd.SQL2.Games.Quake2;
using mxd.SQL2.Items;
using mxd.SQL2.Tools;

#endregion

namespace mxd.SQL2.Games.Kingpin
{
	public class KingpinHandler : GameHandler
	{
		#region ================= Variables

		private Dictionary<string, string> knowngamefolders; // Folder names and titles for official expansions
		private List<VideoModeInfo> rmodes;
		private HashSet<string> nonengines; // Kingpin comes with a lot of unrelated exes..

		#endregion

		#region ================= Properties

		public override string GameTitle => "Kingpin";

		#endregion

		#region ================= Setup

		// Valid Kingpin path if "main\pak0.pak" exist, I guess...
		protected override bool CanHandle(string gamepath)
		{
            return File.Exists(Path.Combine(gamepath, "main\\pak0.pak"));
        }

		// Data initialization order matters (horrible, I know...)!
		protected override void Setup(string gamepath)
		{
			// Default mod path
			defaultmodpath = Path.Combine(gamepath, "main").ToLowerInvariant();

			// Nothing to ignore
			ignoredmapprefix = string.Empty;

			// Demo extensions
			supporteddemoextensions.Add(".dm2");

			// Setup map delegates
			getfoldermaps = DirectoryReader.GetMaps;
			getpakmaps = PAKReader.GetMaps;
            getpk3maps = null; // No PK3 support in Kingpin

            foldercontainsmaps = DirectoryReader.ContainsMaps;
			pakscontainmaps = PAKReader.ContainsMaps;
			pk3scontainmaps = null; // No PK3 support in Kingpin


			getmapinfo = Quake2BSPReader.GetMapInfo;

			// Setup demo delegates
			getfolderdemos = DirectoryReader.GetDemos;
			getpakdemos = PAKReader.GetDemos;
			getpk3demos = null; // No PK3 support in Kingpin

			getdemoinfo = Quake2DemoReader.GetDemoInfo;

			// Setup fullscreen args...
			fullscreenarg[true]  = "1";
			fullscreenarg[false] = "0";

			// Setup launch params
			launchparams[ItemType.ENGINE] = string.Empty;
			launchparams[ItemType.RESOLUTION] = "+vid_fullscreen {1} +gl_mode {0}";//FREDZ gl_mode does not work with set
			launchparams[ItemType.GAME] = string.Empty;
			launchparams[ItemType.MOD] = "+set game {0}";
			launchparams[ItemType.MAP] = "+map {0}";
			launchparams[ItemType.SKILL] = "+set skill {0}";
			launchparams[ItemType.CLASS] = string.Empty;
			launchparams[ItemType.DEMO] = "+map {0}";

			// Setup skills (requires launchparams)
			skills.AddRange(new[]
			{
				new SkillItem("Novice", "0"),
				new SkillItem("Easy", "1"),
                new SkillItem("Medium", "2", true),
                new SkillItem("Hard", "3"),
                new SkillItem("Real", "4", false, true)
            });

            
			// Setup known folders
			knowngamefolders = new Dictionary<string, string>(StringComparer.OrdinalIgnoreCase)
			{
//				{ "MAIN", "Kingpin" },
				{ "COOP", "Cooperative" },
			};

			// Setup fixed gl_mode... Taken from qcommon\vid_modes.h (Kingpin)
			int c = 0;
//            int c = 3; // The first two r_modes are ignored by default Kingpin
            rmodes = new List<VideoModeInfo>
			{
                //FREDZ MH patch resolution settings
                new VideoModeInfo(640, 480, c++), //Mode0
				new VideoModeInfo(800, 600, c++), //Mode1
				new VideoModeInfo(1024, 768, c++),//Mode2
				new VideoModeInfo(1280, 720, c++),//Mode3
				new VideoModeInfo(1280, 1024, c++),//Mode4
                new VideoModeInfo(1360, 768, c++),//Mode5
                new VideoModeInfo(1366, 768, c++),//Mode6
				new VideoModeInfo(1600, 900, c++),//Mode7
				new VideoModeInfo(1680, 1050, c++),//Mode8
				new VideoModeInfo(1920, 1080, c++),//Mode9

//Default Kingpin starts at mode3
/*				new VideoModeInfo(640, 480, c++), //Mode3
				new VideoModeInfo(800, 600, c++), //Mode4
				new VideoModeInfo(960, 720, c++), //Mode5
				new VideoModeInfo(1024, 768, c++),//Mode6
				new VideoModeInfo(1152, 864, c++),//Mode7
				new VideoModeInfo(1280, 960, c++),//Mode8
				new VideoModeInfo(1600, 1200, c++),//Mode9*/
			};
			
			// Setup non-engines...
			nonengines = new HashSet<string>(StringComparer.OrdinalIgnoreCase)
			{
				"kpintro",
				"patchcfg",
				"Kingpin_Multi_Patch",
				"kprespatch",
				"unins000",
			};
			
			// Pass on to base...
			base.Setup(gamepath);
		}

		#endregion

		#region ================= Methods

		public override List<ResolutionItem> GetVideoModes()
		{
			return DisplayTools.GetFixedVideoModes(rmodes);
		}

		public override List<DemoItem> GetDemos(string modpath)
		{
			return GetDemos(modpath, "DEMOS");
		}

		public override List<ModItem> GetMods()
		{
			var result = new List<ModItem>();

			foreach(string folder in Directory.GetDirectories(gamepath))
			{
				// Skip folder if it has no maps or a variant of "gamex86.dll"
				//if(!foldercontainsmaps(folder) && !pakscontainmaps(folder) && !pk3scontainmaps(folder) && !ContainsGameDll(folder))
//				if(!foldercontainsmaps(folder) && !pakscontainmaps(folder) && !ContainsGameDll(folder))//FREDZ
				if(!foldercontainsmaps(folder) && !pakscontainmaps(folder) && !File.Exists(Path.Combine(folder, "gamex86.dll")))//FREDZ
					continue;

				string name = folder.Substring(gamepath.Length + 1);
				bool isbuiltin = (string.Compare(folder, defaultmodpath, StringComparison.OrdinalIgnoreCase) == 0);
				string title = (knowngamefolders.ContainsKey(name) ? knowngamefolders[name] : name);

				result.Add(new ModItem(title, name, folder, isbuiltin));
			}

			// Push known mods above regular ones
			result.Sort((i1, i2) =>
			{
				bool firstknown = (i1.Title != i1.Value);
				bool secondknown = (i2.Title != i2.Value);

				if(firstknown == secondknown) return string.Compare(i1.Title, i2.Title, StringComparison.Ordinal);
				return (firstknown ? -1 : 1);
			});

			return result;
		}

		// Check for a variant of "gamex86.dll". Can be named differently depending on source port...
		// Vanilla, UQE Quake2: gamex86.dll
		// KMQuake2: kmq2gamex86.dll
		// Yamagi Quake2: game.dll
		// Quake 2 Evolved: q2e_gamex86.dll
		// Quake 2 XP: gamex86xp.dll
/*		private bool ContainsGameDll(string folder)
		{
			//TODO? Ideally, we should check for game.dll specific to selected game engine, but that would require too much work, including updating mod list when game engine selection changes. 
			//TODO? So, for now just look for anything resembling game.dll...
			var dlls = Directory.GetFiles(folder, "*.dll");
			foreach (var dll in dlls)
				if (Path.GetFileName(dll).Contains("game"))
					return true;

			return false;
		}*/
		
		protected override bool IsEngine(string filename)
		{
			return !nonengines.Contains(Path.GetFileNameWithoutExtension(filename)) && base.IsEngine(filename);
		}
		
		#endregion
	}
}
