//CTF_Spawn.c - routines for dealing with player spawning

#include "g_local.h"

/*
Buglist:
scores do not reset to zero at map change - FIXED
all deaths by HMG showing as by harpoon - forgot to put in break - FIXED
activate new cloud code - FIXED - BROKEN

Also check teamplay overlay for team indicator at T/L corner of screen
*/

#define VALID_SPAWN_DISTANCE 25.0f	//KP code uses 32

/*
for first time spawning, always spawn player at base - if there are no free spawn points
at base, spawn at neutral spawn point nearest to base
*/
/*
For performance, read the info_player_deathmatch entities at map load time and put them into 1 of 3 arrays
0 = neutral
1 = team 1
2 = team 2

Then a FAST selection routine can pick from the relevant team array or the neutral array as required. In worse case, can use opposite team array.
Determine why firstspawn flag is used to determine a spawn point.
Players should probably always spawn at their base unless all the spawns are occupied.

Will also need corresponding test flag arrays to determine when all slots have been tested.
If a slot is occupied, instead of getting another random slot, just try the next slot up or down.
Repeat until the original slot index is reached.
This method actually removes the requirement for a test flag array.
So:
	Get a random slot in the team array. Store the index (if the array has any slots)
	Test if the slot is occupied
	if free, return the entity pointer at the slot.
	If occupied, randomly select of scanning up or down
		Move to next adjacent slot (up or down, depending on previous selection)
		Test if the slot is occupied
		If free, return the entity pointer. If occupied, go to next slot (loop when reaching start or end of array)
		If reached the start slot again, all slots are full.
			Test using the neutral array.
			If all slots occupied (or none exist)
				Test using the opposite team array
				If all slots occupied (or none exist)
					Use standard KP spawn selection routine (telefrag)

Can probably read the entities as the spawn routine is called, or can scan after the entire entity list has spawned.
Either way will need to reset the array parameters at a map load event
SpawnEntities() in g_spawn.c is the routine to reset the arrays
*/

Spawn_Array_Def Spawn_Array[3];		//0 = neutral, 1 = team 1, 2 = team 2

void CTF_Spawn_Initialise(void)
{//Spawn arrays will be reset here.
	Spawn_Array_Def* Array_Pointer;

	Array_Pointer = Spawn_Array;
	Array_Pointer->Slot_Pointer = Array_Pointer->Slot;
	Array_Pointer->Num_Occupied = 0;

	Array_Pointer = &Spawn_Array[1];
	Array_Pointer->Slot_Pointer = Array_Pointer->Slot;
	Array_Pointer->Num_Occupied = 0;

	Array_Pointer = &Spawn_Array[2];
	Array_Pointer->Slot_Pointer = Array_Pointer->Slot;
	Array_Pointer->Num_Occupied = 0;

//Subsequent calls to the info_player_deathmatch spawn routine will populate the relevant array
//	gi.dprintf("CTF_Spawn_Initialise() called.\n");
}

void CTF_Spawn_Store_Entity(edict_t* ent)
{//info_player_deathmatch spawn routine calls into here to store the data
	Spawn_Array_Def* Array_Pointer;
	Spawn_Slot_Def* Slot;

	switch (ent->style)
	{
		case 1:
		{
			Array_Pointer = &Spawn_Array[1];
			break;
		}
		case 2:
		{
			Array_Pointer = &Spawn_Array[2];
			break;
		}
		default:
		{
			Array_Pointer = Spawn_Array;
			break;
		}
	}
//Store the entity in the array if there is space
	if (Array_Pointer->Num_Occupied < MAX_SPAWN_SLOTS)
	{
//Store the spawn point
		Slot = Array_Pointer->Slot_Pointer;
		Slot->Point = ent;
		Slot->Distance[0] = -1.0f;
		Slot->Distance[1] = -1.0f;
//Update storage parameters
		Array_Pointer->Slot_Pointer++;
		Array_Pointer->Num_Occupied++;
	}
//	gi.dprintf("CTF_Spawn_Store_Entity() called. Style: %d\n", ent->style);
}

void CTF_Spawn_Sort(Spawn_Array_Def* List)
{//Bubble sort
	Spawn_Slot_Def Holder_Slot;
	Spawn_Slot_Def *Current_Slot, *Next_Slot;
	int Num_Scan, Counter;
	qboolean Swapped;

	Num_Scan = List->Num_Occupied - 1;	//1 less as N and N+1 will be compared
	do
	{
		Swapped = false;
		Current_Slot = List->Slot;
		Next_Slot = Current_Slot++;
//Loop through the list and compare
		for (Counter = 0; Counter < Num_Scan; Counter++, Current_Slot++, Next_Slot++)
		{
//Check if current distance < next distance. Sorting near->far on team 1 flag distance
			if (Current_Slot->Distance[0] >= Next_Slot->Distance[0])
				continue;
//Swap
			Swapped = true;
			Holder_Slot = *Next_Slot;
			*Next_Slot = *Current_Slot;
			*Current_Slot = Holder_Slot;
		}
	} while (Swapped == true);
}

void CTF_Spawn_Point_Check_Teams(void)
{
/*
Check if there are slots allocated to both teams. If not, divide up the neutral spawns between the two teams. Modify the style flag and store in the team spawn array.
Neutral spawn list is bubble sorted by distance from team 1 flag.
First half of the list is allocated to team 1, half to team 2
Note that this could still cause some spawns to be in the wrong team base as it does not look for the spawn where the distance to flag 2 < flag 1 as this could
degrade the algorithm to cause all spawns to be allocated to the same team, which this method tries to avoid.
*/
	Spawn_Array_Def *Neutral_List, *Team_1_List, *Team_2_List;
	Spawn_Slot_Def* Slot;
	int Counter, Team, Half_Count;

	Team_1_List = &Spawn_Array[1];
	if (Team_1_List->Num_Occupied > 0)
		return;
	Team_2_List = &Spawn_Array[2];
	if (Team_2_List->Num_Occupied > 0)
		return;
//No team spawns allocated. Assuming all neutral spawns
	Neutral_List = Spawn_Array;
//Sort the spawns based on distance
	CTF_Spawn_Sort(Neutral_List);
//Split up the sorted list
	Slot = Neutral_List->Slot;
	Half_Count = Neutral_List->Num_Occupied / 2;
//If an odd number of spawns then randomly give the spare slot to team 1 or 2
	if (Neutral_List->Num_Occupied & 1)
	{
//		gi.dprintf("Odd number of neutrals. Randomly allocating extra spawn\n");
		if ((rand() % 100) < 50)
			Half_Count++;//Give to team 1
	}
//Split up the list into two lists
//Allocate team 1
	for (Counter = 0; Counter < Half_Count; Counter++, Slot++)
	{
//Set the style to correspond to the team it will be allocated to
		Slot->Point->style = 1;
//Store in the team array
		*Team_1_List->Slot_Pointer = *Slot;
		Team_1_List->Slot_Pointer++;
		Team_1_List->Num_Occupied++;
	}
//Allocate team 2
	for (Counter = Half_Count; Counter < Neutral_List->Num_Occupied; Counter++, Slot++)
	{
//Set the style to correspond to the team it will be allocated to
		Slot->Point->style = 2;
//Store in the team array
		*Team_2_List->Slot_Pointer = *Slot;
		Team_2_List->Slot_Pointer++;
		Team_2_List->Num_Occupied++;
	}
//Log
	gi.dprintf("No team spawns. Reallocated %d neutral spawns to teams %d : %d\n", Spawn_Array[0].Num_Occupied, Spawn_Array[1].Num_Occupied, Spawn_Array[2].Num_Occupied);
//Clear the neutral spawn count
	Neutral_List->Num_Occupied = 0;
}

void CTF_Spawn_Range_Slot(Spawn_Slot_Def* Slot, edict_t* Flag_1, edict_t* Flag_2)
{
	vec3_t Vector;

	if (Flag_1 != NULL)
		Slot->Distance[0] = VectorDistance(Slot->Point->s.origin, Flag_1->s.origin);

	if (Flag_2 != NULL)
		Slot->Distance[1] = VectorDistance(Slot->Point->s.origin, Flag_2->s.origin);
}

qboolean CTF_Spawn_Points_Range(void)
{//Calculate the distance of each spawn point to the flags
	edict_t* Flag_1 = NULL;	//Pointer to team 1 flag
	edict_t* Flag_2 = NULL;	//Pointer to team 2 flag
	Spawn_Array_Def* Array_Pointer;
	Spawn_Slot_Def* Slot;
	int Slot_Counter;

//	gi.dprintf("Spawns: %d : %d : %d\n", Spawn_Array[0].Num_Occupied, Spawn_Array[1].Num_Occupied, Spawn_Array[2].Num_Occupied);

//Find the flags
	Flag_1 = CTF_Flag_Find(1);
	Flag_2 = CTF_Flag_Find(2);
//Check both flags exist (this should probably be checked prior to this point)
	if ((Flag_1 == NULL) || (Flag_2 == NULL))
	{
//Error
		gi.dprintf("Error. Map does not have a flag for each team\n");
//		return false;
	}
//Calculate neutrals
	Array_Pointer = Spawn_Array;
	Slot = Array_Pointer->Slot;
	for (Slot_Counter = 0; Slot_Counter < Array_Pointer->Num_Occupied; Slot_Counter++, Slot++)
		CTF_Spawn_Range_Slot(Slot, Flag_1, Flag_2);
//Calculate team 1
	Array_Pointer++;
	Slot = Array_Pointer->Slot;
	for (Slot_Counter = 0; Slot_Counter < Array_Pointer->Num_Occupied; Slot_Counter++, Slot++)
		CTF_Spawn_Range_Slot(Slot, Flag_1, Flag_2);
//Calculate team 2
	Array_Pointer++;
	Slot = Array_Pointer->Slot;
	for (Slot_Counter = 0; Slot_Counter < Array_Pointer->Num_Occupied; Slot_Counter++, Slot++)
		CTF_Spawn_Range_Slot(Slot, Flag_1, Flag_2);
//Check if spawns need to be reallocated if no team spawns defined
	CTF_Spawn_Point_Check_Teams();
//Done
	return true;
}

edict_t* CTF_Spawn_Slot_Random_Select(Spawn_Array_Def* Array_Pointer)
{
	qboolean Scan_Up;	//Direction of scan if initial spawn is occupied
	int Initial_Index, Scan_Index;
	Spawn_Slot_Def* Slot;
	float Distance;

//Check if any slots
	if (Array_Pointer->Num_Occupied == 0)
		return NULL;
//Get a random slot
	Initial_Index = rand() % Array_Pointer->Num_Occupied;
	Slot = &Array_Pointer->Slot[Initial_Index];
	Distance = PlayersRangeFromSpot(Slot->Point);
//Check if it is free
	if (Distance >= VALID_SPAWN_DISTANCE)
		return Slot->Point;
//Occupied. Scan the other spawn points until a free slot is found
	Scan_Index = Initial_Index;
	Scan_Up = rand() % 1;
	while (true)
	{
		if (Scan_Up == true)
		{
			Scan_Index++;
			if (Scan_Index == Array_Pointer->Num_Occupied)
				Scan_Index = 0;
		}
		else
		{
			Scan_Index--;
			if (Scan_Index < 0)
				Scan_Index = Array_Pointer->Num_Occupied - 1;
		}
//Check if all points have been checked
		if (Scan_Index == Initial_Index)
			break;
//Get the slot
		Slot = &Array_Pointer->Slot[Scan_Index];
		Distance = PlayersRangeFromSpot(Slot->Point);
//Check if it is free
		if (Distance >= VALID_SPAWN_DISTANCE)
			return Slot->Point;
//Occupied. Go to next slot
	}
//No free slots found
	return NULL;
}

edict_t* CTF_Player_Spawn_Point_Select(edict_t *ent)
{
//*** CALL THIS ROUTINE FROM SelectSpawnPoint in P_CLIENT***
	edict_t *Spot = NULL;
	Spawn_Array_Def* Array_Pointer;
	int Index;
	int Selected_Ratio;
	qboolean Base_First;
	int Preference_Ratio;
	int Team;

	Team = ent->client->pers.team;
//If spec, pick a random team
//	if (Team == 0)
//		Team = 1 + (rand() & 1);

/*
If FirstSpawn is true, prefer spawn in the team base
If FirstSpawn is false, randomly select based on a preference ratio which is a % value between 0 and 100
<= Preference_Ratio prefer team spawn
*/

	Preference_Ratio = (int)spawn_ratio->value;
//Sanity
	if (Preference_Ratio < 0)
		Preference_Ratio = 0;
	else
	if (Preference_Ratio > 100)
		Preference_Ratio = 100;

	switch (Preference_Ratio)
	{
		case 0:
		{
			Base_First = false;
			break;
		}
		case 100:
		{
			Base_First = true;
			break;
		}
		default:
		{
//Chance of either base->neutral or neutral->base (ie, try base first, neutral second, or vice-versa)
			Selected_Ratio = rand() % 101;	//101 as we want values between 0 and 100
			if (Selected_Ratio <= Preference_Ratio)
				Base_First = true;
			else
				Base_First = false;
			break;
		}
	}
//Check if in range for base->neutral preference. 0 is a special case in that it is 0% chance of base->neutral
	if (Base_First == true)
	{
//Check team spawns
		Array_Pointer = &Spawn_Array[Team];
		Spot = CTF_Spawn_Slot_Random_Select(Array_Pointer);
		if (Spot != NULL)
			return Spot;
//No free team slots. Try neutrals
		Array_Pointer = &Spawn_Array[0];
		Spot = CTF_Spawn_Slot_Random_Select(Array_Pointer);
		if (Spot != NULL)
			return Spot;
//No free neutral slots. Telefrag in team base. If the telefrag avoid code is active, then the telefrag will be avoided
		Array_Pointer = &Spawn_Array[Team];
		if (Array_Pointer->Num_Occupied > 0)
		{
			Index = rand() % Array_Pointer->Num_Occupied;
			Spot = Array_Pointer->Slot[Index].Point;
			return Spot;
		}
//No free team spawns. Try opposition spawns
		if (Team == 1)
			Array_Pointer = &Spawn_Array[2];
		else
			Array_Pointer = &Spawn_Array[1];
		if (Array_Pointer->Num_Occupied > 0)
		{
			Index = rand() % Array_Pointer->Num_Occupied;
			Spot = Array_Pointer->Slot[Index].Point;
			return Spot;
		}
	}
	else
	{
//Check neutral spawns
		Array_Pointer = &Spawn_Array[0];
		Spot = CTF_Spawn_Slot_Random_Select(Array_Pointer);
		if (Spot != NULL)
			return Spot;
//No free neutral slots. Try team
		Array_Pointer = &Spawn_Array[Team];
		Spot = CTF_Spawn_Slot_Random_Select(Array_Pointer);
		if (Spot != NULL)
			return Spot;
//No free neutral slots. Telefrag in team base. If the telefrag avoid code is active, then the telefrag will be avoided
		Array_Pointer = &Spawn_Array[Team];
		if (Array_Pointer->Num_Occupied > 0)
		{
			Index = rand() % Array_Pointer->Num_Occupied;
			Spot = Array_Pointer->Slot[Index].Point;
			return Spot;
		}
//No free team spawns. Try opposition spawns
		if (Team == 1)
			Array_Pointer = &Spawn_Array[2];
		else
			Array_Pointer = &Spawn_Array[1];
		if (Array_Pointer->Num_Occupied > 0)
		{
			Index = rand() % Array_Pointer->Num_Occupied;
			Spot = Array_Pointer->Slot[Index].Point;
			return Spot;
		}
	}
//No team slots defined. Use standard spawn routine to get info_player_start
	return NULL;
}
