#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <dirent.h>
#include <sys/stat.h>
#include <time.h>
#include <math.h>
#include <ctype.h>

#define MAX_NAME 32

// Quake II valid flag and content masks
#define Q2_FLAG_MASK     0x000003FF   // Light, Sky, Trans33, Trans66, Warp, Flowing, NoDraw, Hint, Skip
#define Q2_CONTENT_MASK  0x0000003F   // Solid, Window, Aux, Lava, Slime, Water, Mist, AreaPortal

// --- WAL header ---
typedef struct {
    char name[MAX_NAME];
    uint32_t width;
    uint32_t height;
    uint32_t offsets[4];
    char nextName[MAX_NAME];
    uint32_t flags;
    uint32_t contents;
    uint32_t value;
} miptex_t;

// --- TGA header (24-bit uncompressed only) ---
#pragma pack(push,1)
typedef struct {
    uint8_t idLength;
    uint8_t colorMapType;
    uint8_t imageType;
    uint16_t cmFirstEntry;
    uint16_t cmLength;
    uint8_t cmEntrySize;
    uint16_t xOrigin;
    uint16_t yOrigin;
    uint16_t width;
    uint16_t height;
    uint8_t pixelDepth;
    uint8_t imageDesc;
} TGAHeader;
#pragma pack(pop)

// --- PAK header ---
typedef struct {
    char name[56];
    int32_t filepos;
    int32_t filelen;
} pakentry_t;

// Global flags
char rootpath[1024] = ".";
int debug = 0;                 // FREDZ reduce log spam

int iniFound = 0;              // set when any textureinfo.ini is successfully opened
int skipIni = 0;               // default: read INI
int onlyIni = 0;               // default: not restricted to INI entries

int skipAlpha = 0;             // default: use alpha channel when present
int onlyAlpha = 0;             // default: not alpha-only

int q2FlagsOnly = 0;           // default: allow all flags
int requirePowerOf2 = 0;       // default: no PoT requirement

int alphaThreshold = 255;      // Quake II: alpha < 255 -> transparent
int useAlphaThreshold = 0;     // set only if -alphathresh is passed

int deletedMode = 0;           // 0 = convert, 1 = delete WALs
int scanMode = 0;
int fixMode = 0;

int extract_colormap_from_pak(const char *pakfile, uint8_t palette[256][3]) {
    FILE *f = fopen(pakfile, "rb");
    if (!f) return -1;

    // Read header
    char magic[4];
    int32_t diroffset, dirlen;
    fread(magic, 1, 4, f);
    fread(&diroffset, 4, 1, f);
    fread(&dirlen, 4, 1, f);

    if (strncmp(magic, "PACK", 4) != 0) {
        fclose(f);
        return -1;
    }

    int numfiles = dirlen / sizeof(pakentry_t);
    pakentry_t *entries = malloc(dirlen);
    fseek(f, diroffset, SEEK_SET);
    fread(entries, sizeof(pakentry_t), numfiles, f);

    // Look for pics/colormap.pcx
    for (int i = 0; i < numfiles; i++) {
        if (strcmp(entries[i].name, "pics/colormap.pcx") == 0) {
            // Read file into buffer
            uint8_t *buf = malloc(entries[i].filelen);
            fseek(f, entries[i].filepos, SEEK_SET);
            fread(buf, 1, entries[i].filelen, f);

            // Parse PCX palette (last 769 bytes: 0x0C + 256*3)
            if (entries[i].filelen >= 769 && buf[entries[i].filelen - 769] == 0x0C) {
                memcpy(palette, buf + entries[i].filelen - 768, 256 * 3);
                free(buf);
                free(entries);
                fclose(f);
                return 0; // success
            }

            free(buf);
            break;
        }
    }

    free(entries);
    fclose(f);
    return -1; // not found
}

// --- Load palette from colormap.pcx ---
int load_or_extract_colormap(uint8_t palette[256][3]) {
	static const char *paths[] = {
		"colormap.pcx",                // current directory
		"main/pics/colormap.pcx",      // Kingpin
		"baseq2/pics/colormap.pcx"     // Quake II
	};

	// --- Try disk paths first ---
	for (int i = 0; i < 3; i++) {
		FILE *f = fopen(paths[i], "rb");
		if (!f)
			continue;
		fseek(f, -769, SEEK_END);
		uint8_t marker = fgetc(f);
		if (marker == 0x0C && fread(palette, 3, 256, f) == 256) {
			fclose(f);
			printf("Loaded palette from %s\n", paths[i]);
			return 0;
		}
		fclose(f);
	}

	// --- Try pak0.pak extraction ---
	if (extract_colormap_from_pak("main/pak0.pak", palette) == 0) {
		printf("Extracted colormap.pcx from main/pak0.pak\n");
		return 0;
	}
	if (extract_colormap_from_pak("baseq2/pak0.pak", palette) == 0) {
		printf("Extracted colormap.pcx from baseq2/pak0.pak\n");
		return 0;
	}

	// --- If all fail ---
	fprintf(stderr,
		"Error: Could not locate a valid 'colormap.pcx'.\n"
		"Searched in:\n"
		"  - Current directory\n"
		"  - main/pics/\n"
		"  - baseq2/pics/\n"
		"  - main/pak0.pak\n"
		"  - baseq2/pak0.pak\n"
		);
	return -1;
}

// --- RGB to palette index ---
int rgb_to_index(uint8_t r, uint8_t g, uint8_t b, uint8_t palette[256][3]) {
	int best = 0, bestDist = 999999;
	for (int i = 0; i < 256; i++) {
		int dr = r - palette[i][0];
		int dg = g - palette[i][1];
		int db = b - palette[i][2];
		int dist = dr*dr + dg*dg + db*db;
		if (dist < bestDist) { bestDist = dist; best = i; }
	}
	return best;
}

// --- Apply textureinfo.ini values ---
int apply_textureinfo(const char *dir, const char *texname, miptex_t *wal) {
	char inipath[512];
	snprintf(inipath, sizeof(inipath), "%s/textureinfo.ini", dir);

	FILE *f = fopen(inipath, "r");
	if (!f)
        return 0;

    iniFound = 1;  // INI exists in this directory

	if (debug)
		printf("Applying textureinfo.ini from %s\n", dir);

	// Normalize slashes and extract basename
	char cleanname[MAX_NAME];
	strncpy(cleanname, texname, MAX_NAME);
	cleanname[MAX_NAME - 1] = '\0';
	for (int i = 0; cleanname[i]; i++) {
		if (cleanname[i] == '\\') cleanname[i] = '/';
	}
	const char *basename = strrchr(cleanname, '/');
	if (!basename) basename = cleanname;
	else basename++; // skip the slash

	char line[256];
	int match = 0;
	int found = 0;
	while (fgets(line, sizeof(line), f)) {
		line[strcspn(line, "\r\n")] = 0; // trim newline

		if (line[0] == '[') {
			char section[64];
			sscanf(line, "[%63[^]]]", section);
			match = (strcmp(section, basename) == 0);
			if (match) {
				found = 1;
				printf("Matched [%s] in %s/textureinfo.ini\n", section, dir);
			}
		}
		else if (match) {
			unsigned int val;
			if (sscanf(line, "Flags=%u", &val) == 1) {
				wal->flags = val;
				printf("  -> Flags = %u\n", val);
				continue;
			}
			if (sscanf(line, "Value=%u", &val) == 1) {
				wal->value = val;
				printf("  -> Value = %u\n", val);
				continue;
			}
			if (sscanf(line, "Contents=%u", &val) == 1) {
				wal->contents = val;
				printf("  -> Contents = %u\n", val);
				continue;
			}
		}
	}
	fclose(f);
	return found;
}

void generate_mipmap(uint8_t *src, int w, int h, uint8_t *dst, uint8_t palette[256][3])
{
	int nw = w / 2, nh = h / 2;
	for (int y = 0; y < nh; y++) {
		for (int x = 0; x < nw; x++) {
			double r = 0, g = 0, b = 0;
			for (int dy = 0; dy < 2; dy++) {
				for (int dx = 0; dx < 2; dx++) {
					int idx = src[(y * 2 + dy)*w + (x * 2 + dx)];
					double lr = pow(palette[idx][0] / 255.0, 2.2);
					double lg = pow(palette[idx][1] / 255.0, 2.2);
					double lb = pow(palette[idx][2] / 255.0, 2.2);
					r += lr; g += lg; b += lb;
				}
			}
			r = pow(r / 4.0, 1.0 / 2.2);
			g = pow(g / 4.0, 1.0 / 2.2);
			b = pow(b / 4.0, 1.0 / 2.2);
			dst[y*nw + x] = rgb_to_index((int)(r * 255), (int)(g * 255), (int)(b * 255), palette);
		}
	}
}

void flip_vertical(uint8_t *data, int width, int height) {
	for (int y = 0; y < height / 2; y++) {
		uint8_t *row_top = data + y * width;
		uint8_t *row_bottom = data + (height - 1 - y) * width;
		for (int x = 0; x < width; x++) {
			uint8_t temp = row_top[x];
			row_top[x] = row_bottom[x];
			row_bottom[x] = temp;
		}
	}
}

int textureinfo_has_entry(const char *dir, const char *texname)
{
    char inipath[512];
    snprintf(inipath, sizeof(inipath), "%s/textureinfo.ini", dir);
    FILE *f = fopen(inipath, "r");
    if (!f)
        return 0; // no INI in this directory

    iniFound = 1;  // INI exists in this directory

    // Normalize slashes and extract basename
    char cleanname[MAX_NAME];
    strncpy(cleanname, texname, MAX_NAME);
    cleanname[MAX_NAME - 1] = 0;
    for (int i = 0; cleanname[i]; i++) {
        if (cleanname[i] == '\\')
            cleanname[i] = '/';
    }

    const char *basename = strrchr(cleanname, '/');
    if (!basename)
        basename = cleanname;
    else
        basename++; // skip the slash

    // Strip extension so INI sections use names without .tga/.wal
    char baseNoExt[MAX_NAME];
    strncpy(baseNoExt, basename, MAX_NAME - 1);
    baseNoExt[MAX_NAME - 1] = 0;
    char *dot = strrchr(baseNoExt, '.');
    if (dot)
        *dot = '\0';

    char line[256];
    while (fgets(line, sizeof(line), f)) {
        line[strcspn(line, "\r\n")] = 0;
        if (line[0] == '[') {
            char section[64];
            if (sscanf(line, "[%63[^]]", section) == 1) {
                if (strcmp(section, baseNoExt) == 0) {
                    fclose(f);
                    return 1; // found
                }
            }
        }
    }

    fclose(f);
    return 0; // not found
}



int scan_tga(const char *filename)
{
	FILE *f = fopen(filename, "rb");
	if (!f) {
		fprintf(stderr, "Error: cannot open %s\n", filename);
		return 0;
	}

	TGAHeader hdr;
	if (fread(&hdr, sizeof(TGAHeader), 1, f) != 1) {
		fprintf(stderr, "Error: %s truncated (cannot read header)\n", filename);
		fclose(f);
		return 0;
	}

    // Check supported type:
    //  2 = uncompressed RGB
    //  3 = uncompressed grayscale (optional)
    // 10 = RLE-compressed RGB
    if (hdr.imageType != 2 && hdr.imageType != 3 && hdr.imageType != 10) {
        fprintf(stderr, "Error: %s has unsupported TGA type (%d)\n",
                filename, hdr.imageType);
        fclose(f);
        return 0;
    }

	// Check pixel depth
	if (hdr.pixelDepth != 24 && hdr.pixelDepth != 32) {
		fprintf(stderr, "Error: %s has unsupported pixel depth (%d)\n", filename, hdr.pixelDepth);
		fclose(f);
		return 0;
	}

	// Check origin flag (must be bottom-left)
	if (hdr.imageDesc & 0x20) {
		fprintf(stderr, "Error: %s has top-left origin. Only bottom-left origin supported.\n", filename);
		fclose(f);
		return 0;
	}

    // Check file size vs expected (uncompressed only)
    struct stat st;
    if (stat(filename, &st) == 0) {
        if (hdr.imageType == 2 || hdr.imageType == 3) {
            size_t expected = sizeof(TGAHeader) + hdr.idLength
                            + (size_t)hdr.width * hdr.height * hdr.pixelDepth / 8;
            if (st.st_size < expected) {
                fprintf(stderr,
                        "Error: %s truncated: expected %zu bytes, got %lld\n",
                        filename, expected, (long long)st.st_size);
                fclose(f);
                return 0;
            }
        } else {
            // For RLE (type 10) just ensure it's not smaller than header + ID
            size_t minSize = sizeof(TGAHeader) + hdr.idLength;
            if (st.st_size < minSize) {
                fprintf(stderr,
                        "Error: %s truncated RLE TGA (too small)\n",
                        filename);
                fclose(f);
                return 0;
            }
        }
    }

	fclose(f);
	return 1; // valid
}

int fix_tga(const char *infile, const char *outfile)//FREDZ working but not on every broken TGA
{
	FILE *fin = fopen(infile, "rb");
	if (!fin) { perror("open TGA"); return 0; }

	TGAHeader hdr;
	if (fread(&hdr, sizeof(TGAHeader), 1, fin) != 1) {
		fprintf(stderr, "Error: %s truncated (cannot read header)\n", infile);
		fclose(fin);
		return 0;
	}

	// Force bottom-left origin
	hdr.imageDesc &= ~0x20;

	// If alpha not needed, force 24-bit
	if (hdr.pixelDepth == 32) {
		fprintf(stderr, "Note: converting %s from 32-bit to 24-bit\n", infile);
		hdr.pixelDepth = 24;
	}

	FILE *fout = fopen(outfile, "wb");
	if (!fout) { perror("open output"); fclose(fin); return 0; }

	fwrite(&hdr, sizeof(TGAHeader), 1, fout);

	// Copy rest of file (naive fix: doesn’t decode RLE, just passes through)
	char buf[4096];
	size_t n;
	while ((n = fread(buf, 1, sizeof(buf), fin)) > 0) {
		fwrite(buf, 1, n, fout);
	}

	fclose(fin);
	fclose(fout);
	return 1; // success
}

// --- Convert one TGA to WAL ---
int convert_tga_to_wal(const char *infile, const char *outfile, uint8_t palette[256][3])
{
    FILE *fin = fopen(infile, "rb");
    if (!fin) {
        perror("open TGA");
        return 0;
    }


    TGAHeader th;
    fread(&th, sizeof(TGAHeader), 1, fin);

    int w = th.width, h = th.height;
    int size = w * h;
    int bytesPerPixel = th.pixelDepth / 8;

    uint8_t *pixels = malloc(size * bytesPerPixel);
    uint8_t *base   = malloc(size);
    if (!pixels || !base)
    {
        fclose(fin);
        free(pixels);
        free(base);
        return 0;
    }

	// --- Prepare WAL name and dirpath ---
    char walname[256];
    strncpy(walname, infile, sizeof(walname)-1);
    walname[sizeof(walname)-1] = '\0';

    // Normalize slashes (convert '\' to '/')
    for (char *p = walname; *p; p++) {
        if (*p == '\\') *p = '/';
    }

    // Strip extension
    char *dot = strrchr(walname, '.');
    if (dot) *dot = '\0';

    // Find "textures/" in the path
    const char *relpath = walname;
    char *texpos = strstr(walname, "textures/");
    if (texpos)
        relpath = texpos + 9; // skip "textures/"


    // Extract directory path from infile
    char dirpath[512];
    strncpy(dirpath, infile, sizeof(dirpath)-1);
    dirpath[sizeof(dirpath)-1] = '\0';
    char *slash = strrchr(dirpath, '/');
    if (!slash)
        slash = strrchr(dirpath, '\\');
    if (slash)
        *slash = '\0'; // terminate at last slash

    // --- Read pixel data ---
    if (th.imageType == 2) {
        fread(pixels, bytesPerPixel, size, fin);
    }
    else if (th.imageType == 10) {
        int i = 0;
        while (i < size) {
            uint8_t packetHeader;
            if (fread(&packetHeader, 1, 1, fin) != 1) {
                fprintf(stderr, "Skipping %s: truncated RLE packet header\n", infile);
                free(pixels);
                free(base);
                return 0;
            }

            int count = (packetHeader & 0x7F) + 1;
            if (count <= 0 || i + count > size) {
                fprintf(stderr, "Skipping %s: invalid RLE packet run (count=%d)\n",
                        infile, count);
                free(pixels);
                free(base);
                return 0;
            }

            if (packetHeader & 0x80) {
                // RLE packet: one pixel repeated 'count' times
                uint8_t pixel[4];
                if (fread(pixel, bytesPerPixel, 1, fin) != 1) {
                    fprintf(stderr, "Skipping %s: truncated RLE pixel data\n", infile);
                    free(pixels);
                    free(base);
                    return 0;
                }
                for (int j = 0; j < count; j++) {
                    memcpy(pixels + (i + j) * bytesPerPixel, pixel, bytesPerPixel);
                }
            } else {
                // Raw packet: 'count' pixels follow
                if (fread(pixels + i * bytesPerPixel,
                          bytesPerPixel, count, fin) != (size_t)count) {
                    fprintf(stderr, "Skipping %s: truncated raw RLE data\n", infile);
                    free(pixels);
                    free(base);
                    return 0;
                }
            }

            i += count;
        }
    }

    fclose(fin);

    // --- Detect alpha usage ---
    int hasAlpha = 0;
    if (th.pixelDepth == 24) {
        if (onlyAlpha)
        {
            if (debug)
                fprintf(stderr, "Skipping %s (24-bit RGB has no alpha channel)\n", infile);
            free(pixels);
            free(base);
            return 0;
        }
    }
    else if (th.pixelDepth == 32) {
        for (int i = 0; i < size; i++)
        {
            uint8_t a = pixels[i*4 + 3];
            if (a < 255)
            {
                hasAlpha = 1;
                break;
            }
        }
        if (skipAlpha) {
            fprintf(stderr,
                    "Skipping %s: 32-bit RGBA has alpha channel and -skipalpha was requested\n",
                    infile);
            free(pixels);
            free(base);
            return 0;
        }
        else if (onlyAlpha) {
            if (!hasAlpha) {
                fprintf(stderr, "Skipping %s (no alpha channel data)\n", infile);
                free(pixels); free(base);
                return 0;
            }
            if (debug)
                fprintf(stderr, "Note: %s is 32-bit RGBA, using alpha channel only (ignoring RGB)\n", infile);
        }
    }
    else
    {
        fprintf(stderr, "Skipping %s (TGA must be 24-bit RGB or 32-bit RGBA)\n", infile);
        free(pixels);
        free(base);
        return 0;
    }

    // --- Dimension check ---
    if (requirePowerOf2 && ((w & (w - 1)) != 0 || (h & (h - 1)) != 0)) {
        fprintf(stderr, "Skipping %s (dimensions %dx%d not power-of-two)\n", infile, w, h);
        free(pixels);
        free(base);
        return 0;
    }

    // --- Conversion loop ---
    int histogram[256] = {0};
    for (int i = 0; i < size; i++) {
        uint8_t r = pixels[i*bytesPerPixel+2];
        uint8_t g = pixels[i*bytesPerPixel+1];
        uint8_t b = pixels[i*bytesPerPixel+0];

        if (bytesPerPixel == 4) {
            uint8_t a = pixels[i*4+3];

            if (onlyAlpha) {
                base[i] = (a < alphaThreshold) ? 255 : 0;
            }
            else
            {
                base[i] = (a < alphaThreshold) ? 255 : rgb_to_index(r, g, b, palette);
            }
        } else {
            base[i] = rgb_to_index(r, g, b, palette);
        }
        histogram[base[i]]++;
    }

	// Prepare WAL header
	miptex_t wal;
	memset(&wal, 0, sizeof(wal));

    // Copy into WAL header (truncate if needed)
    int len = strlen(relpath);
    if (len >= MAX_NAME) len = MAX_NAME - 1;
    memcpy(wal.name, relpath, len);
    wal.name[len] = '\0';

    wal.width = w;
    wal.height = h;
    wal.offsets[0] = sizeof(miptex_t);
    wal.offsets[1] = wal.offsets[0] + size;
    wal.offsets[2] = wal.offsets[1] + size/4;
    wal.offsets[3] = wal.offsets[2] + size/16;

    // --- Apply textureinfo.ini (merged logic) ---
    int matched = 0;

    if (!skipIni || onlyIni) {
        matched = apply_textureinfo(dirpath, wal.name, &wal);
    }

    if (onlyIni && !matched)
    {
        if (debug)
            fprintf(stderr, "Skipping %s (not listed in %s/textureinfo.ini)\n",
                    infile, dirpath);
        free(pixels); free(base);
        return 0;
    }

	// Strip down to Quake II only if requested
	if (q2FlagsOnly) {
		wal.flags &= Q2_FLAG_MASK;
		wal.contents &= Q2_CONTENT_MASK;
	}

    // Generate mipmaps
    uint8_t *mip1 = malloc(size / 4);
    uint8_t *mip2 = malloc(size / 16);
    uint8_t *mip3 = malloc(size / 64);
    generate_mipmap(base, w, h, mip1, palette);
    generate_mipmap(mip1, w/2, h/2, mip2, palette);
    generate_mipmap(mip2, w/4, h/4, mip3, palette);

    flip_vertical(base, w, h);
    flip_vertical(mip1, w/2, h/2);
    flip_vertical(mip2, w/4, h/4);
    flip_vertical(mip3, w/8, h/8);

    // Write WAL file
    FILE *fout = fopen(outfile, "wb");
    if (!fout) {
        perror("open WAL");
        free(pixels); free(base); free(mip1); free(mip2); free(mip3);
        return 0;
    }

    fwrite(&wal, sizeof(miptex_t), 1, fout);
    fwrite(base, 1, size, fout);
    fwrite(mip1, 1, size/4, fout);
    fwrite(mip2, 1, size/16, fout);
    fwrite(mip3, 1, size/64, fout);

    fclose(fout);
    free(pixels); free(base); free(mip1); free(mip2); free(mip3);

    printf("Converted %s -> %s\n", infile, outfile);

    return 1;
}

// Delete the WAL file corresponding to a TGA
void delete_wal_for_tga(const char *infile) {
    // Build WAL filename by replacing extension with .wal
    char walfile[512];
    strncpy(walfile, infile, sizeof(walfile)-1);
    walfile[sizeof(walfile)-1] = '\0';

    // Normalize slashes
    for (char *p = walfile; *p; p++) {
        if (*p == '\\') *p = '/';
    }

    // Strip extension
    char *dot = strrchr(walfile, '.');
    if (dot) *dot = '\0';

    // Append .wal
    strncat(walfile, ".wal", sizeof(walfile)-strlen(walfile)-1);

    // Try to delete
    if (remove(walfile) == 0) {
        printf("Deleted WAL: %s\n", walfile);
    } else {
        fprintf(stderr, "Failed to delete WAL: %s (%s)\n",
                walfile, strerror(errno));
    }
}

// --- Recursively process directories ---
void process_directory(const char *path, uint8_t palette[256][3], int *converted, int *skipped)
{
    DIR *dir = opendir(path);
    if (!dir) return;

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL) {
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0)
            continue;

        char fullpath[1024];
        snprintf(fullpath, sizeof(fullpath), "%s/%s", path, entry->d_name);

        struct stat st;
        if (stat(fullpath, &st) == 0) {
            if (S_ISDIR(st.st_mode)) {
                process_directory(fullpath, palette, converted, skipped);
            } else {
                const char *ext = strrchr(entry->d_name, '.');
                if (ext && strcasecmp(ext, ".tga") == 0)
                {
                    char outpath[1024];
                    strcpy(outpath, fullpath);
                    char *dot = strrchr(outpath, '.');
                    if (dot)
                        strcpy(dot, ".wal");

                    if (deletedMode) {
                        // delete WAL corresponding to this TGA
                        delete_wal_for_tga(fullpath);
                    }
                    else {
                        if (!scan_tga(fullpath)) {
                            fprintf(stderr, "Skipped %s: invalid or unsupported TGA\n", fullpath);
                            (*skipped)++;
                            continue;
                        }

                        if (!convert_tga_to_wal(fullpath, outpath, palette)) {
                            // optional: treat failed conversion as skipped
                            (*skipped)++;
                        } else {
                            (*converted)++;
                        }
                    }

                }
            }
        }
    }
    closedir(dir);
}

void scan_directory(const char *path, int *scanned, int *invalid)
{
    DIR *dir = opendir(path);
    if (!dir) return;

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL) {
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0)
            continue;

        char fullpath[1024];
        snprintf(fullpath, sizeof(fullpath), "%s/%s", path, entry->d_name);

        struct stat st;
        if (stat(fullpath, &st) == 0) {
            if (S_ISDIR(st.st_mode)) {
                scan_directory(fullpath, scanned, invalid);
            } else {
                const char *ext = strrchr(entry->d_name, '.');
                if (ext && strcasecmp(ext, ".tga") == 0) {
                    (*scanned)++;
                    if (!scan_tga(fullpath)) {
                        fprintf(stderr, "Corrupt or unsupported TGA: %s\n", fullpath);
                        (*invalid)++;
                    }
                    // valid TGAs are silent
                }
            }
        }
    }
    closedir(dir);
}

// --- Recursively fix TGAs ---
void fix_directory(const char *path, int *fixed, int *failed)
{
    DIR *dir = opendir(path);
    if (!dir) return;

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL) {
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0)
            continue;

        char fullpath[1024];
        snprintf(fullpath, sizeof(fullpath), "%s/%s", path, entry->d_name);

        struct stat st;
        if (stat(fullpath, &st) == 0) {
            if (S_ISDIR(st.st_mode)) {
                fix_directory(fullpath, fixed, failed);
            } else {
                const char *ext = strrchr(entry->d_name, '.');
                if (ext && strcasecmp(ext, ".tga") == 0) {
                    // Only attempt to fix if scan_tga() says it's bad
                    if (!scan_tga(fullpath)) {
                        char outpath[1024];
                        snprintf(outpath, sizeof(outpath), "%s", fullpath);

                        if (fix_tga(fullpath, outpath)) {
                            printf("Fixed %s -> %s\n", fullpath, outpath);
                            (*fixed)++;
                        } else {
                            fprintf(stderr, "Failed to fix %s\n", fullpath);
                            (*failed)++;
                        }
                    }
                }
            }
        }
    }
    closedir(dir);
}

// --- Main ---
int main(int argc, char **argv)
{
    iniFound = 0;

    time_t now = time(NULL);
    char date[64];
    strftime(date, sizeof(date), "%Y-%m-%d", localtime(&now));

    // Default root depending on game folder
    int pathArgIndex = -1;
    bool pathProvided = false;

    // Parse arguments
    for (int j = 1; j < argc; j++)
    {
        // Skip the directory argument itself
        if (pathProvided && j == pathArgIndex)
            continue;

        if (strcmp(argv[j], "-alphathresh") == 0) {
            if (j + 1 >= argc) {
                fprintf(stderr, "Error: -alphathresh requires a numeric value 0-255.\n");
                return 1;
            }

            // Require the value to start with a digit (or +/-digit if you want)
            const char *val = argv[j + 1];
            if (!isdigit((unsigned char)val[0])) {
                fprintf(stderr, "Error: -alphathresh requires a numeric value 0-255, got '%s'.\n", val);
                return 1;
            }

            alphaThreshold = atoi(argv[++j]);
            useAlphaThreshold = 1;
            printf("Alpha threshold set to %d\n", alphaThreshold);

            if (alphaThreshold < 0 || alphaThreshold > 255) {
                fprintf(stderr,
                    "Warning: alpha threshold %d out of range 0-255. Clamping.\n",
                    alphaThreshold);
                alphaThreshold = (alphaThreshold < 0) ? 0 : 255;
            }
        }

        else if (strcmp(argv[j], "-skipalpha") == 0) {
            skipAlpha = 1; // ignore alpha channel completely
            printf("Alpha channel will be ignored (RGB only)\n");
        }
        else if (strcmp(argv[j], "-onlyalpha") == 0) {
            onlyAlpha = 1; // ignore RGB, use alpha only
            printf("Using alpha channel only (ignoring RGB)\n");
        }
        else if (strcmp(argv[j], "-skipini") == 0)
        {
            skipIni = 1;
            printf("Skipping textureinfo.ini\n");
        }
        else if (strcmp(argv[j], "-onlyini") == 0) {
            onlyIni = 1;
            printf("INI-only mode: textures will be generated from textureinfo.ini without pixel conversion\n");
        }
        else if (strcmp(argv[j], "-powerof2") == 0) {
            requirePowerOf2 = 1;
        }
        else if (strcmp(argv[j], "-q2flags") == 0) {
            q2FlagsOnly = 1;
        }
        else if (strcmp(argv[j], "-deletedwal") == 0) {
            deletedMode = 1;
            printf("Deleted mode: WALs corresponding to TGAs will be removed\n");
        }
        else if (strcmp(argv[j], "-scantga") == 0) {
            scanMode = 1;
        }
        /*
        else if (strcmp(argv[i], "-fixtga") == 0) {
            fixMode = 1;
        }*/
        else if (
            strcmp(argv[j], "-h") == 0 ||
            strcmp(argv[j], "--help") == 0 ||
            strcmp(argv[j], "?") == 0 ||
            strcmp(argv[j], "/?") == 0 ||
            strcmp(argv[j], "help") == 0
        ) {
            printf("Usage: %s [options] [directory]\n", argv[0]);
            printf("If no directory is given, current directory is used.\n\n");
            printf("Options:\n");
            printf("  -alphathresh <N>  : set alpha threshold (default 255).\n");
            printf("                      Pixels with alpha < N map to palette index 255 (transparent).\n");
            printf("  -onlyalpha        : use alpha channel only (ignore RGB images)\n");
            printf("  -skipalpha        : ignore alpha channel (RGB images only)\n");
            printf("  -skipini          : ignore textureinfo.ini completely\n");
            printf("  -onlyini          : convert only TGAs listed in textureinfo.ini (apply INI flags/contents)\n");

            //FREDZ disabled this not really working?
            printf("  -powerof2         : skip textures that are not power-of-two (e.g. 64x64, 128x256)\n");

            printf("  -q2flags          : only apply Quake II-compatible flags (for Wally compatibility)\n");
            printf("  -deletedwal       : delete WALs corresponding to TGAs\n");
            printf("  -scantga          : scan TGA header and report issues (no conversion)\n");
//          printf("  -fixtga           : attempt to repair TGA and overwrite file.\n");
            printf("  -h, --help        : show this help message\n");

            printf("\nBy default, the tool auto-detects the textures folder:\n");
            printf("      Kingpin   main/textures\n");
            printf("      Quake II  baseq2/textures\n");
            printf("If run inside kingpin/main or quake2/baseq2, it will look for the local 'textures' subfolder.\n");
            printf("If no textures folder is found, the tool exits with an error.\n");

            printf("\nCreated by FREDZ on %s\n", date);
            return 0;
        }
        else if (argv[j][0] == '-')
        {
            fprintf(stderr, "Unknown option: %s\n", argv[j]);
            fprintf(stderr, "See %s -h or --help for usage information.\n", argv[0]);
            return 1;
        }
    }

    // Fixed root handling - path first, then options
    for (int i = 1; i < argc; i++) {
        // Skip the value that belongs to -alphathresh
        if (strcmp(argv[i], "-alphathresh") == 0) {
            if (i + 1 < argc) {
                i++;    // skip its numeric parameter
            }
            continue;
        }

        if (argv[i][0] != '-') {
            strncpy(rootpath, argv[i], sizeof(rootpath) - 1);
            rootpath[sizeof(rootpath) - 1] = 0;

            char testpath[2048];
            snprintf(testpath, sizeof(testpath), "%s/textures", rootpath);
            if (access(testpath, 0) != 0) {
                fprintf(stderr, "Error: could not find textures folder at %s\n", testpath);
                return 1;
            }

            pathArgIndex = i;
            pathProvided = true;
            break;
        }
    }

    // Auto-detect if no path provided
    if (!pathProvided) {
        if (access("main/textures", 0) == 0) strcpy(rootpath, "main");
        else if (access("baseq2/textures", 0) == 0) strcpy(rootpath, "baseq2");
        else if (access("textures", 0) == 0) strcpy(rootpath, ".");
        else {
            fprintf(stderr,
                "Error: could not find textures folder (main/textures or baseq2/textures).\n"
                "Run this tool from your Kingpin/Quake2 folder,\n"
                "or pass a path like \"kingpin\\main\" or \"quake2\\baseq2\".\n");
            return 1;
        }
    }

	//conflict cmd error check.
	if (deletedMode) {
		if (skipIni || onlyIni || skipAlpha || onlyAlpha || requirePowerOf2 || q2FlagsOnly || useAlphaThreshold) {
			fprintf(stderr, "Error: -deleted is a standalone mode. All conversion options are disabled in deleted mode.\n");
			return 1;
		}
	}

	// scanMode and fixMode cannot be combined
	if (scanMode && fixMode) {
		fprintf(stderr,
			"Error: -scantga and -fixtga cannot be used together.\n");
		return 1;
	}

    if (scanMode) {
        if (skipIni || onlyIni || skipAlpha || onlyAlpha || requirePowerOf2 || q2FlagsOnly || useAlphaThreshold) {
            fprintf(stderr, "Error: -scantga is a standalone mode. All conversion options are disabled in scan mode.\n");
            return 1;
        }
        else
        {
            int scanned = 0, invalid = 0;
            scan_directory(rootpath, &scanned, &invalid);
            printf("\nScanned %d TGAs, %d corrupt/unsupported\n", scanned, invalid);
            return 0;
        }
    }

#if 0
	//FREDZ Gives to much problems :/
	if (fixMode) {
		if (skipIni || onlyIni || skipAlpha || onlyAlpha || requirePowerOf2 || q2FlagsOnly || useAlphaThreshold) {
			fprintf(stderr, "Error: -fixtga is a standalone mode. All conversion options are disabled in fix mode.\n");
			return 1;
		}
		else
		{
			int fixed = 0, failed = 0;
			fix_directory(root, &fixed, &failed);
			printf("\nFixed %d TGAs, %d failed\n", fixed, failed);
			return 0;
		}
	}
#endif

	if (skipAlpha && onlyAlpha) {
		fprintf(stderr,
			"Error: -skipalpha and -onlyalpha cannot be used together.\n"
			"Choose either to ignore alpha (RGB only) or to use alpha only.\n");
		return 1;
	}

	if (skipAlpha && useAlphaThreshold) {
		fprintf(stderr,
			"Error: -skipalpha and -alphathresh cannot be used together.\n"
			"Choose either to ignore alpha (RGB only) or to use alphathresh only.\n");
		return 1;
	}

	if (skipIni && onlyIni) {
		fprintf(stderr,
			"Error: -skipini and -onlyini cannot be used together.\n"
			"Choose either to skip ini or to use ini-only mode.\n");
		return 1;
	}

    // --- Only load palette if needed ---
    uint8_t palette[256][3];
    if (!scanMode && !fixMode && !deletedMode) {
        // First try pak0.pak under the selected rootpath (e.g. "kingpin\\main")
        char pakPath[1024];
        snprintf(pakPath, sizeof(pakPath), "%s/pak0.pak", rootpath);
        pakPath[sizeof(pakPath) - 1] = '\0';

        // If that fails, fall back to the original logic (current dir, main/pics, baseq2/pics, main/baseq2 pak0.pak)
        if (extract_colormap_from_pak(pakPath, palette) == 0) {
            printf("Extracted colormap.pcx from %s\n", pakPath);
        } else {
            if (load_or_extract_colormap(palette) != 0) {
                return 1;
            }
        }
    }

	// Run conversion
	int converted = 0;
	int skipped = 0;

    process_directory(rootpath, palette, &converted, &skipped);
    printf("\nConverted %d textures, skipped %d invalid TGA\n", converted, skipped);
    if (onlyIni && !iniFound) {
        fprintf(stderr,
            "Warning: -onlyini requested, but no textureinfo.ini was found.\n");
    }
    printf("rootpath = '%s'\n", rootpath);
    printf("Created by FREDZ on %s\n", date);
    return 0;
}
