/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "mm3dfilter.h"

#include "model.h"
#include "texture.h"
#include "log.h"
#include "binutil.h"
#include "misc.h"
#include "endianconfig.h"
#include "mm3dport.h"

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <vector>

using std::list;
using std::string;

// Misfit MM3D format:
//
// File Header
//
// MAGIC_NUMBER    8 bytes  "MISFIT3D"
// MAJOR_VERSION   uint8    0x01
// MINOR_VERSION   uint8    0x04
// MODEL_FLAGS     uint8    0x00 (reserved)
// OFFSET_COUNT    uint8    [offc]
//
//    [OFFSET_COUNT] instances of:
//    Offset header list
//    OFFSET_HEADER   6 bytes * [offc]
//
//    Offset Header
//    OFFSET_TYPE     uint16 (highest bit 0 = Data type A, 1 = data type B)
//    OFFSET_VALUE    uint32 
//
// Data header A (Variable data size)
// DATA_FLAGS      uint16 
// DATA_COUNT      uint32 
//
// Data block A
//
// [DATA_COUNT] instances of:
// DATA_SIZE       uint32 bytes
// DATA_BLOCK_A    [DATA_SIZE] bytes
//
// Data header B (Uniform data size)
// DATA_FLAGS      uint16 
// DATA_COUNT      uint32 
// DATA_SIZE       uint32 
//
// Data block B
//
// [DATA_COUNT] instances of:
// DATA_BLOCK_B   [DATA_SIZE] bytes
//
//
// Offset A types:
//   0x1001          Meta information
//   0x1002          Unknown type information
//   0x0101          Groups
//   0x0141          Embedded textures
//   0x0142          External textures
//   0x0161          Materials
//   0x0191          Canvas background images
//
//   0x0301          Skeletal Animations
//   0x0321          Frame Animations
//   0x0341          Frame Relative Animations
//
// Offset B types:
//   0x0001          Vertices
//   0x0021          Triangles
//   0x0026          Triangle Normals
//   0x0041          Joints
//   0x0046          Joint Vertices
//   0x0106          Smooth Angles
//   0x0121          Texture Coordinates
//
//
// 
// Meta information data
//   KEY             ASCIIZ  <= 1024 (may not be empty)
//   VALUE           ASCIIZ  <= 1024 (may be empty)
//
// Unknown type information
//   OFFSET_TYPE     uint16
//   INFO_STRING     ASCIIZ <= 256 (inform user how to support this data)
//
//
// Vertex data
//   FLAGS           uint16
//   X_COORD         float32
//   Y_COORD         float32
//   Z_COORD         float32
//
//
// Triangle data
//   FLAGS           uint16
//   VERTEX1_INDEX   uint32
//   VERTEX2_INDEX   uint32
//   VERTEX3_INDEX   uint32
//
// Group data
//   FLAGS           uint16
//   NAME            ASCIIZ <= inf
//   TRI_COUNT       uint32
//   TRI_INDICES     uint32 * [TRI_COUNT]
//   SMOOTHNESS      uint8
//   MATERIAL_INDEX  uint32
//   
// Smooth Angles (maximum angle between smoothed normals for a group)
//   GROUP_INDEX     uint32
//   SMOOTHNESS      uint8
//   
// External texture data
//   FLAGS           uint16
//   FILENAME        ASCIIZ
//
// Embedded texture data
//   FLAGS           uint16
//   FORMAT          char8 * 4  ('JPEG', 'PNG ', 'TGA ', etc...)
//   TEXTURE_SIZE    uint32
//   DATA            uint8 * [TEXTURE_SIZE]
//
// Material
//   FLAGS           uint16
//   TEXTURE_INDEX   uint32
//   NAME            ASCIIZ
//   AMBIENT         float32 * 4
//   DIFFUSE         float32 * 4
//   SPECULAR        float32 * 4
//   EMISSIVE        float32 * 4
//   SHININESS       float32
//   
// Canvas background image
//   FLAGS           uint16
//   VIEW_INDEX      uint8
//   SCALE           float32
//   CENTER_X        float32
//   CENTER_X        float32
//   CENTER_X        float32
//   FILENAME        ASCIIZ
//
// Texture coordinates
//   FLAGS           uint16
//   TRI_INDEX       uint32
//   COORD_S         float32 * 3
//   COORD_T         float32 * 3
//
// Joint data
//   FLAGS           uint16
//   NAME            char8 * 40
//   PARENT_INDEX    int32
//   LOCAL_ROT       float32 * 3
//   LOCAL_TRANS     float32 * 3
//
// Joint Vertices
//   VERTEX_INDEX    uint32
//   JOINT_INDEX     int32
//
// Skeletal animation
//   FLAGS           uint16
//   NAME            ASCIIZ
//   FPS             float32
//   FRAME_COUNT     uint32
//
//      [FRAME_COUNT] instances of:
//      KEYFRAME_COUNT  uint32
//
//        [KEYFRAME_COUNT] instances of:
//        JOINT_INDEX     uint32
//        KEYFRAME_TYPE   uint8  (0 = rotation, 1 = translation)
//        PARAM           float32 * 3
//
// Frame animation
//   FLAGS           uint16
//   NAME            ASCIIZ <= 64
//   FPS             float32
//   FRAME_COUNT     uint32
//
//      [FRAME_COUNT] instances of:
//      
//         [VERTEX_COUNT] instances of:
//         COORD_X         float32
//         COORD_Y         float32
//         COORD_Z         float32
// 
// Frame relative animation
//   FLAGS           uint16
//   NAME            ASCIIZ <= 64
//   FPS             float32
//   FRAME_COUNT     uint32
//
//      [FRAME_COUNT] instances of:
//      FVERT_COUNT     uint32
//      
//         [FVERT_COUNT] instances of:
//         VERTEX_INDEX
//         COORD_X_OFFSET  float32
//         COORD_Y_OFFSET  float32
//         COORD_Z_OFFSET  float32
// 

const char     MisfitFilter::MAGIC[] = "MISFIT3D";

const uint8_t  MisfitFilter::WRITE_VERSION_MAJOR = 0x01;
const uint8_t  MisfitFilter::WRITE_VERSION_MINOR = 0x04;

const uint16_t MisfitFilter::OFFSET_TYPE_MASK  = 0x3fff;
const uint16_t MisfitFilter::OFFSET_UNI_MASK   = 0x8000;
const uint16_t MisfitFilter::OFFSET_DIRTY_MASK = 0x4000;

typedef struct MisfitOffset_t
{
   uint16_t offsetType;
   uint32_t offsetValue;
} MisfitOffsetT;

typedef std::vector<MisfitOffsetT>  MisfitOffsetList;

typedef enum MisfitDataTypes_e {

   // A Types
   MDT_Meta,
   MDT_TypeInfo,
   MDT_Groups,
   MDT_EmbTextures,
   MDT_ExtTextures,
   MDT_Materials,
   MDT_CanvasBackgrounds,
   MDT_SkelAnims,
   MDT_FrameAnims,
   MDT_RelativeAnims,

   // B Types
   MDT_Vertices,
   MDT_Triangles,
   MDT_TriangleNormals,
   MDT_Joints,
   MDT_JointVertices,
   MDT_SmoothAngles,
   MDT_TexCoords,

   // End of list
   MDT_EndOfFile,
   MDT_MAX
} MisfitDataTypesE;

typedef enum MisfitFlags_e
{
   MF_HIDDEN    = 1,  // powers of 2
   MF_SELECTED  = 2,

   // Type-specific flags
   MF_MAT_CLAMP_S = 16,
   MF_MAT_CLAMP_T = 32
} MisfitFlagsE;

static const uint16_t _misfitOffsetTypes[MDT_MAX]  = {

// Offset A types
   0x1001,         // Meta information
   0x1002,         // Unknown type information
   0x0101,         // Groups
   0x0141,         // Embedded textures
   0x0142,         // External textures
   0x0161,         // Materials
   0x0191,         // Canvas background images
   0x0301,         // Skeletal Animations
   0x0321,         // Frame Animations
   0x0341,         // Frame Relative Animations

// Offset B types:
   0x0001,         // Vertices
   0x0021,         // Triangles
   0x0026,         // Triangle Normals
   0x0041,         // Joints
   0x0046,         // Joint Vertices
   0x0106,         // Smooth Angles
   0x0121,         // Texture Coordinates

   0x3fff,         // End of file
};

static const char _misfitOffsetNames[MDT_MAX][30] = {
// Offset A types
   "Meta information",
   "Type identity",
   "Groups",
   "Embedded textures",
   "External textures",
   "Materials",
   "Canvas Background Images",
   "Skeletal animations",
   "Frame animations",
   "Frame relative animations",
// Offset B types
   "Vertices",
   "Triangles",
   "Triangle normals",
   "Joints",
   "Joint vertices",
   "Texture coordinates",
// End of file
   "End of file"
};


// byte-align structures
#if defined( __GNUC__ )
#	define BYTE_ALIGNED	__attribute__((packed))
#else
#if defined( _MSC_VER )
#	pragma pack( push, packing )
#	pragma pack( 1 )
#	define BYTE_ALIGNED
#else
#	error do not know how to byte-align for this compiler
#endif // windows
#endif // gcc

typedef float float32_t;

// File header
struct MM3DFILE_Header_t
{
   char magic[8];
   uint8_t versionMajor;
   uint8_t versionMinor;
   uint8_t modelFlags;
   uint8_t offsetCount;
} BYTE_ALIGNED;
typedef struct MM3DFILE_Header_t MM3DFILE_HeaderT;

// Data header B (Uniform data size)
struct MM3DFILE_DataHeaderA_t
{
   uint16_t flags;
   uint32_t count;
} BYTE_ALIGNED;
typedef struct MM3DFILE_DataHeaderA_t MM3DFILE_DataHeaderAT;

// Data header B (Uniform data size)
struct MM3DFILE_DataHeaderB_t
{
   uint16_t flags;
   uint32_t count;
   uint32_t size;
} BYTE_ALIGNED;
typedef struct MM3DFILE_DataHeaderB_t MM3DFILE_DataHeaderBT;

struct MM3DFILE_Vertex_t
{
   uint16_t  flags;
   float32_t coord[3];
} BYTE_ALIGNED;
typedef struct MM3DFILE_Vertex_t MM3DFILE_VertexT;

struct MM3DFILE_Triangle_t
{
   uint16_t  flags;
   uint32_t  vertex[3];
} BYTE_ALIGNED;
typedef struct MM3DFILE_Triangle_t MM3DFILE_TriangleT;

struct MM3DFILE_TriangleNormals_t
{
   uint16_t   flags;
   uint32_t   index;
   float32_t  normal[3][3];
} BYTE_ALIGNED;
typedef struct MM3DFILE_TriangleNormals_t MM3DFILE_TriangleNormalsT;

struct MM3DFILE_Joint_t
{
   uint16_t  flags;
   char      name[40];
   int32_t   parentIndex;
   float32_t localRot[3];
   float32_t localTrans[3];
} BYTE_ALIGNED;
typedef struct MM3DFILE_Joint_t MM3DFILE_JointT;

struct MM3DFILE_JointVertex_t
{
   uint32_t  vertexIndex;
   int32_t   jointIndex;
} BYTE_ALIGNED;
typedef struct MM3DFILE_JointVertex_t MM3DFILE_JointVertexT;

struct MM3DFILE_SmoothAngle_t
{
   uint32_t  groupIndex;
   uint8_t   angle;
} BYTE_ALIGNED;
typedef struct MM3DFILE_SmoothAngle_t MM3DFILE_SmoothAngleT;

struct MM3DFILE_CanvasBackground_t
{
   uint16_t  flags;
   uint8_t   viewIndex;
   float32_t scale;
   float32_t center[3];
} BYTE_ALIGNED;
typedef struct MM3DFILE_CanvasBackground_t MM3DFILE_CanvasBackgroundT;

struct MM3DFILE_SkelKeyframe_t
{
   uint32_t   jointIndex;
   uint8_t    keyframeType;
   float32_t  param[3];
} BYTE_ALIGNED;
typedef struct MM3DFILE_SkelKeyframe_t MM3DFILE_SkelKeyframeT;

struct MM3DFILE_TexCoord_t
{
   uint16_t  flags;
   uint32_t  triangleIndex;
   float32_t sCoord[3];
   float32_t tCoord[3];
} BYTE_ALIGNED;
typedef struct MM3DFILE_TexCoord_t MM3DFILE_TexCoordT;

// Default alignment
#ifdef _MSC_VER
#	pragma pack( pop, packing )
#endif

#undef PACK_STRUCT

struct UnknownData_t
{
   uint16_t offsetType;
   uint32_t offsetValue;
   uint32_t dataLen;
};

typedef struct UnknownData_t UnknownDataT;

typedef std::list<UnknownDataT> UnknownDataList;

static void _addOffset( MisfitDataTypesE type, bool include, MisfitOffsetList & list )
{
   if ( include )
   {
      MisfitOffsetT mo;
      mo.offsetType = _misfitOffsetTypes[ type ];
      mo.offsetValue = 0;
      log_debug( "adding offset type %04X\n", mo.offsetType );
      list.push_back( mo );
   }
}

static void _setOffset( MisfitDataTypesE type, uint32_t offset, MisfitOffsetList & list )
{
   unsigned count = list.size();
   for ( unsigned n = 0; n < count; n++ )
   {
      if ( (list[n].offsetType & MisfitFilter::OFFSET_TYPE_MASK) == _misfitOffsetTypes[type] )
      {
         list[n].offsetValue = offset;
         log_debug( "updated offset for %04X to %08X\n", list[n].offsetType, list[n].offsetValue );
         break;
      }
   }
}

static void _setUniformOffset( MisfitDataTypesE type, bool uniform, MisfitOffsetList & list )
{
   unsigned count = list.size();
   for ( unsigned n = 0; n < count; n++ )
   {
      if ( (list[n].offsetType & MisfitFilter::OFFSET_TYPE_MASK) == _misfitOffsetTypes[type] )
      {
         if ( uniform )
         {
            log_debug( "before uniform: %04X\n", list[n].offsetType );
            list[n].offsetType |= MisfitFilter::OFFSET_UNI_MASK;
            log_debug( "after uniform: %04X\n", list[n].offsetType );
         }
         else
         {
            log_debug( "before variable: %04X\n", list[n].offsetType );
            list[n].offsetType &= MisfitFilter::OFFSET_TYPE_MASK;
            log_debug( "after variable: %04X\n", list[n].offsetType );
         }
         break;
      }
   }
}

bool _offsetIncluded( MisfitDataTypesE type, MisfitOffsetList & list )
{
   unsigned count = list.size();
   for ( unsigned n = 0; n < count; n++ )
   {
      if ( (list[n].offsetType & MisfitFilter::OFFSET_TYPE_MASK) == _misfitOffsetTypes[type] )
      {
         return true;
      }
   }
   return false;
}

unsigned _offsetGet( MisfitDataTypesE type, MisfitOffsetList & list )
{
   unsigned count = list.size();
   for ( unsigned n = 0; n < count; n++ )
   {
      if ( (list[n].offsetType & MisfitFilter::OFFSET_TYPE_MASK) == _misfitOffsetTypes[type] )
      {
         return list[n].offsetValue;
      }
   }
   return 0;
}

bool _offsetIsVariable( MisfitDataTypesE type, MisfitOffsetList & list )
{
   unsigned count = list.size();
   for ( unsigned n = 0; n < count; n++ )
   {
      if ( (list[n].offsetType & MisfitFilter::OFFSET_TYPE_MASK) == _misfitOffsetTypes[type] )
      {
         return ((list[n].offsetType & MisfitFilter::OFFSET_UNI_MASK) == 0);
      }
   }
   return false;
}

MisfitFilter::MisfitFilter()
{
}

MisfitFilter::~MisfitFilter()
{
}

Model::ModelError MisfitFilter::readFile( Model * model, const char * const filename )
{
   if ( filename == NULL || filename[0] == '\0' )
   {
      return Model::ERROR_BAD_ARGUMENT;
   }

   FILE * fp = fopen( filename, "rb" );

   if ( fp == NULL )
   {
      switch ( errno )
      {
         case EACCES:
         case EPERM:
            return Model::ERROR_NO_ACCESS;
         case ENOENT:
            return Model::ERROR_NO_FILE;
         case EISDIR:
            return Model::ERROR_BAD_DATA;
         default:
            return Model::ERROR_FILE_OPEN;
      }
   }

   string modelPath = "";
   string modelBaseName = "";
   string modelFullName = "";

   normalizePath( filename, modelFullName, modelPath, modelBaseName );

   model->setFilename( modelFullName.c_str() );
      
   fseek( fp, 0, SEEK_END );
   unsigned fileLength = ftell( fp );
   unsigned readLength = fileLength;
   fseek( fp, 0, SEEK_SET );

   uint8_t * fileBuf = new uint8_t[fileLength];
   uint8_t * bufPos = fileBuf;

   fread( fileBuf, fileLength, 1, fp );
   fclose( fp );

   MM3DFILE_HeaderT * fileHeader;

   fileHeader = (MM3DFILE_HeaderT *) bufPos;
   bufPos += sizeof(MM3DFILE_HeaderT);

   if ( strncmp( fileHeader->magic, MAGIC, strlen(MAGIC) ) != 0 )
   {
      delete[] fileBuf;
      log_warning( "bad magic number file\n" );
      return Model::ERROR_BAD_MAGIC;
   }

   if ( fileHeader->versionMajor != WRITE_VERSION_MAJOR )
   {
      delete[] fileBuf;
      return Model::ERROR_UNSUPPORTED_VERSION;
   }

   unsigned offsetCount = fileHeader->offsetCount;

   log_debug( "Misfit file:\n" );
   log_debug( "  major:   %d\n", fileHeader->versionMajor );
   log_debug( "  minor:   %d\n", fileHeader->versionMinor );
   log_debug( "  offsets: %d\n", fileHeader->offsetCount );

   log_debug( "Offset information:\n" );

   unsigned t;

   MisfitOffsetList offsetList;

   unsigned lastOffset = 0;
   bool     lastUnknown = false;
   UnknownDataList unknownList;

   readLength = fileLength - sizeof( MM3DFILE_HeaderT );
   for ( t = 0; t < offsetCount; t++ )
   {
      MisfitOffsetT mo;
      bin_read( mo.offsetType,  bufPos, readLength );
                mo.offsetType = ltoh_u16(mo.offsetType);
      bin_read( mo.offsetValue, bufPos, readLength );
                mo.offsetValue = ltoh_u32(mo.offsetValue);

      if ( mo.offsetValue < lastOffset )
      {
         log_error( "Offset are out of order\n" );
         return Model::ERROR_BAD_DATA;
      }

      if ( lastUnknown )
      {
         unknownList.back().dataLen = mo.offsetValue - lastOffset;
         log_warning( "unknown data size = %d\n", unknownList.back().dataLen );
         lastUnknown = false;
      }
      lastOffset = mo.offsetValue;

      offsetList.push_back( mo );

      bool found = false;
      for ( unsigned e = 0; !found && e < MDT_MAX; e++ )
      {
         if ( _misfitOffsetTypes[e] == (mo.offsetType & OFFSET_TYPE_MASK) )
         {
            log_debug( "  %08X %s\n", mo.offsetValue, _misfitOffsetNames[e] );
            found = true;

            if ( e == MDT_EndOfFile )
            {
               if ( mo.offsetValue != fileLength )
               {
                  if ( mo.offsetValue > fileLength )
                  {
                     return Model::ERROR_UNEXPECTED_EOF;
                  }
                  else
                  {
                     log_warning( "EOF offset and file size do not match (%d and %d)\n", mo.offsetValue, fileLength );
                     return Model::ERROR_BAD_DATA;
                  }
               }
            }
         }
      }

      if ( !found )
      {
         log_debug( "  %08X Unknown type %04X\n", mo.offsetValue, mo.offsetType );

         lastUnknown = true;
         UnknownDataT ud;
         ud.offsetType = mo.offsetType;
         ud.offsetValue = mo.offsetValue;
         ud.dataLen = 0;

         unknownList.push_back( ud );
      }
   }

   vector<Model::Vertex *>   & modelVertices  = getVertexList( model );
   vector<Model::Triangle *> & modelTriangles = getTriangleList( model );
   vector<Model::Group *>    & modelGroups    = getGroupList( model );
   vector<Model::Material *> & modelMaterials = getMaterialList( model );
   vector<Model::Joint *>    & modelJoints    = getJointList( model );

   // Used to track whether indices are valid
   bool missingElements = false;

   // Meta data
   if ( _offsetIncluded( MDT_Meta, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_Meta, offsetList );
      uint32_t offset   = _offsetGet( MDT_Meta, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
      bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned m = 0; m < count; m++ )
      {
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
         }

         unsigned len;
         char key[1024];
         char value[1024];

         strncpy( key, (char *) bufPos, sizeof(key) );
         key[ sizeof(key) - 1 ] = '\0';

         len = strlen(key) + 1;
         bufPos     += len;
         readLength -= len;

         strncpy( value, (char *) bufPos, sizeof(value) );
         value[ sizeof(value) - 1 ] = '\0';

         len = strlen(value) + 1;
         bufPos     += len;
         readLength -= len;

         model->addMetaData( key, value );
      }
   }

   // Vertices
   if ( _offsetIncluded( MDT_Vertices, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_Vertices, offsetList );
      uint32_t offset   = _offsetGet( MDT_Vertices, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
      bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned v = 0; v < count; v++ )
      {
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
         }

         MM3DFILE_VertexT * fileVert = (MM3DFILE_VertexT *) bufPos;
         Model::Vertex    * vert = Model::Vertex::get();

         vert->m_boneId = -1;
         vert->m_coord[0] = ltoh_float(fileVert->coord[0]);
         vert->m_coord[1] = ltoh_float(fileVert->coord[1]);
         vert->m_coord[2] = ltoh_float(fileVert->coord[2]);
                        uint16_t vertFlags = ltoh_u16(fileVert->flags);
         vert->m_selected = ((vertFlags & MF_SELECTED) == MF_SELECTED);
         vert->m_visible  = ((vertFlags & MF_HIDDEN) != MF_HIDDEN);

         modelVertices.push_back( vert );

         bufPos     += size;
         readLength -= size;
      }
   }

   // Triangles
   if ( _offsetIncluded( MDT_Triangles, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_Triangles, offsetList );
      uint32_t offset   = _offsetGet( MDT_Triangles, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
      bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned t = 0; t < count; t++ )
      {
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
         }

         MM3DFILE_TriangleT * fileTri = (MM3DFILE_TriangleT *) bufPos;
         Model::Triangle    * tri = Model::Triangle::get();

         tri->m_vertexIndices[0] = ltoh_u32(fileTri->vertex[0]);
         tri->m_vertexIndices[1] = ltoh_u32(fileTri->vertex[1]);
         tri->m_vertexIndices[2] = ltoh_u32(fileTri->vertex[2]);
                        uint16_t triFlags = ltoh_u16(fileTri->flags);
         tri->m_selected = ((triFlags & MF_SELECTED) == MF_SELECTED);
         tri->m_visible  = ((triFlags & MF_HIDDEN) != MF_HIDDEN);

         modelTriangles.push_back( tri );

         bufPos     += size;
         readLength -= size;
      }
   }

#if 0
   // Triangle Normals
   if ( _offsetIncluded( MDT_TriangleNormals, offsetList ) )
   {
      // Just for debugging... we don't actually use any of this

      bool     variable = _offsetIsVariable( MDT_TriangleNormals, offsetList );
      uint32_t offset   = _offsetGet( MDT_TriangleNormals, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
      bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned t = 0; t < count; t++ )
      {
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
         }

         MM3DFILE_TriangleNormalsT * fileTri = (MM3DFILE_TriangleNormalsT *) bufPos;

         log_debug( "triangle %d normals:\n", fileTri->index );

         for ( unsigned v = 0; v < 3; v++ )
         {
            log_debug( "  v %d:  %f %f %f\n", v, 
                  fileTri->normal[v][0],
                  fileTri->normal[v][1],
                  fileTri->normal[v][2] );
         }

         bufPos     += size;
         readLength -= size;
      }
   }
#endif // 0

   // Groups
   if ( _offsetIncluded( MDT_Groups, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_Groups, offsetList );
      uint32_t offset   = _offsetGet( MDT_Groups, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
      bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned g = 0; g < count; g++ )
      {
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
         }

         Model::Group    * grp = Model::Group::get();

         uint16_t flags;
         uint32_t triCount;
         uint8_t  smoothness;
         uint32_t materialIndex;
         char name[1024];

         bin_read( flags, bufPos, readLength );
                        flags = ltoh_u16(flags);

         strncpy( name, (char *) bufPos, sizeof(name));
         name[ sizeof(name) - 1 ] = '\0';
         unsigned nameSize = strlen(name) + 1;
         bufPos     += nameSize;
         readLength -= nameSize;

         bin_read( triCount, bufPos, readLength );
                        triCount = ltoh_u32(triCount);

         for ( unsigned t = 0; t < triCount; t++ )
         {
            uint32_t triIndex = 0;
            bin_read( triIndex, bufPos, readLength );
                                triIndex = ltoh_u32(triIndex);
            grp->m_triangleIndices.push_back( triIndex );
         }

         bin_read( smoothness, bufPos, readLength );
         bin_read( materialIndex, bufPos, readLength );
                        materialIndex = ltoh_u32(materialIndex);

         grp->m_name = name;
         grp->m_smooth = smoothness;
         grp->m_selected = ((flags & MF_SELECTED) == MF_SELECTED);
         grp->m_visible  = ((flags & MF_HIDDEN) != MF_HIDDEN);
         grp->m_materialIndex = materialIndex;

         modelGroups.push_back( grp );
      }
   }

   // External Textures
   std::vector< std::string > texNames;
   if ( _offsetIncluded( MDT_ExtTextures, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_ExtTextures, offsetList );
      uint32_t offset   = _offsetGet( MDT_ExtTextures, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
      bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned t = 0; t < count; t++ )
      {
         log_debug( "reading external texture %d/%d\n", t, count );
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
            log_debug( "read size %d\n", size );
         }

         uint16_t flags;
         char filename[1024];

         bin_read( flags, bufPos, readLength );
                        flags = ltoh_u16(flags);

         strncpy( filename, (char *) bufPos, sizeof(filename));
         filename[ sizeof(filename) - 1 ] = '\0';
         unsigned nameSize = strlen(filename) + 1;
         bufPos     += nameSize;
         readLength -= nameSize;
         replaceBackslash( filename );
         log_debug( "  filename is %s\n", filename );

         std::string fullpath = getAbsolutePath( modelPath.c_str(), filename );

         texNames.push_back( fullpath );
      }
   }

   // Materials
   if ( _offsetIncluded( MDT_Materials, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_Materials, offsetList );
      uint32_t offset   = _offsetGet( MDT_Materials, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
      bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned m = 0; m < count; m++ )
      {
         log_debug( "reading material %d/%d\n", m, count );
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
            log_debug( "read size %d\n", size );
         }

         uint16_t flags = 0;
         uint32_t texIndex = 0;
         char name[1024];

         Model::Material * mat = Model::Material::get();

         bin_read( flags, bufPos, readLength );
                        flags = ltoh_u16(flags);
         bin_read( texIndex, bufPos, readLength );
                        texIndex = ltoh_u32(texIndex);

         strncpy( name, (char *) bufPos, sizeof(name));
         name[ sizeof(name) - 1 ] = '\0';
         unsigned nameSize = strlen(name) + 1;
         bufPos     += nameSize;
         readLength -= nameSize;

         log_debug( "  material name: %s\n", name );

         mat->m_name = name;
         switch ( flags & 0x0f )
         {
            case 0:
               log_debug( "  got external texture %d\n", texIndex );
               mat->m_type = Model::Material::MATTYPE_TEXTURE;
               if ( texIndex < texNames.size() )
               {
                  mat->m_filename = texNames[texIndex];
               }
               else
               {
                  mat->m_filename = "";
               }
               break;
            case 13:
               mat->m_type = Model::Material::MATTYPE_COLOR;
               mat->m_filename = "";
               memset( mat->m_color, 255, sizeof( mat->m_color ) );
               break;
            case 14:
               mat->m_type = Model::Material::MATTYPE_GRADIENT;
               mat->m_filename = "";
               memset( mat->m_color, 255, sizeof( mat->m_color ) );
               break;
            case 15:
               mat->m_type = Model::Material::MATTYPE_BLANK;
               mat->m_filename = "";
               memset( mat->m_color, 255, sizeof( mat->m_color ) );
               break;
            default:
               log_debug( "  got unknown material type\n", texIndex );
               mat->m_type = Model::Material::MATTYPE_BLANK;
               mat->m_filename = "";
               memset( mat->m_color, 255, sizeof( mat->m_color ) );
               break;
         }

         mat->m_sClamp = ( (flags & MF_MAT_CLAMP_S) != 0 );
         mat->m_tClamp = ( (flags & MF_MAT_CLAMP_T) != 0 );

         unsigned i = 0;
         float32_t lightProp = 0;
         for ( i = 0; i < 4; i++ )
         {
            bin_read( lightProp, bufPos, readLength );
                                lightProp = ltoh_float(lightProp);
            mat->m_ambient[i] = lightProp;
         }
         for ( i = 0; i < 4; i++ )
         {
            bin_read( lightProp, bufPos, readLength );
                                lightProp = ltoh_float(lightProp);
            mat->m_diffuse[i] = lightProp;
         }
         for ( i = 0; i < 4; i++ )
         {
            bin_read( lightProp, bufPos, readLength );
                                lightProp = ltoh_float(lightProp);
            mat->m_specular[i] = lightProp;
         }
         for ( i = 0; i < 4; i++ )
         {
            bin_read( lightProp, bufPos, readLength );
                                lightProp = ltoh_float(lightProp);
            mat->m_emissive[i] = lightProp;
         }
         bin_read( lightProp, bufPos, readLength );
                        lightProp = ltoh_float(lightProp);
         mat->m_shininess = lightProp;

         modelMaterials.push_back( mat );
      }
   }

   // Texture coordinates
   if ( _offsetIncluded( MDT_TexCoords, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_TexCoords, offsetList );
      uint32_t offset   = _offsetGet( MDT_TexCoords, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
                bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned c = 0; c < count; c++ )
      {
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
            log_debug( "read size %d\n", size );
         }

         MM3DFILE_TexCoordT * tc = (MM3DFILE_TexCoordT *) bufPos;
                        uint32_t triIndex = ltoh_u32(tc->triangleIndex);

         if ( triIndex < modelTriangles.size() )
         {
            for ( unsigned v = 0; v < 3; v++ )
            {
               modelTriangles[triIndex]->m_s[v] = ltoh_float(tc->sCoord[v]);
               modelTriangles[triIndex]->m_t[v] = ltoh_float(tc->tCoord[v]);
            }
         }

         bufPos     += size;
         readLength -= size;
      }
   }

   // Canvas Background Images
   if ( _offsetIncluded( MDT_CanvasBackgrounds, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_CanvasBackgrounds, offsetList );
      uint32_t offset   = _offsetGet( MDT_CanvasBackgrounds, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
                bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned g = 0; g < count; g++ )
      {
         log_debug( "reading canvas background %d/%d\n", g, count );
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
            log_debug( "read size %d\n", size );
         }

         MM3DFILE_CanvasBackgroundT cb;
         bin_read( cb, bufPos, readLength );
                        cb.flags = ltoh_u16(cb.flags);
                        cb.scale = ltoh_float(cb.scale);
                        cb.center[0] = ltoh_float(cb.center[0]);
                        cb.center[1] = ltoh_float(cb.center[1]);
                        cb.center[2] = ltoh_float(cb.center[2]);

         char name[1024];

         strncpy( name, (char *) bufPos, sizeof(name));
         name[ sizeof(name) - 1 ] = '\0';
         unsigned nameSize = strlen(name) + 1;
         bufPos     += nameSize;
         readLength -= nameSize;
         replaceBackslash( name );
         std::string fileStr = getAbsolutePath( modelPath.c_str(), name );

         model->setBackgroundImage( cb.viewIndex, fileStr.c_str() );
         model->setBackgroundScale( cb.viewIndex, cb.scale );
         model->setBackgroundCenter( cb.viewIndex, 
               cb.center[0], cb.center[1], cb.center[2] );
      }
   }

   // Joints
   if ( _offsetIncluded( MDT_Joints, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_Joints, offsetList );
      uint32_t offset   = _offsetGet( MDT_Joints, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
                bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned j = 0; j < count; j++ )
      {
         log_debug( "reading joint %d/%d\n", j, count );
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
            log_debug( "read size %d\n", size );
         }

         MM3DFILE_JointT * fileJoint = (MM3DFILE_JointT *) bufPos;
         Model::Joint    * joint = Model::Joint::get();

         fileJoint->name[ sizeof(fileJoint->name) - 1 ] = '\0';

         joint->m_name = fileJoint->name;
         joint->m_parent = ltoh_32(fileJoint->parentIndex);
         for ( unsigned i = 0; i < 3; i++ )
         {
            joint->m_localRotation[i] = ltoh_float(fileJoint->localRot[i]);
            joint->m_localTranslation[i] = ltoh_float(fileJoint->localTrans[i]);
         }
                        uint16_t jointFlags = ltoh_u16(fileJoint->flags);
         joint->m_selected = ((jointFlags & MF_SELECTED) == MF_SELECTED);
         joint->m_visible  = ((jointFlags & MF_HIDDEN) != MF_HIDDEN);

         modelJoints.push_back( joint );

         bufPos     += size;
         readLength -= size;
      }

      log_debug( "read %d joints\n" );
   }

   // Joint Vertices
   if ( _offsetIncluded( MDT_JointVertices, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_JointVertices, offsetList );
      uint32_t offset   = _offsetGet( MDT_JointVertices, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
                bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned t = 0; t < count; t++ )
      {
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
            log_debug( "read size %d\n", size );
         }

         MM3DFILE_JointVertexT * fileJv = (MM3DFILE_JointVertexT *) bufPos;
                        uint32_t vertexIndex = ltoh_u32(fileJv->vertexIndex);
                        int32_t jointIndex = ltoh_32(fileJv->jointIndex);
                        
         if ( vertexIndex < modelVertices.size() && (unsigned) jointIndex < modelJoints.size() )
         {
            modelVertices[ vertexIndex ]->m_boneId = jointIndex;
         }
         else
         {
            missingElements = true;
            log_error( "vertex %d or joint %d out of range\n", vertexIndex, jointIndex );
         }

         bufPos     += size;
         readLength -= size;
      }

      log_debug( "read %d joints vertices\n", count );
   }

   // Smooth Angles
   if ( _offsetIncluded( MDT_SmoothAngles, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_SmoothAngles, offsetList );
      uint32_t offset   = _offsetGet( MDT_SmoothAngles, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
                bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned t = 0; t < count; t++ )
      {
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
            log_debug( "read size %d\n", size );
         }

         MM3DFILE_SmoothAngleT fileSa;
         fileSa.groupIndex = 0;
         fileSa.angle = 180;
         bin_read( fileSa, bufPos, readLength );
                        fileSa.groupIndex = ltoh_u32(fileSa.groupIndex);

         if ( fileSa.angle > 180 )
         {
            fileSa.angle = 180;
         }
         if ( fileSa.groupIndex < modelGroups.size() )
         {
            model->setGroupAngle( fileSa.groupIndex, fileSa.angle );
         }
      }

      log_debug( "read %d group smoothness angles\n", count );
   }

   // Skeletal Animations
   if ( _offsetIncluded( MDT_SkelAnims, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_SkelAnims, offsetList );
      uint32_t offset   = _offsetGet( MDT_SkelAnims, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
                bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned s = 0; s < count; s++ )
      {
         log_debug( "reading skel anim %d/%d\n", s, count );
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
            log_debug( "read size %d\n", size );
         }

         uint16_t  flags;
         char      name[1024];
         float32_t fps;
         uint32_t  frameCount;

         bin_read( flags, bufPos, readLength );
                        flags = ltoh_u16(flags);

         strncpy( name, (char *) bufPos, sizeof(name));
         name[ sizeof(name) - 1 ] = '\0';
         unsigned nameSize = strlen(name) + 1;
         bufPos     += nameSize;
         readLength -= nameSize;
         log_debug( "  name is %s\n", name );

         bin_read( fps, bufPos, readLength );
                        fps = ltoh_float(fps);
         bin_read( frameCount, bufPos, readLength );
                        frameCount = ltoh_u32(frameCount);

         unsigned anim = model->addAnimation( Model::ANIMMODE_SKELETAL, name );
         model->setAnimFPS( Model::ANIMMODE_SKELETAL, anim, fps );
         model->setAnimFrameCount( Model::ANIMMODE_SKELETAL, anim, frameCount );

         for ( unsigned f = 0; f < frameCount; f++ )
         {
            uint32_t keyframeCount;
            bin_read( keyframeCount, bufPos, readLength );
                                keyframeCount = ltoh_u32(keyframeCount);

            for ( unsigned k = 0; k < keyframeCount; k++ )
            {
               MM3DFILE_SkelKeyframeT fileKf;
               bin_read( fileKf, bufPos, readLength );
                                        fileKf.jointIndex = ltoh_u32(fileKf.jointIndex);
                                        fileKf.param[0] = ltoh_float(fileKf.param[0]);
                                        fileKf.param[1] = ltoh_float(fileKf.param[1]);
                                        fileKf.param[2] = ltoh_float(fileKf.param[2]);

               model->setSkelAnimKeyframe( anim, f, 
                     fileKf.jointIndex, (fileKf.keyframeType ? false : true), 
                     fileKf.param[0], fileKf.param[1], fileKf.param[2] );
            }
         }
      }
   }

   // Frame Animations
   if ( _offsetIncluded( MDT_FrameAnims, offsetList ) )
   {
      bool     variable = _offsetIsVariable( MDT_FrameAnims, offsetList );
      uint32_t offset   = _offsetGet( MDT_FrameAnims, offsetList );

      bufPos     = &fileBuf[offset];
      readLength = fileLength - offset;

      uint16_t flags = 0;
      uint32_t count = 0;
      bin_read( flags, bufPos, readLength );
                flags = ltoh_u16(flags);
                bin_read( count, bufPos, readLength );
                count = ltoh_u32(count);

      uint32_t size = 0;
      if ( !variable )
      {
         bin_read( size, bufPos, readLength );
                        size = ltoh_u32(size);
         log_debug( "read size %d\n", size );
      }

      for ( unsigned a = 0; a < count; a++ )
      {
         if ( variable )
         {
            bin_read( size, bufPos, readLength );
                                size = ltoh_u32(size);
         }

         uint16_t  flags;
         char      name[1024];
         float32_t fps;
         uint32_t  frameCount;

         bin_read( flags, bufPos, readLength );
                        flags = ltoh_u16(flags);

         strncpy( name, (char *) bufPos, sizeof(name));
         name[ sizeof(name) - 1 ] = '\0';
         unsigned nameSize = strlen(name) + 1;
         bufPos     += nameSize;
         readLength -= nameSize;

         bin_read( fps, bufPos, readLength );
                        fps = ltoh_float(fps);
         bin_read( frameCount, bufPos, readLength );
                        frameCount = ltoh_u32(frameCount);

         unsigned anim = model->addAnimation( Model::ANIMMODE_FRAME, name );
         model->setAnimFPS( Model::ANIMMODE_FRAME, anim, fps );
         model->setAnimFrameCount( Model::ANIMMODE_FRAME, anim, frameCount );

         for ( unsigned f = 0; f < frameCount; f++ )
         {
            unsigned maxVertex = 0;
            float32_t coord[3];
            for ( unsigned v = 0; v < modelVertices.size(); v++ )
            {
               for ( unsigned i = 0; i < 3; i++ )
               {
                  bin_read( coord[i], bufPos, readLength );
                                                coord[i] = ltoh_float(coord[i]);
               }
               model->setFrameAnimVertexCoords( anim, f, v,
                     coord[0], coord[1], coord[2] );
               maxVertex = v;
            }

            maxVertex = maxVertex + 1;
            if ( (maxVertex) != modelVertices.size() )
            {
               missingElements = true;
               log_error( "Vertex count for frame animation %d, frame %d has %d vertices, should be %d\n", anim, f, maxVertex, modelVertices.size() );
            }
         }
      }
   }

   // Read unknown data
   UnknownDataList::iterator it;
   for ( it = unknownList.begin(); it != unknownList.end(); it++ )
   {
      Model::FormatData * fd = new Model::FormatData;
      fd->offsetType = (*it).offsetType;
      bufPos = &fileBuf[ (*it).offsetValue ];

      fd->data = new uint8_t[ (*it).dataLen ];
      fd->len  = (*it).dataLen;
      memcpy( fd->data, bufPos, (*it).dataLen );

      if ( model->addFormatData( fd ) < 0 )
      {
         delete fd;
      }
   }

   // Account for missing elements (vertices, triangles, textures, joints)
   {
      unsigned vcount = modelVertices.size();
      unsigned tcount = modelTriangles.size();
      unsigned gcount = modelGroups.size();
      unsigned jcount = modelJoints.size();
      unsigned mcount = modelMaterials.size();

      for ( unsigned v = 0; v < vcount; v++ )
      {
         if ( modelVertices[v]->m_boneId >= (signed) jcount )
         {
            missingElements = true;
            log_error( "Vertex %d uses missing bone joint %d\n", v, modelVertices[v]->m_boneId );
         }
      }

      for ( unsigned t = 0; t < tcount; t++ )
      {
         for ( unsigned i = 0; i < 3; i++ )
         {
            if ( modelTriangles[t]->m_vertexIndices[i] >= vcount )
            {
               missingElements = true;
               log_error( "Triangle %d uses missing vertex %d\n", t, modelTriangles[t]->m_vertexIndices[i] );
            }
         }
      }

      for ( unsigned g = 0; g < gcount; g++ )
      {
         unsigned count = modelGroups[g]->m_triangleIndices.size();
         for ( unsigned i = 0; i < count; i++ )
         {
            if ( modelGroups[g]->m_triangleIndices[i] >= (signed) tcount )
            {
               missingElements = true;
               log_error( "Group %d uses missing triangle %d\n", g, modelGroups[g]->m_triangleIndices[i] );
            }
         }

         if ( modelGroups[g]->m_materialIndex >= (signed) mcount )
         {
            missingElements = true;
            log_error( "Group %d uses missing texture %d\n", g, modelGroups[g]->m_materialIndex );
         }
      }

      for ( unsigned j = 0; j < jcount; j++ )
      {
         if ( modelJoints[j]->m_parent >= (signed) jcount )
         {
            log_warning( "Joint %d has bad parent joint, checking endianness\n", j );
            if ( htonl( modelJoints[j]->m_parent ) < jcount )
            {
               modelJoints[j]->m_parent = htonl( modelJoints[j]->m_parent );
            }
            else if ( htol_u32( modelJoints[j]->m_parent ) < jcount )
            {
               modelJoints[j]->m_parent = htol_u32( modelJoints[j]->m_parent );
            }
            else
            {
               missingElements = true;
               log_error( "Joint %d has missing parent joint %d\n", j, modelJoints[j]->m_parent );
            }
         }
      }
   }

   delete[] fileBuf;

   if ( missingElements )
   {
      log_warning( "missing elements in file\n" );
      return Model::ERROR_BAD_DATA;
   }
   else
   {
      model->setupJoints();
      return Model::ERROR_NONE;
   }
}

Model::ModelError MisfitFilter::writeFile( Model * model, const char * const filename, ModelFilter::Options * o  )
{
   if ( filename == NULL || filename[0] == '\0' )
   {
      return Model::ERROR_BAD_ARGUMENT;
   }

   FILE * fp = fopen( filename, "wb" );

   if ( fp == NULL )
   {
      switch ( errno )
      {
         case EACCES:
         case EPERM:
            return Model::ERROR_NO_ACCESS;
         case ENOENT:
            return Model::ERROR_NO_FILE;
         case EISDIR:
            return Model::ERROR_BAD_DATA;
         default:
            return Model::ERROR_FILE_OPEN;
      }
   }

   string modelPath = "";
   string modelBaseName = "";
   string modelFullName = "";

   normalizePath( filename, modelFullName, modelPath, modelBaseName );
      
   bool doWrite[ MDT_MAX ];
   unsigned t = 0;

   for ( t = 0; t < MDT_MAX; t++ )
   {
      doWrite[t] = false;
   }

   // Get model data

   vector<Model::Vertex *>      & modelVertices     = getVertexList( model );
   vector<Model::Triangle *>    & modelTriangles    = getTriangleList( model );
   vector<Model::Group *>       & modelGroups       = getGroupList( model );
   vector<Model::Material *>    & modelMaterials    = getMaterialList( model );
   vector<Model::Joint *>       & modelJoints       = getJointList( model );
   vector<Model::SkelAnim *>    & modelSkels        = getSkelList( model );
   vector<Model::FrameAnim *>   & modelFrames       = getFrameList( model );

   int backgrounds = 0;
   for ( t = 0; t < 6; t++ )
   {
      const char * file =  model->getBackgroundImage( t );
      if ( file[0] != '\0' )
      {
         backgrounds++;
      }
   }

   // Find out what sections we need to write
   doWrite[ MDT_Meta ]              = (model->getMetaDataCount() > 0);
   doWrite[ MDT_Vertices ]          = (modelVertices.size() > 0);
   doWrite[ MDT_Triangles ]         = (modelTriangles.size() > 0);
   doWrite[ MDT_TriangleNormals ]   = (modelTriangles.size() > 0);
   doWrite[ MDT_Groups ]            = (modelGroups.size() > 0);
   doWrite[ MDT_Materials ]         = (modelMaterials.size() > 0);
   doWrite[ MDT_ExtTextures ]       =  doWrite[ MDT_Materials ];
   doWrite[ MDT_TexCoords ]         =  doWrite[ MDT_Materials ];
   doWrite[ MDT_CanvasBackgrounds ] = (backgrounds > 0);
   doWrite[ MDT_Joints ]            = (modelJoints.size() > 0);
   doWrite[ MDT_JointVertices ]     =  doWrite[ MDT_Joints ];
   doWrite[ MDT_SmoothAngles ]      = (modelGroups.size() > 0);
   doWrite[ MDT_SkelAnims ]         = (model->getAnimCount( Model::ANIMMODE_SKELETAL ) > 0);
   doWrite[ MDT_FrameAnims ]        = (model->getAnimCount( Model::ANIMMODE_FRAME ) > 0);
   doWrite[ MDT_RelativeAnims ]     = (model->getAnimCount( Model::ANIMMODE_FRAMERELATIVE ) > 0);

   uint8_t modelFlags = 0x00;

   // Write header
   MisfitOffsetList offsetList;

   _addOffset( MDT_Meta,          doWrite[ MDT_Meta ],          offsetList );
   _addOffset( MDT_Vertices,      doWrite[ MDT_Vertices ],      offsetList );
   _addOffset( MDT_Triangles,     doWrite[ MDT_Triangles ],     offsetList );
   _addOffset( MDT_TriangleNormals, doWrite[ MDT_TriangleNormals ],     offsetList );
   _addOffset( MDT_Groups,        doWrite[ MDT_Groups ],        offsetList );
   _addOffset( MDT_Materials,     doWrite[ MDT_Materials ],     offsetList );
   _addOffset( MDT_ExtTextures,   doWrite[ MDT_ExtTextures ],   offsetList );
   _addOffset( MDT_TexCoords,     doWrite[ MDT_TexCoords ],     offsetList );
   _addOffset( MDT_CanvasBackgrounds, doWrite[ MDT_CanvasBackgrounds ], offsetList );
   _addOffset( MDT_Joints,        doWrite[ MDT_Joints ],        offsetList );
   _addOffset( MDT_JointVertices, doWrite[ MDT_JointVertices ], offsetList );
   _addOffset( MDT_SmoothAngles,  doWrite[ MDT_SmoothAngles ],  offsetList );
   _addOffset( MDT_SkelAnims,     doWrite[ MDT_SkelAnims ],     offsetList );
   _addOffset( MDT_FrameAnims,    doWrite[ MDT_FrameAnims ],    offsetList );

   unsigned formatDataCount = model->getFormatDataCount();
   unsigned f = 0;

   for ( f = 0; f < formatDataCount; f++ )
   {
      Model::FormatData * fd = model->getFormatData( f );
      if ( fd->offsetType != 0 )
      {
         MisfitOffsetT mo;
         mo.offsetType = (fd->offsetType | OFFSET_DIRTY_MASK);
         mo.offsetValue = 0;
         offsetList.push_back( mo );
         log_warning( "adding uknown data type %04x\n", mo.offsetType );
      }
   }

   _addOffset( MDT_EndOfFile, true, offsetList );

   uint8_t offsetCount = (uint8_t) offsetList.size();

   fwrite( MAGIC, strlen(MAGIC), 1, fp );
   fwrite( &WRITE_VERSION_MAJOR, sizeof( WRITE_VERSION_MAJOR ), 1, fp );
   fwrite( &WRITE_VERSION_MINOR, sizeof( WRITE_VERSION_MINOR ), 1, fp );
   fwrite( &modelFlags, sizeof( modelFlags ), 1, fp );
   fwrite( &offsetCount, sizeof( offsetCount ), 1, fp );

   for ( t = 0; t < offsetCount; t++ )
   {
      MisfitOffsetT & mo = offsetList[t];
                uint16_t offsetType = htol_u16(mo.offsetType);
      fwrite( &offsetType, sizeof(offsetType), 1, fp );
                uint32_t offsetValue = htol_32(mo.offsetValue);
      fwrite( &offsetValue, sizeof(offsetValue), 1, fp );
   }
   log_debug( "wrote %d offsets\n", offsetCount );

   // Write data

   // Meta data
   if ( doWrite[ MDT_Meta ] )
   {
      _setOffset( MDT_Meta, ftell(fp), offsetList );
      _setUniformOffset( MDT_Meta, false, offsetList );

      unsigned count = model->getMetaDataCount();

      uint16_t flags = htol_u16(0x0000);
      uint32_t c = htol_u32(count);
      
      fwrite( &flags, sizeof(flags), 1, fp );
      fwrite( &c, sizeof(c), 1, fp );

      for ( unsigned m = 0; m < count; m++ )
      {
         char key[1024];
         char value[1024];

         model->getMetaData( m, key, sizeof(key), value, sizeof(value) );

         unsigned keyLen = strlen( key ) + 1;
         unsigned valueLen = strlen( value ) + 1;

         uint32_t s = htol_u32(keyLen + valueLen);
         fwrite( &s, sizeof(s), 1, fp );

         fwrite( key, keyLen, 1, fp );
         fwrite( value, valueLen, 1, fp );
      }
      log_debug( "wrote %d meta data pairs\n", count );
   }

   // Vertices
   if ( doWrite[ MDT_Vertices ] )
   {
      _setOffset( MDT_Vertices, ftell(fp), offsetList );
      _setUniformOffset( MDT_Vertices, true, offsetList );

      unsigned count = modelVertices.size();

      MM3DFILE_DataHeaderBT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      dhdr.size  = htol_u32(sizeof( MM3DFILE_VertexT ));
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      for ( unsigned v = 0; v < count; v++ )
      {
         MM3DFILE_VertexT fileVertex;

         fileVertex.flags  = 0x0000;
         fileVertex.flags |= (modelVertices[v]->m_visible)  ? 0 : MF_HIDDEN;
         fileVertex.flags |= (modelVertices[v]->m_selected) ? MF_SELECTED : 0;
                        fileVertex.flags = htol_u16(fileVertex.flags);

         fileVertex.coord[0] = htol_float(modelVertices[v]->m_coord[0]);
         fileVertex.coord[1] = htol_float(modelVertices[v]->m_coord[1]);
         fileVertex.coord[2] = htol_float(modelVertices[v]->m_coord[2]);

         fwrite( &fileVertex, sizeof(fileVertex), 1, fp );
      }
      log_debug( "wrote %d vertices\n", count );
   }

   // Triangles
   if ( doWrite[ MDT_Triangles ] )
   {
      _setOffset( MDT_Triangles, ftell(fp), offsetList );
      _setUniformOffset( MDT_Triangles, true, offsetList );

      unsigned count = modelTriangles.size();

      MM3DFILE_DataHeaderBT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      dhdr.size  = htol_u32(sizeof( MM3DFILE_TriangleT ));
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      for ( unsigned t = 0; t < count; t++ )
      {
         MM3DFILE_TriangleT fileTriangle;

         fileTriangle.flags = 0x0000;
         fileTriangle.flags |= (modelTriangles[t]->m_visible)  ? 0 : MF_HIDDEN;
         fileTriangle.flags |= (modelTriangles[t]->m_selected) ? MF_SELECTED : 0;
                        fileTriangle.flags = htol_u16(fileTriangle.flags);

         fileTriangle.vertex[0] = htol_u32(modelTriangles[t]->m_vertexIndices[0]);
         fileTriangle.vertex[1] = htol_u32(modelTriangles[t]->m_vertexIndices[1]);
         fileTriangle.vertex[2] = htol_u32(modelTriangles[t]->m_vertexIndices[2]);

         fwrite( &fileTriangle, sizeof(fileTriangle), 1, fp );
      }
      log_debug( "wrote %d triangles\n", count );
   }

   // Triangles
   if ( doWrite[ MDT_TriangleNormals ] )
   {
      _setOffset( MDT_TriangleNormals, ftell(fp), offsetList );
      _setUniformOffset( MDT_TriangleNormals, true, offsetList );

      unsigned count = modelTriangles.size();

      MM3DFILE_DataHeaderBT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      dhdr.size  = htol_u32(sizeof( MM3DFILE_TriangleNormalsT ));
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      for ( unsigned t = 0; t < count; t++ )
      {
         MM3DFILE_TriangleNormalsT fileNormals;

         fileNormals.flags = 0x0000;
         fileNormals.index = t;

         for ( unsigned v = 0; v < 3; v++ )
         {
            fileNormals.normal[v][0] = htol_float(modelTriangles[t]->m_vertexNormals[v][0]);
            fileNormals.normal[v][1] = htol_float(modelTriangles[t]->m_vertexNormals[v][1]);
            fileNormals.normal[v][2] = htol_float(modelTriangles[t]->m_vertexNormals[v][2]);
         }

         fwrite( &fileNormals, sizeof(fileNormals), 1, fp );
      }
      log_debug( "wrote %d triangle normals\n", count );
   }

   // Groups
   if ( doWrite[ MDT_Groups ] )
   {
      _setOffset( MDT_Groups, ftell(fp), offsetList );
      _setUniformOffset( MDT_Groups, false, offsetList );

      unsigned count = modelGroups.size();

      MM3DFILE_DataHeaderAT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      unsigned baseSize = sizeof(uint16_t) + sizeof(uint32_t)
         + sizeof(uint8_t) + sizeof(uint32_t);

      for ( unsigned g = 0; g < count; g++ )
      {
         Model::Group * grp = modelGroups[g];
         unsigned groupSize = baseSize + grp->m_name.length() + 1 
            + (grp->m_triangleIndices.size() * sizeof(uint32_t));

         uint16_t flags = 0x0000;
         flags |= (modelGroups[g]->m_visible)  ? 0 : MF_HIDDEN;
         flags |= (modelGroups[g]->m_selected) ? MF_SELECTED : 0;
         uint32_t triCount = grp->m_triangleIndices.size();

                        groupSize = htol_u32(groupSize);
         fwrite( &groupSize, sizeof(groupSize), 1, fp );
                        flags = ltoh_u16(flags);
         fwrite( &flags, sizeof(flags), 1, fp );
         fwrite( grp->m_name.c_str(), grp->m_name.length() + 1, 1, fp );
                        uint32_t temp32 = htol_u32(triCount);
         fwrite( &temp32, sizeof(triCount), 1, fp );
         for ( unsigned t = 0; t < triCount; t++ )
         {
            uint32_t triIndex = htol_u32(grp->m_triangleIndices[t]);
            fwrite( &triIndex, sizeof(triIndex), 1, fp );
         }
         uint8_t  smoothness = grp->m_smooth;
         uint32_t materialIndex = htol_u32(grp->m_materialIndex);
         fwrite( &smoothness, sizeof(smoothness), 1, fp );
         fwrite( &materialIndex, sizeof(materialIndex), 1, fp );

      }
      log_debug( "wrote %d groups\n", count );
   }

   int texNum = 0;

   // Materials
   if ( doWrite[ MDT_Materials ] )
   {
      _setOffset( MDT_Materials, ftell(fp), offsetList );
      _setUniformOffset( MDT_Materials, false, offsetList );

      unsigned count = modelMaterials.size();

      MM3DFILE_DataHeaderAT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      unsigned baseSize = sizeof(uint16_t) + sizeof(uint32_t)
         + (sizeof(float32_t) * 17);

      for ( unsigned m = 0; m < count; m++ )
      {
         Model::Material * mat = modelMaterials[m];
         unsigned matSize = baseSize + mat->m_name.length() + 1;

         uint16_t flags = 0x0000;
         uint32_t texIndex = texNum;  // TODO deal with embedded textures

         switch ( mat->m_type )
         {
            case Model::Material::MATTYPE_COLOR:
               flags = 0x000d;
               break;
            case Model::Material::MATTYPE_GRADIENT:
               flags = 0x000e;
               break;
            case Model::Material::MATTYPE_BLANK:
               flags = 0x000f;
               break;
            case Model::Material::MATTYPE_TEXTURE:
               flags = 0x0000;
               texNum++;
               break;
            default:
               break;
         }

         if ( mat->m_sClamp )
         {
            flags |= MF_MAT_CLAMP_S;
         }
         if ( mat->m_tClamp )
         {
            flags |= MF_MAT_CLAMP_T;
         }

                        matSize = htol_u32(matSize);
         fwrite( &matSize, sizeof(matSize), 1, fp );
                        flags = htol_u16(flags);
         fwrite( &flags, sizeof(flags), 1, fp );
                        texIndex = htol_u32(texIndex);
         fwrite( &texIndex, sizeof(texIndex), 1, fp );
         fwrite( mat->m_name.c_str(), mat->m_name.length() + 1, 1, fp );

         unsigned i = 0;
         float32_t lightProp = 0;
         for ( i = 0; i < 4; i++ )
         {
            lightProp = htol_float(mat->m_ambient[i]);
            fwrite( &lightProp, sizeof(lightProp), 1, fp );
         }
         for ( i = 0; i < 4; i++ )
         {
            lightProp = htol_float(mat->m_diffuse[i]);
            fwrite( &lightProp, sizeof(lightProp), 1, fp );
         }
         for ( i = 0; i < 4; i++ )
         {
            lightProp = htol_float(mat->m_specular[i]);
            fwrite( &lightProp, sizeof(lightProp), 1, fp );
         }
         for ( i = 0; i < 4; i++ )
         {
            lightProp = htol_float(mat->m_emissive[i]);
            fwrite( &lightProp, sizeof(lightProp), 1, fp );
         }
         lightProp = htol_float(mat->m_shininess);
         fwrite( &lightProp, sizeof(lightProp), 1, fp );

      }
      log_debug( "wrote %d materials with %d internal textures\n", count, texNum );
   }

   // External Textures
   if ( doWrite[ MDT_ExtTextures ] )
   {
      _setOffset( MDT_ExtTextures, ftell(fp), offsetList );
      _setUniformOffset( MDT_ExtTextures, false, offsetList );

      unsigned count = modelMaterials.size();

      MM3DFILE_DataHeaderAT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      unsigned baseSize = sizeof(uint16_t);

      for ( unsigned m = 0; m < count; m++ )
      {
         Model::Material * mat = modelMaterials[m];
         if ( mat->m_type == Model::Material::MATTYPE_TEXTURE )
         {
            std::string fileStr = getRelativePath( modelPath.c_str(), mat->m_filename.c_str() );

            char filename[PATH_MAX];
            strncpy( filename, fileStr.c_str(), PATH_MAX );
            filename[PATH_MAX - 1] = '\0';
            replaceSlash( filename );

            unsigned texSize = htol_u32(baseSize + strlen(filename) + 1);

            uint16_t flags = htol_u16(0x0000);

            fwrite( &texSize, sizeof(texSize), 1, fp );
            fwrite( &flags, sizeof(flags), 1, fp );
            fwrite( filename, strlen(filename) + 1, 1, fp );

            log_debug( "material file is %s\n", filename );
         }
      }
      log_debug( "wrote %d external textures\n", texNum );
   }

   // Texture Coordinates
   // TODO only write texture coordinates for triangles that are grouped
   if ( doWrite[ MDT_TexCoords ] )
   {
      _setOffset( MDT_TexCoords, ftell(fp), offsetList );
      _setUniformOffset( MDT_TexCoords, true, offsetList );

      unsigned count = modelTriangles.size();

      MM3DFILE_DataHeaderBT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      dhdr.size  = htol_u32(sizeof( MM3DFILE_TexCoordT ));
      
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      for ( unsigned t = 0; t < count; t++ )
      {
         Model::Triangle * tri = modelTriangles[t];
         MM3DFILE_TexCoordT tc;

         tc.flags = htol_u16(0x0000);
         tc.triangleIndex = htol_u32(t);
         for ( unsigned v = 0; v < 3; v++ )
         {
            tc.sCoord[v] = htol_float(tri->m_s[v]);
            tc.tCoord[v] = htol_float(tri->m_t[v]);
         }

         fwrite( &tc, sizeof(tc), 1, fp );
      }
      log_debug( "wrote %d texture coordinates\n", count );
   }

   // Canvas Backgrounds
   if ( doWrite[ MDT_CanvasBackgrounds ] )
   {
      _setOffset( MDT_CanvasBackgrounds, ftell(fp), offsetList );
      _setUniformOffset( MDT_CanvasBackgrounds, false, offsetList );

      unsigned count = backgrounds;

      MM3DFILE_DataHeaderAT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      unsigned baseSize = sizeof( MM3DFILE_CanvasBackgroundT );

      for ( unsigned b = 0; b < 6; b++ )
      {
         const char * file = model->getBackgroundImage( b );

         if ( file[0] != '\0' )
         {
            MM3DFILE_CanvasBackgroundT cb;
            cb.flags = htol_u16(0x0000);
            cb.viewIndex = b;

            cb.scale = htol_float(model->getBackgroundScale( b ));
            model->getBackgroundCenter( b, cb.center[0], cb.center[1], cb.center[2] );
                                cb.center[0] = htol_float(cb.center[0]);
                                cb.center[1] = htol_float(cb.center[1]);
                                cb.center[2] = htol_float(cb.center[2]);

            std::string fileStr = getRelativePath( modelPath.c_str(), file );
            unsigned backSize = htol_u32(baseSize + fileStr.size() + 1);

            char * filedup = strdup( fileStr.c_str() );
            replaceSlash( filedup );

            fwrite( &backSize, sizeof(backSize), 1, fp );
            fwrite( &cb, sizeof(cb), 1, fp );
            fwrite( filedup, fileStr.size() + 1, 1, fp );

            free( filedup );
         }
      }
      log_debug( "wrote %d canvas backgrounds\n", count );
   }

   // Joints
   if ( doWrite[ MDT_Joints ] )
   {
      _setOffset( MDT_Joints, ftell(fp), offsetList );
      _setUniformOffset( MDT_Joints, true, offsetList );

      unsigned count = modelJoints.size();

      MM3DFILE_DataHeaderBT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      dhdr.size  = htol_u32(sizeof( MM3DFILE_JointT ));
                
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      for ( unsigned j = 0; j < count; j++ )
      {
         MM3DFILE_JointT fileJoint;
         Model::Joint * joint = modelJoints[j];

         fileJoint.flags = 0x0000;
         fileJoint.flags |= (modelJoints[j]->m_visible)  ? 0 : MF_HIDDEN;
         fileJoint.flags |= (modelJoints[j]->m_selected) ? MF_SELECTED : 0;
                        fileJoint.flags = htol_u16(fileJoint.flags);

         strncpy( fileJoint.name, joint->m_name.c_str(), sizeof(fileJoint.name) );
         fileJoint.name[ sizeof(fileJoint.name) - 1] = '\0';

         fileJoint.parentIndex = htol_u32(joint->m_parent);
         for ( unsigned i = 0; i < 3; i++ )
         {
            fileJoint.localRot[i] = htol_float(joint->m_localRotation[i]);
            fileJoint.localTrans[i] = htol_float(joint->m_localTranslation[i]);
         }

         fwrite( &fileJoint, sizeof(fileJoint), 1, fp );
      }
      log_debug( "wrote %d joints\n", count );
   }

   // Joint Vertices
   if ( doWrite[ MDT_JointVertices ] )
   {
      _setOffset( MDT_JointVertices, ftell(fp), offsetList );
      _setUniformOffset( MDT_JointVertices, true, offsetList );

      unsigned count = 0;
      unsigned vcount = modelVertices.size();

      unsigned v = 0;
      for ( v = 0; v < vcount; v++ )
      {
         if ( modelVertices[v]->m_boneId >= 0 )
         {
            count++;
         }
      }

      MM3DFILE_DataHeaderBT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      dhdr.size  = htol_u32(sizeof( MM3DFILE_JointVertexT ));
                
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      for ( v = 0; v < vcount; v++ )
      {
         if ( modelVertices[v]->m_boneId >= 0 )
         {
            MM3DFILE_JointVertexT fileJv;
            fileJv.vertexIndex = htol_u32(v);
            fileJv.jointIndex = htol_32(modelVertices[v]->m_boneId);

            fwrite( &fileJv, sizeof(fileJv), 1, fp );
         }
      }
      log_debug( "wrote %d joint vertex assignments\n", count );
   }

   // Smooth Angles
   if ( doWrite[ MDT_SmoothAngles ] )
   {
      _setOffset( MDT_SmoothAngles, ftell(fp), offsetList );
      _setUniformOffset( MDT_SmoothAngles, true, offsetList );

      unsigned count = modelGroups.size();

      MM3DFILE_DataHeaderBT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      dhdr.size  = htol_u32(sizeof( MM3DFILE_SmoothAngleT ));
                
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      for ( unsigned t = 0; t < count; t++ )
      {
         MM3DFILE_SmoothAngleT fileSa;
         fileSa.groupIndex = htol_u32(t);
         fileSa.angle = model->getGroupAngle( t );

         fwrite( &fileSa, sizeof(fileSa), 1, fp );
      }
      log_debug( "wrote %d group smoothness angles\n", count );
   }

   // Skel Anims
   if ( doWrite[ MDT_SkelAnims ] )
   {
      _setOffset( MDT_SkelAnims, ftell(fp), offsetList );
      _setUniformOffset( MDT_SkelAnims, false, offsetList );

      unsigned count = modelSkels.size();

      MM3DFILE_DataHeaderAT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      unsigned baseSize = sizeof(uint16_t) + sizeof(float32_t) + sizeof(uint32_t);

      for ( unsigned s = 0; s < count; s++ )
      {
         Model::SkelAnim * sa = modelSkels[s];
         unsigned animSize = baseSize + sa->m_name.length() + 1;

         uint32_t frameCount = sa->m_frameCount;
         uint32_t keyframeCount = 0;
         unsigned f = 0;
         for ( f = 0; f < frameCount; f++ )
         {
            for ( unsigned j = 0; j < sa->m_jointKeyframes.size(); j++ )
            {
               for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
               {
                  if ( sa->m_jointKeyframes[j][k]->m_frame == f )
                  {
                     keyframeCount++;
                  }
               }
            }
         }

         animSize += frameCount    * sizeof(uint32_t);
         animSize += keyframeCount * sizeof(MM3DFILE_SkelKeyframeT);

         uint16_t  flags = htol_u16(0x0000);
         float32_t fps = htol_float(sa->m_fps);

                        animSize = htol_u32(animSize);
         fwrite( &animSize, sizeof(animSize), 1, fp );
         fwrite( &flags, sizeof(flags), 1, fp );
         fwrite( sa->m_name.c_str(), sa->m_name.length() + 1, 1, fp );
         fwrite( &fps, sizeof(fps), 1, fp );
                        uint32_t temp32 = htol_u32(frameCount);
         fwrite( &temp32, sizeof(frameCount), 1, fp );

         for ( f = 0; f < frameCount; f++ )
         {
            keyframeCount = 0;
            unsigned j = 0;
            for ( j = 0; j < sa->m_jointKeyframes.size(); j++ )
            {
               for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
               {
                  if ( sa->m_jointKeyframes[j][k]->m_frame == f )
                  {
                     keyframeCount++;
                  }
               }
            }

                                temp32 = htol_u32(keyframeCount);
            fwrite( &temp32, sizeof(keyframeCount), 1, fp );

            unsigned written = 0;
            for ( j = 0; j < sa->m_jointKeyframes.size(); j++ )
            {
               for ( unsigned k = 0; k < sa->m_jointKeyframes[j].size(); k++ )
               {
                  Model::Keyframe * kf = sa->m_jointKeyframes[j][k];
                  if ( kf->m_frame == f )
                  {
                     MM3DFILE_SkelKeyframeT fileKf;
                     fileKf.jointIndex = htol_u32(j);
                     fileKf.keyframeType = kf->m_isRotation ? 0 : 1;
                     fileKf.param[0] = htol_float(kf->m_parameter[0]);
                     fileKf.param[1] = htol_float(kf->m_parameter[1]);
                     fileKf.param[2] = htol_float(kf->m_parameter[2]);

                     fwrite( &fileKf, sizeof(fileKf), 1, fp );
                     written++;
                  }
               }
            }
         }
      }
      log_debug( "wrote %d skel anims\n", count );
   }

   // Frame Anims
   if ( doWrite[ MDT_FrameAnims ] )
   {
      _setOffset( MDT_FrameAnims, ftell(fp), offsetList );
      _setUniformOffset( MDT_FrameAnims, false, offsetList );

      unsigned count = modelFrames.size();

      MM3DFILE_DataHeaderAT dhdr;
      dhdr.flags = htol_u16(0x0000);
      dhdr.count = htol_u32(count);
      
      fwrite( &dhdr, sizeof(dhdr), 1, fp );

      unsigned baseSize = sizeof(uint16_t) + sizeof(float32_t) + sizeof(uint32_t);

      for ( unsigned a = 0; a < count; a++ )
      {
         Model::FrameAnim * fa = modelFrames[a];
         unsigned animSize = baseSize + fa->m_name.length() + 1;

         uint32_t frameCount = fa->m_frameVertices.size();

         animSize += frameCount * sizeof(uint32_t);
         animSize += frameCount * modelVertices.size() * sizeof(float32_t) * 3;

         uint16_t  flags = htol_u16(0x0000);
         float32_t fps = htol_float(fa->m_fps);

                        animSize = htol_u32(animSize);
         fwrite( &animSize, sizeof(animSize), 1, fp );
         fwrite( &flags, sizeof(flags), 1, fp );
         fwrite( fa->m_name.c_str(), fa->m_name.length() + 1, 1, fp );
         fwrite( &fps, sizeof(fps), 1, fp );
                        uint32_t temp32 = htol_u32(frameCount);
         fwrite( &temp32, sizeof(frameCount), 1, fp );

         for ( unsigned f = 0; f < frameCount; f++ )
         {
            Model::FrameAnimVertexList * list = fa->m_frameVertices[f];
            for ( unsigned v = 0; v < list->size(); v++ )
            {
               for ( unsigned i = 0; i < 3; i++ )
               {
                  float32_t coord = htol_float((*list)[v]->m_coord[i]);
                  fwrite( &coord, sizeof(coord), 1, fp );
               }
            }
         }
      }
      log_debug( "wrote %d skel anims\n", count );
   }

   // Write unknown data (add dirty flag to offset type)
   for ( f = 0; f < formatDataCount; f++ )
   {
      Model::FormatData * fd = model->getFormatData( f );
      uint16_t thisType = (fd->offsetType | OFFSET_DIRTY_MASK);
      if ( fd->offsetType != 0 )
      {
         MisfitOffsetList::iterator it;
         for ( it = offsetList.begin(); it != offsetList.end(); it++ )
         {
            if ( (*it).offsetType == thisType )
            {
               (*it).offsetValue = ftell( fp );
               log_warning( "setting uknown data type %04x offset at %08x\n", (*it).offsetType, (*it).offsetValue );
               fwrite( fd->data, fd->len, 1, fp );
               break;
            }
         }
      }
   }

   // Re-write header with offsets

   _setOffset( MDT_EndOfFile, ftell(fp), offsetList );

   fseek( fp, 12, SEEK_SET );
   for ( t = 0; t < offsetCount; t++ )
   {
      MisfitOffsetT & mo = offsetList[t];
                uint16_t offsetType = htol_u16(mo.offsetType);
      fwrite( &offsetType, sizeof(offsetType), 1, fp );
                uint32_t offsetValue = htol_u32(mo.offsetValue);
      fwrite( &offsetValue, sizeof(offsetValue), 1, fp );
   }
   log_debug( "wrote %d updated offsets\n", offsetCount );

   fclose( fp );

   return Model::ERROR_NONE;
}

bool MisfitFilter::canRead( const char * filename )
{
   return true;
}

bool MisfitFilter::canWrite( const char * filename )
{
   return true;
}

bool MisfitFilter::isSupported( const char * filename )
{
   unsigned len = strlen( filename );

   if ( len >= 5 && strcasecmp( &filename[len-5], ".mm3d" ) == 0 )
   {
      return true;
   }
   else
   {
      return false;
   }
}

list< string > MisfitFilter::getReadTypes()
{
   list<string> rval;
   rval.push_back( "*.mm3d" );
   return rval;
}

list< string > MisfitFilter::getWriteTypes()
{
   list<string> rval;
   rval.push_back( "*.mm3d" );
   return rval;
}

