/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "texwidget.h"

#include "texture.h"
#include "model.h"
#include "log.h"
#include "mm3dport.h"
#include <math.h>

#include "mq3compat.h"
#include <qtimer.h>

#ifdef HAVE_QT4
#include <QMouseEvent>
#endif // HAVE_QT4

using std::vector;
using std::list;

TextureWidget::TextureWidget( QWidget * parent, const char * name )
   : QGLWidget( parent, name ),
     m_sClamp( false ),
     m_tClamp( false ),
     m_materialId( -1 ),
     m_texture( NULL ),
     m_glTexture( 0 ),
     m_textureCount( 1 ),
     m_selecting( false ),
     m_drawBounding( false ),
     m_3d( false ),
     m_animTimer( new QTimer() )
{
   connect( m_animTimer, SIGNAL(timeout()), this, SLOT(animationTimeout()));
}

TextureWidget::~TextureWidget()
{
   m_animTimer->stop();
   delete m_animTimer;
   m_animTimer = NULL;

   makeCurrent();
   glDeleteTextures( 1, (GLuint *) &m_glTexture );
   // Do NOT free m_texture.  TextureManager does that.
}

void TextureWidget::initializeGL()
{
   glEnable( GL_TEXTURE_2D );

   setAutoBufferSwap( false );

   glShadeModel( GL_SMOOTH );
   glDepthFunc( GL_LEQUAL );
   glClearColor( 0.80, 0.80, 0.80, 1.0 );
   glClearDepth( 1.0f );

   glGenTextures( 1, &m_glTexture );

   GLfloat ambient[]  = {  0.8f,  0.8f,  0.8f,  1.0f };
   GLfloat diffuse[]  = {  1.0f,  1.0f,  1.0f,  1.0f };
   GLfloat position[] = {  0.0f,  0.0f,  3.0f,  0.0f };

   glLightModeli( GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE );
   glLightfv( GL_LIGHT0, GL_AMBIENT, ambient );
   glLightfv( GL_LIGHT0, GL_DIFFUSE, diffuse );
   glLightfv( GL_LIGHT0, GL_POSITION, position );

   glEnable( GL_LIGHT0 );
   glEnable( GL_LIGHTING );
}

void TextureWidget::resizeGL( int w, int h )
{
   if ( h == 0 )
      h = 1;

   m_viewportWidth  = w;
   m_viewportHeight = h;

   updateViewport();
}

void TextureWidget::updateViewport()
{
   makeCurrent();

   glViewport( 0, 0, ( GLint ) m_viewportWidth, ( GLint ) m_viewportHeight );

   glMatrixMode( GL_PROJECTION );
   glLoadIdentity( );

   GLfloat ratio = ( GLfloat ) m_viewportWidth / ( GLfloat ) m_viewportHeight;

   glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

   if ( m_3d )
   {
      gluPerspective( 45.0f, ratio, 0.01, 30.0 );
   }
   else
   {
      float x = 1.0;
      float y = x / ratio;
      glOrtho( m_xMin, m_xMax, m_yMin, m_yMax, -1.0, 1.0 );

      m_xOffset = x;
      m_yOffset = y;
   }

   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity( );

   updateGL();
}

void TextureWidget::paintGL()
{
   log_debug( "repainting texture widget\n" );

   makeCurrent();

   glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
   glLoadIdentity( );
   glEnable( GL_LIGHTING );

   if ( m_texture )
   {
      log_debug( "  texture is %s\n", m_texture->m_filename );
      glEnable( GL_TEXTURE_2D );
      glBindTexture( GL_TEXTURE_2D, m_glTexture );
   }
   else
   {
      log_debug( "  no texture\n" );
      glDisable( GL_TEXTURE_2D );
   }

   glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

   if ( m_materialId >= 0 )
   {
      log_debug( "  material %d\n", m_materialId );
      glColor3f( 1.0f, 1.0f, 1.0f );

      if ( m_model )
      {
         float fval[4] = { 0.0f, 0.0f, 0.0f, 0.0f };

         m_model->getTextureAmbient( m_materialId, fval );
         glMaterialfv( GL_FRONT, GL_AMBIENT,
               fval );
         m_model->getTextureDiffuse( m_materialId, fval );
         glMaterialfv( GL_FRONT, GL_DIFFUSE,
               fval );
         m_model->getTextureSpecular( m_materialId, fval );
         glMaterialfv( GL_FRONT, GL_SPECULAR,
               fval );
         m_model->getTextureEmissive( m_materialId, fval );
         glMaterialfv( GL_FRONT, GL_EMISSION,
               fval );
         m_model->getTextureShininess( m_materialId, fval[0] );
         glMaterialf( GL_FRONT, GL_SHININESS,
               fval[0] );
      }
      else
      {
         float fval[4] = { 0.2f, 0.2f, 0.2f, 1.0f };
         glMaterialfv( GL_FRONT, GL_AMBIENT,
               fval );

         fval[0] = fval[1] = fval[2] = 1.0f;
         glMaterialfv( GL_FRONT, GL_DIFFUSE,
               fval );

         fval[0] = fval[1] = fval[2] = 0.0f;
         glMaterialfv( GL_FRONT, GL_SPECULAR,
               fval );

         fval[0] = fval[1] = fval[2] = 0.0f;
         glMaterialfv( GL_FRONT, GL_EMISSION,
               fval );

         glMaterialf( GL_FRONT, GL_SHININESS,
               0.0f );
      }

   }
   else
   {
      float fval[4] = { 0.2f, 0.2f, 0.2f, 1.0f };
      glMaterialfv( GL_FRONT, GL_AMBIENT,
            fval );

      fval[0] = fval[1] = fval[2] = 1.0f;
      glMaterialfv( GL_FRONT, GL_DIFFUSE,
            fval );

      fval[0] = fval[1] = fval[2] = 0.0f;
      glMaterialfv( GL_FRONT, GL_SPECULAR,
            fval );

      fval[0] = fval[1] = fval[2] = 0.0f;
      glMaterialfv( GL_FRONT, GL_EMISSION,
            fval );

      glMaterialf( GL_FRONT, GL_SHININESS,
            0.0f );

      if ( m_texture )
      {
         glColor3f( 1.0f, 1.0f, 1.0f );
      }
      else
      {
         log_debug( "  nothing to draw\n" );
         glColor3f( 0.0f, 0.0f, 0.0f );
      }
   }

   if ( m_materialId >= 0 && m_model && m_model->getMaterialType( m_materialId ) == Model::Material::MATTYPE_COLOR )
   {
      GLubyte r = m_model->getMaterialColor( m_materialId, 0 );
      GLubyte g = m_model->getMaterialColor( m_materialId, 1 );
      GLubyte b = m_model->getMaterialColor( m_materialId, 2 );
      log_debug( "drawing with color %d %d %d\n", r, g, b );
      glDisable( GL_TEXTURE_2D );
      //glDisable( GL_LIGHTING );
      glColor3ub( r, g, b );
   }

   if ( m_materialId >= 0 && m_3d )
   {
      glEnable( GL_DEPTH_TEST );

      PORT_timeval tv;

      PORT_gettimeofday( &tv );
      int ms = tv.tv_msec + (tv.tv_sec & 7) * 1000;

      float yRot = (float) ms / 4000.0 * 360.0;
      float xRot = (float) ms / 8000.0 * 360.0;
      glTranslatef( 0.0, 0.0, -5.0 );
      glRotatef( yRot, 0.0, 1.0, 0.0 );
      glRotatef( xRot, 1.0, 0.0, 0.0 );

      glBegin( GL_QUADS );

      // Front
      glTexCoord2f( 0.0, 0.0 );
      glNormal3f( 0.0, 0.0, 1.0 );
      glVertex3f( -1.0, -1.0, 1.0 );

      glTexCoord2f( 1.0, 0.0 );
      glNormal3f( 0.0, 0.0, 1.0 );
      glVertex3f(  1.0, -1.0, 1.0 );

      glTexCoord2f( 1.0, 1.0 );
      glNormal3f( 0.0, 0.0, 1.0 );
      glVertex3f(  1.0,  1.0, 1.0 );

      glTexCoord2f( 0.0, 1.0 );
      glNormal3f( 0.0, 0.0, 1.0 );
      glVertex3f( -1.0,  1.0, 1.0 );

      // Back
      glTexCoord2f( 0.0, 0.0 );
      glNormal3f( 0.0, 0.0, -1.0 );
      glVertex3f(  1.0, -1.0, -1.0 );

      glTexCoord2f( 1.0, 0.0 );
      glNormal3f( 0.0, 0.0, -1.0 );
      glVertex3f( -1.0, -1.0, -1.0 );

      glTexCoord2f( 1.0, 1.0 );
      glNormal3f( 0.0, 0.0, -1.0 );
      glVertex3f( -1.0,  1.0, -1.0 );

      glTexCoord2f( 0.0, 1.0 );
      glNormal3f( 0.0, 0.0, -1.0 );
      glVertex3f(  1.0,  1.0, -1.0 );

      // Left
      glTexCoord2f( 0.0, 0.0 );
      glNormal3f( 1.0, 0.0, 0.0 );
      glVertex3f(  1.0, -1.0, 1.0 );

      glTexCoord2f( 1.0, 0.0 );
      glNormal3f( 1.0, 0.0, 0.0 );
      glVertex3f( 1.0, -1.0, -1.0 );

      glTexCoord2f( 1.0, 1.0 );
      glNormal3f( 1.0, 0.0, 0.0 );
      glVertex3f( 1.0,  1.0, -1.0 );

      glTexCoord2f( 0.0, 1.0 );
      glNormal3f( 1.0, 0.0, 0.0 );
      glVertex3f(  1.0,  1.0, 1.0 );

      // Right
      glTexCoord2f( 0.0, 0.0 );
      glNormal3f( -1.0, 0.0, 0.0 );
      glVertex3f( -1.0, -1.0, -1.0 );

      glTexCoord2f( 1.0, 0.0 );
      glNormal3f( -1.0, 0.0, 0.0 );
      glVertex3f( -1.0, -1.0, 1.0 );

      glTexCoord2f( 1.0, 1.0 );
      glNormal3f( -1.0, 0.0, 0.0 );
      glVertex3f( -1.0,  1.0, 1.0 );

      glTexCoord2f( 0.0, 1.0 );
      glNormal3f( -1.0, 0.0, 0.0 );
      glVertex3f( -1.0,  1.0, -1.0 );

      // Top
      glTexCoord2f( 0.0, 0.0 );
      glNormal3f( 0.0, 1.0, 0.0 );
      glVertex3f( -1.0, 1.0, -1.0 );

      glTexCoord2f( 1.0, 0.0 );
      glNormal3f( 0.0, 1.0, 0.0 );
      glVertex3f(  1.0, 1.0, -1.0 );

      glTexCoord2f( 1.0, 1.0 );
      glNormal3f( 0.0, 1.0, 0.0 );
      glVertex3f(  1.0,  1.0, 1.0 );

      glTexCoord2f( 0.0, 1.0 );
      glNormal3f( 0.0, 1.0, 0.0 );
      glVertex3f( -1.0,  1.0, 1.0 );

      // Bottom
      glTexCoord2f( 0.0, 0.0 );
      glNormal3f( 0.0, -1.0, 0.0 );
      glVertex3f( 1.0, -1.0, -1.0 );

      glTexCoord2f( 1.0, 0.0 );
      glNormal3f( 0.0, -1.0, 0.0 );
      glVertex3f( -1.0, -1.0, -1.0 );

      glTexCoord2f( 1.0, 1.0 );
      glNormal3f( 0.0, -1.0, 0.0 );
      glVertex3f( -1.0, -1.0, 1.0 );

      glTexCoord2f( 0.0, 1.0 );
      glNormal3f( 0.0, -1.0, 0.0 );
      glVertex3f( 1.0, -1.0, 1.0 );

      glEnd();
      glDisable( GL_DEPTH_TEST );
   }
   else
   {
      glBegin( GL_QUADS );

      glTexCoord2f( m_xMin, m_yMin );
      glNormal3f( 0.0, 0.0, 1.0 );
      glVertex3f( m_xMin, m_yMin, 0.0 );

      glTexCoord2f( m_xMax, m_yMin );
      glNormal3f( 0.0, 0.0, 1.0 );
      glVertex3f(  m_xMax, m_yMin, 0.0 );

      glTexCoord2f( m_xMax, m_yMax );
      glNormal3f( 0.0, 0.0, 1.0 );
      glVertex3f(  m_xMax,  m_yMax, 0.0 );

      glTexCoord2f( m_xMin, m_yMax );
      glNormal3f( 0.0, 0.0, 1.0 );
      glVertex3f( m_xMin,  m_yMax, 0.0 );

      glEnd();
   }

   //glLoadIdentity( );
   glDisable( GL_TEXTURE_2D );
   glDisable( GL_LIGHTING );

   glColor3f( 1.0, 1.0, 1.0 );

   glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );

   glBegin( GL_TRIANGLES );

   for ( unsigned t = 0; t < m_triangles.size(); t++ )
   {
      TextureTriangle * triangle = m_triangles[t];

      for ( unsigned v = 0; v < 3; v++ )
      {
         glVertex3f( m_vertices[ triangle->vertex[v] ]->s, 
               m_vertices[ triangle->vertex[v] ]->t, 
               -0.5 );
      }
   }

   glEnd();

   glPointSize( 3.0 );

   glBegin( GL_POINTS );

   for ( unsigned t = 0; t < m_triangles.size(); t++ )
   {
      for ( unsigned v = 0; v < 3; v++ )
      {
         TextureVertex * vert = m_vertices[ m_triangles[t]->vertex[v] ];
         if ( vert->selected )
         {
            glColor3f( 1.0, 0.0, 0.0 );
         }
         else
         {
            glColor3f( 1.0, 1.0, 1.0 );
         }

         glVertex3f( vert->s, vert->t, -0.5 );
      }
   }

   glEnd();

   if ( m_drawBounding )
   {
      drawSelectBox();
   }

   swapBuffers();
}

void TextureWidget::animationTimeout()
{
   updateGL();
}

void TextureWidget::setModel( Model * model )
{
   m_model = model;
}

void TextureWidget::set3d( bool o )
{
   m_3d = o;

   updateViewport();

   if ( o )
   {
      m_animTimer->start( 30 );
   }
   else
   {
      m_animTimer->stop();
   }
}

void TextureWidget::setTexture( int materialId, Texture * texture )
{
   m_materialId = materialId;
   m_texture = texture;

   if ( m_texture )
   {
      makeCurrent();

      double w = (double) m_textureCount / 2.0;

      m_xMin = 0.5 - w;
      m_xMax = 0.5 + w;
      m_yMin = 0.5 - w;
      m_yMax = 0.5 + w;

      glBindTexture( GL_TEXTURE_2D, m_glTexture );

      glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
            GL_NEAREST );
      glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
            GL_NEAREST );

      glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, 
            (m_sClamp ? GL_CLAMP : GL_REPEAT) );
      glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, 
            (m_tClamp ? GL_CLAMP : GL_REPEAT) );

      GLuint format = m_texture->m_format == Texture::FORMAT_RGBA ? GL_RGBA : GL_RGB;
      gluBuild2DMipmaps( GL_TEXTURE_2D, format,
            m_texture->m_width, m_texture->m_height,
            format, GL_UNSIGNED_BYTE,
            m_texture->m_data );
   }
   else
   {
      if ( m_materialId >= 0 )
      {
         double w = (double) m_textureCount / 2.0;

         m_xMin = 0.5 - w;
         m_xMax = 0.5 + w;
         m_yMin = 0.5 - w;
         m_yMax = 0.5 + w;
      }
      else
      {
         m_xMin = 0.0;
         m_xMax = 1.0;
         m_yMin = 0.0;
         m_yMax = 1.0;
      }
   }

   resizeGL( this->width(), this->height() );
   updateGL();
}

int TextureWidget::addVertex( double s, double t )
{
   int index = m_vertices.size();

   TextureVertex * vert = new TextureVertex;
   vert->s = s;
   vert->t = t;
   vert->selected = true;
   m_vertices.push_back( vert );

   return index;
}

int TextureWidget::addTriangle( int v1, int v2, int v3 )
{
   int vCount = m_vertices.size();

   if (     v1 >= 0 && v1 < vCount
         && v2 >= 0 && v2 < vCount
         && v3 >= 0 && v3 < vCount )
   {
      int index = m_triangles.size();

      TextureTriangle * triangle = new TextureTriangle;
      triangle->vertex[0] = v1;
      triangle->vertex[1] = v2;
      triangle->vertex[2] = v3;
      m_triangles.push_back( triangle );

      return index;
   }

   return -1;
}

void TextureWidget::mousePressEvent( QMouseEvent * e )
{
   m_lastXPos = e->pos().x();
   m_lastYPos = e->pos().y();

   switch ( m_operation )
   {
      case MouseSelect:
         if ( !( (e->state() & ShiftButton) || (e->button() &  RightButton) ) )
         {
            clearSelected();
         }
         m_xSel1 =  (m_lastXPos / (double) this->width()) * (m_xMax - m_xMin) + m_xMin;
         m_ySel1 =  (1.0 - (m_lastYPos / (double) this->height())) * (m_yMax - m_yMin) + m_yMin;
         m_selecting = ( e->button() & RightButton ) ? false : true;
         m_drawBounding = true;
         break;
      case MouseMove:
         break;
      case MouseScale:
         startScale(
               (m_lastXPos / (double) this->width()) * (m_xMax - m_xMin) + m_xMin , 
               (1.0 - (m_lastYPos / (double) this->height())) * (m_yMax - m_yMin) + m_yMin );
         break;
      default:
         log_error( "Unknown mouse operation: %d\n", m_operation );
         break;
   }
}

void TextureWidget::mouseReleaseEvent( QMouseEvent * e )
{
   int x = e->pos().x();
   int y = e->pos().y();

   switch ( m_operation )
   {
      case MouseSelect:
         m_drawBounding = false;
         updateSelectRegion( 
               (x / (double) this->width()) * (m_xMax - m_xMin) + m_xMin, 
               (1.0 - (y / (double) this->height())) * (m_yMax - m_yMin) + m_yMin );
         selectDone();
         break;
      case MouseMove:
         moveSelectedVertices( 
               ((x - m_lastXPos) / (double) this->width()) * (m_xMax - m_xMin), 
               (-(y - m_lastYPos) / (double) this->height()) * (m_yMax - m_yMin) );
         emit updateCoordinatesSignal();
         break;
      case MouseScale:
         break;
      default:
         log_error( "Unknown mouse operation: %d\n", m_operation );
         break;
   }
}

void TextureWidget::mouseMoveEvent( QMouseEvent * e )
{
   int x = e->pos().x();
   int y = e->pos().y();

   switch ( m_operation )
   {
      case MouseSelect:
         /*
         updateSelectRegion( 
               x / (double) this->width(), 
               1.0 - (y / (double) this->height()) );
               */
         updateSelectRegion( 
               (x / (double) this->width()) * (m_xMax - m_xMin) + m_xMin, 
               (1.0 - (y / (double) this->height())) * (m_yMax - m_yMin) + m_yMin );
         break;
      case MouseMove:
         moveSelectedVertices( 
               ((x - m_lastXPos) / (double) this->width()) * (m_xMax - m_xMin), 
               (-(y - m_lastYPos) / (double) this->height()) * (m_yMax - m_yMin) );
         emit updateCoordinatesSignal();
         break;
      case MouseScale:
         scaleSelectedVertices(
               (x / (double) this->width()) * (m_xMax - m_xMin) + m_xMin, 
               (1.0 - (y / (double) this->height())) * (m_yMax - m_yMin) + m_yMin );
         emit updateCoordinatesSignal();
         break;
      default:
         log_error( "Unknown mouse operation: %d\n", m_operation );
         break;
   }

   m_lastXPos = x;
   m_lastYPos = y;
}

void TextureWidget::moveSelectedVertices( double x, double y )
{
   for ( unsigned t = 0; t < m_vertices.size(); t++ )
   {
      if ( m_vertices[t]->selected )
      {
         m_vertices[t]->s += x;
         m_vertices[t]->t += y;
      }
   }
   updateGL();
}

void TextureWidget::updateSelectRegion( double x, double y )
{
   m_xSel2 = x;
   m_ySel2 = y;
   updateGL();
}

void TextureWidget::selectDone()
{
   if ( m_xSel1 > m_xSel2 )
   {
      double temp = m_xSel2;
      m_xSel2 = m_xSel1;
      m_xSel1 = temp;
   }
   if ( m_ySel1 > m_ySel2 )
   {
      double temp = m_ySel2;
      m_ySel2 = m_ySel1;
      m_ySel1 = temp;
   }
   for ( unsigned v = 0; v < m_vertices.size(); v++ )
   {
      if (     m_vertices[v]->s >= m_xSel1 && m_vertices[v]->s <= m_xSel2 
            && m_vertices[v]->t >= m_ySel1 && m_vertices[v]->t <= m_ySel2 )
      {
         m_vertices[v]->selected = m_selecting;
      }
   }
   updateGL();
}

void TextureWidget::setTextureCount( unsigned c )
{
   m_textureCount = c;
   setTexture( m_materialId, m_texture );
}

void TextureWidget::setMouseOperation( MouseOperation op )
{
   log_debug( "texture widget operation = %d\n", (int) op);
   m_operation = op;
}

void TextureWidget::drawSelectBox()
{
   glEnable( GL_COLOR_LOGIC_OP );
   glColor3f( 1.0, 1.0, 1.0 );
   glLogicOp( GL_XOR );
   glBegin( GL_LINES );

   glVertex3f( m_xSel1, m_ySel1, -0.75 );
   glVertex3f( m_xSel1, m_ySel2, -0.75 );

   glVertex3f( m_xSel1, m_ySel2, -0.75 );
   glVertex3f( m_xSel2, m_ySel2, -0.75 );

   glVertex3f( m_xSel2, m_ySel2, -0.75 );
   glVertex3f( m_xSel2, m_ySel1, -0.75 );

   glVertex3f( m_xSel2, m_ySel1, -0.75 );
   glVertex3f( m_xSel1, m_ySel1, -0.75 );

   glEnd();
   glLogicOp( GL_COPY );
   glDisable( GL_LOGIC_OP );
}

void TextureWidget::clearSelected()
{
   for ( unsigned v = 0; v < m_vertices.size(); v++ )
   {
      m_vertices[v]->selected = false;
   }
}

void TextureWidget::clearCoordinates()
{
   while ( m_triangles.size() )
   {
      delete m_triangles.back();
      m_triangles.pop_back();
   }

   while ( m_vertices.size() )
   {
      delete m_vertices.back();
      m_vertices.pop_back();
   }
}

void TextureWidget::getCoordinates( int tri, float * s, float * t )
{
   if ( !t || !s || tri >= (signed) m_triangles.size() )
   {
      return;
   }

   for ( int v = 0; v < 3; v++ )
   {
      s[v] = m_vertices[ m_triangles[ tri ]->vertex[v] ]->s;
      t[v] = m_vertices[ m_triangles[ tri ]->vertex[v] ]->t;
   }
}

void TextureWidget::startScale( double x, double y )
{
   log_debug( "starting scale at %f,%f\n", x, y );

   m_scaleList.clear();
   bool first = true;

   double minX = 0;
   double minY = 0;
   double maxX = 0;
   double maxY = 0;

   for ( unsigned t = 0; t < m_vertices.size(); t++ )
   {
      if ( m_vertices[t]->selected )
      {
         // update range
         if ( first )
         {
            minX = m_vertices[t]->s;
            minY = m_vertices[t]->t;
            maxX = m_vertices[t]->s;
            maxY = m_vertices[t]->t;

            first = false;
         }
         else
         {
            if ( m_vertices[t]->s < minX ) { minX = m_vertices[t]->s; };
            if ( m_vertices[t]->t < minY ) { minY = m_vertices[t]->t; };
            if ( m_vertices[t]->s > maxX ) { maxX = m_vertices[t]->s; };
            if ( m_vertices[t]->t > maxY ) { maxY = m_vertices[t]->t; };
         }

         ScaleVertices sv;
         sv.index = t;
         sv.x = m_vertices[t]->s;
         sv.y = m_vertices[t]->t;

         m_scaleList.push_back( sv );
      }
   }

   double minmin = distance( x, y, minX, minY );
   double minmax = distance( x, y, minX, maxY );
   double maxmin = distance( x, y, maxX, minY );
   double maxmax = distance( x, y, maxX, maxY );

   if ( minmin > minmax )
   {
      if ( minmin > maxmin )
      {
         if ( minmin > maxmax )
         {
            m_farX = minX;
            m_farY = minY;
         }
         else
         {
            m_farX = maxX;
            m_farY = maxY;
         }
      }
      else // maxmin > minmin
      {
         if ( maxmin > maxmax )
         {
            m_farX = maxX;
            m_farY = minY;
         }
         else
         {
            m_farX = maxX;
            m_farY = maxY;
         }
      }
   }
   else // minmax > minmin
   {
      if ( minmax > maxmin )
      {
         if ( minmax > maxmax )
         {
            m_farX = minX;
            m_farY = maxY;
         }
         else
         {
            m_farX = maxX;
            m_farY = maxY;
         }
      }
      else // maxmin > minmax
      {
         if ( maxmin > maxmax )
         {
            m_farX = maxX;
            m_farY = minY;
         }
         else
         {
            m_farX = maxX;
            m_farY = maxY;
         }
      }
   }

   m_startLengthX = fabs( x - m_farX );
   m_startLengthY = fabs( y - m_farY );
   log_debug( "far distance is %f,%f\n", m_startLengthX, m_startLengthY );
}

void TextureWidget::scaleSelectedVertices( double x, double y )
{
   log_debug( "scaling at %f,%f\n", x, y );

   double lengthX = fabs( x - m_farX );
   double lengthY = fabs( y - m_farY );

   log_debug( "far distance is %f,%f\n", lengthX, lengthY );

   ScaleVerticesList::iterator it;
   for( it = m_scaleList.begin(); it != m_scaleList.end(); it++ )
   {
      double x = (*it).x;
      double y = (*it).y;

      x -= m_farX;
      y -= m_farY;

      x *= (lengthX / m_startLengthX);
      y *= (lengthY / m_startLengthY);

      x += m_farX;
      y += m_farY;

      m_vertices[(*it).index]->s = x;
      m_vertices[(*it).index]->t = y;
   }

   updateGL();
}

double TextureWidget::distance( const double & x1, const double & y1, const double & x2, const double & y2 )
{
   double xDiff = x2 - x1;
   double yDiff = y2 - y1;
   return sqrt( xDiff*xDiff + yDiff*yDiff );
}

double TextureWidget::max( const double & a, const double & b )
{
   return ( a > b ) ? a : b;
}

