/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "model.h"
#include "toolbox.h"
#include "tool.h"
#include "glmath.h"
#include "decalmgr.h"
#include "decal.h"
#include "log.h"
#include "statusbar.h"
#include "modelstatus.h"
#include "texmgr.h"
#include "modelviewport.h"
#include "3dmprefs.h"
#include "mm3dport.h"

#include "mq3compat.h"

#include "pixmap/arrow.xpm"
#include "pixmap/crosshairrow.xpm"

#include <qfont.h>
#include <qtimer.h>
#include <math.h>
#include <stdarg.h>

#ifdef HAVE_QT4
#include <QMouseEvent>
#include <QWheelEvent>
#endif // HAVE_QT4

#define VP_ZOOMSCALE 0.75

#define MM3D_ENABLEALPHA

static int const SCROLL_SIZE =  16;

struct _ScrollButton_t
{
   int x;
   int y;
   int texIndex;
   float s1;
   float t1;
   float s2;
   float t2;
   float s3;
   float t3;
   float s4;
   float t4;
};
typedef struct _ScrollButton_t ScrollButtonT;

static ScrollButtonT s_buttons[ ModelViewport::ScrollButtonMAX ] =
{
   { -18, -18, 1,   0.0f, 0.0f,   1.0f, 0.0f,   1.0f, 1.0f,  0.0f, 1.0f  }, // Pan
   { -52, -18, 0,   0.0f, 1.0f,   0.0f, 0.0f,   1.0f, 0.0f,  1.0f, 1.0f  }, // Left
   { -35, -18, 0,   0.0f, 0.0f,   0.0f, 1.0f,   1.0f, 1.0f,  1.0f, 0.0f  }, // Right
   { -18, -35, 0,   0.0f, 0.0f,   1.0f, 0.0f,   1.0f, 1.0f,  0.0f, 1.0f  }, // Up
   { -18, -52, 0,   0.0f, 1.0f,   1.0f, 1.0f,   1.0f, 0.0f,  0.0f, 0.0f  }, // Down
};

static Matrix s_mat;

GLvoid glPrintf( int displayList, const char *fmt, ... )
{
#ifndef WIN32 
    if ( fmt )
    {
       char line[1024];
       va_list ap;

       va_start( ap, fmt );
       vsprintf( line, fmt, ap );
       va_end( ap );

       glPushAttrib( GL_LIST_BIT );

       glListBase( displayList - FONT_START );

       glCallLists( strlen( line ), GL_UNSIGNED_BYTE, line );

       glPopAttrib( );
    }
#endif
}

ModelViewport::ModelViewport( QWidget * parent, const char * name )
   : QGLWidget( parent, name ),
     m_model( NULL ),
     m_viewDirection( ViewFront ),
     m_centerX( 0.0 ),
     m_centerY( 0.0 ),
     m_rotX( 0.0 ),
     m_rotY( 0.0 ),
     m_rotZ( 0.0 ),
     m_zoomLevel( 32.0 ),
     m_far( 10000.0 ),
     m_near( 1.0 ),
     m_farOrtho( 1000000.0 ),
     m_nearOrtho( 0.01 ),
     m_viewportWidth( 0 ),
     m_viewportHeight( 0 ),
     m_scrollTimer( new QTimer() ),
     m_overlayButton( ScrollButtonMAX ),
     m_displayList( -1 ),
     m_capture( false ),
     m_texturesLoaded( false ),
     m_viewOptions( ViewTexture ),
     m_toolbox( NULL )
{
   m_glxFontData = GlxFont_Allocate();

   setAutoBufferSwap( false );

   setFocusPolicy( WheelFocus );
   setMinimumSize( 220, 180 );

   double rot[3] = { 45 * PIOVER180, 0, 0 };
   s_mat.setRotation( rot );

   m_backColor.setRgb( 130, 200, 200 );

   setAcceptDrops( true );
   setMouseTracking( true );

   QPixmap arrow( arrow_xpm );
   QPixmap cross( crosshairrow_xpm );

   QImage img;

   makeCurrent();

   glGenTextures( 2, m_scrollTextures );

   img = arrow.convertToImage();
   makeTextureFromImage( img, m_scrollTextures[0] );

   img = cross.convertToImage();
   makeTextureFromImage( img, m_scrollTextures[1] );

   connect( m_scrollTimer, SIGNAL(timeout()), this, SLOT(scrollTimeout()));
}

ModelViewport::~ModelViewport()
{
   log_debug( "deleting model viewport\n" );

   makeCurrent();

   glDeleteTextures( 1, &m_backgroundTexture );
   glDeleteTextures( 2, m_scrollTextures );

   freeTextures();

   if ( m_glxFontData )
   {
      GlxFont_Free( m_glxFontData );
      m_glxFontData = NULL;
   }
   if ( m_displayList >= 0 )
   {
      glDeleteLists( m_displayList, FONT_SIZE );
      m_displayList = -1;
   }

   delete m_scrollTimer;
}

void ModelViewport::freeTextures()
{
   log_debug( "freeing texture for viewport\n" );
   makeCurrent();
   if ( m_model )
   {
      m_model->removeContext( (unsigned) this );
   }
}

void ModelViewport::initializeGL()
{
   glShadeModel( GL_SMOOTH );
   glClearColor( m_backColor.red() / 256.0, 
         m_backColor.green() / 256.0, 
         m_backColor.blue() / 256.0, 1.0f );
   glClearDepth( 1.0f );
   glEnable( GL_DEPTH_TEST );
   glDepthFunc( GL_LEQUAL );
   glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

   GLfloat ambient[]  = {  0.8f,  0.8f,  0.8f,  1.0f };
   GLfloat diffuse[]  = {  0.9f,  0.9f,  0.9f,  1.0f };
   GLfloat position[] = {  0.0f,  0.0f,  1.0f,  0.0f };

   glLightModeli( GL_LIGHT_MODEL_LOCAL_VIEWER, GL_FALSE );
   glLightfv( GL_LIGHT0, GL_AMBIENT, ambient );
   glLightfv( GL_LIGHT0, GL_DIFFUSE, diffuse );
   glLightfv( GL_LIGHT0, GL_POSITION, position );
   //glEnable( GL_LIGHT0 );

   {
      GLint texSize = 0;
      glGetIntegerv( GL_MAX_TEXTURE_SIZE, &texSize );
      log_debug( "max texture size is %dx%d\n", texSize, texSize );
   }

   glGenTextures( 1, &m_backgroundTexture );

   checkGlErrors();

   {
#ifdef WIN32
#if 0
      log_debug( "getting windows font\n" );
      QFont        fixedFont( "fixed", 8 );
      HDC glFont = qt_display_dc();

      log_debug( "creating display list\n" );
      m_displayList = glGenLists(FONT_SIZE);
      if ( m_displayList != 0 )
      {
         log_debug( "setting up font bitmap\n" );
         wglUseFontBitmaps( glFont, FONT_START, FONT_SIZE, m_displayList );
      }
#endif // 0
#else
      log_debug( "getting linux font\n" );

      log_debug( "creating display list\n" );
      m_displayList = glGenLists(FONT_SIZE);
      if ( m_displayList != 0 )
      {
         log_debug( "setting up glx font\n" );
         m_glxFontData = GlxFont_Allocate();
         GlxFont_GetFont( m_glxFontData, m_displayList, "fixed" );
      }

      //fontInfo = XLoadQueryFont( dpy, "-*-Courier New-bold-i-normal-*-*-720-*-*-*-*-iso8859-1" );
#endif
   }

#ifdef MM3D_ENABLEALPHA
   glEnable( GL_BLEND );
   glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
   if ( ! glIsEnabled( GL_BLEND) )
   {
      log_warning( "alpha not supported\n" );
      glDisable( GL_BLEND );
      glGetError(); // clear errors
   }
#endif // MM3D_ENABLEALPHA
}

void ModelViewport::resizeGL( int w, int h )
{
   if ( h == 0 )
   {
      h = 1;
   }

   m_viewportWidth  = w;
   m_viewportHeight = h;

   adjustViewport();
}

void ModelViewport::paintGL()
{
   //LOG_PROFILE();

   if ( m_inOverlay )
   {
      setViewportDraw();
   }

   if ( m_capture )
   {
      glClearColor( 130.0 / 256.0, 200.0 / 256.0, 200.0 / 256.0, 1.0f );
   }

   float viewPoint[4] = { 0.0f, 0.0f, 0.0f, 1.0f };

   if ( m_viewDirection == View3d )
   {
      glEnable( GL_LIGHT0 );
      glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
      glLoadIdentity( );

      viewPoint[0] = m_centerX;
      viewPoint[1] = m_centerY;
      viewPoint[2] = (m_zoomLevel * 2.0);

      glTranslatef( -viewPoint[0], -viewPoint[1], -viewPoint[2] );
      glRotatef( m_rotZ, 0.0, 0.0, 1.0);
      glRotatef( m_rotY, 0.0, 1.0, 0.0);
      glRotatef( m_rotX, 1.0, 0.0, 0.0);

      Matrix m;
      m.setRotationInDegrees( 0.0f, 0.0f, -m_rotZ );
      m.apply( viewPoint );
      m.setRotationInDegrees( 0.0f, -m_rotY, 0.0f );
      m.apply( viewPoint );
      m.setRotationInDegrees( -m_rotX, 0.0f, 0.0f );
      m.apply( viewPoint );
   }
   else
   {
      glDisable( GL_LIGHT0 );
      glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
      glLoadIdentity( );

      viewPoint[0] = 0.0f;
      viewPoint[1] = 0.0f;
      viewPoint[2] = 500000.0f;

      glTranslatef( -viewPoint[0], -viewPoint[1], -viewPoint[2] );

      Matrix m;
      switch ( m_viewDirection )
      {
         case ViewFront:
            // do nothing
            break;
         case ViewBack:
            glRotatef( 180.0, 0.0, 1.0, 0.0);
            m.setRotationInDegrees( 0.0f, -180.0f, 0.0f );
            break;
         case ViewLeft:
            glRotatef( -90.0, 0.0, 1.0, 0.0);
            m.setRotationInDegrees( 0.0f, 90.0f, 0.0f );
            break;
         case ViewRight:
            glRotatef(  90.0, 0.0, 1.0, 0.0);
            m.setRotationInDegrees( 0.0f, -90.0f, 0.0f );
            break;
         case ViewTop:
            glRotatef(  90.0, 1.0, 0.0, 0.0);
            m.setRotationInDegrees( -90.0f, 0.0f, 0.0f );
            break;
         case ViewBottom:
            glRotatef( -90.0, 1.0, 0.0, 0.0);
            m.setRotationInDegrees( 90.0f, 0.0f, 0.0f );
            break;
         default:
            log_error( "Unknown ViewDirection: %d\n", m_viewDirection );
            swapBuffers();
            return;
            break;
      }

      m.apply( viewPoint );
   }

   if ( !m_capture )
   {
      drawGridLines();
   }

   if ( m_model )
   {
      glColor3f( 0.7, 0.7, 0.7 );
      if ( m_viewDirection == View3d )
      {
         glEnable( GL_LIGHTING );
         int opt = Model::DO_TEXTURE | Model::DO_SMOOTHING 
            | ( (g_prefs( "ui_render_bad_textures" ).intValue() == 0) ? 0 : Model::DO_BADTEX);

         bool drawSelections 
            = (g_prefs( "ui_render_3d_selections" ).intValue() == 0) 
            ? false : true;

         switch ( m_viewOptions )
         {
            case ViewWireframe:
               opt = Model::DO_WIREFRAME;
               drawSelections = true;
               break;
            case ViewFlat:
               opt = Model::DO_NONE;
               break;
            case ViewSmooth:
               opt = Model::DO_SMOOTHING;
               break;
            case ViewAlpha:
               opt = opt | Model::DO_ALPHA;
               break;
            default:
               break;
         }

         if ( drawSelections )
         {
            glDisable( GL_LIGHTING );
            m_model->drawLines();
            m_model->drawPoints();
         }

         if ( opt != Model::DO_WIREFRAME )
         {
            glEnable( GL_LIGHTING );
            m_model->draw( opt, (unsigned) this, viewPoint );

         }

         if ( drawSelections )
         {
            glDisable( GL_LIGHTING );
            glDisable( GL_DEPTH_TEST );
            m_model->drawJoints();
         }
      }
      else
      {
         // Draw background
         drawBackground();

         glClear( GL_DEPTH_BUFFER_BIT );

         m_model->drawLines();
         m_model->drawPoints();

         int drawMode = m_model->getCanvasDrawMode();
         if ( drawMode != ViewWireframe )
         {
            glEnable( GL_LIGHTING );
            glEnable( GL_LIGHT0 );
            int opt = Model::DO_TEXTURE | Model::DO_SMOOTHING 
               | ( (g_prefs( "ui_render_bad_textures" ).intValue() == 0) ? 0 : Model::DO_BADTEX);
            switch ( drawMode )
            {
               case ViewFlat:
                  opt = Model::DO_NONE;
                  break;
               case ViewSmooth:
                  opt = Model::DO_SMOOTHING;
                  break;
               case ViewAlpha:
                  opt = opt | Model::DO_ALPHA;
                  break;
               default:
                  break;
            }
            m_model->draw( opt, (unsigned) this, viewPoint );

            glDisable( GL_LIGHTING );
         }

         glDisable( GL_DEPTH_TEST );
         m_model->drawJoints();
         for ( DecalList::iterator it = m_decals.begin(); it != m_decals.end(); it++ )
         {
            (*it)->draw();
         }
         glEnable( GL_DEPTH_TEST );
      }
   }

   glDisable( GL_LIGHTING );
   glDisable( GL_TEXTURE_2D );

   if ( !m_capture )
   {
      drawOrigin();
   }

   if ( this->hasFocus() )
   {
      drawOverlay();
   }

   checkGlErrors();

   swapBuffers();
}

void ModelViewport::drawGridLines()
{
   if ( m_viewDirection == View3d )
   {
      double max = 25.0;
      double inc = 5.0;
      double x;
      double z;

      glColor3f( 0.55f, 0.55f, 0.55f );

      glBegin( GL_LINES );

      for ( x = -max; x <= max; x += inc )
      {
         glVertex3f( x, 0, -max );
         glVertex3f( x, 0, +max );
      }

      for ( z = -max; z <= max; z += inc )
      {
         glVertex3f( -max, 0, z );
         glVertex3f( +max, 0, z );
      }

      glEnd();
   }
   else
   {
      double unitWidth = 1.0;
      double maxDimension = (m_width > m_height) ? m_width : m_height;

      while ( (maxDimension / unitWidth) > 8 )
      {
         unitWidth *= 4.0;
      }
      while ( (maxDimension / unitWidth) < 2 )
      {
         unitWidth /= 4.0;
      }

      double xRangeMin = 0;
      double xRangeMax = 0;
      double yRangeMin = 0;
      double yRangeMax = 0;

      double xStart = 0;
      double yStart = 0;

      double x = 0;
      double y = 0;

      glColor3f( 0.55f, 0.55f, 0.55f );

      glBegin( GL_LINES );
      
      switch ( m_viewDirection )
      {
         case ViewFront:
            xRangeMin = m_centerX - (m_width / 2.0);
            xRangeMax = m_centerX + (m_width / 2.0);
            yRangeMin = m_centerY - (m_height / 2.0);
            yRangeMax = m_centerY + (m_height / 2.0);

            xStart = unitWidth * ((int) (xRangeMin / unitWidth));
            yStart = unitWidth * ((int) (yRangeMin / unitWidth));

            for ( x = xStart; x < xRangeMax; x += unitWidth )
            {
               glVertex3f( x, yRangeMin, 0 );
               glVertex3f( x, yRangeMax, 0 );
            }

            for ( y = yStart; y < yRangeMax; y += unitWidth )
            {
               glVertex3f( xRangeMin, y, 0 );
               glVertex3f( xRangeMax, y, 0 );
            }
            break;

         case ViewBack:
            xRangeMin = -m_centerX - (m_width / 2.0);
            xRangeMax = -m_centerX + (m_width / 2.0);
            yRangeMin = m_centerY - (m_height / 2.0);
            yRangeMax = m_centerY + (m_height / 2.0);

            xStart = unitWidth * ((int) (xRangeMin / unitWidth));
            yStart = unitWidth * ((int) (yRangeMin / unitWidth));

            for ( x = xStart; x < xRangeMax; x += unitWidth )
            {
               glVertex3f( x, yRangeMin, 0 );
               glVertex3f( x, yRangeMax, 0 );
            }

            for ( y = yStart; y < yRangeMax; y += unitWidth )
            {
               glVertex3f( xRangeMin, y, 0 );
               glVertex3f( xRangeMax, y, 0 );
            }
            break;

         case ViewLeft:
            xRangeMin = -m_centerX - (m_width / 2.0);
            xRangeMax = -m_centerX + (m_width / 2.0);
            yRangeMin = m_centerY - (m_height / 2.0);
            yRangeMax = m_centerY + (m_height / 2.0);

            xStart = unitWidth * ((int) (xRangeMin / unitWidth));
            yStart = unitWidth * ((int) (yRangeMin / unitWidth));

            for ( x = xStart; x < xRangeMax; x += unitWidth )
            {
               glVertex3f( 0, yRangeMin, x );
               glVertex3f( 0, yRangeMax, x );
            }

            for ( y = yStart; y < yRangeMax; y += unitWidth )
            {
               glVertex3f( 0, y, xRangeMin );
               glVertex3f( 0, y, xRangeMax );
            }
            break;

         case ViewRight:
            xRangeMin = m_centerX - (m_width / 2.0);
            xRangeMax = m_centerX + (m_width / 2.0);
            yRangeMin = m_centerY - (m_height / 2.0);
            yRangeMax = m_centerY + (m_height / 2.0);

            xStart = unitWidth * ((int) (xRangeMin / unitWidth));
            yStart = unitWidth * ((int) (yRangeMin / unitWidth));

            for ( x = xStart; x < xRangeMax; x += unitWidth )
            {
               glVertex3f( 0, yRangeMin, x );
               glVertex3f( 0, yRangeMax, x );
            }

            for ( y = yStart; y < yRangeMax; y += unitWidth )
            {
               glVertex3f( 0, y, xRangeMin );
               glVertex3f( 0, y, xRangeMax );
            }
            break;

         case ViewTop:
            xRangeMin = m_centerX - (m_width / 2.0);
            xRangeMax = m_centerX + (m_width / 2.0);
            yRangeMin = -m_centerY - (m_height / 2.0);
            yRangeMax = -m_centerY + (m_height / 2.0);

            xStart = unitWidth * ((int) (xRangeMin / unitWidth));
            yStart = unitWidth * ((int) (yRangeMin / unitWidth));

            for ( x = xStart; x < xRangeMax; x += unitWidth )
            {
               glVertex3f( x, 0, yRangeMin );
               glVertex3f( x, 0, yRangeMax );
            }

            for ( y = yStart; y < yRangeMax; y += unitWidth )
            {
               glVertex3f( xRangeMin, 0, y );
               glVertex3f( xRangeMax, 0, y );
            }
            break;

         case ViewBottom:
            xRangeMin = m_centerX - (m_width / 2.0);
            xRangeMax = m_centerX + (m_width / 2.0);
            yRangeMin = m_centerY - (m_height / 2.0);
            yRangeMax = m_centerY + (m_height / 2.0);

            xStart = unitWidth * ((int) (xRangeMin / unitWidth));
            yStart = unitWidth * ((int) (yRangeMin / unitWidth));

            for ( x = xStart; x < xRangeMax; x += unitWidth )
            {
               glVertex3f( x, 0, yRangeMin );
               glVertex3f( x, 0, yRangeMax );
            }

            for ( y = yStart; y < yRangeMax; y += unitWidth )
            {
               glVertex3f( xRangeMin, 0, y );
               glVertex3f( xRangeMax, 0, y );
            }
            break;

         default:
            log_error( "Unhandled view direction: %d\n", m_viewDirection );
            break;
      }

      glEnd();

      glColor3f( 0.35f, 0.35f, 0.35f );
      glRasterPos3f( xRangeMin, yRangeMin, 0.0f );
      switch ( m_viewDirection )
      {
         case ViewFront:
            break;

         case ViewBack:
            glRasterPos3f( xRangeMax, yRangeMin, 0.0f );
            break;

         case ViewLeft:
            glRasterPos3f( 0.0f, yRangeMin, xRangeMax );
            break;

         case ViewRight:
            glRasterPos3f( 0.0f, yRangeMin, xRangeMin );
            break;

         case ViewTop:
            glRasterPos3f( xRangeMin, 0.0f, yRangeMax );
            break;

         case ViewBottom:
            glRasterPos3f( xRangeMin, 0.0f, yRangeMin );
            break;

         default:
            log_error( "Unhandled view direction: %d\n", m_viewDirection );
            break;
      }
      glPrintf( m_displayList, "%g", unitWidth );
   }
}

void ModelViewport::drawOrigin()
{
   glDisable( GL_DEPTH_TEST );

   glBegin( GL_LINES );

   glColor3f( 1, 0, 0 );
   glVertex3f( 0, 0, 0 );
   glVertex3f( m_zoomLevel / 10, 0, 0 );
   glColor3f( 0, 1, 0 );
   glVertex3f( 0, 0, 0 );
   glVertex3f( 0, m_zoomLevel / 10, 0 );
   glColor3f( 0, 0, 1 );
   glVertex3f( 0, 0, 0 );
   glVertex3f( 0, 0, m_zoomLevel / 10 );

   glEnd();

   glEnable( GL_DEPTH_TEST );
}

void ModelViewport::drawBackground()
{
   glDisable( GL_LIGHTING );
   glColor3f( 1.0f, 1.0f, 1.0f );

   updateBackground();

   if ( m_backgroundFile[0] != '\0' )
   {
      if ( m_viewDirection != View3d )
      {
         int index = (int) m_viewDirection - 1;

         float cenX  = 0.0f;
         float cenY  = 0.0f;
         float cenZ  = 0.0f;

         float minX  = 0.0f;
         float minY  = 0.0f;
         float minZ  = 0.0f;
         float maxX  = 0.0f;
         float maxY  = 0.0f;
         float maxZ  = 0.0f;

         float normX = 0.0f;
         float normY = 0.0f;
         float normZ = 0.0f;

         float w = m_texture->m_origWidth;
         float h = m_texture->m_origHeight;
         float dimMax = w > h ? w : h;

         float scale  = m_model->getBackgroundScale( index );
         m_model->getBackgroundCenter( index, cenX, cenY, cenZ );

         glBindTexture( GL_TEXTURE_2D, m_backgroundTexture );
         glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP  );
         glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP  );
         glEnable( GL_TEXTURE_2D );

         glBegin( GL_QUADS );

         switch ( m_viewDirection )
         {
            case ViewFront:
               minZ  =  maxZ = -((m_farOrtho / 2.0f) - 0.1);
               minX  = -scale * (w / dimMax) + cenX;
               maxX  =  scale * (w / dimMax) + cenX;
               minY  = -scale * (h / dimMax) + cenY;
               maxY  =  scale * (h / dimMax) + cenY;
               normZ =  1.0f;
               break;

            case ViewBack:
               minZ  =  maxZ = ((m_farOrtho / 2.0f) - 0.1);
               minX  =  scale * (w / dimMax) + cenX;
               maxX  = -scale * (w / dimMax) + cenX;
               minY  = -scale * (h / dimMax) + cenY;
               maxY  =  scale * (h / dimMax) + cenY;
               normZ = -1.0f;
               break;

            case ViewRight:
               minX  =  maxX = ((m_farOrtho / 2.0f) - 0.1);
               minZ  = -scale * (w / dimMax) + cenZ;
               maxZ  =  scale * (w / dimMax) + cenZ;
               minY  = -scale * (h / dimMax) + cenY;
               maxY  =  scale * (h / dimMax) + cenY;
               normX =  1.0f;
               break;

            case ViewLeft:
               minX  =  maxX = -((m_farOrtho / 2.0f) - 0.1);
               minZ  =  scale * (w / dimMax) + cenZ;
               maxZ  = -scale * (w / dimMax) + cenZ;
               minY  = -scale * (h / dimMax) + cenY;
               maxY  =  scale * (h / dimMax) + cenY;
               normX = -1.0f;
               break;

            case ViewTop:
               minY  =  maxY = -((m_farOrtho / 2.0f) - 0.1);
               minX  = -scale * (w / dimMax) + cenX;
               maxX  =  scale * (w / dimMax) + cenX;
               minZ  =  scale * (h / dimMax) + cenZ;
               maxZ  = -scale * (h / dimMax) + cenZ;
               normY =  1.0f;
               break;

            case ViewBottom:
               minY  =  maxY = ((m_farOrtho / 2.0f) - 0.1);
               minX  = -scale * (w / dimMax) + cenX;
               maxX  =  scale * (w / dimMax) + cenX;
               minZ  = -scale * (h / dimMax) + cenZ;
               maxZ  =  scale * (h / dimMax) + cenZ;
               normY = -1.0f;
               break;

            default:
               break;
         }

         if ( m_viewDirection == ViewLeft || m_viewDirection == ViewRight )
         {
            glNormal3f( normX, normY, normZ );
            glTexCoord2f( 0.0f, 0.0f );
            glVertex3f( minX,  minY, minZ );
            glNormal3f( normX, normY, normZ );
            glTexCoord2f( 0.0f, 1.0f );
            glVertex3f( maxX,  maxY, minZ );
            glNormal3f( normX, normY, normZ );
            glTexCoord2f( 1.0f, 1.0f );
            glVertex3f( maxX,  maxY, maxZ );
            glNormal3f( normX, normY, normZ );
            glTexCoord2f( 1.0f, 0.0f );
            glVertex3f( minX,  minY, maxZ );
         }
         else
         {
            glNormal3f( normX, normY, normZ );
            glTexCoord2f( 0.0f, 0.0f );
            glVertex3f( minX,  minY, minZ );
            glNormal3f( normX, normY, normZ );
            glTexCoord2f( 1.0f, 0.0f );
            glVertex3f( maxX,  minY, minZ );
            glNormal3f( normX, normY, normZ );
            glTexCoord2f( 1.0f, 1.0f );
            glVertex3f( maxX,  maxY, maxZ );
            glNormal3f( normX, normY, normZ );
            glTexCoord2f( 0.0f, 1.0f );
            glVertex3f( minX,  maxY, maxZ );
         }

         glEnd();
      }
   }
}

void ModelViewport::drawOverlay()
{
   setViewportOverlay();

   glDisable( GL_LIGHTING );
   glColor3f( 1.0f, 1.0f, 1.0f );

   glEnable( GL_TEXTURE_2D );

   int w = this->width();
   int h = this->height();

   /*
   glVertex3f( w - , h, 0 );
   glVertex3f( w - SCROLL_SIZE, h, 0 );
   glVertex3f( w - SCROLL_SIZE, h - SCROLL_SIZE, 0 );
   glVertex3f( w, h - SCROLL_SIZE, 0 );

   glVertex3f( w - SCROLL_ALL_X, h - SCROLL_ALL_Y, 0 );
   glVertex3f( w - SCROLL_ALL_X + SCROLL_SIZE, h - SCROLL_ALL_Y, 0 );
   glVertex3f( w - SCROLL_ALL_X + SCROLL_SIZE, h - SCROLL_ALL_Y + SCROLL_SIZE, 0 );
   glVertex3f( w - SCROLL_ALL_X, h - SCROLL_ALL_Y + SCROLL_SIZE, 0 );
   */

   int sx = 0;
   int sy = 0;
   int size = SCROLL_SIZE;

   for ( int b = 0; b < ScrollButtonMAX; b++ )
   {
      ScrollButtonT * sbt = &s_buttons[b];
      sx = sbt->x;
      sy = sbt->y;

      glBindTexture( GL_TEXTURE_2D, m_scrollTextures[ sbt->texIndex ] );

      glBegin( GL_QUADS );

      glTexCoord2f( sbt->s1, sbt->t1 );
      glVertex3f( w + sx, h + sy, 0 );
      glTexCoord2f( sbt->s2, sbt->t2 );
      glVertex3f( w + sx + size, h + sy, 0 );
      glTexCoord2f( sbt->s3, sbt->t3 );
      glVertex3f( w + sx + size, h + sy + size, 0 );
      glTexCoord2f( sbt->s4, sbt->t4 );
      glVertex3f( w + sx, h + sy + size, 0 );

      glEnd();
   }

   glDisable( GL_TEXTURE_2D );
}

void ModelViewport::makeTextureFromImage( const QImage & i, GLuint & t )
{
   glBindTexture( GL_TEXTURE_2D, t );

   glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
         GL_NEAREST );
   glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER,
         GL_NEAREST );

   int w = i.width();
   int h = i.height();
   unsigned pixelCount = w * i.height();
   uint8_t * data = new uint8_t[ pixelCount * 3 ];
   for ( int y = 0; y < h; y ++ )
   {
      for ( int x = 0; x < w; x++ )
      {
         QRgb p = i.pixel( x, h - y - 1 );
         data[ ((y * w + x)*3) + 0 ] = qRed( p );
         data[ ((y * w + x)*3) + 1 ] = qGreen( p );
         data[ ((y * w + x)*3) + 2 ] = qBlue( p );
      }
   }

   gluBuild2DMipmaps( GL_TEXTURE_2D, GL_RGB,
         w, h, 
         GL_RGB, GL_UNSIGNED_BYTE,
         data );

   delete data;
}

void ModelViewport::updateBackground()
{
   int index = (int) m_viewDirection - 1;
   std::string filename = m_model->getBackgroundImage( index );
   if ( strcmp( filename.c_str(), m_backgroundFile.c_str() ) != 0 )
   {
      m_backgroundFile = filename;

      if ( m_backgroundFile[0] != '\0' )
      {
         m_texture = TextureManager::getInstance()->getTexture( m_backgroundFile.c_str() );
         if ( !m_texture )
         {
            model_status( m_model, StatusError, STATUSTIME_LONG, "Could not load background %s", m_backgroundFile.c_str() );
            m_texture = TextureManager::getInstance()->getDefaultTexture( m_backgroundFile.c_str() );
         }
         glBindTexture( GL_TEXTURE_2D, m_backgroundTexture );
         glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
         glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

         GLuint format = m_texture->m_format == Texture::FORMAT_RGBA ? GL_RGBA : GL_RGB;
         gluBuild2DMipmaps( GL_TEXTURE_2D, format,
               m_texture->m_width, m_texture->m_height,
               format, GL_UNSIGNED_BYTE,
               m_texture->m_data );
      }
   }
}

void ModelViewport::adjustViewport()
{
   setViewportDraw();

   updateGL();
}

void ModelViewport::setViewportDraw()
{
   makeCurrent();

   m_inOverlay = false;

   glViewport( 0, 0, ( GLint ) m_viewportWidth, ( GLint ) m_viewportHeight );

   glMatrixMode( GL_PROJECTION );
   glLoadIdentity( );

   GLfloat ratio = ( GLfloat ) m_viewportWidth / ( GLfloat ) m_viewportHeight;
   if ( m_viewDirection == View3d )
   {
      m_far  = m_zoomLevel * 2000.0;
      m_near = m_zoomLevel * 0.002;
      gluPerspective( 45.0f, ratio, m_near, m_far );

      float x = 0.0;
      float y = 0.0;
      if ( m_viewportHeight > m_viewportWidth )
      {
         x = m_zoomLevel;
         y = x / ratio;
      }
      else
      {
         y = m_zoomLevel;
         x = y * ratio;
      }

      m_width = x * 2.0;
      m_height = y * 2.0;
   }
   else
   {
      float x = 0.0;
      float y = 0.0;
      if ( m_viewportHeight > m_viewportWidth )
      {
         x = m_zoomLevel;
         y = x / ratio;
      }
      else
      {
         y = m_zoomLevel;
         x = y * ratio;
      }
      glOrtho( m_centerX - x, m_centerX + x, 
            m_centerY - y, m_centerY + y, 
            m_nearOrtho, m_farOrtho );

      m_width = x * 2.0;
      m_height = y * 2.0;
   }

   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity( );
}

void ModelViewport::setViewportOverlay()
{
   makeCurrent();

   m_inOverlay = true;

   glViewport( 0, 0, ( GLint ) m_viewportWidth, ( GLint ) m_viewportHeight );

   glMatrixMode( GL_PROJECTION );
   glLoadIdentity( );

   glOrtho( 0, this->width(), 
         0, this->height(), 
         m_nearOrtho, m_farOrtho );

   glMatrixMode( GL_MODELVIEW );
   glLoadIdentity( );
}

void ModelViewport::wheelEvent( QWheelEvent * e )
{
   if ( e->delta() > 0 )
   {
      zoomIn();
   }
   else
   {
      zoomOut();
   }
}

void ModelViewport::zoomIn()
{
   m_zoomLevel *= (VP_ZOOMSCALE);

   QString zoomStr;
   zoomStr.sprintf( "%f", m_zoomLevel );
   emit zoomLevelChanged( zoomStr );

   makeCurrent();
   adjustViewport();
}

void ModelViewport::zoomOut()
{
   if ( (m_zoomLevel / VP_ZOOMSCALE) < 250000 )
   {
      m_zoomLevel /= VP_ZOOMSCALE;
   }

   QString zoomStr;
   zoomStr.sprintf( "%f", m_zoomLevel );
   emit zoomLevelChanged( zoomStr );

   makeCurrent();
   adjustViewport();
}

void ModelViewport::scrollUp()
{
   log_debug( "scroll up\n" );
   m_centerY += m_zoomLevel * 0.10f;
   makeCurrent();
   adjustViewport();
}

void ModelViewport::scrollDown()
{
   log_debug( "scroll down\n" );
   m_centerY -= m_zoomLevel * 0.10f;
   makeCurrent();
   adjustViewport();
}

void ModelViewport::scrollLeft()
{
   m_centerX -= m_zoomLevel * 0.10f;
   makeCurrent();
   adjustViewport();
}

void ModelViewport::scrollRight()
{
   m_centerX += m_zoomLevel * 0.10f;
   makeCurrent();
   adjustViewport();
}

static int tempButton = 0;

void ModelViewport::mousePressEvent( QMouseEvent * e )
{
   //printf( "press = %d\n", e->button() );
   e->accept();
   tempButton = e->button();

   int w = this->width();
   int h = this->height();

   int bx = e->pos().x();
   int by = h - e->pos().y();

   m_overlayButton = ScrollButtonMAX;

   int sx = 0;
   int sy = 0;
   int size = SCROLL_SIZE;

   for ( int b = 0; m_overlayButton == ScrollButtonMAX && b < ScrollButtonMAX; b++ )
   {
      sx = s_buttons[b].x;
      sy = s_buttons[b].y;

      if (     (bx >= w + sx) && (bx <= w + sx + size) 
            && (by >= h + sy) && (by <= h + sy + size) )
      {
         m_overlayButton = (ScrollButtonE) b;

         switch ( m_overlayButton )
         {
            case ScrollButtonPan:
               m_scrollStartPosition = e->pos();
               break;
            case ScrollButtonUp:
               log_debug( "scroll button up" );
               scrollUp();
               m_scrollTimer->start( 300, true );
               break;
            case ScrollButtonDown:
               log_debug( "scroll button down" );
               scrollDown();
               m_scrollTimer->start( 300, true );
               break;
            case ScrollButtonLeft:
               scrollLeft();
               m_scrollTimer->start( 300, true );
               break;
            case ScrollButtonRight:
               scrollRight();
               m_scrollTimer->start( 300, true );
               break;
            default:
               break;
         }
      }
   }

   if ( m_overlayButton == ScrollButtonMAX )
   {
      if ( e->button() == MidButton || (m_viewDirection == View3d && e->button() == LeftButton) )
      {
         m_scrollStartPosition = e->pos();
      }
      else
      {
         int button = constructButtonState( e );

         ::Tool * tool = m_toolbox->getCurrentTool();
         tool->mouseButtonDown( this, button, e->pos().x(), e->pos().y() );
      }
   }
}

void ModelViewport::mouseReleaseEvent( QMouseEvent * e )
{
   //printf( "release = %d\n", e->button() );
   if ( m_overlayButton == ScrollButtonMAX )
   {
      e->accept();
      if ( tempButton == MidButton || (m_viewDirection == View3d && tempButton == LeftButton) )
      {
         // do nothing
      }
      else
      {
         int button = constructButtonState( e );

         ::Tool * tool = m_toolbox->getCurrentTool();
         tool->mouseButtonUp( this, button, e->pos().x(), e->pos().y() );
         m_model->operationComplete( tool->getName( 0 ) );

         if ( m_model->getSelectedBoneJointCount() > 0 )
         {
            m_model->setDrawJoints( 
                  (Model::DrawJointMode) g_prefs( "ui_draw_joints" ).intValue() );
            updateView();
         }
      }
   }
   else
   {
      m_overlayButton = ScrollButtonMAX;
      m_scrollTimer->stop();

      model_status( m_model, StatusNormal, STATUSTIME_SHORT, "Use the middle mouse button to drag/pan the viewport" );
   }
   tempButton = NoButton;
}

void ModelViewport::mouseMoveEvent( QMouseEvent * e )
{
   e->accept();

   if ( ! this->hasFocus() )
   {
      this->setFocus();
   }

   {
      int x = e->pos().x();
      int y = e->pos().y();

      double xcoord = 0.0;
      double ycoord = 0.0;
      double zcoord = 0.0;

      getXValue( x, y, &xcoord );
      getYValue( x, y, &ycoord );
      getZValue( x, y, &zcoord );

      char str[80];
      PORT_snprintf( str, sizeof(str), "%g, %g, %g", xcoord, ycoord, zcoord );
      model_status( m_model, StatusNormal, STATUSTIME_NONE, str );
   }

   if ( m_overlayButton == ScrollButtonMAX )
   {
      if ( tempButton )
      {
         if ( tempButton == MidButton || (m_viewDirection == View3d && tempButton == LeftButton) )
         {
            if ( m_viewDirection == View3d && (tempButton == LeftButton) )
            {
#if 1
               QPoint curPos = e->pos();

               double xDiff = -(m_scrollStartPosition.x() - curPos.x());
               double yDiff = -(m_scrollStartPosition.y() - curPos.y());

               Matrix mcur;
               Matrix mcurinv;
               double rot[3];
               rot[0] = m_rotX * PIOVER180;
               rot[1] = m_rotY * PIOVER180;
               rot[2] = m_rotZ * PIOVER180;
               mcur.setRotation( rot );
               mcurinv = mcur.getInverse();

               Vector yvec;
               yvec.set( 0, 0.0 );
               yvec.set( 1, 1.0 );
               yvec.set( 2, 0.0 );
               yvec.set( 3, 0.0 );

               Vector xvec;
               xvec.set( 0, 1.0 );
               xvec.set( 1, 0.0 );
               xvec.set( 2, 0.0 );
               xvec.set( 3, 0.0 );

               Matrix mx;
               Matrix my;
               double rotY = (double) xDiff / (double) this->width()  * 3.14159 * 2.0;
               double rotX = (double) yDiff / (double) this->height() * 3.14159 * 2.0;

               yvec = yvec * mcurinv;
               my.setRotationOnAxis( yvec.getVector(), rotY );

               xvec = xvec * mcurinv;
               mx.setRotationOnAxis( xvec.getVector(), rotX );

               mcur = mx * mcur;
               mcur = my * mcur;
               mcur.getRotation( rot );
               m_rotX = rot[0] / PIOVER180;
               m_rotY = rot[1] / PIOVER180;
               m_rotZ = rot[2] / PIOVER180;

               //m_rotY += xDiff * PIOVER180 * 14.0;
               //m_rotX += yDiff * PIOVER180 * 14.0;

               m_scrollStartPosition = curPos;
#else
               QPoint curPos = e->pos();

               double xDiff = -(m_scrollStartPosition.x() - curPos.x());
               double yDiff = -(m_scrollStartPosition.y() - curPos.y());

               m_rotY += xDiff * PIOVER180 * 14.0;
               m_rotX += yDiff * PIOVER180 * 14.0;

               m_scrollStartPosition = curPos;
#endif

               // And finally, update the view
               makeCurrent();
               adjustViewport();
            }
            else
            {
               //printf( "adjusting translation\n" );
               QPoint curPos = e->pos();

               double xDiff = m_scrollStartPosition.x() - curPos.x();
               m_centerX += xDiff * (m_width / m_viewportWidth);

               double yDiff = m_scrollStartPosition.y() - curPos.y();
               yDiff = -yDiff;  // Adjust for difference between pixel and GL coordinates
               m_centerY += yDiff * (m_height / m_viewportHeight);

               m_scrollStartPosition = curPos;

               makeCurrent();
               adjustViewport();
            }
         }
         else
         {
            //printf( "tool mouse event\n" );
            int button = constructButtonState( e );

            ::Tool * tool = m_toolbox->getCurrentTool();
            tool->mouseButtonMove( this, button, e->pos().x(), e->pos().y() );
         }
      }
   }
   else
   {
      if ( m_overlayButton == ScrollButtonPan )
      {
         //printf( "adjusting translation\n" );
         QPoint curPos = e->pos();

         double xDiff = m_scrollStartPosition.x() - curPos.x();
         m_centerX += xDiff * (m_width / m_viewportWidth);

         double yDiff = m_scrollStartPosition.y() - curPos.y();
         yDiff = -yDiff;  // Adjust for difference between pixel and GL coordinates
         m_centerY += yDiff * (m_height / m_viewportHeight);

         m_scrollStartPosition = curPos;

         makeCurrent();
         adjustViewport();
      }
   }
}

void ModelViewport::keyPressEvent( QKeyEvent * e )
{
   switch ( e->key() )
   {
      case Key_Equal:
      case Key_Plus:
         {
            {
               zoomIn();
            }
         }
         break;
      case Key_Minus:
      case Key_Underscore:
         {
            {
               zoomOut();
            }
         }
         break;
      case Key_QuoteLeft:
         {
            int newDir = 0;
            switch ( m_viewDirection )
            {
               case ViewFront:
                  newDir = ViewBack;
                  break;
               case ViewBack:
                  newDir = ViewFront;
                  break;
               case ViewRight:
                  newDir = ViewLeft;
                  break;
               case ViewLeft:
                  newDir = ViewRight;
                  break;
               case ViewTop:
                  newDir = ViewBottom;
                  break;
               case ViewBottom:
                  newDir = ViewTop;
                  break;
               default:
                  break;
            }

            emit viewDirectionChanged( newDir );
            //viewChangeEvent( newDir );
         }
         break;
      case Key_0:
         m_centerX = 0.0;
         m_centerY = 0.0;
         makeCurrent();
         adjustViewport();
         break;
      case Key_Up:
         log_debug( "scroll key up" );
         scrollUp();
         break;
      case Key_Down:
         log_debug( "scroll key down" );
         scrollDown();
         break;
      case Key_Left:
         scrollLeft();
         break;
      case Key_Right:
         scrollRight();
         break;
      default:
         QGLWidget::keyPressEvent( e );
         break;
   }
}

void ModelViewport::viewChangeEvent( int dir )
{
   float x = 0.0;
   float y = 0.0;
   float z = 0.0;

   switch ( m_viewDirection )
   {
      case ViewFront:
         x = m_centerX;
         y = m_centerY;
         break;
      case ViewBack:
         x = -m_centerX;
         y = m_centerY;
         break;
      case ViewRight:
         z = -m_centerX;
         y = m_centerY;
         break;
      case ViewLeft:
         z = m_centerX;
         y = m_centerY;
         break;
      case ViewTop:
         x = m_centerX;
         z = m_centerY;
         break;
      case ViewBottom:
         x = m_centerX;
         z = -m_centerY;
         break;
      default:
         break;
   }

   switch ( dir )
   {
      case ViewFront:
         m_centerX = x;
         m_centerY = y;
         break;
      case ViewBack:
         m_centerX = -x;
         m_centerY = y;
         break;
      case ViewRight:
         m_centerX = -z;
         m_centerY = y;
         break;
      case ViewLeft:
         m_centerX = z;
         m_centerY = y;
         break;
      case ViewTop:
         m_centerX = x;
         m_centerY = z;
         break;
      case ViewBottom:
         m_centerX = x;
         m_centerY = -z;
         break;
      default:
         break;
   }

   m_viewDirection = (ViewDirection) dir;
   makeCurrent();
   adjustViewport();
}

void ModelViewport::setZoomLevel( double zoom )
{
   m_zoomLevel = zoom;
   makeCurrent();
   adjustViewport();

   QString zoomStr;
   zoomStr.sprintf( "%f", m_zoomLevel );
   emit zoomLevelChanged( zoomStr );
}

void ModelViewport::wireframeEvent()
{
   m_viewOptions = ViewWireframe;
   updateGL();
}

void ModelViewport::flatEvent()
{
   m_viewOptions = ViewFlat;
   updateGL();
}

void ModelViewport::smoothEvent()
{
   m_viewOptions = ViewSmooth;
   updateGL();
}

void ModelViewport::textureEvent()
{
   m_viewOptions = ViewTexture;
   updateGL();
}

void ModelViewport::alphaEvent()
{
   m_viewOptions = ViewAlpha;
   updateGL();
}

void ModelViewport::scrollTimeout()
{
   switch ( m_overlayButton )
   {
      case ScrollButtonUp:
         scrollUp();
         break;
      case ScrollButtonDown:
         scrollDown();
         break;
      case ScrollButtonLeft:
         scrollLeft();
         break;
      case ScrollButtonRight:
         scrollRight();
         break;
      default:
         m_scrollTimer->stop();
         return;
   }

   m_scrollTimer->start( 100 );
}

void ModelViewport::focusInEvent( QFocusEvent * e )
{
   m_backColor.setRgb( 150, 210, 210 );
   makeCurrent();
   glClearColor( m_backColor.red() / 256.0, 
         m_backColor.green() / 256.0, 
         m_backColor.blue() / 256.0, 1.0f );
   updateGL();
}

void ModelViewport::focusOutEvent( QFocusEvent * e )
{
   m_backColor.setRgb( 130, 200, 200 );
   makeCurrent();
   glClearColor( m_backColor.red() / 256.0, 
         m_backColor.green() / 256.0, 
         m_backColor.blue() / 256.0, 1.0f );
   updateGL();
}

/*
void ModelViewport::dragEnterEvent( QDragMoveEvent * e )
{
   log_debug( "got drag enter event\n" );
   if ( QUriDrag::canDecode( e ) )
   {
      log_debug( "is URI\n" );
   }
   if ( QTextDrag::canDecode( e ) )
   {
      log_debug( "is Text\n" );
   }
   if ( QImageDrag::canDecode( e ) )
   {
      log_debug( "is Image\n" );
   }
   //e->accept( QRect( 0, 0, this->width(), this->height() ) );
}
*/

bool ModelViewport::getXValue( int x, int y, double * val )
{
   switch ( m_viewDirection )
   {
      case ViewFront:
      case ViewTop:
      case ViewBottom:
         *val = (m_centerX - (m_width / 2.0)) + (((double) x / (double) m_viewportWidth) * m_width);
         break;
      case ViewBack:
         *val = ((-m_centerX) + (m_width / 2.0)) + (((double) (-x) / (double) m_viewportWidth) * m_width);
         break;
      case ViewRight:
      case ViewLeft:
      default:
         return false;
   }

   return true;
}

bool ModelViewport::getYValue( int x, int y, double * val )
{
   switch ( m_viewDirection )
   {
      case ViewFront:
      case ViewBack:
      case ViewRight:
      case ViewLeft:
         *val = (m_centerY + (m_height / 2.0)) - (((double) y / (double) m_viewportHeight) * m_height);
         break;
      case ViewTop:
      case ViewBottom:
      default:
         return false;
         break;
   }

   return true;
}

bool ModelViewport::getZValue( int x, int y, double * val )
{
   switch ( m_viewDirection )
   {
      case ViewTop:
         *val = ((-m_centerY) - (m_height / 2.0)) - (((double) (-y) / (double) m_viewportHeight) * m_height);
         break;
      case ViewBottom:
         *val = (m_centerY + (m_height / 2.0)) - (((double) y / (double) m_viewportHeight) * m_height);
         //*val = (m_centerY + (m_height / 2.0)) - (((double) y / (double) m_viewportHeight) * m_height);
         break;
      case ViewRight:
         *val = (m_centerX - (m_width / 2.0)) + (((double) x / (double) m_viewportWidth) * m_width);
         break;
      case ViewLeft:
         *val = ((-m_centerX) + (m_width / 2.0)) + (((double) (-x) / (double) m_viewportWidth) * m_width);
         break;
      case ViewFront:
      case ViewBack:
      default:
         return false;
   }

   return true;
}

int ModelViewport::constructButtonState( QMouseEvent * e )
{
   int button = 0;

   //switch ( e->button() )
   switch ( tempButton )
   {
      case LeftButton:
         button = ::Tool::BS_Left;
         break;
      case MidButton:
         button = ::Tool::BS_Middle;
         break;
      case RightButton:
         button = ::Tool::BS_Right;
         break;
      default:
         break;
   }

#ifdef HAVE_QT4
   if ( e->modifiers() & ShiftButton )
#else
   if ( e->state() & ShiftButton )
#endif // HAVE_QT4
   {
      button |= ::Tool::BS_Shift;
   }

#ifdef HAVE_QT4
   if ( e->modifiers() & AltButton )
#else
   if ( e->state() & AltButton )
#endif // HAVE_QT4
   {
      button |= ::Tool::BS_Alt;
   }

#ifdef HAVE_QT4
   if ( e->modifiers() & ControlButton )
#else
   if ( e->state() & ControlButton )
#endif // HAVE_QT4
   {
      button |= ::Tool::BS_Ctrl;
   }

   return button;
}

void ModelViewport::updateView()
{
   updateGL();
   StatusBar * bar = StatusBar::getStatusBarFromModel( m_model );
   bar->setVertices(   m_model->getVertexCount(),    m_model->getSelectedVertexCount() );
   bar->setFaces(      m_model->getTriangleCount(),  m_model->getSelectedTriangleCount() );
   bar->setGroups(     m_model->getGroupCount() );
   bar->setBoneJoints( m_model->getBoneJointCount(), m_model->getSelectedBoneJointCount() );
   bar->setTextures(   m_model->getTextureCount() );
}

void ModelViewport::update3dView()
{
   if ( m_viewDirection == View3d )
   {
      updateView();
   }
}

void ModelViewport::addDecal( Decal * decal )
{
   m_decals.push_back( decal );
   updateGL();
}

void ModelViewport::removeDecal( Decal * decal )
{
   m_decals.remove( decal );
   updateGL();
}

void ModelViewport::frameArea( double x1, double y1, double z1, double x2, double y2, double z2 )
{
   double centerX = (x1 + x2) / 2.0;
   double centerY = (y1 + y2) / 2.0;
   double centerZ = (z1 + z2) / 2.0;

   double width  = fabs( x1 - x2 );
   double height = fabs( y1 - y2 );
   double depth  = fabs( z1 - z2 );

   if ( width < 0.001 )
   {
      width = 0.001;
   }
   if ( height < 0.001 )
   {
      height = 0.001;
   }
   if ( depth < 0.001 )
   {
      depth = 0.001;
   }

   width  *= 1.20;
   height *= 1.20;
   depth  *= 1.20;

   double viewWidth   = 0.0;
   double viewHeight  = 0.0;

   switch ( m_viewDirection )
   {
      case ViewFront:
         m_centerX   = centerX;
         m_centerY   = centerY;
         viewWidth   = width;
         viewHeight  = height;
         break;
      case ViewBack:
         m_centerX   = -centerX;
         m_centerY   = centerY;
         viewWidth   = width;
         viewHeight  = height;
         break;
      case ViewRight:
         m_centerX   = centerZ;
         m_centerY   = centerY;
         viewWidth   = depth;
         viewHeight  = height;
         break;
      case ViewLeft:
         m_centerX   = -centerZ;
         m_centerY   = centerY;
         viewWidth   = depth;
         viewHeight  = height;
         break;
      case ViewTop:
         m_centerX   = centerX;
         m_centerY   = -centerZ;
         viewWidth   = width;
         viewHeight  = depth;
         break;
      case ViewBottom:
         m_centerX   = centerX;
         m_centerY   = centerZ;
         viewWidth   = width;
         viewHeight  = depth;
         break;
      case View3d:
         {
            double rotation[3];
            rotation[0] = m_rotX * PIOVER180;
            rotation[1] = m_rotY * PIOVER180;
            rotation[2] = m_rotZ * PIOVER180;
            Matrix m;
            m.setRotation( rotation );

            double v[3];
            v[0] = centerX;
            v[1] = centerY;
            v[2] = centerZ;

            m.inverseRotateVector( v );

            m_centerX   = v[0];
            m_centerY   = v[1];
            viewWidth   = height;
            viewHeight  = width;
         }
         break;
      default:
         return;
         break;
   }

   if ( viewWidth > viewHeight )
   {
      if ( m_viewDirection == View3d )
      {
         m_zoomLevel = viewWidth;
      }
      else
      {
         m_zoomLevel = viewWidth / 2.0;
      }
   }
   else
   {
      if ( m_viewDirection == View3d )
      {
         m_zoomLevel = viewHeight;
      }
      else
      {
         m_zoomLevel = viewHeight / 2.0;
      }
   }

   QString zoomStr;
   zoomStr.sprintf( "%f", m_zoomLevel );
   emit zoomLevelChanged( zoomStr );

   makeCurrent();
   adjustViewport();
}

void ModelViewport::checkGlErrors()
{
   int error = glGetError();
   if ( error )
   {
      switch ( error )
      {
         case GL_INVALID_VALUE:
            model_status( m_model, StatusNormal, STATUSTIME_NONE, "OpenGL error = Invalid Value" );
            break;
         case GL_INVALID_ENUM:
            model_status( m_model, StatusNormal, STATUSTIME_NONE, "OpenGL error = Invalid Enum" );
            break;
         case GL_INVALID_OPERATION:
            model_status( m_model, StatusNormal, STATUSTIME_NONE, "OpenGL error = Invalid Operation" );
            break;
         case GL_STACK_OVERFLOW:
            model_status( m_model, StatusNormal, STATUSTIME_NONE, "OpenGL error = Stack Overflow" );
            break;
         case GL_STACK_UNDERFLOW:
            model_status( m_model, StatusNormal, STATUSTIME_NONE, "OpenGL error = Stack Underflow" );
            break;
         case GL_OUT_OF_MEMORY:
            model_status( m_model, StatusNormal, STATUSTIME_NONE, "OpenGL error = Out Of Memory" );
            break;
         default:
            model_status( m_model, StatusNormal, STATUSTIME_NONE, "OpenGL error = %d", error );
            break;
      }
   }
}

void ModelViewport::copyContentsToTexture( Texture * tex )
{
   makeCurrent();

   m_capture = true;
   if ( tex )
   {
      unsigned w = this->width();
      unsigned h = this->height();

      // make sure texture can hold data

      if ( tex->m_data )
      {
         unsigned bpp = (tex->m_format == Texture::FORMAT_RGB) ? 3 : 4;
         if ( (tex->m_width * tex->m_height * bpp) < (w * h * 4) )
         {
            delete[] tex->m_data;
            tex->m_data = NULL;
         }
      }

      if ( tex->m_data == NULL )
      {
         tex->m_data = new uint8_t[ w * h * 4 ];
      }

      tex->m_width  = w;
      tex->m_height = h;
      tex->m_format = Texture::FORMAT_RGBA;

      makeCurrent();
      updateGL();

      glReadPixels( 0, 0, w, h, GL_RGBA, GL_UNSIGNED_BYTE, tex->m_data );
   }

   makeCurrent();
   glClearColor( m_backColor.red() / 256.0, 
         m_backColor.green() / 256.0, 
         m_backColor.blue() / 256.0, 1.0f );

   m_capture = false;
}

