/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "cylindertool.h"

#include "pixmap/cylindertool.xpm"

#include "model.h"
#include "glmath.h"
#include "log.h"
#include "modelstatus.h"
#include "mq3compat.h"

#include <math.h>

#include <qwidget.h>

using std::vector;
using std::list;

CylinderTool::CylinderTool()
   : m_segments( 1 ),
     m_sides( 8 ),
     m_inverted( false )
{
}

CylinderTool::~CylinderTool()
{
}

void CylinderTool::activated( int arg, Model * model, QMainWindow * mainwin )
{
   log_debug( "cylinder activated\n" );
   m_widget = new CylinderToolWidget( this, mainwin );
#ifdef HAVE_QT4
   //mainwin->addDockWindow( m_widget, DockBottom );
#endif
   m_widget->show();
}

void CylinderTool::deactivated()
{
   m_widget->close();
}

void CylinderTool::mouseButtonDown( Parent * parent, int buttonState, int x, int y )
{
   Model * model = parent->getModel();

   model->unselectAll();

   int index  = 0;
   int xindex = 0;
   int yindex = 1;
   int zindex = 2;
   double pos[3] = {0,0,0};
   double rad[3] = {0,0,0};

   m_inverted = false;

   if ( !parent->getXValue( x, y, &pos[0] ) )
   {
      index = 1;

      xindex = 0;
      yindex = 1;
      zindex = 2;
   }
   if ( !parent->getYValue( x, y, &pos[1] ) )
   {
      index = 2;

      xindex = 2;
      yindex = 1;
      zindex = 0;
   }
   if ( !parent->getZValue( x, y, &pos[2] ) )
   {
      index = 0;

      xindex = 2;
      yindex = 1;
      zindex = 0;
   }

   m_startX = pos[0];
   m_startY = pos[1];
   m_startZ = pos[2];

   unsigned triStart = model->getTriangleCount();

   unsigned points   = m_sides;
   CylinderVertices ev;
   unsigned s;
   unsigned r;
   // Create external cylinder
   for ( s = 0; s <= m_segments; s++ )
   {
      double scale = (double) m_scale / 100.0;
      scale = (double) m_scale / 100.0;
      scale = (1.0 - scale) * ((double) s / (double) m_segments) 
         //* (sqrt( 10000.0 * (double) s / (double) m_segments ) / 100.0)
         //* ( sin( ((double) s / (double) m_segments) * 3.14159 / 2.0 ) )
         + scale;
      for ( r = 0; r < points; r++ )
      {
         ev.coords[xindex] = sin( ((double) r / (double) points) * 3.14159 * 2.0 ) * scale;
         ev.coords[yindex] = cos( ((double) r / (double) points) * 3.14159 * 2.0 ) * scale;
         ev.coords[zindex] = (double) s / (double) (m_segments);
         ev.v = model->addVertex( ev.coords[0], ev.coords[1], ev.coords[2] );
         m_vertices.push_back( ev );
      }

      if ( s > 0 )
      {
         for ( r = 0; r < points; r++ )
         {
            unsigned next = r + 1;
            if ( next >= points )
            {
               next = 0;
            }

            model->addTriangle( m_vertices[((s-1)*points) + r].v, 
                  m_vertices[ ((s-1)*points) + next].v, 
                  m_vertices[ (s*points) + r ].v );
            model->addTriangle( m_vertices[((s-1)*points) + next].v, 
                  m_vertices[ (s*points) + next].v, 
                  m_vertices[ (s*points) + r ].v );
         }
      }
   }

#if 1
   unsigned off = m_vertices.size();

   // Create internal cylinder
   if ( m_width < 100 )
   {
      double rad = 1.0 - (double) m_width / 100.0;
      if ( rad > 0.999 )
      {
         rad = 0.999;
      }
      for ( s = 0; s <= m_segments; s++ )
      {
         double scale = (double) m_scale / 100.0;
         scale = (1.0 - scale) * ((double) s / (double) m_segments) + scale;
         for ( r = 0; r < points; r++ )
         {
            ev.coords[xindex] = sin( ((double) r / (double) points) * 3.14159 * 2.0 ) * rad * scale;
            ev.coords[yindex] = cos( ((double) r / (double) points) * 3.14159 * 2.0 ) * rad * scale;
            ev.coords[zindex] = (double) s / (double) (m_segments);
            ev.v = model->addVertex( ev.coords[0], ev.coords[1], ev.coords[2] );
            m_vertices.push_back( ev );
         }

         if ( s > 0 )
         {
            for ( r = 0; r < points; r++ )
            {
               unsigned next = r + 1;
               if ( next >= points )
               {
                  next = 0;
               }
               model->addTriangle( m_vertices[ off + ((s-1)*points) + next ].v, 
                     m_vertices[ off + ((s-1)*points) + r ].v, 
                     m_vertices[ off + (s*points) + r ].v );
               model->addTriangle( m_vertices[ off + ((s-1)*points) + next ].v, 
                     m_vertices[ off + (s*points) + r ].v, 
                     m_vertices[ off + (s*points) + next ].v );
            }
         }
      }

      // Add ends
      unsigned off2 = m_vertices.size() - points;
      for ( r = 0; r < points; r++ )
      {
         unsigned next = r + 1;
         if ( next >= points )
         {
            next = 0;
         }
         model->addTriangle( m_vertices[ next ].v, m_vertices[ r ].v, m_vertices[ off + r ].v );
         model->addTriangle( m_vertices[ next ].v, m_vertices[ off + r ].v, m_vertices[ off + next ].v );

         model->addTriangle( m_vertices[ (off - points) + r ].v, m_vertices[ (off - points) + next ].v, m_vertices[ off2 + r ].v );
         model->addTriangle( m_vertices[ (off - points) + next ].v, m_vertices[ off2 + next ].v, m_vertices[ off2 + r ].v );
      }
   }
   else // No internal cylinder
   {
      off = m_vertices.size() - points;
      // Add ends
      ev.coords[xindex] = 0.0;
      ev.coords[yindex] = 0.0;
      ev.coords[zindex] = 0.0;
      ev.v = model->addVertex( ev.coords[0], ev.coords[1], ev.coords[2] );

      for ( r = 0; r < points; r++ )
      {
         unsigned next = r + 1;
         if ( next >= points )
         {
            next = 0;
         }
         model->addTriangle( ev.v, m_vertices[ next ].v, m_vertices[ r ].v );
      }
      m_vertices.push_back( ev );

      ev.coords[xindex] = 0.0;
      ev.coords[yindex] = 0.0;
      ev.coords[zindex] = 1.0;
      ev.v = model->addVertex( ev.coords[0], ev.coords[1], ev.coords[2] );
      m_vertices.push_back( ev );

      for ( r = 0; r < points; r++ )
      {
         unsigned next = r + 1;
         if ( next >= points )
         {
            next = 0;
         }
         model->addTriangle( ev.v, m_vertices[ off + r].v, m_vertices[ off + next].v );
      }
      m_vertices.push_back( ev );

   }
#endif // 0 

   unsigned triEnd = model->getTriangleCount();

   for ( unsigned t = triStart; t < triEnd; t++ )
   {
      model->selectTriangle( t );
   }

   updateVertexCoords( model, pos[0], pos[1], pos[2], rad[0], rad[1], rad[2] );

   parent->updateAllViews();

   model_status( model, StatusNormal, STATUSTIME_SHORT, "Cylinder created" );
}

void CylinderTool::mouseButtonUp( Parent * parent, int buttonState, int x, int y )
{
   m_vertices.clear();
}

void CylinderTool::mouseButtonMove( Parent * parent, int buttonState, int x, int y )
{
   Model * model = parent->getModel();

   int index = 0;
   double pos[3] = {0,0,0};
   double rad[3] = {0,0,0};

   if ( !parent->getXValue( x, y, &pos[0] ) ) { index = 1; }
   if ( !parent->getYValue( x, y, &pos[1] ) ) { index = 2; }
   if ( !parent->getZValue( x, y, &pos[2] ) ) { index = 0; }

   bool doInvert = false;

   switch ( index )
   {
      case 0:
         rad[0] = (pos[0] - m_startX);
         rad[1] = fabs(m_startY - pos[1]) / 2;
         rad[2] = fabs(m_startY - pos[1]) / 2;

         pos[0] = m_startX;
         pos[1] = (pos[1] + m_startY) / 2;
         pos[2] = 0;

         if ( rad[0] < 0.0 && m_inverted == false )
         {
            doInvert = true;
         }
         else if ( rad[0] >= 0.0 && m_inverted )
         {
            doInvert = true;
         }
         break;
      case 1:
         rad[0] = fabs(m_startY - pos[1]) / 2;
         rad[1] = fabs(m_startY - pos[1]) / 2;
         rad[2] = (pos[2] - m_startZ);

         pos[0] = 0;
         pos[1] = (pos[1] + m_startY) / 2;
         pos[2] = m_startZ;

         if ( rad[2] > 0.0 && m_inverted == false )
         {
            doInvert = true;
         }
         else if ( rad[2] <= 0.0 && m_inverted )
         {
            doInvert = true;
         }
         break;
      case 2:
         rad[0] = (pos[0] - m_startX );
         rad[1] = fabs(m_startZ - pos[2]) / 2;
         rad[2] = fabs(m_startZ - pos[2]) / 2;

         pos[0] = m_startX;
         pos[1] = 0;
         pos[2] = (pos[2] + m_startZ) / 2;

         if ( rad[0] < 0.0 && m_inverted == false )
         {
            doInvert = true;
         }
         else if ( rad[0] >= 0.0 && m_inverted )
         {
            doInvert = true;
         }
         break;
      default:
         break;
   }

   if ( doInvert )
   {
      m_inverted = !m_inverted;
      list<int> selectedList = model->getSelectedTriangles();
      list<int>::iterator it;
      for ( it = selectedList.begin(); it != selectedList.end(); it++ )
      {
         model->invertNormals( *it );
      }
   }

   updateVertexCoords( model, pos[0], pos[1], pos[2], rad[0], rad[1], rad[2] );

   parent->updateAllViews();
}

const char ** CylinderTool::getPixmap()
{
   return (const char **) cylindertool_xpm;
}

void CylinderTool::updateVertexCoords( Model * model, double x, double y, double z,
      double xrad, double yrad, double zrad )
{
   vector<CylinderVertices>::iterator it;

   for ( it = m_vertices.begin(); it != m_vertices.end(); it++ )
   {
      model->moveVertex( (*it).v, 
            (*it).coords[0]*xrad + x, 
            (*it).coords[1]*yrad + y, 
            (*it).coords[2]*zrad + z );
   }
}

void CylinderTool::setSegmentsValue( int newValue )
{
   log_debug( "segments: %d\n", newValue );
   m_segments = newValue;
}

void CylinderTool::setSidesValue( int newValue )
{
   log_debug( "sides: %d\n", newValue );
   m_sides = newValue;
}

void CylinderTool::setWidthValue( int newValue )
{
   log_debug( "width: %d\n", newValue );
   m_width = newValue;
}

void CylinderTool::setScaleValue( int newValue )
{
   log_debug( "scale: %d\n", newValue );
   m_scale = newValue;
}

void CylinderTool::setShapeValue( int newValue )
{
   log_debug( "shape: %d\n", newValue );
   m_shape = newValue;
}

