#define WIN32_LEAN_AND_MEAN
#define ERR(x) error(__FILE__, __LINE__, x)
#define AVIERR(x) if ((x) != AVIERR_OK) ERR(#x)

#include <windows.h>
#include <vfw.h> /* don't forget to link with vfw32.lib and user32.lib! */
#include <stdio.h>
#include <stdlib.h>
#include <shellapi.h> 
#include <malloc.h>
#include <mx/mx.h>
#include <mx/gl.h>
#include <mx/mxTga.h>
#include "mdxviewer.h"
#include "GlWindow.h"
#include "pakviewer.h"


MDXViewer *g_mdxViewer = 0;
char loadmod[256];

#ifdef __cplusplus
extern "C" {
#endif
int SaveAsMD2(char *filename, mdx_model_t *model);
#ifdef __cplusplus
}
#endif

static char recentFiles[8][256] = { "", "", "", "", "", "", "", "" };

long Skins=0;
long Vertices=0;
long Triangles=0;
long GlCommands=0;
long Frames=0;


void error(char *file, int line, char *err)
{
	FILE *error;
	error = fopen("error.log", "w");
	fprintf(error, "%s:%d: %s\n", file, line, err);
	fclose(error);
	exit(1);
}

void
MDXViewer::initRecentFiles ()
{
	for (int i = 0; i < 8; i++)
	{
		if (strlen (recentFiles[i]))
		{
			UI_mb->modify (IDC_MODEL_RECENTMODELS1 + i, IDC_MODEL_RECENTMODELS1 + i, recentFiles[i]);
		}
		else
		{
			UI_mb->modify (IDC_MODEL_RECENTMODELS1 + i, IDC_MODEL_RECENTMODELS1 + i, "(empty)");
			UI_mb->setEnabled (IDC_MODEL_RECENTMODELS1 + i, false);
		}
	}
}



void
MDXViewer::loadRecentFiles ()
{
	char tmpStr[256];
	FILE *file = fopen ("KP_Viewer.ini", "rb"); //hypov8 todo: .dat? .ini should be a text file
	if (file)
	{
		//hypov8 fail safe read of ini
		for (int i = 0; i < 8; i++) 
		{
			int rByte = fread(tmpStr, sizeof(recentFiles[i]), 1, file);
			if (rByte==1) //make sure its 1
			{
				tmpStr[255] = 0; //null term string
				memcpy(recentFiles[i], tmpStr, sizeof(recentFiles[i]));
			}
		}
		fclose(file);
	}
}



void
MDXViewer::saveRecentFiles ()
{
	char path[256];

	strcpy_s (path, sizeof(path), mx::getApplicationPath ());
	strcat_s (path,sizeof(path), "/KP_Viewer.ini"); //hypov8 todo: .dat? .ini should be a text file

	FILE *file = fopen (path, "wb");
	if (file)
	{
		fwrite (recentFiles, sizeof recentFiles, 1, file);
		fclose (file);
	}
}


MDXViewer::MDXViewer ()
: mxWindow (0, 0, 0, 0, 0, "Kingpin Model Viewer 1.1.3", mxWindow::Normal) //hypov8 version
{
	// create menu stuff
	UI_mb = new mxMenuBar (this);
	mxMenu *menuModel = new mxMenu ();
	mxMenu *menuPack = new mxMenu (); //hypov8
	mxMenu *menuSkin = new mxMenu ();
	mxMenu *menuOptions = new mxMenu ();
	mxMenu *menuView = new mxMenu ();
	mxMenu *menuHelp = new mxMenu ();

	UI_mb->addMenu ("Model", menuModel);
	UI_mb->addMenu ("Pak ", menuPack); //hypov8
	UI_mb->addMenu ("Textures", menuSkin);
	UI_mb->addMenu ("Options", menuOptions);
	UI_mb->addMenu ("Help", menuHelp);

	mxMenu *menuRecentModels = new mxMenu ();
	menuRecentModels->add ("(empty)", IDC_MODEL_RECENTMODELS1);
	menuRecentModels->add ("(empty)", IDC_MODEL_RECENTMODELS2);
	menuRecentModels->add ("(empty)", IDC_MODEL_RECENTMODELS3);
	menuRecentModels->add ("(empty)", IDC_MODEL_RECENTMODELS4);

	mxMenu *menuRecentPakFiles = new mxMenu ();
	menuRecentPakFiles->add ("(empty)", IDC_MODEL_RECENTPAKFILES1);
	menuRecentPakFiles->add ("(empty)", IDC_MODEL_RECENTPAKFILES2);
	menuRecentPakFiles->add ("(empty)", IDC_MODEL_RECENTPAKFILES3);
	menuRecentPakFiles->add ("(empty)", IDC_MODEL_RECENTPAKFILES4);


	menuModel->add ("Load Model...", IDC_MODEL_LOADMODEL);
	menuModel->add ("Merge Model...", IDC_MODEL_MERGEMODEL);
	//menuModel->add ("Unload Models", IDC_MODEL_UNLOADMODEL);//hypov8 not really needed
	menuModel->addSeparator ();
	menuModel->addMenu ("Recent Models", menuRecentModels);
	menuModel->addSeparator ();
	menuModel-> add ("Save as MD2...", IDC_MODEL_MD2);
	menuModel->addSeparator ();
	menuModel->add ("Exit", IDC_MODEL_EXIT);

	menuPack->add ("Open PAK file...", IDC_MODEL_OPENPAKFILE);
	menuPack->add ("Close PAK file", IDC_MODEL_CLOSEPAKFILE);
	menuPack->addSeparator ();
	menuPack->addMenu ("Recent PAK files", menuRecentPakFiles);

	menuSkin->add ("Load Main Model Skin...", IDC_SKIN_MODELSKIN);
	menuSkin->add ("Load Model 2 Skin...", IDC_SKIN_MODELSKIN2);
	menuSkin->add ("Load Model 3 Skin...", IDC_SKIN_MODELSKIN3);
	menuSkin->add ("Load Model 4 Skin...", IDC_SKIN_MODELSKIN4);
	menuSkin->add ("Load Model 5 Skin...", IDC_SKIN_MODELSKIN5);
	menuSkin->add ("Load Model 6 Skin...", IDC_SKIN_MODELSKIN6);
	menuSkin->addSeparator ();
	menuSkin->add ("Load Background Texture...", IDC_SKIN_BACKGROUND);
	menuSkin->add ("Load Water Texture...", IDC_SKIN_WATER);
#ifdef WIN32
	menuSkin->addSeparator ();
	menuSkin->add ("Make Screenshot...", IDC_SKIN_SCREENSHOT);	//hypov8 slow
	//menuSkin->add ("Make AVI File...", IDC_SKIN_AVI);				//hypov8 buggy
#endif
	menuOptions->add ("Background Color...", IDC_OPTIONS_BGCOLOR);
	menuOptions->add ("Wireframe Color...", IDC_OPTIONS_WFCOLOR);
	menuOptions->add ("Shade Color...", IDC_OPTIONS_FGCOLOR);
	menuOptions->add ("Light Color...", IDC_OPTIONS_LIGHTCOLOR);
	menuOptions->addSeparator ();
	menuOptions->add ("Center Model", IDC_OPTIONS_CENTERMODEL);
	menuOptions->addSeparator ();
	menuOptions->add ("Generate Light Normals", IDC_OPTIONS_GENERATELIGHTNORMALS);

#ifdef WIN32
	menuHelp->add ("Goto Homepage...", IDC_HELP_GOTOHOMEPAGE);
	menuHelp->addSeparator ();
#endif
	menuHelp->add ("About", IDC_HELP_ABOUT);

	// create tabcontrol with subdialog windows
	UI_tab = new mxTab (this, 0, 0, 0, 0);
#ifdef WIN32
	SetWindowLong ((HWND)UI_tab->getHandle (), GWL_EXSTYLE, 0/* WS_EX_CLIENTEDGE*/); //hypov8
#endif
	/*UI_bar = new mxProgressBar (this, 0, 0, 0, 0, 0);
#ifdef WIN32
	SetWindowLong ((HWND)UI_bar->getHandle (), GWL_EXSTYLE, WS_EX_CLIENTEDGE);
#endif*/

	//Window *wAnim = new mxWindow (this, 0, 0, 0, 0);
	mxWindow *wInfo = new mxWindow (this, 0, 0, 0, 0);
	mxWindow *wView = new mxWindow (this, 0, 0, 0, 0);

	// and add them to the tabcontrol
	UI_tab->add (wView, "View");
	//_tab->add (wAnim, "Animation");
	UI_tab->add (wInfo, "Model Info");

	///////////////////////////////////
	// Create widgets for the View Tab
	///////////////////////////////////
	cRenderMode = new mxChoice (wView, 5, 5, 125, 22, IDC_RENDERMODE); //view mode
	cRenderMode->add ("Wireframe");
	cRenderMode->add ("Flat shaded");
	cRenderMode->add ("Smooth Shaded");
	cRenderMode->add ("Textured");
	cRenderMode->select (0);
	mxChoice *cTextureLimit = new mxChoice(wView, 130, 5, 60, 22, IDC_TEXTURELIMIT); //max texture rez
	cTextureLimit->add("512");
	cTextureLimit->add("256");
	cTextureLimit->add("128");
	cTextureLimit->select(0);
	mxToolTip::add(cTextureLimit, "Choose Texture Size Limit");

	//brightness slider
	mxLabel *lblBrightness = new mxLabel (wView, 65, 34, 60, 22, "Brightness"); //label
	mxSlider *slBrightness = new mxSlider (wView, 115, 31, 80, 22, IDC_BRIGHTNESS); //slider
	slBrightness->setRange (0, 100);
	slBrightness->setValue (5);
	mxToolTip::add (slBrightness, "Brightness - don't forget to reload the skin!");

	//check boxes
	cbWater = new mxCheckBox (wView, 5, 30, 50, 22, "Water", IDC_WATER);
	cbLight = new mxCheckBox (wView, 5, 55, 50, 22, "Light", IDC_LIGHT);
	mxCheckBox *cbShininess = new mxCheckBox (wView, 55, 55, 50, 22, "Shine", IDC_SHININESS);
	cbBackground = new mxCheckBox (wView, 110, 55, 80, 22, "Background", IDC_BACKGROUND);

	cbLight->setChecked (false);
	cbShininess->setChecked (false);

	///////////////////////////////////
	// Create widgets for the Animation
	///////////////////////////////////
	int ofT = 205;
	cAnim = new mxChoice (wView, 5+ofT, 5, 170, 22, IDC_ANIMATION);

	mxCheckBox *cbInterp = new mxCheckBox (wView, 5 + ofT, 29, 70, 22, "Interpolate", IDC_INTERPOLATE);
	cbInterp->setChecked (true);

	b1sPerson = new mxButton(wView, 95 + ofT, 31, 75, 20, "1st Person", IDC_1ST_PERSON); // button 1st person
	b1sPerson->setEnabled(true);
	mxToolTip::add(b1sPerson, "Set viewport to match ingame");

	mxSlider *slPitch = new mxSlider (wView, 5 + ofT, 55, 170, 22, IDC_PITCH); //slider
	slPitch->setRange (1, 200);
	slPitch->setValue (125);
	mxToolTip::add (slPitch, "Frame Animation Speed (Pitch)");

	///////////////////////////////////
	// animation navigator
	///////////////////////////////////
	int ofT2 = 220;
	bPause = new mxButton (wView, 180 + ofT2, 5, 79, 22, "Pause", IDC_PAUSE);

	bDecFrame = new mxButton (wView, 181 + ofT2, 31, 20, 20, "<", IDC_DECFRAME);
	bDecFrame->setEnabled (false);
	mxToolTip::add (bDecFrame, "Decrease Current Frame");

	leFrame = new mxLineEdit (wView, 202 + ofT2, 30, 35, 22, "0");
	leFrame->setEnabled (false);
	mxToolTip::add (leFrame, "Current Frame");

	bSetFrame = new mxButton (wView, 202 + ofT2, 51, 35, 20, "Set", IDC_FRAME);
	bSetFrame->setEnabled (false);
	mxToolTip::add (bSetFrame, "Set Current Frame");

	bIncFrame = new mxButton (wView, 238 + ofT2, 31, 20, 20, ">", IDC_INCFRAME);
	bIncFrame->setEnabled (false);
	mxToolTip::add (bIncFrame, "Increase Current Frame");


	////////////////////////////////////////
	// Create widgets for the Model Info Tab
	lModelInfo1 = new mxLabel (wInfo, 5, 5, 150, 125, "No Model.");
	lModelInfo2 = new mxLabel (wInfo, 150, 5, 150, 125, "");

	///////////////////////////
	// create the OpenGL window
	glw = new GlWindow (this, 0, 0, 0, 0, "", mxWindow::Normal);
#ifdef WIN32
	SetWindowLong ((HWND) glw->getHandle (), GWL_EXSTYLE, WS_EX_CLIENTEDGE);
#endif

	// finally create the pakviewer window
	pakViewer = new PAKViewer (this);

	loadRecentFiles ();
	initRecentFiles ();

	setBounds (20, 20, 690, 550);
	setVisible (true);

	//hypov8 file->open with
	if(loadmod[0]!=NULL)
	{
		if (!loadModel (loadmod, 0))
		{
			char str[256];
			sprintf_s (str, sizeof(str), "Error reading model: %s", loadmod);
			mxMessageBox (this, str, "ERROR", MX_MB_OK | MX_MB_ERROR);
		}
	}
}



MDXViewer::~MDXViewer ()
{
	saveRecentFiles ();
}

void
MDXViewer::reset_modelData(void)
{
	Skins = 0;
	Vertices = 0;
	Triangles = 0;
	GlCommands = 0;
	Frames = 0;
	setModelInfo(0, TEXTURE_MODEL_0);
	glw->loadModel(0, TEXTURE_MODEL_0);
	glw->loadModel(0, TEXTURE_MODEL_1);
	glw->loadModel(0, TEXTURE_MODEL_2);
	glw->loadModel(0, TEXTURE_MODEL_3);
	glw->loadModel(0, TEXTURE_MODEL_4);
	glw->loadModel(0, TEXTURE_MODEL_5);

	glw->loadTexture(0, TEXTURE_MODEL_0);
	glw->loadTexture(0, TEXTURE_MODEL_1);
	glw->loadTexture(0, TEXTURE_MODEL_2);
	glw->loadTexture(0, TEXTURE_MODEL_3);
	glw->loadTexture(0, TEXTURE_MODEL_4);
	glw->loadTexture(0, TEXTURE_MODEL_5);

	//glw->loadTexture(0, TEXTURE_BACKGROUND);
	//glw->loadTexture(0, TEXTURE_WATER);

	cAnim->removeAll();
	glw->setFrameInfo(0, 0);
	glw->redraw();
	//mIndex = 0;
}


int
MDXViewer::handleEvent (mxEvent *event)
{
	switch (event->event)
	{
	case mxEvent::Action:
	{
		switch (event->action)
		{
		case IDC_MODEL_LOADMODEL:
		case IDC_MODEL_MERGEMODEL:
		{
			int mIndex = glw->getModelIndex();
			if (mIndex < 6)
			{
				const char *ptr = mxGetOpenFileName(this, FILE_TYPE_MDX/*, "", "*.mdx"*/);

				if (ptr)
				{
					int i;
					char path[256];

					if (event->action == IDC_MODEL_LOADMODEL)
					{
						reset_modelData(); //hypov8
						mIndex = 0;
					}

					if (!loadModel(ptr, mIndex))
					{
						char str[300];

						sprintf_s(str, sizeof(str), "Error reading model: %s", ptr);
						mxMessageBox(this, str, "ERROR", MX_MB_OK | MX_MB_ERROR);
						break;
					}


					// now update recent files list
					strcpy_s(path, sizeof(path), "[m] ");
					strcat_s(path, sizeof(path), ptr);
					path[255] = 0; //hypov8 null

					for (i = 0; i < 4; i++)
					{
						if (!_stricmp(recentFiles[i], path))
							break;
					}

					// swap existing recent file
					if (i < 4)
					{
						char tmp[256];
						strcpy_s(tmp, 256, recentFiles[0]);
						strcpy_s(recentFiles[0], 256, recentFiles[i]);
						strcpy_s(recentFiles[i], 256, tmp);
					}

					// insert recent file
					else
					{
						for (i = 3; i > 0; i--)
							strcpy_s(recentFiles[i], 256, recentFiles[i - 1]);

						strcpy_s(recentFiles[0], 256, path);
					}

					initRecentFiles();
				}
				//else cancled?
			}
			else 
			{
				HWND hwID = (HWND)g_mdxViewer->getHandle();
				//max models
				MessageBox(hwID, "Limit of 6 models reached.\nUse \"Load Model\" instead ", "Note", MB_OK);
			}
		}
		break;

		case IDC_MODEL_UNLOADMODEL:
		case IDC_MODEL_UNLOADWEAPON:
			reset_modelData(); //hypov8
			break;

		case IDC_MODEL_OPENPAKFILE:
		case IDC_MODEL_OPENPAKFILE2:
		{
			const char *ptr = mxGetOpenFileName (this, FILE_TYPE_PAK/*,"", "*.pak"*/);
			if (ptr)
			{
				pakViewer->setLoadEntirePAK (event->action == IDC_MODEL_OPENPAKFILE);
				if (!pakViewer->openPAKFile (ptr))
				{
					mxMessageBox (this, "Error loading PAK file", "ERROR", MX_MB_OK | MX_MB_ERROR);
					break;
				}

				// update recent pak file list

				int i;

				for (i = 4; i < 8; i++)
				{
					if (!_stricmp(recentFiles[i], ptr))
						break;
				}

				// swap existing recent file
				if (i < 8)
				{
					char tmp[256];
					strcpy_s (tmp, 256, recentFiles[4]);
					strcpy_s (recentFiles[4], 256, recentFiles[i]);
					strcpy_s (recentFiles[i], 256, tmp);
				}

				// insert recent file
				else
				{
					for (i = 7; i > 4; i--)
						strcpy_s (recentFiles[i], 256, recentFiles[i - 1]);

					strcpy_s (recentFiles[4], 256, ptr);
				}

				initRecentFiles ();

				redraw ();
			}
		}
		break;

		case IDC_MODEL_CLOSEPAKFILE:
		{
			pakViewer->closePAKFile ();
			redraw ();
		}
		break;

		case IDC_MODEL_RECENTMODELS1:
		case IDC_MODEL_RECENTMODELS2:
		case IDC_MODEL_RECENTMODELS3:
		case IDC_MODEL_RECENTMODELS4:
		{
			int i = event->action - IDC_MODEL_RECENTMODELS1;
			bool isModel = recentFiles[i][1] == 'm';
			char *ptr = &recentFiles[i][4];

			if (!loadModel (ptr, isModel ? 0:1))
			{
				char str[300];

				sprintf_s (str, sizeof(str), "Error reading model: %s", ptr);
				mxMessageBox (this, str, "ERROR", MX_MB_OK | MX_MB_ERROR);
				break;
			}

			// update recent model list

			char tmp[256];			
			strcpy_s (tmp, sizeof(tmp), recentFiles[0]);
			strcpy_s (recentFiles[0], 256, recentFiles[i]);
			strcpy_s (recentFiles[i], 256, tmp);

			initRecentFiles ();
		}
		break;

		case IDC_MODEL_RECENTPAKFILES1:
		case IDC_MODEL_RECENTPAKFILES2:
		case IDC_MODEL_RECENTPAKFILES3:
		case IDC_MODEL_RECENTPAKFILES4:
		{
			int i = event->action - IDC_MODEL_RECENTPAKFILES1 + 4;
			pakViewer->setLoadEntirePAK (true);
			pakViewer->openPAKFile (recentFiles[i]);

			char tmp[256];			
			strcpy_s (tmp, recentFiles[4]);
			strcpy_s (recentFiles[4], 256, recentFiles[i]);
			strcpy_s (recentFiles[i], 256, tmp);

			initRecentFiles ();

			redraw ();
		}
		break;
		case IDC_MODEL_MD2:
		{
			mdx_model_t *model = glw->getModel(0);
			if (model)
			{
				const char *file = mxGetSaveFileName(this, FILE_TYPE_MD2/*, "*.md2"*/);
				if (file)
				{
					int blah;
					const char *test = NULL;
					if ((test = strstr(file, ".md2")) == NULL)
						strcat((char *)file, ".md2");

					blah = SaveAsMD2((char *)file, model); //glw->getModel(0) //hypov8
					if (blah != 1)
					{
						char str[256];

						sprintf_s(str, sizeof(str), "Error saving model: %s", file);
						mxMessageBox(this, str, "ERROR", MX_MB_OK | MX_MB_ERROR);
					}
				}
			}
		}
			break;
		case IDC_MODEL_EXIT:
			mx::setIdleWindow (0);
			mx::quit ();
			break;

		case IDC_SKIN_MODELSKIN:
		case IDC_SKIN_MODELSKIN2:
		case IDC_SKIN_MODELSKIN3:
		case IDC_SKIN_MODELSKIN4:
		case IDC_SKIN_MODELSKIN5:
		case IDC_SKIN_MODELSKIN6:
		{
//			const char *ptr = mxGetOpenFileName (this, 0, "*.pcx");
			const char *ptr = mxGetOpenFileName (this, FILE_TYPE_TGA/*, "", "*.tga"*/);
			if (ptr)
			{
				if(event->action==IDC_SKIN_MODELSKIN)
					glw->loadTexture (ptr, TEXTURE_MODEL_0);
				else if(event->action== IDC_SKIN_MODELSKIN2)
					glw->loadTexture (ptr, TEXTURE_MODEL_1);
				else if(event->action==IDC_SKIN_MODELSKIN3)
					glw->loadTexture (ptr, TEXTURE_MODEL_2);
				else if(event->action==IDC_SKIN_MODELSKIN4)
					glw->loadTexture (ptr, TEXTURE_MODEL_3);
				else if(event->action==IDC_SKIN_MODELSKIN5)
					glw->loadTexture (ptr, TEXTURE_MODEL_4);
				else if(event->action==IDC_SKIN_MODELSKIN6)
					glw->loadTexture (ptr, TEXTURE_MODEL_5);
				
				setRenderMode (3); //allow viewing skins
				glw->redraw ();
			}
		}
		break;

		case IDC_SKIN_BACKGROUND:
		case IDC_SKIN_WATER:
		{
			const char *ptr = mxGetOpenFileName (this, FILE_TYPE_TGA/*, "", "*.tga"*/);
			if (!ptr)
				break;

			if (glw->loadTexture (ptr, event->action == IDC_SKIN_BACKGROUND ? TEXTURE_BACKGROUND:TEXTURE_WATER))
			{
				if (event->action == IDC_SKIN_BACKGROUND)
				{
					cbBackground->setChecked (true);
					glw->setFlag (F_BACKGROUND, true);
				}
				else
				{
					cbWater->setChecked (true);
					glw->setFlag (F_WATER, true);
				}

				//setRenderMode (3);
				glw->redraw ();
			}
			else
				mxMessageBox (this, "Error loading texture.", "Kingpin Model Viewer", MX_MB_OK | MX_MB_ERROR);
		}
		break;

#ifdef WIN32
		case IDC_SKIN_SCREENSHOT:
		{
			const char *ptr = mxGetSaveFileName (this, FILE_TYPE_TGA/*, "*.tga"*/);
			if (ptr)
			{
				const char *test=NULL;
				if((test = strstr(ptr, ".tga"))==NULL)
					strcat((char *)ptr, ".tga");
				makeScreenShot (ptr);
			}
		//	int test = MakeAVI();
		}
		break;
		case IDC_SKIN_AVI:
		{
			int index = cAnim->getSelectedIndex ();
			if (index >= 0)
			{
				// set the animation
				int start, end, test;
				//FILE *te;
				initAVIAnimation (glw->getModel (0), index - 1, &start, &end);
				if(start==0 && end==0)
				{
					start = 1; 
					end = glw->getModel (0)->header.numFrames;
				}
			//	te = fopen("test.txt", "w+");
			//	fprintf(te, "%d start, %d end\n", start, end);
			//	fclose(te);
				if((test = MakeAVI(start, end))==0)
					mxMessageBox (this, "Error creating AVI File.", "Kingpin Model Viewer", MX_MB_OK | MX_MB_ERROR);
				
				glw->redraw ();
			}

		}
		break;
#endif

		case IDC_OPTIONS_BGCOLOR:
		case IDC_OPTIONS_FGCOLOR:
		case IDC_OPTIONS_WFCOLOR:
		case IDC_OPTIONS_LIGHTCOLOR:
		{
			float r, g, b;
			int ir, ig, ib;

			if (event->action == IDC_OPTIONS_BGCOLOR)
				glw->getBGColor (&r, &g, &b);
			else if (event->action == IDC_OPTIONS_FGCOLOR)
				glw->getFGColor (&r, &g, &b);
			else if (event->action == IDC_OPTIONS_WFCOLOR)
				glw->getWFColor (&r, &g, &b);
			else if (event->action == IDC_OPTIONS_LIGHTCOLOR)
				glw->getLightColor (&r, &g, &b);

			ir = (int) (r * 255.0f);
			ig = (int) (g * 255.0f);
			ib = (int) (b * 255.0f);
			if (mxChooseColor (this, &ir, &ig, &ib))
			{
				if (event->action == IDC_OPTIONS_BGCOLOR)
					glw->setBGColor ((float) ir / 255.0f, (float) ig / 255.0f, (float) ib / 255.0f);
				else if (event->action == IDC_OPTIONS_FGCOLOR)
					glw->setFGColor ((float) ir / 255.0f, (float) ig / 255.0f, (float) ib / 255.0f);
				else if (event->action == IDC_OPTIONS_WFCOLOR)
					glw->setWFColor ((float) ir / 255.0f, (float) ig / 255.0f, (float) ib / 255.0f);
				else if (event->action == IDC_OPTIONS_LIGHTCOLOR)
					glw->setLightColor ((float) ir / 255.0f, (float) ig / 255.0f, (float) ib / 255.0f);

				glw->redraw ();
			}
		}
		break;

		case IDC_OPTIONS_CENTERMODEL:
		{
			centerModel ();
			glw->redraw ();
		}
		break;

		case IDC_OPTIONS_GENERATELIGHTNORMALS:
			int i;
			for (i = 0; i < 5; i++)
			{
				mdx_generateLightNormals(glw->getModel(i));
			}
			glw->redraw ();
			break;

#ifdef WIN32
		case IDC_HELP_GOTOHOMEPAGE:
			ShellExecute (0, "open", "http://Kingpin.info", 0, 0, SW_SHOW);
			break;
#endif

		case IDC_HELP_ABOUT:
			mxMessageBox (this,
				"Info:" 
				"\tLoads mdx and md2 models. Support for 6 models.\n"
				"\tAbility to export .mdx models to .md2.\n"
				"\tPak viewer that loads models and textures.\n"
				"\n"
				"Keys:"
				"\tMouse-Left:\tdrag to rotate.\n"
				"\tMouse-Right:\tdrag to zoom.\n"
				"\tMouse-Middle:\tdrag to pan x-y (slow).\n"
				"\tMouse-Left+Shift:\tdrag to pan x-y (fast).\n"
				"\n"
				"bugs:"
				"\tOdd size TGA files not supported(eg 129* 64).\n"
				"\tNeed to left click first in PAK viewer.\n"
				"\n"
				"Thanks:"
				"\tTiCaL: Creator of MDX Viewer v1.1.\n"
				"\tMete Cirigan: for MD2 Viewer code.\n"
				"\tChris Cookson: for helping TiCaL.\n"
				"\n"
				"Build:\t" __DATE__ ".\n"
				"Email:\thypov8@kingpin.info\n"
				"Web:\thttp://Kingpin.info/",
				"Kingpin Model Viewer 1.1.3 by Hypov8", //title //hypov8 version
				MX_MB_OK | MX_MB_INFORMATION);
			break;

		// 
		// widget actions
		//
		//

		//
		// Model Panel
		//

		case IDC_RENDERMODE:
			setRenderMode (cRenderMode->getSelectedIndex ());
			glw->redraw ();
			break;

		case IDC_WATER:
			glw->setFlag (F_WATER, cbWater->isChecked ());
			glw->redraw ();
			break;

		case IDC_LIGHT:
			glw->setFlag (F_LIGHT, cbLight->isChecked ());
			glw->redraw ();
			break;

		case IDC_BRIGHTNESS:
			glw->setBrightness (((mxSlider *) event->widget)->getValue ());
			break;

		case IDC_SHININESS:
			glw->setFlag (F_SHININESS, ((mxCheckBox *) event->widget)->isChecked ());
			glw->redraw ();
			break;

		case IDC_BACKGROUND:
			glw->setFlag (F_BACKGROUND, ((mxCheckBox *) event->widget)->isChecked ());
			glw->redraw ();
			break;

		case IDC_TEXTURELIMIT:
		{
			int tl[3] = { 512, 256, 128 };
			int index = ((mxChoice *) event->widget)->getSelectedIndex ();
			if (index >= 0)
				glw->setTextureLimit (tl[index]);
		}
		break;

		//
		// Animation Panel
		//
		case IDC_ANIMATION:
		{
			int index = cAnim->getSelectedIndex ();
			if (index >= 0)
			{
				// set the animation
				initAnimation (glw->getModel (0), index - 1);

				// if we pause, update current frame in leFrame
				if (glw->getFlag (F_PAUSE))
				{
					char str[32];
					int frame = glw->getCurrFrame ();
					sprintf_s (str, sizeof(str), "%d", frame);
					leFrame->setLabel (str);
					glw->setFrameInfo (frame, frame);
				}

				glw->redraw ();
			}
		}
		break;

		case IDC_INTERPOLATE:
			glw->setFlag (F_INTERPOLATE, ((mxCheckBox *) event->widget)->isChecked ());
			break;

		case IDC_PITCH:
			glw->setPitch (200 -(float) ((mxSlider *) event->widget)->getValue () ); //hypov8 UI: invert
			break;

		case IDC_PAUSE: // Pause/Play
		{
			bool pause = !glw->getFlag (F_PAUSE);
			static int startFrame = 0, endFrame = 0, currFrame = 0, currFrame2 = 0;
			static float pol = 0;
			static int index;

			glw->setFlag (F_PAUSE, pause);
			bDecFrame->setEnabled (pause);
			leFrame->setEnabled (pause);
			bIncFrame->setEnabled (pause);
			bSetFrame->setEnabled (pause);

			if (pause)
			{
				char str[32];

				// store current settings
				startFrame = glw->getStartFrame ();
				endFrame = glw->getEndFrame ();
				currFrame = glw->getCurrFrame ();
				currFrame2 = glw->getCurrFrame2 ();
				pol = glw->d_pol;

				sprintf_s (str, sizeof(str), "%d", glw->getCurrFrame ());
				leFrame->setLabel (str);
				bPause->setLabel ("Play");

				index = cAnim->getSelectedIndex ();
			}
			else
			{

				glw->d_startFrame = startFrame;
				glw->d_endFrame = endFrame;
				glw->d_currFrame = currFrame;
				glw->d_currFrame2 = currFrame2;
				glw->d_pol = pol;

				bPause->setLabel ("Pause");

				int index2 = cAnim->getSelectedIndex ();
				if (index != index2 && index2 >= 0)
					initAnimation (glw->getModel (0), index2 - 1);
			}
		}
		break;

		case IDC_DECFRAME:
		{
			int frame = glw->getCurrFrame () - 1;
			glw->setFrameInfo (frame, frame);

			char str[32];
			sprintf_s (str, sizeof(str), "%d", glw->getCurrFrame ());
			leFrame->setLabel (str);
			glw->redraw ();
		}
		break;

		case IDC_FRAME:
		{
			const char *ptr = leFrame->getLabel ();
			if (ptr)
			{
				int frame = atoi (ptr);
				glw->setFrameInfo (frame, frame);

				char str[32];
				sprintf_s (str, sizeof(str), "%d", glw->getCurrFrame ());
				leFrame->setLabel (str);
				glw->redraw ();
			}
		}
		break;

		case IDC_INCFRAME:
		{
			int frame = glw->getCurrFrame () + 1;
			glw->setFrameInfo (frame, frame);

			char str[32];
			sprintf_s (str, sizeof(str), "%d", glw->getCurrFrame ());
			leFrame->setLabel (str);
			glw->redraw ();
			
		}
		break;

		case IDC_1ST_PERSON:
		{
			//hypov8 set viewport to 1st person
			glw->d_rotX =0;
			glw->d_rotY =90;
			glw->d_transX=0;
			glw->d_transY=0;
			glw->d_transZ=3; //move fov back slightly

			glw->redraw ();
			
		}
		break;
		}//end switch
	} // mxEvent::Action
	break;

	case mxEvent::Size:
	{
		int w = event->width;
		int h = event->height;
		int y = UI_mb->getHeight ();
#ifdef WIN32
#define HEIGHT 104 //hypov8 was 120 with progress bar
#else
#define HEIGHT 140
		h -= 40;
#endif

		//resize
		if (pakViewer->isVisible ())
		{
			w -= 170;
			pakViewer->setBounds (w, y, 170, h); //hypov8 UI: 
		}
		//fix tga bug for avi file....always make glw even width
		if (w&1) 
			w+=1;

		glw->setBounds (0, y, w/*- pakW*/, h - HEIGHT);
		//UI_bar->setBounds (0, y + h - 16, w-2, HEIGHT-105); //progress bar!!
		UI_tab->setBounds (0, y + h - HEIGHT, w-2, HEIGHT);
		
	}
	break;
	} // event->event


	return 1;
}



void
MDXViewer::redraw ()
{
	mxEvent event;
	event.event = mxEvent::Size;
	event.width = w2 ();
	event.height = h2 ();
	handleEvent (&event);
}



void
MDXViewer::makeScreenShot (const char *filename)
{
#ifdef WIN32
	glw->redraw ();
	int w = glw->w2 ();
	int h = glw->h2 ();

	mxImage *image = new mxImage ();
	if (image->create (w, h, 24))
	{
#if 0
		glReadBuffer (GL_FRONT);
		glReadPixels (0, 0, w, h, GL_RGB, GL_UNSIGNED_BYTE, image->data);
#else
		HDC hdc = GetDC ((HWND) glw->getHandle ());
		byte *data = (byte *) image->data;
		int i = 0;
		for (int y = 0; y < h; y++)
		{
			for (int x = 0; x < w; x++)
			{
				COLORREF cref = GetPixel (hdc, x, y);
				data[i++] = (byte) ((cref >> 0)& 0xff);
				data[i++] = (byte) ((cref >> 8) & 0xff);
				data[i++] = (byte) ((cref >> 16) & 0xff);
			}
		}
		ReleaseDC ((HWND) glw->getHandle (), hdc);
#endif
		if (!mxTgaWrite (filename, image))
			mxMessageBox (this, "Error writing screenshot.", "Kingpin Model Viewer", MX_MB_OK | MX_MB_ERROR);

		delete image;
	}
#endif
}



void
MDXViewer::setRenderMode (int mode)
{
	if (mode >= 0)
	{
		cRenderMode->select (mode);
		glw->setRenderMode (mode);

		// disable light, if not needed
	//	glw->setFlag (F_LIGHT, mode != 0);
	//	cbLight->setChecked (mode != 0);
	}
}



void
MDXViewer::centerModel ()
{
	if (glw->getModel (0))
	{
		float minmax[6];//, minmax2[6], minmax3[6], minmax4[6], minmax5[6], minmax6[6];
//		float model1=0, model2=0, model3=0, model4=0, model5=0, model6=0, average;
		
//		int count=0;
		mdx_getBoundingBox (glw->getModel (0), minmax);
/*		model1 = (minmax[3]+minmax[2])/2;
		
		if(glw->getModel (1)){
			mdx_getBoundingBox (glw->getModel (1), minmax2);
			count++;
			model2 = (minmax2[3]+minmax2[2])/2;
		}
		if(glw->getModel (2)){
			mdx_getBoundingBox (glw->getModel (2), minmax3);
			count++;
			model3 = (minmax3[3]+minmax3[2])/2;
		}
		if(glw->getModel (3)){
			mdx_getBoundingBox (glw->getModel (3), minmax4);
			count++;
			model4 = (minmax4[3]+minmax4[2])/2;
		}
		if(glw->getModel (4)){
			mdx_getBoundingBox (glw->getModel (4), minmax5);
			count++;
			model5 = (minmax5[3]+minmax5[2])/2;
		}
		if(glw->getModel (5)){
			mdx_getBoundingBox (glw->getModel (5), minmax6);
			count++;
			model6 = (minmax6[3]+minmax6[2])/2;
		}
		
		average = (model1+model2+model3+model4+model5+model6)/count;*/
		
		// center vertically
		glw->d_transY = (minmax[3] + minmax[2]) / 2;
//		glw->d_transY = average;

		// adjust distance
		float dx = minmax[1] - minmax[0];
		float dy = minmax[3] - minmax[2];
		float dz = minmax[5] - minmax[4];

		float d = dx;
		if (dy > d)
			d = dy;

		if (dz > d)
			d = dz;

		glw->d_transZ = d * 1.2f;
		glw->d_transX = 0;
		glw->d_rotX = glw->d_rotY = 0.0f;
	}
}



bool
MDXViewer::loadModel (const char *ptr, int pos)
{
	mdx_model_t *model;

	model = glw->loadModel (ptr,  pos);
	
	if (!model)
		return false;

	if (pos == TEXTURE_MODEL_0)
	{	
		initAnimation (model, -1);
		setModelInfo (model, pos);
		glw->loadTexture (modelTexNames[pos], pos); //hypov8
		//setRenderMode (0);
		centerModel ();					
	}
	else if (pos == TEXTURE_MODEL_1)
	{
		setModelInfo (model, pos);
		glw->loadTexture (modelTexNames[pos], pos);
	}
	else if (pos == TEXTURE_MODEL_2)
	{
		setModelInfo (model, pos);
		glw->loadTexture (modelTexNames[pos], pos);
	}
	else if (pos == TEXTURE_MODEL_3)
	{
		setModelInfo (model, pos);
		glw->loadTexture (modelTexNames[pos], pos);
	}
	else if (pos == TEXTURE_MODEL_4)
	{
		setModelInfo (model, pos);
		glw->loadTexture (modelTexNames[pos], pos);
	}
	else if (pos == TEXTURE_MODEL_5)
	{
		setModelInfo (model, pos);
		glw->loadTexture (modelTexNames[pos], pos);
	}

	glw->redraw ();

	return true;
}



void
MDXViewer::setModelInfo (mdx_model_t *model, int pos)
{
	static char str[1024];
	static char str2[1024];

	if(pos == TEXTURE_MODEL_0) //first model
	{
		if (model)
		{
			sprintf_s (str, sizeof(str),
				"Skins: %d\n"
				"Vertices: %d\n"
				"Triangles: %d\n"
				"GlCommands: %d\n"
				"Frames: %d\n",
				model->header.numSkins,
				model->header.numVertices,
				model->header.numTriangles,
				model->header.numGlCommands,
				model->header.numFrames
				);

				Skins = model->header.numSkins;
				Vertices = model->header.numVertices;
				Triangles = model->header.numTriangles;
				GlCommands = model->header.numGlCommands;
				Frames = model->header.numFrames;

			sprintf_s (str2, sizeof(str2),
				"BBox (Frame 0):\n"
				"min (%5.1f, %5.1f, %5.1f)\n"
				"max (%5.1f, %5.1f, %5.1f)\n",
				model->min[0],model->min[1],model->min[2],
				model->max[0], model->max[1], model->max[2]);
		}
		else
			strcpy_s (str, sizeof(str), "No Models.");
		
		
	}
	else //rest of models
	{
		if (model)
		{
			Skins += model->header.numSkins;
			Vertices += model->header.numVertices;
			Triangles += model->header.numTriangles;
			GlCommands += model->header.numGlCommands;
			sprintf_s (str, sizeof(str),
				"Skins: %d\n"
				"Vertices: %d\n"
				"Triangles: %d\n"
				"GlCommands: %d\n"
				"Frames: %d\n",
				Skins,
				Vertices,
				Triangles,
				GlCommands,
				Frames
				);
		}
	}
	lModelInfo1->setLabel (str);
	lModelInfo2->setLabel (str2);
}



void
MDXViewer::initAnimation (mdx_model_t *model, int animation)
{
	cAnim->removeAll ();

	if (!model)
		return;

	int count = mdx_getAnimationCount (model);

	//if (model->header.numFrames==1)
	//	bPause->setEnabled(false);
	//else
	//	bPause->setEnabled(true);

	cAnim->add ("<All Animations>");

	for (int i = 0; i < count; i++)
		cAnim->add (mdx_getAnimationName (model, i));

	int startFrame, endFrame;
	mdx_getAnimationFrames (model, animation, &startFrame, &endFrame);
	glw->setFrameInfo (startFrame, endFrame);

	if (animation == -1)
		glw->setFrameInfo (0, model->header.numFrames - 1);

	cAnim->select (animation + 1);
}

void
MDXViewer::initAVIAnimation (mdx_model_t *model, int animation, int *startFrame, int *endFrame)
{
	if (!model)
		return;

	mdx_getAnimationFrames (model, animation, startFrame, endFrame);
	
}


int MDXViewer::MakeAVI(int start, int end)
{
#if 1 //hypov8
	return 0;
#else
	if(!glw->getModel(0)) return 0;
		
	char *outfn = (char *)malloc(256);
	int fps = 0;
	int img_width, img_height;
	int frame_num, avi_frame;
	char pattern[256];
	int frame_begin, frame_end, frame_step;
	char fn[1024];

	mxImage *in = new mxImage ();
	mxImage *tga = new mxImage ();

	LPBITMAPINFOHEADER bih;
	PAVIFILE af = NULL;
	AVISTREAMINFO asi;
	PAVISTREAM as = NULL, ascomp = NULL;
	AVICOMPRESSOPTIONS opts;
	AVICOMPRESSOPTIONS FAR * aopts[1] = {&opts};
	HANDLE dib;
	int memsize;
	int test2 = NULL;
	char *test3 = NULL;

	outfn = (char *)mxGetSaveFileName (this, FILE_TYPE_AVI/*, "*.avi"*/);
	if(!outfn)
		return 0;

	if((test3 = strstr(outfn, ".avi"))==NULL)
 			strcat_s(outfn, 256, ".avi");
	
	fps = 10;
	//strcpy_s(pattern, sizeof(pattern), "shot%04d.tga"); //hypov8 moved below

	//frame_begin = 1;
	frame_begin = start;
	//frame_end = glw->getModel(0)->header.numFrames;
	frame_end = end;
	frame_step = 1;

	if (frame_end < frame_begin) 
		ERR("Invalid end frame!");
	if (frame_step < 1) 
		ERR("Invalid frame step!");
	
	//check for existance and format of first frame 
	//sprintf_s(fn, sizeof(fn),pattern, frame_begin);
	sprintf_s(fn, sizeof(fn), "shot%04d.tga", frame_begin);

	char *test = (char *)malloc(256);
	GetTempPath (256, test);
	strcat_s(test,256, "/");
	strcat_s(test,256, fn);
	makeScreenShot (test);
	
	tga = mxTgaRead(test);
	if (!tga) return 0;

	img_width = tga->width;
	img_height = tga->height;
	remove (test);

	// allocate DIB 
	memsize = sizeof(BITMAPINFOHEADER) + (img_width*img_height*3);
	dib = GlobalAlloc(GHND,memsize);
	bih = (LPBITMAPINFOHEADER)GlobalLock(dib);

	bih->biSize = sizeof(BITMAPINFOHEADER);
	bih->biWidth = img_width;
	bih->biHeight = img_height; // negative = top-down 
	bih->biPlanes = 1;
	bih->biBitCount = 24; // B,G,R 
	bih->biCompression = BI_RGB;
	bih->biSizeImage = memsize - sizeof(BITMAPINFOHEADER);
	bih->biXPelsPerMeter = 0;
	bih->biYPelsPerMeter = 0;
	bih->biClrUsed = 0;
	bih->biClrImportant = 0;

	if (HIWORD(VideoForWindowsVersion()) < 0x010a)
		ERR("VFW is too old");

	AVIFileInit();
	AVIERR( AVIFileOpen(&af, outfn, OF_WRITE | OF_CREATE, NULL) );

	// header 
	memset(&asi, 0, sizeof(asi));
	asi.fccType	= streamtypeVIDEO;// stream type
	asi.fccHandler	= 0;
	asi.dwScale	= 1;
	asi.dwRate	= fps;
	asi.dwSuggestedBufferSize  = bih->biSizeImage;
	SetRect(&asi.rcFrame, 0, 0, img_width, img_height);

	AVIERR( AVIFileCreateStream(af, &as, &asi) );

	memset(&opts, 0, sizeof(opts));

	if (!AVISaveOptions(NULL, 0, 1, &as,
		(LPAVICOMPRESSOPTIONS FAR *) &aopts))
	{
		return 0;
	}

	AVIERR( AVIMakeCompressedStream(&ascomp, as, &opts, NULL) );

	AVIERR( AVIStreamSetFormat(ascomp, 0,
			       bih,	    // stream format
			       bih->biSize) ); // format size

	avi_frame = 0;
	frame_num = 1;

	int current = glw->getCurrFrame ();
	
	//bar->setTotalSteps(glw->getModel(0)->header.numFrames);
	bar->setTotalSteps(end-start);

	mxImage *flip = new mxImage ();

	int cnt=0;
	//for (int frame = 1;frame<=glw->getModel(0)->header.numFrames;frame++,frame_num++)
	for (int frame = start;frame<=end;frame++,frame_num++, cnt++)
	{		
		//Increment progress bar
		bar->setValue(cnt);
				
		//init strings
		char *filename = (char *)malloc(256);
		char *str = (char *)malloc(32);
		char *path = (char *)malloc(256);

		//Set to first frame and take a shot
		glw->setFrameInfo (frame, frame);
		sprintf_s (str, sizeof(byte) * 32, "%d", glw->getCurrFrame ());
		leFrame->setLabel (str);
		glw->redraw ();
		sprintf_s (filename, sizeof(byte)*256, "/shot%04d.tga", frame);
		GetTempPath (256, path);
		strcat_s (path, 256, filename);
		makeScreenShot (path);

		//Create an AVI file
		in = mxTgaRead(path);
		if (!in) return 0;

		//Flip the image
		flip->data = (unsigned char *)malloc(in->height*in->width*3);
		flip->height = in->height;
		flip->width = in->width;

		int len = (flip->height*flip->width*3)-1;

		for (int i = 0;i <= (flip->height*flip->width * 3) - 1;i++, len--)
			flip->data[i] = in->data[len];
		
		for (long y=0; y<flip->height; y++) 
		{ 
			long left = y*flip->width*3; 
			long right = left + flip->width*3 - 3; 
			for (long x=0; x<flip->width*0.5; x++) 
			{ 
				unsigned char tempRed   = flip->data[left]; 
				unsigned char tempGreen = flip->data[left+1]; 
				unsigned char tempBlue  = flip->data[left+2]; 
				
				flip->data[left]   = flip->data[right]; 
				flip->data[left+1] = flip->data[right+1]; 
				flip->data[left+2] = flip->data[right+2]; 
				
				flip->data[right]   = tempRed; 
				flip->data[right+1] = tempGreen; 
				flip->data[right+2] = tempBlue; 
				
				left  += 3; 
				right -= 3; 
			} 
		} 

		memcpy((LPBYTE)(bih) + sizeof(BITMAPINFOHEADER), 
			flip->data, bih->biSizeImage);

		AVIERR( AVIStreamWrite(ascomp,	// stream pointer
			avi_frame,		// time of this frame
			1,			// number of frames to write
			(LPBYTE)(bih) + sizeof(BITMAPINFOHEADER),
			bih->biSizeImage,	// size of this frame
			0,			// flags
			NULL,
			NULL) );
	
		avi_frame++;

		//Clean up the mess
		remove (path);
		delete [] str;
		delete [] filename;
		delete [] path;

	}

	//Close AVI Stuff
	if (ascomp) AVIStreamClose(ascomp);
	if (as) AVIStreamClose(as);
	if (af) AVIFileClose(af);
	GlobalUnlock(dib);
	AVIFileExit();

	//Set to initial frame
	char *str = (char *)malloc(32);
	glw->setFrameInfo (current, current);
	sprintf_s (str, sizeof(byte) * 32, "%d", glw->getCurrFrame ());
	leFrame->setLabel (str);
	glw->redraw ();
	delete [] str;

	free(in->data);
	free(tga->data);
	free(flip->data);
	
	in->~mxImage();
	tga->~mxImage();
	flip->~mxImage();

	delete in;
	delete tga;
	delete flip;
	
	bar->setValue(0);
	int n = _heapmin();
	return 1;
#endif
}


int
main (int argc, char *argv[])
{
	int ret;
	//
	// make sure, we start in the right directory
	//
	SetCurrentDirectory (mx::getApplicationPath ());

	mx::init (argc, argv);

	if(argc>1)
	{

		//loadmod = (char *)malloc(256);
		strcpy_s(loadmod, sizeof(loadmod), argv[1]);
	}

	g_mdxViewer = new MDXViewer ();
	g_mdxViewer->setMenuBar (g_mdxViewer->getMenuBar ());

	ret =  mx::run ();
	mx::cleanup();

	return ret;
}
