#ifndef MParameterObject_Header
#define MParameterObject_Header

#include "ModelGeneric.h"

namespace Aztec {
  
  // forward declare the scene class.
  class MParameterObject;
  class MScene;
  
  typedef MRefCountedPtr<MParameterObject> MParameterObjectPtr;
  typedef MRefCountedPtr<MScene> MScenePtr;
}

#include "MStr.h"
#include "MMath.h"
#include "MDAGNode.h"

namespace Aztec {

  /**
   * MParameterObject represents a parameter of some type. A
   * parameter is a value of a specific type, and has three names 
   * associated with it. It has a short name, long name and a friendly name.
   * The names are used to identify the parameter so it can be set 
   * without knowledge of the object that contains the parameters.
   */
  class MGENEXPORT MParameterObject : public virtual MDAGNode {
  public:
    static const AztecFlags FLAG_VISIBLE;
    static const AztecFlags FLAG_READONLY;
    static const AztecFlags FLAG_CONNECTABLE;
    static const AztecFlags FLAG_STORABLE;

    // TODO: change these to static const ints once we move from
    // the VC6 compiler.
    typedef enum { TYPE_INTEGER, 
                   TYPE_FLOAT,
                   TYPE_BOOLEAN,
                   TYPE_STRING,
                   TYPE_VECTOR,
                   TYPE_MATRIX,
                   TYPE_OBJECT,
                   TYPE_IMAGE,
                   TYPE_PARTICLE_ACTION,
                   TYPE_AGGREGATE,
                   TYPE_ARRAY } TypeEnum;

    typedef enum { MEANING_ANY,
                   MEANING_LENGTH,
                   MEANING_MASS,
                   MEANING_POSITION,
                   MEANING_ORIENTATION,
                   MEANING_SCALING,
                   MEANING_COLOUR,
                   MEANING_TIME,
                   MEANING_FILENAME,
                   MEANING_IMAGE_FILENAME } MeaningEnum;

    /**
     * This returns the type of the data. This could be an integer, floating
     * point, string, an object, mesh, matrix, any basic data type.
     */
    virtual int getDataType() const = 0;

    /**
     * This is the specific meaning of the data. For example,
     * even though two string parameters are interchangable, one
     * might mean a name of something, the other might be a filename.
     * Both are string types, but essentially mean different things.
     */
    virtual int getDataMeaning() const = 0;

    /**
     * Gets the short name of this parameter.
     */
    virtual MStr getShortName() const = 0;

    /**
     * Gets the long name of this parameter.
     */
    virtual MStr getLongName() const = 0;

    /**
     * Gets the friendly name of this parameter.
     */
    virtual MStr getFriendlyName() const = 0;

    /**
     * This gets the DAGNode which contains this parameter. This method does 
     * not return the parameter object list that contains this parameter.
     */
    virtual MBaseObject* getOwner() const = 0;

    /**
     * Sets the owning Parameter object list that the parameter belongs to.
     */
    virtual void setOwner(MBaseObject *newOwner) = 0;

    /**
     * Is this parameter used for receiving information or for outputting
     * information.
     *
     * @return true if this parameter acts as an input to its owning 
     *         parameter list, or false if it acts as an output.
     */
    virtual bool isInputParameter() = 0;

    // Getting and setting values
    virtual bool getValueString(MStr &value) = 0;
    virtual bool getValueInteger(int &value) = 0;
    virtual bool getValueFloat(float &value) = 0;
    virtual bool getValueBoolean(bool &value) = 0;
    virtual bool getValueVector(MVector3 &value) = 0;

    virtual bool setValueString(const MStr &value) = 0;
    virtual bool setValueInteger(int value) = 0;
    virtual bool setValueFloat(float value) = 0;
    virtual bool setValueBoolean(bool value) = 0;
    virtual bool setValueVector(const MVector3 &value) = 0;

    /**
     * This retrieves the value from the source parameter, and places it into
     * this parameter. This performs no other operation other than copying
     * the value. All other information about a parameter stays the same,
     * it is only the value that changes.
     */
    virtual bool setValueParameter(const MParameterObjectPtr &value) = 0;

    virtual bool createKeyAt(long time) = 0;

    /**
     * This method performs a precise as possible copy from the source 
     * parameter into this one. This will include any information such as
     * input parameters, wether we are an input or output parameter, and
     * any other normally immutable data. This method is primarily used for
     * keeping track of undo information, and it is generally undesirable 
     * to be called unless in special circumstances.
     */
    virtual bool setFromParameter(MParameterObjectPtr src) = 0;

    virtual bool isConnectable() = 0;
    virtual bool isConnected() = 0;
    virtual bool setInputParameter(const MParameterObjectPtr &input) = 0;
    virtual MParameterObjectPtr getInputParameter() = 0;

    /**
     * This creates a new parameter that is a separate copy of this parameter,
     * and contains all the information important to this parameter. When the
     * setFromParameter() method is called, with the copy as the source, the
     * parameter must return to its original state.
     */
    virtual MParameterObjectPtr createCopy() = 0;

    // helper methods
    /**
     * This tests the given name against this parameters short, long and friendly name.
     */
    bool isNamed(const MStr &name);

    bool isVisible();

    void setVisible(bool visible);

    /**
     * Check sto see if this parameter should be written out or read in when 
     * saving files. 
     *
     * @return true if the parameter should be saved, false if it shouldn't.
     */
    bool isStorable();

    /**
     * This sets whether or not a parameter should be saved out or not,
     *
     * @param storable True if the parameter shuold be saved out with a file, 
     *                 false if it shouldn't.
     */
    void setStorable(bool storable);

    template <class ParamType>
    static MParameterObjectPtr createParameterCopy(ParamType *source) {
      MParameterObjectPtr newParam = new ParamType(source->getShortName(), source->getLongName(), source->getFriendlyName());
      newParam->setFromParameter(source);
      return newParam;
    }
  };

  class MGENEXPORT MParameterObjectImpl : public virtual MParameterObject {
  public:
    MParameterObjectImpl(const MStr &shortName, 
                         const MStr &longName, 
                         const MStr &friendlyName,
                         const MParameterObjectPtr &inputParameter);

    ~MParameterObjectImpl();

    void setInputParameter(bool input);

    // MParameterObject methods
    MStr getShortName() const;
    MStr getLongName() const;
    MStr getFriendlyName() const;

    MBaseObject* getOwner() const;
    void setOwner(MBaseObject *newOwner);
    bool isInputParameter();

    bool createKeyAt(long time);
    bool setFromParameter(MParameterObjectPtr src);

    bool isConnectable();
    bool isConnected();
    bool disconnect();
    bool setInputParameter(const MParameterObjectPtr &input);
    MParameterObjectPtr getInputParameter();

    // MBaseObject methods;
    bool doUpdateObject();

  protected:
    MStr shortName;
    MStr longName;
    MStr friendlyName;
    MParameterObjectPtr connectedInput;
    MBaseObject* owner;
    bool isInputParam;
    MBaseObjectPtr connectedInputOwner;
  };

}

#endif
