#ifndef MLISTSTREES_H
#define MLISTSTREES_H

#include "ModelGeneric.h"

namespace Aztec {
  class MObjectNode;
  class MListObjectNode;
  class MTreeObjectNode;
  class MBaseObjectList;
  class MBaseObjectTree;
  class MSortedNumberList;
  
  typedef MRefCountedPtr<MObjectNode> MObjectNodePtr;
  typedef MRefCountedPtr<MListObjectNode> MListObjectNodePtr;
  typedef MRefCountedPtr<MTreeObjectNode> MTreeObjectNodePtr;
  typedef MRefCountedPtr<MBaseObjectList> MBaseObjectListPtr;
  typedef MRefCountedPtr<MBaseObjectTree> MBaseObjectTreePtr;
  typedef MRefCountedPtr<MSortedNumberList> MSortedNumberListPtr;
}

#include "MBaseObject.h"
#include <vector>
#include <stack>

#include <MBaseUndoNode.h>
#include <MUndoManager.h>

namespace Aztec {
  
  //----------------------------------------------------------------------------------------
  //  MListObjectNode
  //----------------------------------------------------------------------------------------
  // Because this class does no actual processing, their memebers are public.
  class MGENEXPORT MListObjectNode : public MObjectNode, public MUndoableObject {
  public:
    MListObjectNode();
    virtual ~MListObjectNode();

    inline const MListObjectNodePtr getPrev() const {
      return m_Prev;
    }

    inline void setPrev(const MListObjectNodePtr &prev) {
      doEnsureUndo(undo, this);
      m_Prev = prev;
    }

    inline const MListObjectNodePtr getNext() const {
      return m_Next;
    }

    inline void setNext(const MListObjectNodePtr &next) {
      doEnsureUndo(undo, this);
      m_Next = next;
    }

    // MUndoableObject methods
    void finishWithUndoNode();

  private:
    MListObjectNodePtr m_Prev, m_Next;

    class Undo : public MBaseUndoNode {
    public:
      Undo(MListObjectNode *obj);

      MUndoableObject* getObject();
      void undo();
      void redo();

    private:
      MListObjectNodePtr prev;
      MListObjectNodePtr next;
      MRefCountedPtr<MListObjectNode> object;
    };

    MRefCountedPtr<Undo> undo;

    friend class Undo;
  };
  
  //----------------------------------------------------------------------------------------
  //  MTreeObjectNode
  //----------------------------------------------------------------------------------------
  // Because this class does no actual processing, their memebers are public.
  class MGENEXPORT MTreeObjectNode : public MObjectNode, public MUndoableObject {
  protected:
    
  public:
    MTreeObjectNode();
    virtual ~MTreeObjectNode();
    
    inline const MTreeObjectNodePtr getParent() const {
      return m_Parent;
    }
    inline const MTreeObjectNodePtr getFirstChild() const {
      return m_FirstChild;
    }
    inline const MTreeObjectNodePtr getNextSibling() const {
      return m_NextSibling;
    }
    inline const MTreeObjectNodePtr getPrevSibling() const {
      return m_PrevSibling;
    }

    inline void setParent(const MTreeObjectNodePtr &node) {
      doEnsureUndo(undo, this);
      m_Parent = node;
    }
    inline void setFirstChild(const MTreeObjectNodePtr &node) {
      doEnsureUndo(undo, this);
      m_FirstChild = node;
    }
    inline void setNextSibling(const MTreeObjectNodePtr &node) {
      doEnsureUndo(undo, this);
      m_NextSibling = node;
    }
    inline void setPrevSibling(const MTreeObjectNodePtr &node) {
      doEnsureUndo(undo, this);
      m_PrevSibling = node;
    }

    int isChildOf(const MTreeObjectNodePtr &Node);
    int isChildOfObject(const MBaseObjectPtr &Obj);
    int isChildOfFlagged(AztecFlags Flags);

    // MUndoableObject methods
    void finishWithUndoNode();

  private:
    MTreeObjectNodePtr m_Parent, m_FirstChild, m_NextSibling, m_PrevSibling;

    class Undo : public MBaseUndoNode {
    public:
      Undo(MTreeObjectNode *obj);

      MUndoableObject* getObject();
      void undo();
      void redo();

    private:
      MTreeObjectNodePtr parent;
      MTreeObjectNodePtr child;
      MTreeObjectNodePtr prev;
      MTreeObjectNodePtr next;
      MRefCountedPtr<MTreeObjectNode> object;
    };

    MRefCountedPtr<Undo> undo;

    friend class Undo;
  };
  
  
  typedef int (*MEnumFunc)(MObjectNodePtr Node, DWORD LParam);
  
  //----------------------------------------------------------------------------------------
  //  MBaseObjectList
  //----------------------------------------------------------------------------------------
  class MGENEXPORT ListData {
  public:
    ListData();
    ListData(const ListData &src);
    virtual ~ListData();

    ListData& operator=(const ListData &rhs);

    MListObjectNodePtr m_Head, m_Tail, m_CurPos;
    int                m_Count;
    std::stack<MListObjectNodePtr> m_PosStack;
  };

  class MGENEXPORT MBaseObjectList : public MBaseObject, public ListData {
    
  public:
    // Construction / Destruction
    MBaseObjectList();
    virtual ~MBaseObjectList();
    
    // Class related
    virtual MStr getClassName() {return MStr("MBaseObjectList");};
    virtual MStr getParentClassName() {return MStr("MBaseObject");};
    virtual MBaseObjectPtr createNew();
    
    MListObjectNodePtr addHead(MBaseObjectPtr Obj);
    MListObjectNodePtr addTail(MBaseObjectPtr Obj);
    MListObjectNodePtr insertAfter(MListObjectNodePtr After, MBaseObjectPtr Obj);
    MListObjectNodePtr findNode(MBaseObjectPtr Obj);
    void removeNode(MListObjectNodePtr Node);
    
    MBaseObjectPtr getHead();
    MBaseObjectPtr getTail();
    MListObjectNodePtr getHeadNode();
    MListObjectNodePtr getTailNode();
    MBaseObjectPtr removeHead();
    MBaseObjectPtr removeTail();
    void removeAllNodes();
    
    MListObjectNodePtr findObjectWithClass(MStr Name);
    MObjectNodePtr enumList(MEnumFunc Func, DWORD lParam);
    
    void beginIteration();
    void endIteration();
    void restartIteration();
    MBaseObjectPtr getNext();
    
    int getCount();
    bool isEmpty();

    // MUndoableObject methods
    void finishWithUndoNode();

  private:
    class Undo : public MBaseUndoNode {
    public:
      Undo(MBaseObjectList *node);

      MUndoableObject* getObject();
      void undo();
      void redo();

    private:
      ListData stored;
      MRefCountedPtr<MBaseObjectList> object;
    };

    MRefCountedPtr<Undo> nodeUndo;
    bool undoEnabled;
    void ensureUndo();

    friend class Undo;

  };
  
  
  enum MTreeTraverseType {ttParentChildSib};
  
  class MGENEXPORT TreeData {
  public:
    TreeData();
    TreeData(const TreeData &src);
    virtual ~TreeData();

    TreeData& operator=(const TreeData &rhs);

    MTreeObjectNodePtr m_Root;
    int                m_Count;
  };

  //----------------------------------------------------------------------------------------
  //  MBaseObjectTree
  //----------------------------------------------------------------------------------------

  class MGENEXPORT MBaseObjectTree : public MBaseObject, public TreeData {
  protected:
    class MGENEXPORT Iterator {
    public:
      Iterator() { };
      Iterator(const Iterator &iter) {
        m_TraverseType = iter.m_TraverseType;
        m_TraverseStack = iter.m_TraverseStack;
        m_LastNode = iter.m_LastNode;
      }

      MTreeTraverseType  m_TraverseType;
      std::vector<MTreeObjectNodePtr> m_TraverseStack;
      MTreeObjectNodePtr m_LastNode;
    };

    Iterator m_CurIter;
    std::stack<Iterator> m_IterStack;

  public:
    MBaseObjectTree();
    virtual ~MBaseObjectTree();
    
    // Class related
    virtual MStr getClassName() {return MStr("MBaseObjectTree");};
    virtual MStr getParentClassName() {return MStr("MBaseObject");};
    virtual MBaseObjectPtr createNew();
    
    // Adds the given object to the tree, which is a child to the node Parent. if Parent is NULL,
    // Then the new node is placed at the root level of the tree.
    MTreeObjectNodePtr addNode(MBaseObjectPtr Obj, MTreeObjectNodePtr Parent = NULL);
    // This deletes the given node, but moving all of its children up one level to where the 
    // parent is
    void deleteNodeKeepChildren(MTreeObjectNodePtr Node);
    // This deletes the given node, along with all of its children.
    void deleteNodeAndChildren(MTreeObjectNodePtr Node);
    
    // Sets the parent of Node to ParentNode. If ParentNode is NULL, the Node becoes a root
    // node sibling
    void setParent(MTreeObjectNodePtr Node, MTreeObjectNodePtr ParentNode = NULL);
    
    // Searches Node and its children for Obj
    MTreeObjectNodePtr findObject(MBaseObjectPtr Obj, MTreeObjectNodePtr Node = NULL);
    MBaseObjectPtr findObject(const MStr &Name);
    MTreeObjectNodePtr findObjectNode(const MStr &Name);
    
    void beginIteration(MTreeTraverseType  Type = ttParentChildSib);
    void endIteration();
    void restartIteration();
    MBaseObjectPtr getNext();
    MBaseObjectPtr getCurrent();
    MTreeObjectNodePtr getCurrentNode();
    
    int getCount() {return m_Count;};

    // MUndoableObject methods
    void finishWithUndoNode();

  private:
    class Undo : public MBaseUndoNode {
    public:
      Undo(MBaseObjectTree *node);

      MUndoableObject* getObject();
      void undo();
      void redo();

    private:
      TreeData stored;
      MRefCountedPtr<MBaseObjectTree> object;
    };

    MRefCountedPtr<Undo> nodeUndo;
    bool undoEnabled;
    void ensureUndo();

    friend class Undo;

  };
  
  
  //----------------------------------------------------------------------------------------
  //  MSortedNumberList
  //----------------------------------------------------------------------------------------
  class MGENEXPORT MSortedNumberList : public MBaseObject
  {
  protected:
    int      m_Num, m_NumAllocated;
    int      *m_Data;
    
  public:
    MSortedNumberList();
    virtual ~MSortedNumberList();
    
    int add(int Num);      // Adds a number to the list. Returns 0 if it already exists
    int remove(int Num);   // Adds a number to the list. Returns 0 if it does not exists
    int find(int Num);         // returns the index of the item in question. -1 if not found
    void removeAll();          // Removes all items in the list.
    int getItem(int Index);  // Returns the item at the given index.
    int getNumItems() {return m_Num;};
    
  protected:
    // Inserts the given value at the given location, shifting all numbers after it 
    // up one spot. Does no checking or sorting
    void insertAt(int Index, int Value);   
  };
  
  
}

#endif
