
#ifndef QGLCanvas_Header
#define QGLCanvas_Header

#include <string>
#include <iostream>

#include <qgl.h>

#include "gui/MGLCanvas.h"
#include "gui/MApplication.h"


/**
 * Custom QGLWidget class for passing signals into AztecGUI
 */
class CQGLCanvas_ : public QGLWidget {

  // Qt MOC pre-processor macro
  Q_OBJECT

public:
  CQGLCanvas_(QWidget* parent, const Aztec::MGLCanvasPtr ref)
    : QGLWidget(parent),
      m_Ref(ref) {
    // Connect the signals
    connect(this, SIGNAL(resizeEvent(QResizeEvent*)), SLOT(onResize(QResizeEvent*)));
/*    connect(this, SIGNAL(showEvent(QShowEvent*)), SLOT(onShow(QShowEvent*)));
    connect(this, SIGNAL(paintEvent(QPaintEvent*)), SLOT(onPaint(QPaintEvent*)));*/
    connect(this, SIGNAL(mousePressEvent(QMouseEvent*)), SLOT(onMousePress(QMouseEvent*)));
    connect(this, SIGNAL(mouseReleaseEvent(QMouseEvent*)), SLOT(onMouseRelease(QMouseEvent*)));
    connect(this, SIGNAL(mouseMoveEvent(QMouseEvent*)), SLOT(onMouseMove(QMouseEvent*)));
    connect(this, SIGNAL(wheelEvent(QWheelEvent*)), SLOT(onWheelEvent(QWheelEvent*)));

    // Disable automatic buffer swapping
    setAutoBufferSwap(false);
  };

  ~CQGLCanvas_() {
    m_Ref = 0l;
  }

protected:
  /**
   * A reference to the MGLCanvas
   */
  Aztec::MGLCanvasPtr m_Ref;

  /**
   * State backup
   */
  Aztec::MShiftState m_State;


  /**
   * Virtual Qt methods (overridden)
   */
  void initializeGL() { m_Ref->initOpenGL(); };
  void resizeGL(int w, int h) { m_Ref->onResize(w, h); };
  void paintGL() { m_Ref->onPaint(); };

signals:
  /**
   * Qt signals
   */
  void resizeEvent(QResizeEvent* ev);
/*  void showEvent(QShowEvent* ev);
  void paintEvent(QPaintEvent* ev);*/
  void mousePressEvent(QMouseEvent* ev);
  void mouseReleaseEvent(QMouseEvent* ev);
  void mouseMoveEvent(QMouseEvent* ev);
  void wheelEvent(QWheelEvent* ev);

protected slots:
  /**
   * These methods pass the signals into AztecGUI
   */
  void onResize(QResizeEvent* ev) {
    QGLWidget::resizeEvent(ev);
    m_Ref->onResize(ev->size().width(), ev->size().height());
  };

/*  void onShow(QShowEvent* ev) {
    QGLWidget::showEvent(ev);
    raise();
    m_Ref->onShow();
  };

  void onPaint(QPaintEvent* ev) {
    QGLWidget::paintEvent(ev);
    m_Ref->onPaint();
  }*/

  void onMousePress(QMouseEvent* ev) {

    m_State = Aztec::MShiftState(ev->button() == Qt::LeftButton,
                                 ev->button() == Qt::MidButton,
                                 ev->button() == Qt::RightButton,
                                (ev->state() & Qt::AltButton) != 0,
                                (ev->state() & Qt::ControlButton) != 0,
                                (ev->state() & Qt::ShiftButton) != 0);

    Aztec::MMouseEvent event(Aztec::MMouseEvent::MBUTTON_DOWN, 0,
                             Aztec::MPoint2D(ev->x(), ev->y()), m_State, m_Ref);

    // TODO: Take care of time value (?)
    if (m_State.leftMouseDown) {
      event = Aztec::MMouseEvent(Aztec::MMouseEvent::LBUTTON_DOWN, 0,
                                 Aztec::MPoint2D(ev->x(), ev->y()), m_State, m_Ref);
    }
    else if (m_State.rightMouseDown) {
      event = Aztec::MMouseEvent(Aztec::MMouseEvent::RBUTTON_DOWN, 0,
                                 Aztec::MPoint2D(ev->x(), ev->y()), m_State, m_Ref);
    }
    else if (m_State.middleMouseDown) {
      event = Aztec::MMouseEvent(Aztec::MMouseEvent::MBUTTON_DOWN, 0,
                                 Aztec::MPoint2D(ev->x(), ev->y()), m_State, m_Ref);
    }

    // Fire off all AGUI handlers
    bool handled = Aztec::MApp::getInstance()->onMousePressed(event);

    m_Ref->fireMouseListeners(event);

    if (!handled) m_Ref->onMousePressed(event);
    else ev->ignore();
  };

  void onMouseRelease(QMouseEvent* ev) {
    static Aztec::MShiftState state(false, false, false, false, false, false);

    Aztec::MMouseEvent event(Aztec::MMouseEvent::MBUTTON_UP, 0,
                             Aztec::MPoint2D(ev->x(), ev->y()), state, m_Ref);

    // TODO: Take care of time value (?)
    if (m_State.leftMouseDown) {
      event = Aztec::MMouseEvent(Aztec::MMouseEvent::LBUTTON_UP, 0,
                                 Aztec::MPoint2D(ev->x(), ev->y()), state, m_Ref);
    }
    else if (m_State.rightMouseDown) {
      event = Aztec::MMouseEvent(Aztec::MMouseEvent::RBUTTON_UP, 0,
                                 Aztec::MPoint2D(ev->x(), ev->y()), state, m_Ref);
    }
    else if (m_State.middleMouseDown) {
      event = Aztec::MMouseEvent(Aztec::MMouseEvent::MBUTTON_UP, 0,
                                 Aztec::MPoint2D(ev->x(), ev->y()), state, m_Ref);
    }

    // Fire off all AGUI handlers
    bool handled = Aztec::MApp::getInstance()->onMouseReleased(event);

    m_Ref->fireMouseListeners(event);

    if (!handled) m_Ref->onMouseReleased(event);
    else ev->ignore();
  };

  void onMouseMove(QMouseEvent* ev) {
    // TODO: Take care of time value (?)
    Aztec::MMouseEvent event(Aztec::MMouseEvent::MOVE, 0,
                             Aztec::MPoint2D(ev->x(), ev->y()),
                             m_State, m_Ref);

    // Fire off all AGUI handlers
    bool handled = Aztec::MApp::getInstance()->onMouseMove(event);

    m_Ref->fireMouseListeners(event);

    if (!handled) m_Ref->onMouseMove(event);
    else ev->ignore();
  };

  void onWheelEvent(QWheelEvent* ev) {
    // TODO: Implementation (note: remember to ignore the event if we
    //                       aren't using it, so Qt can send it upwards
    //                       to the parent widgets)
  }

};


#endif

