#include <Aztec3DPCH.h>
#include <views/AztecGraphView.h>

#include <controls/MultiSplitter.h>
#include <controls/FourWaySplitterMethod.h>
#include <controls/CommandButton.h>
#include <controls/ToolButton.h>
#include <tools/MGraphSelectTool.h>
#include <tools/MToolManager.h>

// AztecGUI includes
#include <gui/MBorderLayout.h>
#include <gui/MButton.h>
#include <gui/MFlowLayout.h>

// AztecLib inclues
#include <MListsTrees.h>
#include <MScene.h>


namespace AztecGUI {

  static const int toolbarHeight = 33;

  static const int ParamListID = 10002;

  class GraphViewTree : public MTreeControl {
  public:
    GraphViewTree(AztecGraphView *v) {
      view = v;
    }

    bool onItemSelected(const MTreeItemPtr &item) {
      view->onSelChanged(item);
      return true;
    }

    AztecGraphView *view;
  };


  AztecGraphView::AztecGraphView()
    : AztecView("Graph View")
  {
    m_SplitPos = 100;
    splitBarWidth = 4;
    topShelf = NULL;
    paramListTree = NULL;
  }

  AztecGraphView::~AztecGraphView()
  {
  }

  void AztecGraphView::onCreate() {
    AztecView::onCreate();

    Aztec::MContainerPtr contentPane = new Aztec::MContainer();
    addComponent(contentPane, Aztec::MBorderLayout::CENTRE);
    contentPane->setLayoutManager(new Aztec::MBorderLayout());

    MultiSplitterPtr splitter = new MultiSplitter();
    splitter->setSplitterMethod(new FourWaySplitter(6, 0, FourWaySplitter::TwoVertical, 0.25));

    MContainerPtr splitContainer = new MContainer();

    contentPane->addComponent(splitContainer, MBorderLayout::CENTRE);
    splitContainer->setLayoutManager(new MBorderLayout());
    splitContainer->addComponent(splitter, MBorderLayout::CENTRE);

    if (topShelf == NULL) {
      topShelf = new MContainer();
      contentPane->addComponent(topShelf, MBorderLayout::NORTH);
      topShelf->setMinimumSize(MSize2D(32,32));
      topShelf->setLayoutManager(new Aztec::MFlowLayout());
      topShelf->addComponent(new ToolButton("Select Keys", "graphView", "GraphSelectTool"));
      topShelf->addComponent(new ToolButton("Move Keys", "graphView", "GraphMoveTool"));
      topShelf->addComponent(new ToolButton("Insert Keys", "graphView", "GraphInsertKeyTool"));
      topShelf->addComponent(new CommandButton("Merge Selected Keys", "Scene.graphMergeSelectedKeys()"));
    }
    if (paramListTree == NULL) {
      paramListTree = new GraphViewTree(this);

      splitter->addComponent(paramListTree);
    }

    m_Graph = new GraphComponent(this);
    splitter->addComponent(m_Graph);

    MToolManager::getInstance()->setTool(new MGraphSelectTool(), getViewGroup());
  }

  template <class SET1, class SET2>
  bool areEqual(const SET1 &set1, const SET2 &set2) {
    typename SET1::const_iterator it;

    if (set1.size() != set2.size()) {
      return false;
    }

    int count = 0;

    for (it = set1.begin(); it != set1.end(); ++it) {
      if (set2.find(*it) == set2.end()) {
        return false;
      }
    }

    // if we managed to get here, we must be equal
    return true;
  }

  AztecViewPtr AztecGraphView::createCopy() {
    AztecGraphViewPtr result = new AztecGraphView();

    // TODO: Copy important information about cameras and whatever else here.

    return result;
  }

  std::string AztecGraphView::getViewGroup() const {
    return "graphView";
  }

  MToolTypePtr AztecGraphView::getSelectTool() {
    return new MGraphSelectTool();
  }


  void AztecGraphView::drawView() {
    updateTreeFromScene();

    MRect2D clientRect = getClientSize();

    std::vector<GraphPair> values;
    {
      MBaseObjectTreePtr Tree = MScene::getGlobalScene()->getObjectList();
      MBaseObjectPtr BaseObj;
      MSceneObjectPtr Obj;
    
      KeyParamSet::iterator keyIt;

      for (keyIt = selectedParams.begin(); keyIt != selectedParams.end(); ++keyIt) {
        MFloatKeyListPtr floatKeyList = AZTEC_CAST(MFloatKeyList, (*keyIt));

        if (floatKeyList != NULL) {
          values.push_back(GraphPair(floatKeyList, paramColourMap[floatKeyList]));
          continue;
        }

        MVector3KeyListPtr vectorKeyList = AZTEC_CAST(MVector3KeyList, (*keyIt));
        if (vectorKeyList != NULL) {
          values.push_back(GraphPair(vectorKeyList->getXValue(), paramColourMap[vectorKeyList->getXValue()]));
          values.push_back(GraphPair(vectorKeyList->getYValue(), paramColourMap[vectorKeyList->getYValue()]));
          values.push_back(GraphPair(vectorKeyList->getZValue(), paramColourMap[vectorKeyList->getZValue()]));
          continue;
        }
      }
    }

    m_Graph->setGraphValues(values);
    m_Graph->refresh();
  }

/*  DWORD AztecGraphView::ViewPopupMenu(int x, int y)
  {
     CMenu    Menu;
     CMenu    *Popup;
     DWORD    Choice;

     Menu.LoadMenu(IDR_GRAPHVIEW_POPUP);
     Popup = Menu.GetSubMenu(0);
     InitPopupMenu(Popup);

     // Check the carious menu items depending on the current flags etc.

     Choice = Popup->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_NONOTIFY | TPM_RETURNCMD, x, y, this);

     Popup->DestroyMenu();

     bool     Update;
     int      Result;

     Result = HandlePopupCommand(Choice);

     // The command was handled, but the view was changed, so we must return.
     if (Result == -1)
        return 1;
      
     if (Result == 1)
        Update = true;
     else 
        Update = false;

     if (Update) {
       Invalidate();
       UpdateWindow();
     }
   
     return Choice;
  }

  int AztecGraphView::HandlePopupCommand(DWORD Cmd)
  {
     int Result;

     Result = MBaseViewWnd::HandlePopupCommand(Cmd);

     if (Result) {
        return Result;
     }

     switch (Cmd)
     {
     case ID_POPUP_SELECTTOOL:
       g_KeyList.ExecuteAction("GraphSelectTool");
       break;

     case ID_POPUP_MOVETOOL:
       g_KeyList.ExecuteAction("GraphMoveTool");
       break;

     case ID_POPUP_INSERTKEYSTOOL:
       g_KeyList.ExecuteAction("GraphInsertKeyTool");
       break;

     case ID_POPUP_DELETESELECTEDKEYS:
       g_KeyList.ExecuteAction("KObjectDeleteSelected");
       break;

     case ID_POPUP_ZOOMIN:
       g_KeyList.ExecuteAction("KViewportZoomIn");
       break;

     case ID_POPUP_ZOOMOUT:
       g_KeyList.ExecuteAction("KViewportZoomOut");
       break;

     case ID_POPUP_ZOOMTOALL:
       g_KeyList.ExecuteAction("KViewportFrameAll");
       break;

     case ID_POPUP_ZOOMTOSELECTED:
       g_KeyList.ExecuteAction("KViewportFrameSelected");
       break;

     default:
       return 0;
     }

     return 1;
  }
*/
  GraphComponent* AztecGraphView::getGraphComponent() {
    return &*m_Graph;
  }

  void AztecGraphView::doZoomToFit(bool doSelection) {
    std::vector<GraphPair> graphValues;

    getGraphComponent()->getGraphValues(graphValues);

    float minTime;
    float maxTime;
    bool timeRangeSet = false;

    float minValue;
    float maxValue;
    bool valueRangeSet = false;

    for (unsigned i = 0; i < graphValues.size(); ++i) {
      MKeyableValue *keyableValue = AZTEC_CAST(MKeyableValue, graphValues[i].value);

      // if we dont have a keyable value, then we can't get a minimum and a maximum
      // time

      if (keyableValue == NULL) {
        continue;
      }

      std::vector<int> keyTimes;

      if (doSelection) {
        keyableValue->getSelectedKeyTimes(keyTimes);
      } else {
        keyableValue->getKeyTimes(keyTimes);
      }

      if (keyTimes.size() > 0) {
        if (!timeRangeSet) {
          minTime = keyTimes.front();
          maxTime = keyTimes.back();
          timeRangeSet = true;
        } else {
          if (keyTimes.front() < minTime) {
            minTime = keyTimes.front();
          }
          if (keyTimes.back() > maxTime) {
            maxTime = keyTimes.back();
          }
        }
      } else {
        if (!timeRangeSet) {
          minTime = MScene::getGlobalScene()->getStartTime();
          maxTime = MScene::getGlobalScene()->getEndTime();
          timeRangeSet = true;
        }
      }

      // if we have a float key list, then we will be able to get the values 
      // from the list to zoom to them.
      MFloatKeyList *floatList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);

      if (floatList != NULL) {
        if (floatList->getKeyCount() == 0) {
          float value = floatList->getValueAtTime(MScene::getGlobalScene()->getTime());

          if (!valueRangeSet) {
            minValue = 0.0;
            maxValue = 0.0;
            valueRangeSet = true;
          } 
          if (value < minValue) {
            minValue = value;
          }
          if (value > maxValue) {
            maxValue = value;
          }
        } else {
          for (int i = 0; i < floatList->getKeyCount(); ++i) {
            float value = floatList->getValueAtIndex(i);

            if (doSelection && !floatList->getKeyAtIndex(i)->isSelected()) {
              continue;
            }

            if (!valueRangeSet) {
              minValue = value;
              maxValue = value;
              valueRangeSet = true;
            } else {
              if (value < minValue) {
                minValue = value;
              }
              if (value > maxValue) {
                maxValue = value;
              }
            }

          }
        }
      }
    }

    if (timeRangeSet) {
      getGraphComponent()->setGraphRange(minTime - MScene::getGlobalScene()->getTicksPerFrame() * 2, 
                                         maxTime + MScene::getGlobalScene()->getTicksPerFrame() * 2);
    }

    if (valueRangeSet) {
      minValue -= 0.5 + 0.1 * (maxValue - minValue);
      maxValue += 0.5 + 0.1 * (maxValue - minValue);
      getGraphComponent()->setValueRange(minValue, maxValue);
    }

  }

  void AztecGraphView::deleteSelectedKeys() {
    std::vector<GraphPair> graphValues;

    getGraphComponent()->getGraphValues(graphValues);

    for (int i = 0; i < graphValues.size(); ++i) {
      MKeyableValue *keyableValue = AZTEC_CAST(MKeyableValue, graphValues[i].value);

      if (keyableValue != NULL) {
        keyableValue->deleteSelectedKeys();
      }
    }
  }

  void AztecGraphView::zoomIn() {
    float min, max;
  
    getGraphComponent()->getGraphRange(min, max);
  
    float middle = 0.5*(max + min);
    float diff = 0.5*(max - min);
  
    getGraphComponent()->setGraphRange(middle - diff/1.2, middle + diff/1.2);
  }

  void AztecGraphView::zoomOut() {
    float min, max;

    getGraphComponent()->getGraphRange(min, max);

    float middle = 0.5*(max + min);
    float diff = 0.5*(max - min);

    getGraphComponent()->setGraphRange(middle - diff*1.2, middle + diff*1.2);
  }

  void AztecGraphView::zoomToFitAll() {
    doZoomToFit(false);
  }

  void AztecGraphView::zoomToFitSelected() {
    doZoomToFit(true);
  }

  bool AztecGraphView::selectParent() {
    return false;
  }

  bool AztecGraphView::selectChild() {
    return false;
  }

  bool AztecGraphView::selectSiblingNext() {
    return false;
  }

  bool AztecGraphView::selectSiblingPrev() {
    return false;
  }

  bool AztecGraphView::selectNone() {
    return false;
  }

  bool AztecGraphView::selectAll() {
    return false;
  }

  bool AztecGraphView::selectInverse() {
    return false;
  }

  bool AztecGraphView::selectAllChildren() {
    return false;
  }

  bool AztecGraphView::anythingSelected() {
    return getGraphComponent()->isAnythingSelected();
  }

  MVector3 AztecGraphView::getSelectionCentre() {
    return MVector3(0,0,0);
  }

  bool AztecGraphView::deleteSelected() {
    this->deleteSelectedKeys();
    return true;
  }

  void AztecGraphView::onSelChanged(const MTreeItemPtr &item) {
    MKeyableValuePtr param = getParameterFromTree(item);

    if (item->getSelected()) {
      selectedParams.insert(param);
    } else {
      selectedParams.erase(param);
    }

    refresh();
  }

  void AztecGraphView::updateTreeFromScene() {

    MBaseObjectTreePtr Tree = MScene::getGlobalScene()->getObjectList();
    MBaseObjectPtr BaseObj;
    MSceneObjectPtr Obj;
    std::set<MParameterObjectPtr> objParams;
    std::set<MSceneObjectPtr> sceneObjs;

    Tree->beginIteration();
    while (( BaseObj = Tree->getNext() ) != NULL ) {
      Obj = AZTEC_CAST(MSceneObject, BaseObj);
  
      if (Obj != NULL && Obj->isFlagged(OBJECTFLAG_SELECTED)) {
        sceneObjs.insert(Obj);
      }
    }
    Tree->endIteration();

    bool createList = false;

    if (!areEqual(sceneObjs, selectedObjects)) {
      createList = true;
    }

    if (createList) {
      // Here we have to deselect everything otherwise the windows
      // common control library spits the dummy everynow and then
      // and it crashes!!!!
      // This means we have to store our selections remporarily as well.
      KeyParamSet tempSelectedParams = selectedParams;

      paramListTree->selectItem(NULL);
      paramListTree->clear();

      selectedObjects.clear();
      itemParamMap.clear();
      paramItemMap.clear();
      paramColourMap.clear();

      std::set<MSceneObjectPtr>::iterator sceneIt;
      for (sceneIt = sceneObjs.begin(); sceneIt != sceneObjs.end(); ++sceneIt) {
        MTreeItemPtr item = paramListTree->addItem((*sceneIt)->getName().c_str());

        selectedObjects[*sceneIt] = item;

        addParameters(item, (*sceneIt)->getParamList());

        MShapeObjectPtr shape = (*sceneIt)->getShapeObject();

        if (shape != NULL) {
          addParameters(item, shape->getParamList());
        }

        paramListTree->expandItem(item, true);
      }

      if (tempSelectedParams.size() != 0) {
        MKeyableValuePtr selectedKey = *tempSelectedParams.begin();

        MTreeItemPtr item = getTreeItem(selectedKey);

        if (item != NULL) {
          paramListTree->selectItem(item);
        }  else {
          tempSelectedParams.clear();
        }
      }
    }

  }

  void AztecGraphView::addParameters(MTreeItemPtr parentItem, const MParameterObjectListPtr &paramList) {
    for (int i = 0; i < paramList->getNumParams(); ++i) {
      MParameterObjectPtr param = paramList->getParameter(i);
      MKeyParameter *keyParam = AZTEC_CAST(MKeyParameter, param);

      // we are only interested in animatable keys
      if (keyParam != NULL) {
        MTreeItemPtr paramItem = paramListTree->addItem(param->getFriendlyName().c_str(), parentItem);

        addParameter(paramItem, keyParam->getKeyableValue(), MVector3(0.5f,0.8f,0.5f));

        MVector3KeyParameterPtr vector3Param = AZTEC_CAST(MVector3KeyParameter, keyParam);
        if (vector3Param != NULL) {
          addParameters(paramItem, vector3Param);
        }
      }
    }
  }

  void AztecGraphView::addParameters(MTreeItemPtr parentItem, const MVector3KeyParameterPtr &vectorParam) {
    std::string vectorNames[3];

    if (vectorParam->getDataMeaning() == MParameterObject::MEANING_COLOUR) {
      vectorNames[0] = "Red";
      vectorNames[1] = "Green";
      vectorNames[2] = "Blue";
    } else {
      vectorNames[0] = "X";
      vectorNames[1] = "Y";
      vectorNames[2] = "Z";
    }

    MVector3KeyListPtr vectorKeyList = AZTEC_CAST(MVector3KeyList, vectorParam->getVector3Value());
    if (vectorKeyList != NULL) {
      MFloatKeyableValuePtr param;

      param = AZTEC_CAST(MFloatKeyableValue, vectorKeyList->getXValue());
      if (param != NULL) {
        MTreeItemPtr item = paramListTree->addItem(vectorNames[0].c_str(), parentItem);
        addParameter(item, param, MVector3(1,0,0));
      }
      param = AZTEC_CAST(MFloatKeyableValue, vectorKeyList->getYValue());
      if (param != NULL) {
        MTreeItemPtr item = paramListTree->addItem(vectorNames[1].c_str(), parentItem);
        addParameter(item, param, MVector3(0,1,0));
      }
      param = AZTEC_CAST(MFloatKeyableValue, vectorKeyList->getZValue());
      if (param != NULL) {
        MTreeItemPtr item = paramListTree->addItem(vectorNames[2].c_str(), parentItem);
        addParameter(item, param, MVector3(0,0,1));
      }
    }
  }

  void AztecGraphView::addParameter(MTreeItemPtr item, const MKeyableValuePtr &param, const MVector3 &colour) {
    itemParamMap[item] = param;
    paramItemMap[param] = item;
    paramColourMap[param] = colour;
  }

  MKeyableValuePtr AztecGraphView::getParameterFromTree(MTreeItemPtr item) {
    // try and find this parameter in the tree view
    ItemParamMap::iterator treeIt;

    treeIt = itemParamMap.find(item);

    if (treeIt == itemParamMap.end()) {
      return NULL;
    } else {
      return treeIt->second;
    }
  }

  MTreeItemPtr AztecGraphView::getTreeItem(const MKeyableValuePtr &param) {
    // try and find this parameter in the tree view
    std::map<MKeyableValuePtr, MTreeItemPtr>::iterator treeIt;

    treeIt = paramItemMap.find(param);

    if (treeIt == paramItemMap.end()) {
      return NULL;
    } else {
      return treeIt->second;
    }
  }

}
