#include <Aztec3DPCH.h>

#include "tools/ExtrudeTool.h"

// Aztec2 includes
#include <views/Aztec3DView.h>
#include <utils/AztecGLUtils.h>
#include <config/UIConfig.h>
#include <functions/mesh/MeshFunctions.h>

// AztecLib includes
#include "MSystemManager.h"
#include "MEditableMesh.h"
#include "MAnimMesh.h"
#include "MUIManager.h"
#include <MComponentisedObject.h>

// standard includes
#include <math.h>
#include <GL/gl.h>
#include <set>


namespace AztecGUI {
  using std::set;

  Aztec::MVector3    m_ExtrudeVec;
  float       m_Dist;

  MExtrudeTool::MExtrudeTool()
  {
    m_RequiresSel = true;
    m_Dist = 0.0f;
    m_ExtrudeVec.set(1,0,0);
    m_Extruding = false;
  }

  std::string MExtrudeTool::getName() {
    return "toolExtrude";
  }

  int MExtrudeTool::drawTool(bool select, const Aztec::MComponentPtr &component) {
    // we want to draw an arrow at the centre of the selection, 
    // and and pointing in the average normal direction.

    // update our normal. If nothing was selected, we don't draw anything
    if (!m_Extruding && !updateExtrudeNormal()) {
      return 1;
    }

    AztecGLCanvasPtr GLWnd = AztecGLView::getGLCanvasFor(component);

    if (GLWnd == NULL) {
      return 0;
    }
  
    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_DEPTH_TEST);
  
  
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
  
    // Perform the viewport transformation
  
    GLWnd->doCameraTransform();
    glMatrixMode(GL_MODELVIEW);
  
    glPushMatrix();

    Aztec::MScene::getGlobalScene()->GLTransform(Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

    glPushMatrix();

    float ScaleFact = GLWnd->getScalingFactor(m_CurrentPos);
    glTranslatef(m_CurrentPos.x, m_CurrentPos.y, m_CurrentPos.z);
  
    glScalef(ScaleFact, ScaleFact, ScaleFact);
  
    glMultMatrixf((float*)m_GLTransform.m);

    // draw the axis icon
    glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.5f, 2.0f, select, DRAWAXIS_Z | DRAWAXIS_ARROWS);

    // if we are extruding the faces, draw an arrow where we started from.
    if (m_Extruding) {
      glPopMatrix();
      glPushMatrix();

      ScaleFact = GLWnd->getScalingFactor(m_StartPos);
      glTranslatef(m_StartPos.x, m_StartPos.y, m_StartPos.z);
      glScalef(ScaleFact, ScaleFact, ScaleFact);
      glMultMatrixf((float*)m_GLTransform.m);

      glDrawAxisIcon(UIConfig::get3DWidgetSize(), 0.2f, 2.0f, false, DRAWAXIS_ALLGRAY | DRAWAXIS_Z | DRAWAXIS_ARROWS);
    }
  
    glPopMatrix();
    glPopMatrix();
    glPopMatrix();
  
    glPopAttrib();
  
    return 1;
  }

  bool MExtrudeTool::updateExtrudeNormal() {
    Aztec::MSceneObjectPtr sceneObj;
    Aztec::MEditableMeshPtr meshObj;
  
    sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

    if (sceneObj != NULL && sceneObj->getShapeObject() != NULL) {
      meshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
    }

    if (meshObj == NULL) {
      return false;
    }

    Aztec::MVector3 Norm, centre;
    int n, count;
  
    count = 0;
    Norm.set(0,0,0);

    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      return false;
    }

    int numComps = meshObj->getComponentCount(Aztec::MUIManager::getComponentMode());

    // go through and extract the average normal and centre
    for(n = 0; n < numComps; n++) {
      if (!meshObj->isComponentFlagged(Aztec::MUIManager::getComponentMode(), n)) {
        continue;
      }

      centre += meshObj->getComponentCentre(Aztec::MUIManager::getComponentMode(), n);
      Norm += meshObj->getComponentNormal(Aztec::MUIManager::getComponentMode(), n);
      count++;
    }
    if (count) {
      Norm.normalize();
      centre /= (float)count;
    } else {
      return false;
    }
  
    m_StartPos = centre;
    m_CurrentPos = m_StartPos;
    m_ExtrudeCentre = centre;
    m_ExtrudeVec = Norm;

    // transform the drawing to out z axis lines up with the normal.
    {
      Aztec::MVector3  x,y,z;

      m_GLTransform.identity();
      z = m_ExtrudeVec;

      // construct two otrhonormal vectors
      y.set(-z.y, z.z, -z.x);
      x = z.crossProduct(y);
      y = z.crossProduct(x);

      m_GLTransform.m[0][0] = x.x; m_GLTransform.m[0][1] = x.y; m_GLTransform.m[0][2] = x.z;
      m_GLTransform.m[1][0] = y.x; m_GLTransform.m[1][1] = y.y; m_GLTransform.m[1][2] = y.z;
      m_GLTransform.m[2][0] = z.x; m_GLTransform.m[2][1] = z.y; m_GLTransform.m[2][2] = z.z;
    }

    return true;
  }

  #define TEMP_MARK_TODO 0x80
  #define TEMP_MARK_DONE 0x40

  static void resetTempMarks(Aztec::MEditableMeshPtr mesh) {
    // any edge that has already been split on this vertex will
    // be marked with the TEMP_MARK_TODO flag. Right now we unflag all the 
    // vertices in the mesh.

    mesh->unsetVertexFlags(TEMP_MARK_TODO | TEMP_MARK_DONE);
  }

  // 
  /**
   * Splits the edges around the vertices of the mesh with the given flags.
   *
   * @param mesh The mesh whose edges we are splitting
   * @param flags The flags that the vertices have that we want to split
   * @param dist The distance from each vertex to split connected edges. 
   * @param distIsFraction If true, the dist parameter is considered to be
   *                       a fraction. If false, it is a distance in local space.
   * 
   */
  static void splitEdgesAroundVerts(const Aztec::MEditableMeshPtr &mesh, Aztec::AztecFlags flags, float dist = 0.5, bool distIsFraction = true) {
    EdgeConnectEdges inputEdges;
    std::vector<Edge> newEdges;

    for (int i = 0; i < mesh->getNumVerts(); ++i) {
      if (mesh->isVertexFlagged(i, VERTEX_SELECTED)) {
        std::vector<int> others;
        mesh->getVerticesConnecting(i, others);
      
        for (int j = 0; j < others.size(); ++j) {
          inputEdges[Edge(i, others[j])] = -1;
        }
      }
    }

    edgeConnectMesh(mesh, 1, inputEdges, newEdges);
  }

  /**
   * Splits the edges around the vertices of the mesh with the given flags.
   *
   * @param mesh The mesh whose edges we are splitting
   * @param flags The flags that the vertices have that we want to split
   * @param dist The distance from each vertex to split connected edges. 
   * @param distIsFraction If true, the dist parameter is considered to be
   *                       a fraction. If false, it is a distance in local space.
   * 
   */
  static void splitEdgesAroundEdges(Aztec::MEditableMeshPtr mesh, Aztec::AztecFlags flags, float dist = 0.5, bool distIsFraction = true) {

    EdgeConnectEdges inputEdges;
    std::vector<Edge> newEdges, keepEdges;

    for (int i = 0; i < mesh->getNumTris(); ++i) {
      for (int j = 0; j < 3; ++j) {
        if (mesh->isTriangleEdgeFlagged(i, j, flags)) {
          int a, b;
          mesh->getVertsFromEdge(i, j, &a, &b);
          keepEdges.push_back(Edge(a, b));

          std::vector<int> others;
          mesh->getVerticesConnecting(a, others);
      
          for (int k = 0; k < others.size(); ++k) {
            inputEdges[Edge(a, others[k])] = -1;
          }

          mesh->getVerticesConnecting(b, others);
          for (int k = 0; k < others.size(); ++k) {
            inputEdges[Edge(b, others[k])] = -1;
          }
        
        }

      }
    }

    // now go and deselect all the edges we don't want to connects.
    for (int i = 0; i < keepEdges.size(); ++i) {
      inputEdges.erase(keepEdges[i]);
    }

    edgeConnectMesh(mesh, 1, inputEdges, newEdges);

  }

  int MExtrudeTool::onMouseDown(const Aztec::MMouseEvent &event) {
    Aztec::MSceneObjectPtr sceneObj;
    Aztec::MEditableMeshPtr MeshObj;
  
    sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

    if (sceneObj == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires one scene object to work with.");
      return TOOLRESULT_DRAWNONE;
    }
    MeshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
    if (MeshObj == NULL) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires Mesh objects to work with.");
      return TOOLRESULT_DRAWNONE;
    }

    if (!updateExtrudeNormal()) {
      m_Extruding = false;
      Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extrude requires some faces selected.");
    
      return TOOLRESULT_DRAWNONE;
    }

    m_Extruding = true;
  
    {
      Aztec::MSceneObjectPtr sceneObj;
      Aztec::MEditableMeshPtr MeshObj;
    
      sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

      if (sceneObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires one scene object to work with.");
        return TOOLRESULT_DRAWNONE;
      }
      MeshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
      if (MeshObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires Mesh objects to work with.");
        return TOOLRESULT_DRAWNONE;
      }
    
      AztecGLCanvasPtr glWnd = AztecGLView::getGLCanvasFor(event.getComponent());
    
      if (glWnd == NULL) {
        m_StartDist = 0.0;
      } else {
        Aztec::MRay viewRay, normalRay;
        Aztec::MVector3  e1,e2;

        e1 = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(MeshObj), m_ExtrudeCentre);
        e2 = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(MeshObj), m_ExtrudeCentre + m_ExtrudeVec);

        normalRay.set(e1, e2-e1);
        viewRay = glWnd->getRay(event.getX(), event.getY());

        Aztec::MMath::distanceBetween(normalRay, viewRay, &m_StartDist, NULL);
      }
    }
  

    m_Dist = m_StartDist;

    // if we are in point mode, we get split the dges around the selected points
    // and move the one point.
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {
      // TODO: Make the radius of extrusion customisable.
      splitEdgesAroundVerts(MeshObj, VERTEX_SELECTED, 2, false);

      // now go over all the selected verts and mark them for change so we
      // move them with the tool
      for (int i = MeshObj->getNumVerts() - 1; i >= 0 ; --i) {
        if (MeshObj->isVertexFlagged(i, VERTEX_SELECTED)) {
          MeshObj->setVertexFlag(i, VERTEX_FLAGFORCHANGE);
        } else {
          MeshObj->unsetVertexFlag(i, VERTEX_FLAGFORCHANGE);
        }
      }

    // extrude the faces if that is the mode we are in
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {
      // Extrude the faces initially by a small amount.
      MeshObj->extrudeFaces(TRIANGLE_SELECTED, 0.0f, m_ExtrudeVec);
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE) {
      MeshObj->unsetVertexFlags(VERTEX_FLAGFORCHANGE | VERTEX_SELECTED);

      for (int i = MeshObj->getNumTris() - 1; i >= 0 ; --i) {
        for (int e = 0; e < 3; ++e) {
          if (MeshObj->isTriangleEdgeFlagged(i, e, EDGE_SELECTED)) {
            MeshObj->setVertexFlag(MeshObj->getTriangleVertex(i, e), VERTEX_FLAGFORCHANGE | VERTEX_SELECTED);
            MeshObj->setVertexFlag(MeshObj->getTriangleVertex(i, (e+1)%3), VERTEX_FLAGFORCHANGE | VERTEX_SELECTED);
          }
        }
      }
      // TODO: Make the radius of extrusion customisable.
      // DOESN'T WORK WITH VEREX_FLAGFORCHANGE!!!! WHY NOT
      splitEdgesAroundEdges(MeshObj, VERTEX_SELECTED, 2, false);
      MeshObj->unsetVertexFlags(VERTEX_FLAGFORCHANGE);
      for (int i = MeshObj->getNumTris() - 1; i >= 0 ; --i) {
        for (int e = 0; e < 3; ++e) {
          if (MeshObj->isTriangleEdgeFlagged(i, e, EDGE_SELECTED)) {
            MeshObj->setVertexFlag(MeshObj->getTriangleVertex(i, e), VERTEX_FLAGFORCHANGE);
            MeshObj->setVertexFlag(MeshObj->getTriangleVertex(i, (e+1)%3), VERTEX_FLAGFORCHANGE);
          }
        }
      }
    }
  
    // Take a snapshot, so we can alter the distance of the vertices later on nice and easily.
    MeshObj->takeSnapshot();
  
    return TOOLRESULT_DRAWALL;
  }


  int MExtrudeTool::onMouseMove(const Aztec::MMouseEvent &event) {
    // if we are extruding, keep track of how the mouse has moved etc.
  
    if (m_Extruding) {
      Aztec::MSceneObjectPtr sceneObj;
      Aztec::MEditableMeshPtr MeshObj;
    
      sceneObj = AZTEC_CAST(Aztec::MSceneObject, Aztec::MScene::getGlobalScene()->getSelectedObjectList()->getHead());

      if (sceneObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires one scene object to work with.");
        return TOOLRESULT_DRAWNONE;
      }
      MeshObj = AZTEC_CAST(Aztec::MEditableMesh, sceneObj->getShapeObject()->convertToMesh());
      if (MeshObj == NULL) {
        Aztec::MSystemManager::getInstance()->logOutput("Error: Face Extude requires Mesh objects to work with.");
        return TOOLRESULT_DRAWNONE;
      }
    
      AztecGLCanvasPtr glWnd = AztecGLView::getGLCanvasFor(event.getComponent());
    
      if (glWnd == NULL) {
        m_Dist = (float)(m_CurPos.x - m_DownPos.x) / 25.0f;
      } else {
        Aztec::MRay viewRay, normalRay;
        Aztec::MVector3  e1,e2;

        e1 = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(MeshObj), m_ExtrudeCentre);
        e2 = Aztec::MScene::getGlobalScene()->objectToWorldSpace(Aztec::MBaseObjectPtr(MeshObj), m_ExtrudeCentre + m_ExtrudeVec);

        normalRay.set(e1, e2-e1);
        viewRay = glWnd->getRay(event.getX(), event.getY());

        Aztec::MMath::distanceBetween(normalRay, viewRay, &m_Dist, NULL);
      }

      m_Dist -= m_StartDist;
      m_CurrentPos = m_StartPos + m_Dist * m_ExtrudeVec;


      MeshObj->retreiveSnapshot();
    
      // Move the marked vertices by the specified amount.
    
      for (int n=0; n<MeshObj->getNumVerts(); n++) {
        if (MeshObj->isVertexFlagged(n, VERTEX_FLAGFORCHANGE)) {
          MeshObj->setVertexPosition(n, MeshObj->getVertexPosition(n) + m_Dist * m_ExtrudeVec);
        }
      }

      MeshObj->calculateNormals();
    
      return TOOLRESULT_DRAWALL;
    }
  
    return TOOLRESULT_DRAWNONE;
  }

  int MExtrudeTool::onMouseUp(const Aztec::MMouseEvent &event) {
    if (m_Extruding) {
      // If the extrusion didn't aceheive anything, just undo it.
      if (fabs(m_Dist) < 0.01) {
        Aztec::getSystemManager()->getUndoManager()->undo();
      } else {
        Aztec::MBaseObjectPtr BaseObj;
        Aztec::MSceneObjectPtr Obj;
  
        Aztec::MScene::getGlobalScene()->getObjectList()->beginIteration();
        while (( BaseObj = Aztec::MScene::getGlobalScene()->getObjectList()->getNext() ) != NULL ) {
          Obj = AZTEC_CAST(Aztec::MSceneObject, BaseObj);
          if (Obj == NULL) {
            continue;
          }
    
          Aztec::MTreeObjectNodePtr ObjNode;
          Aztec::MComponentisedObjectPtr compObj = Obj->getComponentObject();
    
          ObjNode = Aztec::MScene::getGlobalScene()->getObjectList()->getCurrentNode();
    
          if (compObj != NULL && compObj->isInComponentMode()) {
            Aztec::MAnimMeshPtr AnimMesh = AZTEC_CAST(Aztec::MAnimMesh, compObj);
            if (AnimMesh != NULL) {
              AnimMesh->updateKey(Aztec::MScene::getGlobalScene()->getTime(), Aztec::MSystemManager::getInstance()->getSettings()->m_Animate);
            }
          }
        }
        Aztec::MScene::getGlobalScene()->getObjectList()->endIteration();
      }
    
      m_Extruding = false;
      updateExtrudeNormal();
    }
  
    return TOOLRESULT_DRAWALL;
  }

}
