#include <Aztec3DPCH.h>

// Aztec2 includes
#include <controls/UVCanvas.h>
#include <tools/MToolManager.h>
#include <utils/AztecGLUtils.h>
#include <views/AztecViewManager.h>

// AztecGUICommon includes
#include <config/UIConfig.h>
#include <config/ColourConstants.h>

// AztecLib includes
#include <MSystemManager.h>
#include <MUIManager.h>

// standard includes
#include <math.h>
#include <iostream>


namespace AztecGUI {

  using namespace Aztec;

  UVCanvas::UVCanvas(AztecView *parentView)
    : Aztec3DSceneCanvas(parentView)
  {
    perspectiveView = false;
    zoomFactor = 100;
    cameraFocus.set(0.5,0.5,0.5);
  }

  UVCanvas::~UVCanvas() {
  }
  
  void UVCanvas::draw3D(Aztec::MSelectMethod method, 
                        std::vector<Aztec::MSelectionItem> &items, 
                        float left, 
                        float top, 
                        float right, 
                        float bottom) 
  {
    makeCurrent();

    Aztec::MVector3 bgColour = UIConfig::getColour(Colours::VIEW_BACKGROUND);

    glClearColor(bgColour.x, bgColour.y, bgColour.z, 0.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_DEPTH_TEST);

    initProjectionMatrix(method, left, top, right, bottom);
 
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    doCameraTransform();

    glColor3f(1,1,1);
    
    // now draw the scene.
    MSceneViewFlags flags;
    flags.m_ShadingMode = glWireframe;
    flags.m_TexturingMode = glNone;
    flags.m_LightingMode = glUnlit;
    flags.m_CullBackface = true;
    flags.m_TextureMode = GL_LINEAR;
    flags.m_SelectMethod = method;
    flags.m_SelectBuf = NULL;
    flags.m_WireColor.set(1,1,1,1);
    flags.m_FlatColor.set(1,1,1,1);

    if (method != smNone) {
      flags.m_SelectBuf = new UINT[16384];
      glSelectBuffer(16384, flags.m_SelectBuf);
      glRenderMode(GL_SELECT);
      glInitNames();
    }


    // draw the view
    std::vector<Aztec::MSceneObjectPtr> objects;
    drawUVMeshes(method, objects);

    if (method == smNone) {
      draw3DExtras();
    }

    // if we had selection information, clear it out.
    if (method != smNone) {
      int numHits = glRenderMode(GL_RENDER);

      Aztec::MScene::getGlobalScene()->processSelection(flags.m_SelectBuf, numHits, objects, &items);

      delete[] flags.m_SelectBuf;
      flags.m_SelectBuf = NULL;
    }
    

    if (method == smNone) {

      glFlush();
      glFinish();

      swapBuffers();
    }
  }

  void UVCanvas::draw3DExtras() {
    // draw our grid.
//    drawGrid(gridBasisX, gridBasisY);


    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();

    // draw the current tool, but only if we are not in selection mode.
    drawManipulators(false);


    // set up all the drawing to be orthographic
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
  
    glLoadIdentity();
  
    MSize2D size = this->getSize();
    glOrtho(0,size.getWidth(),0,size.getHeight(),-20,20);
  
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();



    // now draw the little cute axis icon in the bottom left
    glPushMatrix();
  
    glEnable(GL_DEPTH_TEST);
    glTranslatef(20,20,0);
    glRotateCamera();
    glDrawAxisIcon(getFont(), 20, 1, 1, false, DRAWAXIS_X | DRAWAXIS_Y | DRAWAXIS_LABELS, "U", "V");
    glDisable(GL_DEPTH_TEST);
  
    glPopMatrix();

    // now draw our view title.
    glPushMatrix();
    glEnable(GL_BLEND);

    MVector3 titleColour(1,1,0.5);
    if (AztecViewManager::getCurrentView() == parentView) {
      titleColour = UIConfig::getColour(Colours::VIEW_TITLE_CURRENT);
    } else {
      titleColour = UIConfig::getColour(Colours::VIEW_TITLE);
    }
    getFont()->setForegroundColor(titleColour.x, titleColour.y, titleColour.z);

    // these postitions were just picked so they looked nice, they may appear 
    // dodgy in some circumstances!
    getFont()->draw(5, size.getHeight() - 13, parentView->getName().c_str());

    glPopMatrix();


    // Clean up this mess matrix wise
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    
  }

  void UVCanvas::drawUVMeshes(Aztec::MSelectMethod method, std::vector<MSceneObjectPtr> &objects) {
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
    
    glColor4f(1,1,1,1);
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    
    // Draw the skin mesh border
    glBegin(GL_LINE_STRIP);
    glVertex2f(0,0);
    glVertex2f(1,0);
    glVertex2f(1,1);
    glVertex2f(0,1);
    glVertex2f(0,0);
    glEnd();
    
               
    glDisable(GL_DEPTH_TEST);

    glColor4f(1,1,1,1);

    // Draw the texture mesh
    {
      // we use -1 as the delimieter that starts off a new selection.
      glPushName(-1);

      MBaseObjectPtr Obj;
      MMeshPtr Mesh;  
      int meshNumber = 0;
      
      scene->getSelectedObjectList()->beginIteration();
      objects.clear();
      // push back a single empty object so the first object starts at index 1.
      objects.push_back(NULL);
      
      while ((Obj = scene->getSelectedObjectList()->getNext()) != NULL) {
        MSceneObjectPtr sceneObj;
        if ((sceneObj = AZTEC_CAST(MSceneObject, Obj)) == NULL) {
          continue;
        }
        if (sceneObj->getShapeObject() != NULL) {
          Mesh = AZTEC_CAST(MMesh, sceneObj->getShapeObject()->convertToMesh());
        }
        objects.push_back(sceneObj);
        meshNumber++;
        
        if (Mesh != NULL) {
          // put our object number on the name stack if necessary.
          if (method != smNone) {
            glPushName(meshNumber);
          }
          
          MMeshPtr TextureMesh;
          
          TextureMesh = Mesh->getTextureMesh();
          
          if (TextureMesh != NULL) {
            int   nTri, nVert;
            
            // Only draw triangles if not in a selection mode
            if (method == smNone) {
              glDisable(GL_LINE_SMOOTH);
              glLineWidth(1);
              
              glColor4f(1,1,1,1);
              for (int n=0; n<3; n++) {
                glBegin(GL_TRIANGLES);
                for (nTri = TextureMesh->getNumTris()-1; nTri>=0; nTri--) {
                  MVector3 V[3];
                  
                  V[0] = TextureMesh->getVertexPosition(TextureMesh->getTriangleVertex(nTri, 0));
                  V[1] = TextureMesh->getVertexPosition(TextureMesh->getTriangleVertex(nTri, 1));
                  V[2] = TextureMesh->getVertexPosition(TextureMesh->getTriangleVertex(nTri, 2));
                  
                  glVertex2fv((float*)&V[0]);
                  glVertex2fv((float*)&V[1]);
                  glVertex2fv((float*)&V[2]);
                }
                glEnd();
              }
            }
            
            // Object mode is basically groupd vertex selecting, so only deal with 
            // the selection buffer here
            if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
              if (method != smNone) {
                glPushName(SELECTITEM_UV_POINT);
                for (nVert = TextureMesh->getNumVerts()-1; nVert>=0; nVert--) {
                  MVector3 pos = TextureMesh->getVertexPosition(nVert);
                  
                  glPushName(nVert);
                  glBegin(GL_POINTS);
                  glVertex2f(pos.x, pos.y);
                  glEnd();
                  glPopName();
                }
                glPopName();
              }
            }
            
            // If we are in point mode, draw points
            if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {
              glColor3f(0.0f,0.8f,1.0f);
              glPointSize(5);
              
              if (method == smNone) {
                glBegin(GL_POINTS);
              }
              
              for (nVert = TextureMesh->getNumVerts()-1; nVert>=0; nVert--) {
                glColor3f(0.1f, 0.8f, 0.1f);
                
                if (TextureMesh->isVertexFlagged(nVert, VERTEX_SELECTED)) {
                  glColor3f(1.0f, 0.1f, 0.1f);
                }
                
                if (method != smNone) {
                  glPushName(SELECTITEM_UV_POINT);
                  glPushName(nVert);
                  glBegin(GL_POINTS);
                }
                
                MVector3 pos = TextureMesh->getVertexPosition(nVert);
                glVertex2f(pos.x, pos.y);
                
                if (method != smNone) {
                  glEnd();
                  glPopName();
                  glPopName();
                }
              }
              
              if (method == smNone) {
                glEnd();
              }
              
            }
            
            // Face mode deals with whole triangles. This handles both selection buffers
            // and drawing the selected triangles.
            if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {
              glEnable(GL_BLEND);
              //                  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
              glBlendFunc(GL_SRC_ALPHA, GL_SRC_COLOR);
              glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
              glColor4f(1.0f,0.8f,0.3f, 0.2f);
              
              if (method == smNone) {
                glBegin(GL_TRIANGLES);
              }
              
              for (nTri = TextureMesh->getNumTris()-1; nTri>=0; nTri--) {
                MVector3 V[3];
                
                V[0] = TextureMesh->getVertexPosition(TextureMesh->getTriangleVertex(nTri, 0));
                V[1] = TextureMesh->getVertexPosition(TextureMesh->getTriangleVertex(nTri, 1));
                V[2] = TextureMesh->getVertexPosition(TextureMesh->getTriangleVertex(nTri, 2));
                
                if (method != smNone) {
                  glPushName(SELECTITEM_UV_FACE);
                  glPushName(nTri);
                  glBegin(GL_TRIANGLES);
                }
                
                // Only draw the triangle  if we are drawing the select buffer, or if the triangle is selected
                if (method != smNone || TextureMesh->isTriangleFlagged(nTri, TRIANGLE_SELECTED)) {
                  glVertex2fv((float*)&V[0]);
                  glVertex2fv((float*)&V[1]);
                  glVertex2fv((float*)&V[2]);
                }
                
                if (method != smNone) {
                  glEnd();
                  glPopName();
                  glPopName();
                }
              }
              if (method == smNone) {
                glEnd();
              }
            }
            
            // Wsfw mode deals with only single edges of triangles. This handles
            // both selection and drawing the selected triangles.
            if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE)
            {
              int      NumEdges;
              
              NumEdges = 0;
              
              glColor4f(1.0f,0.0f,0.0f, 1.0f);
              
              if (method == smNone) {
                glBegin(GL_LINES);
              }
              
              //                  for (nTri = TextureMesh->getNumTris()-1; nTri>=0; nTri--)
              for (nTri = 0; nTri<TextureMesh->getNumTris(); nTri++) {
                MVector3 V[3];
                
                V[0] = TextureMesh->getVertexPosition(TextureMesh->getTriangleVertex(nTri, 0));
                V[1] = TextureMesh->getVertexPosition(TextureMesh->getTriangleVertex(nTri, 1));
                V[2] = TextureMesh->getVertexPosition(TextureMesh->getTriangleVertex(nTri, 2));
                
                if (method != smNone) {
                  glPushName(SELECTITEM_UV_EDGE);
                  glPushName(NumEdges++);
                  glBegin(GL_LINES);
                  glVertex2fv((float*)&V[0]);
                  glVertex2fv((float*)&V[1]);
                  glEnd();
                  glPopName();
                  
                  glPushName(NumEdges++);
                  glBegin(GL_LINES);
                  glVertex2fv((float*)&V[1]);
                  glVertex2fv((float*)&V[2]);
                  glEnd();
                  glPopName();
                  
                  glPushName(NumEdges++);
                  glBegin(GL_LINES);
                  glVertex2fv((float*)&V[2]);
                  glVertex2fv((float*)&V[0]);
                  glEnd();
                  glPopName();
                  glPopName();
                    
                } else {
                  
                  if (TextureMesh->isTriangleEdgeFlagged(nTri, 0, EDGE_SELECTED)) {
                    glVertex2fv((float*)&V[0]);
                    glVertex2fv((float*)&V[1]);
                  }
                  if (TextureMesh->isTriangleEdgeFlagged(nTri, 1, EDGE_SELECTED)) {
                    glVertex2fv((float*)&V[1]);
                    glVertex2fv((float*)&V[2]);
                  }
                  if (TextureMesh->isTriangleEdgeFlagged(nTri, 2, EDGE_SELECTED)) {
                    glVertex2fv((float*)&V[2]);
                    glVertex2fv((float*)&V[0]);
                  }
                }
              }
              if (method == smNone) {
                glEnd();
              }
            }
          }
          // take our object number on the name stack if necessary.
          if (method != smNone) {
            glPopName();
          }
        }
      }
      scene->getSelectedObjectList()->endIteration();

      glPopName();
    }

  }

}

