// GraphViewWnd.cpp : implementation file
//

#include <AztecMainPCH.h>
#include "AztecMain.h"
#include "GraphViewWnd.h"

#include <tools/MGraphSelectTool.h>

#include "MdlGlobs.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

static const int toolbarHeight = 33;

static const int ParamListID = 10002;


CGraphViewWnd::CGraphViewWnd()
{
  paintDC = NULL;
  m_SplitPos = 100;
  splitBarWidth = 4;
  splitDragging = false;
  topShelf = NULL;
  paramListTree = NULL;
}

CGraphViewWnd::~CGraphViewWnd()
{
  if (topShelf != NULL) {
    delete topShelf;
  }

  if (paramListTree != NULL) {
    delete paramListTree;
  }
}


BEGIN_MESSAGE_MAP(CGraphViewWnd, CWnd)
	//{{AFX_MSG_MAP(CGraphViewWnd)
  ON_WM_SIZE()
  ON_WM_PAINT()
	ON_WM_CLOSE()
  ON_WM_DESTROY()
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

void CGraphViewWnd::OnPaint() 
{
  CPaintDC dc(this); // device context for painting

  paintDC = dc.m_hDC;
  DrawView();
  paintDC = NULL;
}

MBaseObjectPtr CGraphViewWnd::createNew()
{
   CGraphViewWnd  *Wnd;

   Wnd = new CGraphViewWnd;
   Wnd->setName(getName());

   return (MBaseObject*)Wnd;
}


void CGraphViewWnd::ViewCreate() {
  MBaseViewWnd::ViewCreate();

  RECT clientRect;
  GetClientRect(&clientRect);

  if (m_Graph == NULL) {
    m_Graph = new CGraphComponent(this);
    m_Graph->Create(WS_VISIBLE, CRect(0,0,clientRect.right,clientRect.bottom), this);
//    m_Graph->InitOpenGL(m_Graph->m_hWnd);
    m_Graph->ShowWindow(SW_SHOW);
  }
  if (topShelf == NULL) {
    topShelf = new CShelfDlg;
    topShelf->Create(IDD_EMPTYDIALOG, this);
    topShelf->SetWindowPos(NULL, 0,0, clientRect.right, 24, SWP_NOZORDER);
    topShelf->LoadFromFile(g_ProgSet.m_PrefsPath + "GraphView.buttons");
    topShelf->ShowWindow(SW_SHOW);
  }
  if (paramListTree == NULL) {
    RECT rect;

    rect.left = 0;
    rect.top = 0;
    rect.right = 100;
    rect.bottom = clientRect.bottom;
    paramListTree = new CTreeCtrl();
    paramListTree->Create(
      WS_DLGFRAME | WS_VISIBLE | TVS_DISABLEDRAGDROP | TVS_HASLINES | 
      TVS_HASBUTTONS | TVS_LINESATROOT | TVS_SHOWSELALWAYS, 
      rect, this, ParamListID);
  }

  g_KeyList.ExecuteAction("GraphSelectTool");
}

template <class SET1, class SET2>
bool areEqual(const SET1 &set1, const SET2 &set2) {
  SET1::const_iterator it;

  if (set1.size() != set2.size()) {
    return false;
  }

  int count = 0;

  for (it = set1.begin(); it != set1.end(); ++it) {
    if (set2.find(*it) == set2.end()) {
      return false;
    }
  }

  // if we managed to get here, we must be equal
  return true;
}

void CGraphViewWnd::DrawView() {
  updateTreeFromScene();

  MBaseViewWnd::DrawView();

  CRect clientRect;
  GetClientRect(&clientRect);

  // Draw the split bar
  {
    RECT rect;
    GetClientRect(&rect);

    rect.left = clientRect.left + m_SplitPos;
    rect.right = rect.left + splitBarWidth + 1;

    ::FillRect(paintDC, &rect, (HBRUSH)::GetStockObject(GRAY_BRUSH));
  }

  std::vector<GraphPair> values;
  {
    MBaseObjectTreePtr Tree = g_Scene->getObjectList();
    MBaseObjectPtr BaseObj;
    MSceneObjectPtr Obj;
    RECT clientRect;
    
    GetClientRect(&clientRect);

    KeyParamSet::iterator keyIt;

    for (keyIt = selectedParams.begin(); keyIt != selectedParams.end(); ++keyIt) {
      MFloatKeyListPtr floatKeyList = AZTEC_CAST(MFloatKeyList, (*keyIt));

      if (floatKeyList != NULL) {
        values.push_back(GraphPair(floatKeyList, MVector3(0.5f,0.8f,0.5f)));
        continue;
      }

      MVector3KeyListPtr vectorKeyList = AZTEC_CAST(MVector3KeyList, (*keyIt));
      if (vectorKeyList != NULL) {
        values.push_back(GraphPair(vectorKeyList->getXValue(), MVector3(1,0,0)));
        values.push_back(GraphPair(vectorKeyList->getYValue(), MVector3(0,1,0)));
        values.push_back(GraphPair(vectorKeyList->getZValue(), MVector3(0,0,1)));
        continue;
      }
    }
  }

  m_Graph->setGraphValues(values);

  if (m_Graph != NULL) {
    m_Graph->DrawView();
  }
}

DWORD CGraphViewWnd::ViewPopupMenu(int x, int y)
{
   CMenu    Menu;
   CMenu    *Popup;
   DWORD    Choice;

   Menu.LoadMenu(IDR_GRAPHVIEW_POPUP);
   Popup = Menu.GetSubMenu(0);
   InitPopupMenu(Popup);

   // Check the carious menu items depending on the current flags etc.

   Choice = Popup->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_NONOTIFY | TPM_RETURNCMD, x, y, this);

   Popup->DestroyMenu();

   bool     Update;
   int      Result;

   Result = HandlePopupCommand(Choice);

   // The command was handled, but the view was changed, so we must return.
   if (Result == -1)
      return 1;
      
   if (Result == 1)
      Update = true;
   else 
      Update = false;

   if (Update) {
     Invalidate();
     UpdateWindow();
   }
   
   return Choice;
}

int CGraphViewWnd::HandlePopupCommand(DWORD Cmd)
{
   int Result;

   Result = MBaseViewWnd::HandlePopupCommand(Cmd);

   if (Result) {
      return Result;
   }

   switch (Cmd)
   {
   case ID_POPUP_SELECTTOOL:
     g_KeyList.ExecuteAction("GraphSelectTool");
     break;

   case ID_POPUP_MOVETOOL:
     g_KeyList.ExecuteAction("GraphMoveTool");
     break;

   case ID_POPUP_INSERTKEYSTOOL:
     g_KeyList.ExecuteAction("GraphInsertKeyTool");
     break;

   case ID_POPUP_DELETESELECTEDKEYS:
     g_KeyList.ExecuteAction("KObjectDeleteSelected");
     break;

   case ID_POPUP_ZOOMIN:
     g_KeyList.ExecuteAction("KViewportZoomIn");
     break;

   case ID_POPUP_ZOOMOUT:
     g_KeyList.ExecuteAction("KViewportZoomOut");
     break;

   case ID_POPUP_ZOOMTOALL:
     g_KeyList.ExecuteAction("KViewportFrameAll");
     break;

   case ID_POPUP_ZOOMTOSELECTED:
     g_KeyList.ExecuteAction("KViewportFrameSelected");
     break;

   default:
     return 0;
   }

   return 1;
}

CGraphComponent* CGraphViewWnd::getGraphComponent() {
  return &*m_Graph;
}

void CGraphViewWnd::doZoomToFit(bool doSelection) {
  std::vector<GraphPair> graphValues;

  getGraphComponent()->getGraphValues(graphValues);

  float minTime;
  float maxTime;
  bool timeRangeSet = false;

  float minValue;
  float maxValue;
  bool valueRangeSet = false;

  for (int i = 0; i < graphValues.size(); ++i) {
    MKeyableValue *keyableValue = AZTEC_CAST(MKeyableValue, graphValues[i].value);

    // if we dont have a keyable value, then we can't get a minimum and a maximum
    // time

    if (keyableValue == NULL) {
      continue;
    }

    std::vector<int> keyTimes;

    if (doSelection) {
      keyableValue->getSelectedKeyTimes(keyTimes);
    } else {
      keyableValue->getKeyTimes(keyTimes);
    }

    if (keyTimes.size() > 0) {
      if (!timeRangeSet) {
        minTime = keyTimes.front();
        maxTime = keyTimes.back();
        timeRangeSet = true;
      } else {
        if (keyTimes.front() < minTime) {
          minTime = keyTimes.front();
        }
        if (keyTimes.back() > maxTime) {
          maxTime = keyTimes.back();
        }
      }
    } else {
      if (!timeRangeSet) {
        minTime = g_Scene->getStartTime();
        maxTime = g_Scene->getEndTime();
        timeRangeSet = true;
      }
    }

    // if we have a float key list, then we will be able to get the values 
    // from the list to zoom to them.
    MFloatKeyList *floatList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);

    if (floatList != NULL) {
      if (floatList->getKeyCount() == 0) {
        float value = floatList->getValueAtTime(g_Scene->getTime());

        if (!valueRangeSet) {
          minValue = 0.0;
          maxValue = 0.0;
          valueRangeSet = true;
        } 
        if (value < minValue) {
          minValue = value;
        }
        if (value > maxValue) {
          maxValue = value;
        }
      } else {
        for (int i = 0; i < floatList->getKeyCount(); ++i) {
          float value = floatList->getValueAtIndex(i);

          if (doSelection && !floatList->getKeyAtIndex(i)->isSelected()) {
            continue;
          }

          if (!valueRangeSet) {
            minValue = value;
            maxValue = value;
            valueRangeSet = true;
          } else {
            if (value < minValue) {
              minValue = value;
            }
            if (value > maxValue) {
              maxValue = value;
            }
          }

        }
      }
    }
  }

  if (timeRangeSet) {
    getGraphComponent()->setGraphRange(minTime - g_Scene->getTicksPerFrame() * 2, 
                                       maxTime + g_Scene->getTicksPerFrame() * 2);
  }

  if (valueRangeSet) {
    minValue -= 0.5 + 0.1 * (maxValue - minValue);
    maxValue += 0.5 + 0.1 * (maxValue - minValue);
    getGraphComponent()->setValueRange(minValue, maxValue);
  }

}

void CGraphViewWnd::deleteSelectedKeys() {
  std::vector<GraphPair> graphValues;

  getGraphComponent()->getGraphValues(graphValues);

  for (int i = 0; i < graphValues.size(); ++i) {
    MKeyableValue *keyableValue = AZTEC_CAST(MKeyableValue, graphValues[i].value);

    if (keyableValue != NULL) {
      keyableValue->deleteSelectedKeys();
    }
  }
}

void CGraphViewWnd::onMouseDown(int x, int y, MShiftState Shift) {
  MBaseViewWnd::onMouseDown(x,y,Shift);

  if (x >= m_SplitPos || x <= m_SplitPos + splitBarWidth) {
    splitDragging = true;
    shiftingSplitPos = m_SplitPos;
    lastXPos = x;
  }
}

void CGraphViewWnd::onMouseUp(int x, int y, MShiftState Shift) {
  MBaseViewWnd::onMouseUp(x,y,Shift);

  splitDragging = false;
}


void CGraphViewWnd::onMouseMove(int x, int y, MShiftState Shift) {
  if (x >= m_SplitPos || x <= m_SplitPos + splitBarWidth) {
    ::SetCursor(LoadCursor(NULL, IDC_SIZEWE));
  } else {
    ::SetCursor(LoadCursor(NULL, IDC_ARROW));
  }

  if (splitDragging) {
    shiftingSplitPos += x - lastXPos;
    m_SplitPos = shiftingSplitPos;
    lastXPos = x;

    RECT rect;
    GetClientRect(&rect);

    if (m_SplitPos < 0) {
      m_SplitPos = 0;
    }
    if (m_SplitPos > rect.right - splitBarWidth) {
      m_SplitPos = rect.right - splitBarWidth;
    }

    OnSize(0, rect.right, rect.bottom);
  }

  MBaseViewWnd::onMouseMove(x,y,Shift);
}

void CGraphViewWnd::zoomIn() {
  float min, max;
  
  getGraphComponent()->getGraphRange(min, max);
  
  float middle = 0.5*(max + min);
  float diff = 0.5*(max - min);
  
  getGraphComponent()->setGraphRange(middle - diff/1.2, middle + diff/1.2);
}

void CGraphViewWnd::zoomOut() {
  float min, max;

  getGraphComponent()->getGraphRange(min, max);

  float middle = 0.5*(max + min);
  float diff = 0.5*(max - min);

  getGraphComponent()->setGraphRange(middle - diff*1.2, middle + diff*1.2);
}

void CGraphViewWnd::zoomToFitAll() {
  doZoomToFit(false);
}

void CGraphViewWnd::zoomToFitSelected() {
  doZoomToFit(true);
}

void CGraphViewWnd::OnSize(UINT nType, int cx, int cy) 
{
  GetClientRect(&m_PopupRect);

  CRect clientRect;
  GetClientRect(&clientRect);

  if (topShelf != NULL) {
    if (topShelf->m_hWnd != NULL) {
      topShelf->SetWindowPos(NULL,
                             clientRect.left, clientRect.top,
                             clientRect.Width(), toolbarHeight,
                             SWP_NOZORDER);
      clientRect.top += toolbarHeight;
    }
  }

  if (paramListTree != NULL && paramListTree->m_hWnd != NULL) {
    int right = m_SplitPos;
    int left = clientRect.left;
    paramListTree->SetWindowPos(NULL, 
                               clientRect.left, clientRect.top,
                               m_SplitPos, clientRect.Height(),
                               SWP_NOZORDER);

    clientRect.left += m_SplitPos + splitBarWidth;
  }
  if (m_Graph != NULL) {
    if (m_Graph->m_hWnd != NULL) {
      m_Graph->SetWindowPos(NULL, 
                            clientRect.left, clientRect.top,
                            clientRect.Width(), clientRect.Height(), 
                            SWP_NOZORDER);
    }
  }
}


void CGraphViewWnd::OnClose() 
{
   g_ViewList.DeleteView(this);
	
//	CWnd::OnClose();
}

void CGraphViewWnd::OnDestroy() {
  if (paramListTree != NULL) {
    paramListTree->DestroyWindow();
    delete paramListTree;
    paramListTree = NULL;
  }

  MBaseViewWnd::OnDestroy();
}

BOOL CGraphViewWnd::OnNotify(WPARAM wParam, LPARAM lParam, LRESULT *pResult) {
  if (wParam == ParamListID) {

    NMTREEVIEW *nm = (NMTREEVIEW*)lParam;

    if (nm->hdr.code == TVN_SELCHANGED) {
      HTREEITEM sel = paramListTree->GetSelectedItem();

      MKeyableValuePtr param = getParameterFromTree(sel);

      selectedParams.clear();
      if (param != NULL) {
        selectedParams.insert(param);
      }

      Invalidate();
      UpdateWindow();

      return TRUE;
    }
  }
  return FALSE;
}

void CGraphViewWnd::updateTreeFromScene() {
  MBaseObjectTreePtr Tree = g_Scene->getObjectList();
  MBaseObjectPtr BaseObj;
  MSceneObjectPtr Obj;
  std::set<MParameterObjectPtr> objParams;
  std::set<MSceneObjectPtr> sceneObjs;

  Tree->beginIteration();
  while (( BaseObj = Tree->getNext() ) != NULL ) {
    Obj = AZTEC_CAST(MSceneObject, BaseObj);
  
    if (Obj != NULL && Obj->isFlagged(OBJECTFLAG_SELECTED)) {
      sceneObjs.insert(Obj);
    }
  }
  Tree->endIteration();

  bool createList = false;

  if (!areEqual(sceneObjs, selectedObjects)) {
    createList = true;
  }

  if (createList) {
    // Here we have to deselect everything otherwise the windows
    // common control library spits the dummy everynow and then
    // and it crashes!!!!
    // This means we have to store our selections remporarily as well.
    KeyParamSet tempSelectedParams = selectedParams;

    paramListTree->SelectItem(NULL);
    paramListTree->DeleteAllItems();

    selectedObjects.clear();
    itemParamMap.clear();
    paramItemMap.clear();

    std::set<MSceneObjectPtr>::iterator sceneIt;
    for (sceneIt = sceneObjs.begin(); sceneIt != sceneObjs.end(); ++sceneIt) {
      HTREEITEM item;
      
      item = paramListTree->InsertItem((*sceneIt)->getName().c_str());

      selectedObjects.insert(std::make_pair((*sceneIt), item));

      addParameters(item, (*sceneIt)->getParamList());

      MShapeObjectPtr shape = (*sceneIt)->getShapeObject();

      if (shape != NULL) {
        addParameters(item, shape->getParamList());
      }

      paramListTree->Expand(item, TVE_EXPAND);
    }

    if (tempSelectedParams.size() != 0) {
      MKeyableValuePtr selectedKey = *tempSelectedParams.begin();

      HTREEITEM item = getTreeItem(selectedKey);

      if (item != NULL) {
        paramListTree->SelectItem(item);
      }  else {
        tempSelectedParams.clear();
      }
    }
  }
}

void CGraphViewWnd::addParameters(HTREEITEM parentItem, const MParameterObjectListPtr &paramList) {
  for (int i = 0; i < paramList->getNumParams(); ++i) {
    MParameterObjectPtr param = paramList->getParameter(i);
    MKeyParameter *keyParam = AZTEC_CAST(MKeyParameter, param);

    // we are only interested in animatable keys
    if (keyParam != NULL) {
      HTREEITEM paramItem = paramListTree->InsertItem(param->getFriendlyName().c_str(), parentItem);

      addParameter(paramItem, keyParam->getKeyableValue());

      MVector3KeyParameterPtr vector3Param = AZTEC_CAST(MVector3KeyParameter, keyParam);
      if (vector3Param != NULL) {
        addParameters(paramItem, vector3Param);
      }
    }
  }
}

void CGraphViewWnd::addParameters(HTREEITEM parentItem, const MVector3KeyParameterPtr &vectorParam) {
  std::string vectorNames[3];

  if (vectorParam->getDataMeaning() == MParameterObject::MEANING_COLOUR) {
    vectorNames[0] = "Red";
    vectorNames[1] = "Green";
    vectorNames[2] = "Blue";
  } else {
    vectorNames[0] = "X";
    vectorNames[1] = "Y";
    vectorNames[2] = "Z";
  }

  MVector3KeyListPtr vectorKeyList = AZTEC_CAST(MVector3KeyList, vectorParam->getVector3Value());
  if (vectorKeyList != NULL) {
    MFloatKeyableValuePtr param;

    param = AZTEC_CAST(MFloatKeyableValue, vectorKeyList->getXValue());
    if (param != NULL) {
      HTREEITEM item = paramListTree->InsertItem(vectorNames[0].c_str(), parentItem);
      addParameter(item, param);
    }
    param = AZTEC_CAST(MFloatKeyableValue, vectorKeyList->getYValue());
    if (param != NULL) {
      HTREEITEM item = paramListTree->InsertItem(vectorNames[1].c_str(), parentItem);
      addParameter(item, param);
    }
    param = AZTEC_CAST(MFloatKeyableValue, vectorKeyList->getZValue());
    if (param != NULL) {
      HTREEITEM item = paramListTree->InsertItem(vectorNames[2].c_str(), parentItem);
      addParameter(item, param);
    }
  }
}

void CGraphViewWnd::addParameter(HTREEITEM item, const MKeyableValuePtr &param) {
  itemParamMap.insert(std::make_pair(item, param));
  paramItemMap.insert(std::make_pair(param, item));
}

MKeyableValuePtr CGraphViewWnd::getParameterFromTree(HTREEITEM item) {
  // try and find this parameter in the tree view
  std::map<HTREEITEM, MKeyableValuePtr>::iterator treeIt;

  treeIt = itemParamMap.find(item);

  if (treeIt == itemParamMap.end()) {
    return NULL;
  } else {
    return treeIt->second;
  }
}

HTREEITEM CGraphViewWnd::getTreeItem(const MKeyableValuePtr &param) {
  // try and find this parameter in the tree view
  std::map<MKeyableValuePtr, HTREEITEM>::iterator treeIt;

  treeIt = paramItemMap.find(param);

  if (treeIt == paramItemMap.end()) {
    return NULL;
  } else {
    return treeIt->second;
  }
}

