#include <AztecMainPCH.h>
#include <MXYZToolType.h>

// Model Lib includes
#include <MUIManager.h>
#include <MEditableMesh.h>
#include <MAnimMesh.h>

// AztecMain includes
#include <MDLGlobs.h>

// standard includes
#include <math.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//---------------
//  MXYZToolType
//---------------
// Goes through the vertices of a given mesh if they will be altered by an action
void MXYZToolType::MarkMeshVerticesComponents(MMeshPtr Mesh, AztecFlags CompMode) {
  if (Mesh == NULL)
    return;
  
  if (CompMode == MComponentisedObject::POINT_TYPE) {
    for (int i = Mesh->getNumVerts()-1; i >= 0; i--) {
      if (Mesh->isVertexFlagged(i, VERTEX_SELECTED)) {
        Mesh->setVertexFlag(i, VERTEX_FLAGFORCHANGE);
      } else {
        Mesh->unsetVertexFlag(i, VERTEX_FLAGFORCHANGE);
      }
    }
  }
  if (CompMode == MComponentisedObject::FACET_TYPE) {
    MTriangle const *Tri;
    
    Mesh->unsetVertexFlags(VERTEX_FLAGFORCHANGE);

    for (int i = Mesh->getNumTris()-1; i >= 0; i--) {
      Tri = Mesh->getTriangle(i);
      if (Tri == NULL)
        continue;

      if (Tri->isFlagged(TRIANGLE_SELECTED)) {
        for (int n=0; n<3; n++) {
          Mesh->setVertexFlag(Tri->getVertex(n), VERTEX_FLAGFORCHANGE);
        }
      }
    }
    
  }
  if (CompMode == MComponentisedObject::EDGE_TYPE) {
    int         i;
    
    Mesh->unsetVertexFlags(VERTEX_FLAGFORCHANGE);
    for (i=Mesh->getNumTris()-1; i>=0; i--) {
      MTriangle const *Tri = Mesh->getTriangle(i);
      if (Tri == NULL)
        continue;

      for (int edge = 0; edge < 3; ++edge) {
        if (Tri->isEdgeFlagged(edge, EDGE_SELECTED)) {
          Mesh->setVertexFlag(Tri->getVertex(edge), VERTEX_FLAGFORCHANGE);
          Mesh->setVertexFlag(Tri->getVertex((edge+1) % 3), VERTEX_FLAGFORCHANGE);
        }
      }
    }
    
  }
}

void MXYZToolType::PrepareSelectedObjects(MVector3 &Origin, MMatrix4 &SelObjXFormMat) {
  MBaseObjectPtr BaseSelObj;
  MSceneObjectPtr SelObj;
  MTreeObjectNodePtr SelObjNode;
  MTransformObjectPtr XForm;
  
  BaseSelObj = g_Scene->getSelectedObjectList()->getTail();
  
  
  g_Scene->getSelectedObjectList()->beginIteration();
  
  while (( BaseSelObj = g_Scene->getSelectedObjectList()->getNext() ) != NULL )
  {
    SelObjNode = g_Scene->getObjectList()->findObject(BaseSelObj);
    
    SelObj = AZTEC_CAST(MSceneObject, BaseSelObj);
    
    if (SelObj != NULL) {
      // set up which vertices need to be moved if in component mode
      MShapeObjectPtr outObj = SelObj->getShapeObject();
      MEditableComponentisedObjectPtr compObj = SelObj->getEditableComponentObject();
      if (compObj != NULL && compObj->isInComponentMode()) {
        compObj->storeComponentPositions(compObj->getComponentMode());

        MEditableMeshPtr mesh = AZTEC_CAST(MEditableMesh, SelObj->getShapeObject()->convertToMesh());
        if (mesh != NULL) {
          MarkMeshVerticesComponents(mesh, mesh->getComponentMode());
        }
      }
      
      {
        MTreeObjectNodePtr ParentNode;
        
        ParentNode = NULL;
        
        if (SelObjNode != NULL) {
          ParentNode = SelObjNode->getParent();
        }
        
        XForm = g_Scene->getTransform(SelObjNode);
        
        if (XForm != NULL) {
          // Tell the Transformobject to keep a backup set of values for reference.
          XForm->holdValues();
          Origin = g_Scene->getSelectionCentre();
        }
        g_Scene->getWorldTransformMatrix(SelObjNode, SelObjXFormMat);
      }
    }
  }   
  g_Scene->getSelectedObjectList()->endIteration();
}

void MXYZToolType::UpdateKeys(const MStr &UndoName, AztecFlags KeyFlags, bool holdValues)
{
  MBaseObjectPtr BaseObj;
  MSceneObjectPtr Obj;
  
  g_Scene->getObjectList()->beginIteration();
  while (( BaseObj = g_Scene->getObjectList()->getNext() ) != NULL ) {
    Obj = AZTEC_CAST(MSceneObject, BaseObj);
    if (Obj == NULL) {
      continue;
    }
    
    MComponentisedObjectPtr compObj = Obj->getComponentObject();
    
    if (compObj != NULL && compObj->isInComponentMode()) {
      MEditableComponentisedObjectPtr editableCompObj = Obj->getEditableComponentObject();
      editableCompObj->updateComponentAnimation(
        editableCompObj->getComponentMode(), 
        MComponentisedObject::COMPONENT_SELECTED,
        g_Scene->getTime(),
        g_SysMan->getSettings()->m_Animate);

    } else if (Obj->isFlagged(OBJECTFLAG_SELECTED)) {
      MTransformObjectPtr XForm;
      
      XForm = Obj->getTransformObject();
      
      if (XForm != NULL) {
        XForm->updateKey(Obj->getTime(), KeyFlags, g_SysMan->getSettings()->m_Animate);
        if (holdValues) {
          XForm->holdValues();
        }
      }
    }
  }
  g_Scene->getObjectList()->endIteration();
  
}

void MXYZToolType::CreateUndoNode(const MStr &UndoName, AztecFlags KeyFlags)
{
}

void MXYZToolType::GetPlaneParams(MVector3 &Origin, MVector3 &Dir, MVector3 &ViewNorm, MMatrix4 &XFormMat)
{
  COpenGLWnd     *GLWnd = AZTEC_CAST(COpenGLWnd, g_CurView);
  
  if (GLWnd == NULL)
    return;
  
  GLWnd->m_ViewXFormInv.transform(MVector3(0,0,1),ViewNorm);
  
  ViewNorm.x = -ViewNorm.x;
  ViewNorm.y = -ViewNorm.y;
  ViewNorm.z = ViewNorm.z;
  
  PrepareSelectedObjects(Origin, XFormMat);
  MMatrix4 mat(m_AxisTransform);
  mat.inverse();
  if (m_PickedManip == 1) // X Axis
  {
    Dir.set(0,0,1);
    if (fabs(Dir * ViewNorm) < 0.01)
      Dir.set(0,1,0);
    Dir = m_AxisTransform * Dir;
  }
  else if (m_PickedManip == 2) // Y Axis
  {
    Dir.set(0,0,1);
    if (fabs(Dir * ViewNorm) < 0.01)
      Dir.set(1,0,0);
    Dir = m_AxisTransform * Dir;
  }
  else if (m_PickedManip == 3) // Z Axis
  {
    Dir.set(1,0,0);
    if (fabs(Dir * ViewNorm) < 0.01) {
      Dir.set(0,1,0);
    }
    Dir = m_AxisTransform * Dir;
  } else {
    Dir.set(ViewNorm);
  }
}

void MXYZToolType::GetPlaneParams(MConstraintType &con, const MMatrix4 &axisTransform, AztecFlags flags) {
  COpenGLWnd     *GLWnd = AZTEC_CAST(COpenGLWnd, g_CurView);
  
  if (GLWnd == NULL) {
    return;
  }

  MVector3 ViewNorm = GLWnd->getViewNormal();

  MVector3 Origin;
  MMatrix4 XFormMat;
  
  PrepareSelectedObjects(Origin, XFormMat);
  if (m_PickedManip == 1) {
    con.setConstraint(MRay(Origin, axisTransform*MVector3(1,0,0)));
  } else if (m_PickedManip == 2) {
    con.setConstraint(MRay(Origin, axisTransform*MVector3(0,1,0)));
  } else if (m_PickedManip == 3) {
    con.setConstraint(MRay(Origin, axisTransform*MVector3(0,0,1)));
  } else {
    if ((flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_X) {
      con.setConstraint(MPlane(Origin, axisTransform*MVector3(1,0,0)));
    } else if ((flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_Y) {
      con.setConstraint(MPlane(Origin, axisTransform*MVector3(0,1,0)));
    } else if ((flags & DRAWAXIS_MIDDLE_TYPE) == DRAWAXIS_MIDDLE_NORMAL_Z) {
      con.setConstraint(MPlane(Origin, axisTransform*MVector3(0,0,1)));
    } else {
      con.setConstraint(MPlane(Origin, ViewNorm));
    }
  }
}

int MXYZToolType::onMouseDown(int X, int Y, const MShiftState &Shift) {
  MToolType::onMouseDown(X,Y,Shift);
  m_DownVec = m_Constraint.getConstrainedPoint(m_CurRay);
  m_CurVec = m_Constraint.getConstrainedPoint(m_CurRay);
  return TOOLRESULT_DRAWNONE;
}

int MXYZToolType::onMouseUp(int X, int Y, const MShiftState &Shift) {
  MToolType::onMouseUp(X,Y,Shift);
  m_UpVec = m_Constraint.getConstrainedPoint(m_CurRay);
  m_CurVec = m_Constraint.getConstrainedPoint(m_CurRay);
  return TOOLRESULT_DRAWNONE;
}

int MXYZToolType::onMouseMove(int X, int Y, const MShiftState &Shift) {
  MToolType::onMouseMove(X,Y,Shift);
  m_CurVec = m_Constraint.getConstrainedPoint(m_CurRay);
  return TOOLRESULT_DRAWNONE;
}

void MXYZToolType::getAxisMatrix(COpenGLWnd *view) {
  if (MUIManager::getAxisMode() == MUIManager::WORLD_AXIS) {
    m_AxisTransform.identity();
  } else if (MUIManager::getAxisMode() == MUIManager::SCREEN_AXIS) {
    if (view == NULL) {
      m_AxisTransform.identity();
    } else {
      m_AxisTransform = MMatrix4(view->m_ModelMat);
    }
  } else if (MUIManager::getAxisMode() == MUIManager::OBJECT_AXIS) {
    MSceneObjectPtr sceneObj;
    MBaseObjectPtr obj;

    sceneObj = AZTEC_CAST(MSceneObject, g_Scene->getSelectedObjectList()->getTail());

    if (MUIManager::getAxisMode() == MUIManager::PARENT_AXIS && sceneObj != NULL) {
      sceneObj = sceneObj->getParent();
    }

    if (sceneObj != NULL) {
      g_Scene->getWorldTransformMatrix( g_Scene->getObjectList()->findObject(sceneObj), m_AxisTransform );
    } else {
      m_AxisTransform.identity();
    }
  } else if (MUIManager::getAxisMode() == MUIManager::LOCAL_AXIS) {

    MSceneObjectPtr sceneObj;
    MBaseObjectPtr obj;

    sceneObj = AZTEC_CAST(MSceneObject, g_Scene->getSelectedObjectList()->getTail());

    if (MUIManager::getAxisMode() == MUIManager::PARENT_AXIS && sceneObj != NULL) {
      sceneObj = sceneObj->getParent();
    }

    if (sceneObj != NULL) {
      // if we are not in component mode, use the object's axis, otherwise
      // use the components to figure things out.
      if (MUIManager::getComponentMode() == MComponentisedObject::OBJECT_TYPE) {
        g_Scene->getWorldTransformMatrix( g_Scene->getObjectList()->findObject(sceneObj), m_AxisTransform );
      } else {
        MMatrix4 objXForm;
        MVector3 avgNormal(0,0,0.001f); // just a little bit off centre to allow no selections to not cause problems.
        int compType = MUIManager::getComponentMode();

        g_Scene->getWorldTransformMatrix( g_Scene->getObjectList()->findObject(sceneObj), objXForm );

        MComponentisedObjectPtr compObj = sceneObj->getComponentObject();
        if (compObj != NULL) {
        // iterate over the object's components
          for (int n = 0; n < compObj->getComponentCount(compType); ++n) {
            if (!compObj->isComponentFlagged(compType, n)) {
              continue;
            }

            avgNormal += compObj->getComponentNormal(compType, n);
          }
        }

        avgNormal.normalize();

        // now we have an average normal, work out the matrix
        // which transform the z axis onto that normal
        MMatrix4 xform;
        {
          MVector3 zAxis(0,0,1), axis;
          xform.identity();
          float angle(acos(zAxis * avgNormal));

          if (fabs(angle) > 0.01) {
            axis = zAxis / avgNormal;
            axis.normalize();

            // rotate with vector takes radians
            xform.rotateWithVector(axis, angle);
          }
        }

        m_AxisTransform = xform * objXForm;

      }
    } else {
      m_AxisTransform.identity();
    }
  } else {
    // just a catchall here
    m_AxisTransform.identity();

  }

  m_AxisTransform.normaliseRotation();
  m_AxisTransform.m[0][3] = 0;
  m_AxisTransform.m[1][3] = 0;
  m_AxisTransform.m[2][3] = 0;
  m_AxisTransform.m[3][3] = 1;
  m_AxisTransform.m[3][2] = 0;
  m_AxisTransform.m[3][1] = 0;
  m_AxisTransform.m[3][0] = 0;

  m_AxisTransformInv = m_AxisTransform;
  m_AxisTransformInv.inverse();

}

void MXYZToolType::updateVectors() {
}

//------------------
//  MConstraintType
//------------------
void MConstraintType::setConstraint(const MPlane &plane) {
  constraintType = ctPlane;
  m_Plane = plane;
}

void MConstraintType::setConstraint(const MRay &ray) {
  constraintType = ctRay;
  m_Ray = ray;
}

MVector3 MConstraintType::getConstrainedPoint(const MRay &srcRay) {
  if (constraintType == ctPlane) {
    return srcRay.intersectWithPlane(m_Plane);
  } else if (constraintType == ctRay) {
    double dist;

    MMath::distanceBetween(m_Ray, srcRay, &dist);

    return m_Ray.Org + dist * m_Ray.Dir;
  }

  return MVector3();
}

MVector3 MConstraintType::getConstraintNormal() {
  if (constraintType == ctPlane) {
    return MVector3(m_Plane.A, m_Plane.B, m_Plane.C);
  } else if (constraintType == ctRay) {
    return m_Ray.Dir;
  }

  return MVector3();
}

