#include <StdAfx.h>

#include <animation/MIKControllerCCD.h>
#include <MSystemManager.h>

#include <math.h>

namespace Aztec {

  MIKControllerCCD::MIKControllerCCD() {
  }

  MStr MIKControllerCCD::getClassName() {
    return "IKControllerCCD";
  }

  MBaseObjectPtr MIKControllerCCD::createNew() {
    MIKControllerPtr result = new MIKControllerCCD();
    result->setFrom(this);
    
    return result;
  }

  typedef std::vector<MSceneObjectPtr> BoneList;

  static void resetXForm(const MSceneObjectPtr &bone) {
    MVector3 rotation;

    if (bone->findParameter("initialRotate")->getValueVector(rotation)) {
      bone->findParameter("Rotate")->setValueVector(rotation);
    }
  }

  static void resetXForms(const BoneList &bones) 
  {
    for (BoneList::const_iterator it = bones.begin(); it != bones.end(); ++it) {
      resetXForm(*it);
    }
  }

  /**
   * This gets all the bones from the end bone to the root bone, and places
   * it into the given container. The resulting container DOES NOT contain
   * endBone.
   */
  static void getBones(const MSceneObjectPtr &rootBone, 
                       const MSceneObjectPtr &endBone,
                       BoneList &list) {
    list.clear();

    if (endBone == NULL || rootBone == NULL) {
      return;
    }

    MSceneObjectPtr current = endBone->getParent();

    while (current != rootBone) {
      list.push_back(current);  
      current = current->getParent();
    }

    list.push_back(rootBone);  
  }

  static void rotateBone(const MSceneObjectPtr &bone, 
                         const MVector3 &worldAxis,
                         float angle)
  {
    MVector3 Vec;
    MVector3 axis = worldAxis;
    MMatrix4 mat;
    MVector3 x,y,z,nx,ny,nz;

    Vec = bone->getRotateVector(MSystemManager::getInstance()->getScene()->getTime());
    Vec *= (float)(M_PI / 180.0);
    mat.convertFromEuler(Vec);

    x.set(mat.m[0][0], mat.m[0][1], mat.m[0][2]);
    y.set(mat.m[1][0], mat.m[1][1], mat.m[1][2]);
    z.set(mat.m[2][0], mat.m[2][1], mat.m[2][2]);

    axis.normalize();
    rotateWithVector(x, axis, angle, nx);
    rotateWithVector(y, axis, angle, ny);
    rotateWithVector(z, axis, angle, nz);

    mat.m[0][0] = nx.x; mat.m[0][1] = nx.y; mat.m[0][2] = nx.z;
    mat.m[1][0] = ny.x; mat.m[1][1] = ny.y; mat.m[1][2] = ny.z;
    mat.m[2][0] = nz.x; mat.m[2][1] = nz.y; mat.m[2][2] = nz.z;

    mat.convertToEuler(Vec);
    Vec *= (float)(180.0 / M_PI);
    bone->setRotateVector(Vec);
    bone->setParamByName("Rot", Vec.convertToString(), false);
    
  }

  static void orientBone(const MSceneObjectPtr &bone, 
                         const MSceneObjectPtr &end, 
                         const MVector3 &target, 
                         float dampen = 1.0) {

    MScenePtr scene = MSystemManager::getInstance()->getScene();

    MVector3 origin = scene->objectToWorldSpace(MBaseObjectPtr(bone), MVector3(0,0,0));
    MVector3 currentEnd = scene->objectToWorldSpace(MBaseObjectPtr(end), MVector3(0,0,0));

    MVector3 originToTarget = target - origin;
    MVector3 originToEndEffector = currentEnd - origin;

    originToTarget.normalize();
    originToEndEffector.normalize();
    float dot = originToTarget * originToEndEffector;
    if (dot < -1.0) {
      dot = -1.0;
    } else if (dot > 1.0) {
      dot = 1.0;
    }
    float angBetween = (float)acos(dot);
    MVector3 axisBetween = originToTarget / originToEndEffector;

    axisBetween.normalize();
    rotateBone(bone, axisBetween, -dampen * angBetween);

  }


  bool MIKControllerCCD::doUpdateObject() {
    if (getEndEffector() == NULL || 
        getStartBone() == NULL || 
        getEndBone() == NULL) 
    {
      return false;
    }

    if (lastRoot != getStartBone() || lastEnd != getEndBone()) {
      lastRoot = getStartBone();
      lastEnd = getEndBone();
    }


    MScenePtr scene = MSystemManager::getInstance()->getScene();
    MVector3 endEffector;

    // Get the position of the end effector in world space
    endEffector = scene->objectToWorldSpace(MBaseObjectPtr(getEndEffector()), MVector3(0,0,0));

    // get all the bones in the chain from end to start.
    BoneList bones;
    getBones(getStartBone(), lastEnd, bones);

    // reset all the rotations back to the way they started.
    resetXForms(bones);


    // do a number of iterations until we converge on a solution
    for (int i = 0; i < 20; ++i) {
      int boneCount = 0;

      // iterate up the bone chain, adjusting the bones to their proper places
      for (BoneList::const_iterator it = bones.begin(); it != bones.end(); ++it) {
        ++boneCount;
        orientBone((*it), lastEnd, endEffector, (float)boneCount / (float)bones.size());
      }

      MVector3 dist = scene->objectToWorldSpace(MBaseObjectPtr(lastEnd), MVector3(0,0,0)) - endEffector;
      if (dist.length2() < 0.01) {
        break;
      }

    }

    unsetFlag(OBJECTFLAG_NEEDS_UPDATE);

    return true;
  }

}
