#include "StdAfx.h"

#include "MTextFileStream.h"

#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

  const int MTextFileReader::QUOTED_STRINGS = 0x0001;

  //--------------------------
  // MTextFileReader
  //--------------------------

  MTextFileReader::MTextFileReader() { 
    m_FileHandle = NULL;
    m_Flags = QUOTED_STRINGS;
  }

  MTextFileReader::MTextFileReader(const MStr &filename, int flags) { 
    m_FileHandle = NULL;
    open(filename);
    m_Flags = flags;
  }

  MTextFileReader::~MTextFileReader() {
    close();
  }

  bool MTextFileReader::open(const MStr &filename) { 
    if (isOpen()) {
      close();
    }

    // open the file for reading in text mode
    m_FileHandle = ::fopen(filename.c_str(), "rt");

    return (m_FileHandle != NULL);
  }

  void MTextFileReader::close() { 
    if (isOpen()) {
      ::fclose(m_FileHandle);
      m_FileHandle = NULL;
    }
  }

  bool MTextFileReader::isOpen() { 
    return m_FileHandle != NULL;
  }

  bool MTextFileReader::isAtEOF() { 
    return feof(m_FileHandle) != 0;
  }

  MStr MTextFileReader::readLine() {
    MStr result;
    char readChar = '\x0';
    int size;

    while (!isAtEOF() && readChar != '\n') {
      size = ::fread(&readChar, 1, 1, m_FileHandle);

      if (size == 1 && readChar != '\n') {
        result += readChar;
      }
    }
    return result;
  }

  char MTextFileReader::readChar() { 
    return fgetc(m_FileHandle);
  }

  unsigned char MTextFileReader::readUnsignedChar() {
    MStr str = readString();
    return (unsigned char)str.ToDWORD();
  }

  short MTextFileReader::readShort() {
    MStr str = readString();
    return (short)str.ToInt();
  }

  unsigned short MTextFileReader::readUnsignedShort() {
    MStr str = readString();
    return (unsigned short)str.ToDWORD();
  }

  int MTextFileReader::readInt() { 
    MStr  str;
    str = readString();

    return str.ToInt();
  }

  unsigned int MTextFileReader::readUnsignedInt() {
    MStr str = readString();
    return str.ToDWORD();
  }

  float MTextFileReader::readFloat() {
    MStr str = readString();
    return str.ToFloat();
  }

  double MTextFileReader::readDouble() {
    // TODO: return an actual double.
    MStr str = readString();
    return str.ToFloat();
  }

  bool MTextFileReader::readBoolean() {
    MStr str = readString();
    return (str.compareNoCase("true") == 0);
  }

  MStr MTextFileReader::readString() { 
    if (m_Flags & QUOTED_STRINGS) {
      char  charRead;
      MStr  result;

      // skip any whitespace.
      while (isspace(charRead = readChar())) {
        if (isAtEOF()) {
          return result;
        }
      } 

      if (isAtEOF()) {
        return result;
      }

      // if we encountered a quoted string, deal with it.
      if (charRead == '\"') {
        // keep going until we find another quote
        while ((charRead = readChar()) != '\"') {
          result += MStr(charRead);

          if (isAtEOF()) {
            return result;
          }
        }

        // we have our result, just return it.
        return result;
      } else {
        result += MStr(charRead);
      
        // read in until we find some whitespace.
        // skip any whitespace.
        while (!isspace(charRead = readChar())) { 
          if (isAtEOF()) {
            return result;
          }
          result += MStr(charRead);
        } 
  
        //return the result
        return result;
      }

    } else {
      char  result[1024];

      fscanf(m_FileHandle, "%s", result);

      return result;
    }
  }

  int MTextFileReader::readBuffer(void *buffer, int bufSize) {
    // throw an assertion as this isn't implemented for text files yet
    assert(0);
    return 0;
  }


  //--------------------------
  // MTextFileWriter
  //--------------------------

  MTextFileWriter::MTextFileWriter() { 
    m_FileHandle = NULL;
  }

  MTextFileWriter::MTextFileWriter(const MStr &filename, bool append) {
    m_FileHandle = NULL;
    open(filename, append);
  }

  MTextFileWriter::~MTextFileWriter() {
    close();
  }

  bool MTextFileWriter::open(const MStr &filename, bool append) {
    if (isOpen()) {
      close();
    }

    // open the file writing for in text mode
    if (append) {
      m_FileHandle = ::fopen(filename.c_str(), "at");
    } else {
      m_FileHandle = ::fopen(filename.c_str(), "wt");
    }

    return (m_FileHandle != NULL);
  }

  void MTextFileWriter::close() { 
    if (isOpen()) {
      ::fflush(m_FileHandle);
      ::fclose(m_FileHandle);
      m_FileHandle = NULL;
    }
  }

  bool MTextFileWriter::isOpen() { 
    return m_FileHandle != NULL;
  }

  bool MTextFileWriter::isAtEOF() { 
    return feof(m_FileHandle) != 0;
  }

  void MTextFileWriter::writeChar(char value) {
    fwrite(&value, 1, 1, m_FileHandle);
  }

  void MTextFileWriter::writeUnsignedChar(unsigned char value) {
    writeUnsignedInt(value);
  }

  void MTextFileWriter::writeShort(short value) {
    writeInt(value);
  }

  void MTextFileWriter::writeUnsignedShort(unsigned short value) {
    writeUnsignedInt(value);
  }
  
  void MTextFileWriter::writeInt(int value) {
    char  str[32];

    sprintf(str, "%i", value);

    writeString(str);
  }

  void MTextFileWriter::writeUnsignedInt(unsigned int value) {
    char  str[32];

    sprintf(str, "%u", value);

    writeString(str);
  }

  void MTextFileWriter::writeFloat(float value) {
    char str[32];

    sprintf(str, "%.4f", value);
    writeString(str);
  }

  void MTextFileWriter::writeDouble(double value) {
    char str[32];

    sprintf(str, "%.4f", value);
    writeString(str);
  }

  void MTextFileWriter::writeBoolean(bool value) {
  }

  void MTextFileWriter::writeString(const MStr &value) {
    fwrite(value.c_str(), 1, strlen(value.c_str()), m_FileHandle);
  }

  void MTextFileWriter::writeString(const char *value) {
    fwrite(value, 1, strlen(value), m_FileHandle);
  
  }

  void MTextFileWriter::writeBuffer(const void *buffer, int bufSize) {
    // assert here because there is no implementation for this for text file streams yet.
    assert(0);
  }

}
