#include "StdAfx.h"
#include <MPluginManager.h>
#include <MSystemManager.h>

namespace Aztec {

  template <class TranslatorContainer>
  static typename TranslatorContainer::value_type canImport(const TranslatorContainer &container, const MStr &filename) {
    typename TranslatorContainer::const_iterator it;
    for (it = container.begin(); it != container.end(); ++it) {
      if ((*it)->canImportFile(filename)) {
        return *it;
      }
    }

    return NULL;
  }

  int MPluginManager::registerObject(const MBaseObjectPtr &object) {
    if (object == NULL) {
      return 0;
    }
    
    MSystemManager::getInstance()->logOutput("SysMan: Registering Class '%s' ...", (LPCTSTR)object->getClassName());
    
    // now try and insert the new object and its class name.
    std::pair<ObjectMap::iterator, bool> result = objectTemplates.insert(ObjectMap::value_type(object->getClassName().c_str(), object));

    // if the insertion failed, report an error
    if (!result.second && result.first->second != NULL) {
      MSystemManager::getInstance()->logOutput("SysMan: Registering Class '%s' Failed. Class already exists", object->getClassName().c_str());
      return 0;
    } else {
      result.first->second = object;
    }

    return 1;
  }

  MBaseObjectPtr MPluginManager::createObjectFromDefault(const MStr &className) {
    // try to find the object by class name.
    MBaseObjectPtr templateObj = objectTemplates[className.c_str()];
    MBaseObjectPtr newObject;

    if (templateObj != NULL) {
      newObject = templateObj->createNew();
    } else {
      // if we don't get an exact class name match, try to matching against object name.
      for (ObjectMap::iterator it = objectTemplates.begin(); it != objectTemplates.end(); ++it) {
        MNamedObjectPtr namedObject = AZTEC_CAST(MNamedObject, it->second);
        if (namedObject != NULL && namedObject->getName().compareNoCase(className) == 0) {
          newObject = it->second->createNew();
          break;
        }
      }
    }

    return newObject;
  }


  int MPluginManager::registerSceneTranslator(const MSceneTranslatorPtr &translator) {
    MSystemManager::getInstance()->logOutput("SysMan: Registering Scene Translator '%s' ", (LPCTSTR)translator->getClassName());
    sceneTranslators.push_back(AZTEC_CAST(MSceneTranslator, translator->createNew()));
    return 1;
  }

  int MPluginManager::getSceneTranslatorCount() {
    return sceneTranslators.size();
  }

  MSceneTranslatorPtr MPluginManager::getSceneTranslator(int index) {
    return sceneTranslators[index];
  }

  MSceneTranslatorPtr MPluginManager::getSceneTranslatorThatImports(const MStr &filename) {
    return canImport(sceneTranslators, filename);
  }

  MSceneTranslatorPtr MPluginManager::getSceneTranslatorThatExports(const MStr &filename) {
    MSceneTranslatorPtr Trans;
    MStr extension;
    filename.SplitPath(NULL, NULL, NULL, &extension);
    extension.toLower();

    for (unsigned int i = 0; i < sceneTranslators.size(); ++i) {
      // get the file filter for the translator.
      MStr filefilter = sceneTranslators[i]->getFilter().c_str();
      filefilter.toLower();

      // try to find our file extension in the filter
      if (filefilter.findSubstring(extension) != -1) {
        // since we have a match, just return that.
        return Trans;
      }
    }
    
    return NULL;
  }


  int MPluginManager::registerImageTranslator(const MImageTranslatorPtr &translator) {
    MSystemManager::getInstance()->logOutput("SysMan: Registering Image Translator '%s' ", (LPCTSTR)translator->getClassName());
    imageTranslators.push_back(AZTEC_CAST(MImageTranslator, translator->createNew()));
    return 1;
  }

  int MPluginManager::getImageTranslatorCount() {
    return imageTranslators.size();
  }

  MImageTranslatorPtr MPluginManager::getImageTranslator(int index) {
    return imageTranslators[index];
  }

  MImageTranslatorPtr MPluginManager::getImageTranslatorThatImports(const MStr &filename) {
    if (imageTranslators.size() == 0) {
      return NULL;
    }
    if (filename.GetLength() == 0) {
      return NULL;
    }
   
    ImageTranslatorVector possibleTranslators;
    ImageTranslatorVector::iterator it;

    MImageTranslatorPtr Trans;
    MStr ext;

    filename.SplitPath(NULL, NULL, NULL, &ext);
    ext.toLower();

    for (it = imageTranslators.begin(); it != imageTranslators.end(); ++it) {
      MStr filefilter = (*it)->getFilter().c_str();
      filefilter.toLower();

      // if we can find our file extension in the filter, then we have a filename match.
      if (filefilter.findSubstring(ext) != -1) {
        possibleTranslators.push_back(*it);
      }
    }

    return canImport(possibleTranslators, filename);
  }

  int MPluginManager::registerMaterialTranslator(const MMaterialTranslatorPtr &translator) {
    MSystemManager::getInstance()->logOutput("SysMan: Registering Material Translator '%s' ", (LPCTSTR)translator->getClassName());
    materialTranslators.push_back(AZTEC_CAST(MMaterialTranslator, translator->createNew()));
    return 1;
  }

  int MPluginManager::getMaterialTranslatorCount() {
    return materialTranslators.size();
  }

  MMaterialTranslatorPtr MPluginManager::getMaterialTranslator(int index) {
    return materialTranslators[index];
  }

  MMaterialTranslatorPtr MPluginManager::getMaterialTranslatorThatImports(const MStr &filename) {
    return canImport(materialTranslators, filename);
  }

 
  int MPluginManager::registerPrimitive(const MNamedObjectPtr &namedObject, const MStr &menuName) {
    if (namedObject == NULL) {
      return 0;
    }
    
    MStr     Str;
    
    MSystemManager::getInstance()->logOutput("SysMan: Registering Primtive '%s' ...", namedObject->getClassName().c_str());
    
    if (registerObject(namedObject)) {
      namedObject->setName(menuName);
      primitives.push_back(namedObject);
      return 1;
    } else {    
      MSystemManager::getInstance()->logOutput("SysMan: Registering Primtive '%s' Failed. Class already registered as primitive", namedObject->getClassName().c_str());
      return 0;
    }
  }

  int MPluginManager::getPrimitiveCount() {
    return primitives.size();
  }

  MNamedObjectPtr MPluginManager::getPrimitive(int index) {
    return primitives[index];
  }

  template <class TranslatorContainer>
    static TranslatorContainer getTranslators(
      const TranslatorContainer &container,
      MPluginManager::CapabilityEnum canImport, 
      MPluginManager::CapabilityEnum canExport)
  {
    // if both criteria are DONT_CARE, then we will be returning everything, 
    // so just do that now.
    if (canImport == MPluginManager::DONT_CARE && canExport == MPluginManager::DONT_CARE) {
      return container;
    }

    TranslatorContainer result;

    // iterate over our store translators, and add in ones that match our 
    // criteria.
    typename TranslatorContainer::const_iterator it;
    for (it = container.begin(); it != container.end(); ++it) {
      // If we care wether the translator can import or not, check the 
      // ability of the translator
      if (canImport != MPluginManager::DONT_CARE) {

        // Here is our criteria 
        //   MUST && translator->canImport()  : add the translator in
        //   MUST_NOT && !translator->canImport()  : add the translator in
        //   otherwise dont add it in
        if ( !(canImport == MPluginManager::MUST && (*it)->canImport()) &&
             !(canImport == MPluginManager::MUST_NOT && !(*it)->canImport()) ) 
        {
          continue;
        }
      }

      if (canExport != MPluginManager::DONT_CARE) {

        // Here is our criteria 
        //   MUST && translator->canImport()  : add the translator in
        //   MUST_NOT && !translator->canImport()  : add the translator in
        //   otherwise dont add it in
        if ( !(canExport == MPluginManager::MUST && (*it)->canExport()) &&
             !(canExport == MPluginManager::MUST_NOT && !(*it)->canExport()) ) 
        {
          continue;
        }
      }

      // if we have gotten here, then it has passed all our criteria, 
      // so add it in.
      result.push_back(*it);
    }

    return result;

  }


  MPluginManager::ImageTranslatorVector MPluginManager::getImageTranslators(
      MPluginManager::CapabilityEnum canImport, 
      MPluginManager::CapabilityEnum canExport) 
  {
    return getTranslators(imageTranslators, canImport, canExport);
  }


  MPluginManager::SceneTranslatorVector MPluginManager::getSceneTranslators(
      MPluginManager::CapabilityEnum canImport, 
      MPluginManager::CapabilityEnum canExport) 
  {
    return getTranslators(sceneTranslators, canImport, canExport);
  }

  
  void MPluginManager::registerRenderer(const MSceneRendererPtr &renderer) {
    renderers.insert(RendererMap::value_type(renderer->getName(), renderer));
  }

  int MPluginManager::getRendererCount() {
    return renderers.size();
  }

  MSceneRendererPtr MPluginManager::getRenderer(int index) {
    RendererMap::iterator it = renderers.begin();
    std::advance(it, index);
    return it->second;
  }

  MSceneRendererPtr MPluginManager::getRenderer(const std::string &name) {
    return renderers[name];
  }

  
}

