//---------------------------------------------------------------------------
#include "StdAfx.h"

#include "MMath.h"

#include <stdio.h>
#include <math.h>
#include <float.h>

#include "EulerAngles.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define PI 3.1415926535897932384626

namespace Aztec {

  //---------
  // MMath
  //---------
  double MMath::distanceBetween(const MRay &ray1, const MRay &ray2, double *ray1Dist, double *ray2Dist) {
    // This implementation was borrowed from the superb work of the Magic Math library.
    MVector3 kDiff = ray1.Org - ray2.Org;
    double fA00 = ray1.Dir.length2();
    double fA01 = -ray1.Dir.dotProduct(ray2.Dir);
    double fA11 = ray2.Dir.length2();
    double fB0 = kDiff.dotProduct(ray1.Dir);
    double fC = kDiff.length2();
    double fDet = fabs(fA00*fA11-fA01*fA01);
    double fB1, fS, fT, fSqrDist;
    
    if ( fDet >= 0.00001 )
    {
      // lines are not parallel
      fB1 = -kDiff.dotProduct(ray2.Dir);
      double fInvDet = 1.0/fDet;
      fS = (fA01*fB1-fA11*fB0)*fInvDet;
      fT = (fA01*fB0-fA00*fB1)*fInvDet;
      fSqrDist = fS*(fA00*fS+fA01*fT+2.0*fB0) +
        fT*(fA01*fS+fA11*fT+2.0*fB1)+fC;
    }
    else
    {
      // lines are parallel, select any closest pair of points
      fS = -fB0/fA00;
      fT = 0.0;
      fSqrDist = fB0*fS+fC;
    }
    
    if ( ray1Dist )
      *ray1Dist = fS;
    
    if ( ray2Dist )
      *ray2Dist = fT;
    
    return fabs(fSqrDist);
  }

  //----------------------------------------------------------------------------------------
  //  MVector3
  //----------------------------------------------------------------------------------------
  
  float MVector3::dotProduct(const MVector3& B) const {
    return x*B.x + y*B.y + z*B.z;
  }
  
  MVector3 MVector3::crossProduct(const MVector3& B) const {
    MVector3 temp;
    
    temp.x = y*B.z - z*B.y;
    temp.y = z*B.x - x*B.z;
    temp.z = x*B.y - y*B.x;
    
    return temp;
  }
  
  MVector3 MVector3::addVector(const MVector3& B) const {
    MVector3 temp;
    
    temp.x = x + B.x;
    temp.y = y + B.y;
    temp.z = z + B.z;
    
    return temp;
  }
  
  MVector3 MVector3::subVector(const MVector3& B) const {
    MVector3 temp;
    
    temp.x = x - B.x;
    temp.y = y - B.y;
    temp.z = z - B.z;
    
    return temp;
  }
  
  MVector3 MVector3::scalarbyVector(float a) const {
    MVector3 temp;
    
    temp.x = x * a;
    temp.y = y * a;
    temp.z = z * a;
    
    return temp;
  }
  
  void MVector3::normalize() {
    float l = length();
    
    if (l > 0) {
      x /= l;
      y /= l;
      z /= l;
    } else {
      x = 1;
      y = 0;
      z = 0;
    }
  }
  
  float MVector3::length() const {
    return (float)(sqrt( x*x + y*y + z*z));
  }
  
  float MVector3::length2() const {
    return (float)(x*x + y*y + z*z);
  }
  
  MVector3& MVector3::operator= (const MVector4 &v) {
    x = v.x; 
    y = v.y; 
    z = v.z; 
    return *this;
  };
  
  // This allows array-like access for our elements x, y and z.
  const float& MVector3::operator[] (int i) const {
    i = i%3;
    switch(i) {
    case 0:
      return x;
      break;
    case 1:
      return y;
      break;
    case 2:
      return z;
      break;
    default:
      return x;
    }
  }
  
  // This allows array-like access for our elements x, y and z.
  float& MVector3::operator[] (int i) {
    i = i%3;
    switch(i) {
    case 0:
      return x;
      break;
    case 1:
      return y;
      break;
    case 2:
      return z;
      break;
    default:
      return x;
    }
  }
  
  
  // String conversion operations
  MStr MVector3::convertToString(const MStr &FormatStr) const {
    MStr  Str;
    return Str.Format((LPCTSTR)FormatStr, x,y,z);
  }
  
  bool MVector3::convertFromString(const MStr &Str) {
    char  XStr[32],YStr[32],ZStr[32];
    char  OrigStr[96], *TempStr;
    
    memset(OrigStr, 0,96);
    strcpy(OrigStr, (LPCTSTR)Str);
    
    
    TempStr = OrigStr + 95;
    while (TempStr >= OrigStr) {
      if (*TempStr == ' '|| *TempStr == '\t')
        *TempStr = '\x0';
      
      TempStr--;
    }
    
    
    TempStr = OrigStr;
    while (*TempStr == '\x0' && (TempStr < OrigStr+95))
      TempStr++;
    
    strcpy(XStr, TempStr);
    
    while (*TempStr != '\x0' && (TempStr < OrigStr+95))
      TempStr++;
    while (*TempStr == '\x0' && (TempStr < OrigStr+95))
      TempStr++;
    
    strcpy(YStr, TempStr);
    
    while (*TempStr != '\x0' && (TempStr < OrigStr+95))
      TempStr++;
    while (*TempStr == '\x0' && (TempStr < OrigStr+95))
      TempStr++;
    
    strcpy(ZStr, TempStr);
    
    
    if (strcmp(XStr, ".") != 0)
      x = (float)atof(XStr);
    
    if (strcmp(YStr, ".") != 0)
      y = (float)atof(YStr);
    if (strcmp(ZStr, ".") != 0)
      z = (float)atof(ZStr);
   
    return true;
  }
  
  void rotateWithVector(const MVector3 &x, MVector3 u, float ang, MVector3 &Res) {
    MVector3 h,v,uxx;
    
    u.normalize();
    
    uxx = (u / x) * (float)sin(ang);
    
    h = u * (x * u);
    
    v = (x - h) * (float)cos(ang);
    
    Res = (h + v) + uxx;
  }
  
  MVector3 operator* (float f, const MVector3 &V) {
    return V.scalarbyVector(f);
  }
  
  //----------------------------------------------------------------------------------------
  //  Matrix3
  //----------------------------------------------------------------------------------------
  
  void Matrix3::set(const MVector3& x, const MVector3& y, const MVector3& z)
  {
    m1 = x;
    m2 = y;
    m3 = z;
  }
  
  void Matrix3::set(float m11, float m12, float m13,
    float m21, float m22, float m23,
    float m31, float m32, float m33)
  {
    m1.x = m11;
    m1.y = m12;
    m1.z = m13;
    m2.x = m21;
    m2.y = m22;
    m2.z = m23;
    m3.x = m31;
    m3.y = m32;
    m3.z = m33;
  }
  
  Matrix3& Matrix3::operator+= (Matrix3& m)
  {
    m1.x += m.m1.x; m1.y += m.m1.y; m1.z += m.m1.z;
    m2.x += m.m2.x; m2.y += m.m2.y; m2.z += m.m2.z;
    m3.x += m.m3.x; m3.y += m.m3.y; m3.z += m.m3.z;
    return *this;
  }
  
  Matrix3& Matrix3::operator-= (Matrix3& m)
  {
    m1.x -= m.m1.x; m1.y -= m.m1.y; m1.z -= m.m1.z;
    m2.x -= m.m2.x; m2.y -= m.m2.y; m2.z -= m.m2.z;
    m3.x -= m.m3.x; m3.y -= m.m3.y; m3.z -= m.m3.z;
    return *this;
  }
  
  Matrix3& Matrix3::operator*= (Matrix3& m)
  {
    Matrix3 r;
    r.m1.x = m1.x*m.m1.x + m1.y*m.m2.x + m1.z*m.m3.x;
    r.m1.y = m1.x*m.m1.y + m1.y*m.m2.y + m1.z*m.m3.y;
    r.m1.z = m1.x*m.m1.z + m1.y*m.m2.z + m1.z*m.m3.z;
    r.m2.x = m2.x*m.m1.x + m2.y*m.m2.x + m2.z*m.m3.x;
    r.m2.y = m2.x*m.m1.y + m2.y*m.m2.y + m2.z*m.m3.y;
    r.m2.z = m2.x*m.m1.z + m2.y*m.m2.z + m2.z*m.m3.z;
    r.m3.x = m3.x*m.m1.x + m3.y*m.m2.x + m3.z*m.m3.x;
    r.m3.y = m3.x*m.m1.y + m3.y*m.m2.y + m3.z*m.m3.y;
    r.m3.z = m3.x*m.m1.z + m3.y*m.m2.z + m3.z*m.m3.z;
    *this = r;
    return *this;
  }
  
  Matrix3& Matrix3::operator*= (float s)
  {
    m1 *= s;
    m2 *= s;
    m3 *= s;
    return *this;
  }
  
  void Matrix3::identity()
  {
    m1.y = m1.z = 0;
    m2.x = m2.z = 0;
    m3.x = m3.y = 0;
    m1.x = m2.y = m3.z = 1;
  }
  
  void Matrix3::transpose()
  {
    float swap;
    swap = m1.y; m1.y = m2.x; m2.x = swap;
    swap = m1.z; m1.z = m3.x; m3.x = swap;
    swap = m2.z; m2.z = m3.y; m3.y = swap;
  }
  
  void Matrix3::inverse()
  {
    float s = (float)(1.0/determinant());
    Matrix3 C;
    C.m1.x =  (m2.y*m3.z - m2.z*m3.y);
    C.m1.y = -(m2.x*m3.z - m2.z*m3.x);
    C.m1.z =  (m2.x*m3.y - m2.y*m3.x);
    C.m2.x = -(m1.y*m3.z - m1.z*m3.y);
    C.m2.y =  (m1.x*m3.z - m1.z*m3.x);
    C.m2.z = -(m1.x*m3.y - m1.y*m3.x);
    C.m3.x =  (m1.y*m2.z - m1.z*m2.y);
    C.m3.y = -(m1.x*m2.z - m1.z*m2.x);
    C.m3.z =  (m1.x*m2.y - m1.y*m2.x);
    C.transpose();
    *this = C;
    (*this) *= s;
  }
  
  float Matrix3::determinant()
  {
    return
      m1.x * (m2.y*m3.z - m2.z*m3.y)
      -m1.y * (m2.x*m3.z - m2.z*m3.x)
      +m1.z * (m2.x*m3.y - m2.y*m3.x);
  }
  
  void Matrix3::transform (float x, float y, float z, MVector3 &t) const
  {
    t.x = m1.x*x+m1.y*y+m1.z*z;
    t.y = m2.x*x+m2.y*y+m2.z*z;
    t.z = m3.x*x+m3.y*y+m3.z*z;
  }
  
  void Matrix3::transform (MVector3& f) const
  {
    transform(f.x, f.y, f.z, f);
  }
  
  void Matrix3::transform (const MVector3 &Src, MVector3& Dest) const
  {
    transform(Src.x, Src.y, Src.z, Dest);
  }
  
  Matrix3& Matrix3::makeRotationMatrix(float X, float Y, float Z)
  {
    static Matrix3 A,B,C;
    
    A.identity();
    B.identity();
    C.identity();
    
    A.m1.x = (float)(cos(X));
    A.m1.y = (float)(sin(X));
    A.m2.x = (float)(-sin(X));
    A.m2.y = (float)(cos(X));
    
    B.m2.y = (float)(cos(-Y));
    B.m2.z = (float)(sin(-Y));
    B.m3.y = (float)(-sin(-Y));
    B.m3.z = (float)(cos(-Y));
    
    C.m1.x = (float)(cos(Z));
    C.m1.z = (float)(sin(Z));
    C.m3.x = (float)(-sin(Z));
    C.m3.z = (float)(cos(Z));
    
    B *= A;
    
    *this = B;
    return *this;
  }
  
  Matrix3& Matrix3::makeRotationMatrix(MVector3& A)
  {
    return makeRotationMatrix(A.x, A.y, A.z);
  }
  
  void Matrix3::splitHoriz(MVector3& A, MVector3& B, MVector3& C)
  {
    A = m1;
    B = m2;
    C = m3;
  }
  
  void Matrix3::splitVert(MVector3& A, MVector3& B, MVector3& C)
  {
    A.x = m1.x;
    A.y = m2.x;
    A.z = m3.x;
    B.x = m1.y;
    B.y = m2.y;
    B.z = m3.y;
    C.x = m1.z;
    C.y = m2.z;
    C.z = m3.z;
  }
  
  void Matrix3::makeHoriz(MVector3& A, MVector3& B, MVector3& C)
  {
    set(A,B,C);
  }
  
  void Matrix3::makeVert(MVector3& A, MVector3& B, MVector3& C)
  {
    set(A.x, B.x, C.x, A.y, B.y, C.y, A.z, B.z, C.z);
  }
  
  // This allows array-like access for our elements m1, m2 and m3.
  MVector3& Matrix3::operator[] (int i)
  {
    i = i%3;
    switch(i)
    {
    case 0:
      return m1;
      break;
    case 1:
      return m2;
      break;
    case 2:
      return m3;
      break;
    default:
      return m1;
      break;
    }
  }
  
  //---------------------------------------------------------------------------
  bool operator==(const MVector3 &LHS, const MVector3 &RHS) {
    if (LHS.x == RHS.x && LHS.y == RHS.y && LHS.z == RHS.z)
      return true;
    
    return false;
  }
  
  //----------------------------------------------------------------------------------------
  //  MRay
  //----------------------------------------------------------------------------------------
  
  MVector3 MRay::intersectWithPlane(const MPlane &P) const {
    // This is the intersection between the Plane P, and the line parameterised by x(t), y(t), z(t)
    // This intersection produces:
    //            t = (a*ox + b*oy + c*oz - d) / (a*dx + b*dy + c*dz)
    
    float       t;
    MVector3    Res;
    
    t = (P.A*Org.x + P.B*Org.y + P.C*Org.z + P.D) / (P.A*Dir.x + P.B*Dir.y + P.C*Dir.z);
    
    Res = Org;
    Res -= t*Dir;
    
    return Res;
  }
  
  void MRay::set(const MVector3 &P1, const MVector3 &P2, float Mult)
  {
    MVector3  Dir;
    
    Dir = P2;
    Dir -= P1;
    Dir.normalize();
    Dir *= Mult;
    
    set(P1, Dir);
  }
  
  //----------------------------------------------------------------------------------------
  //  MPlane
  //----------------------------------------------------------------------------------------
  
  MPlane::MPlane(const MVector3 &Origin, const MVector3 &Normal)
  {
    A = Normal.x;
    B = Normal.y;
    C = Normal.z;
    
    D = -(A*Origin.x + B*Origin.y + C*Origin.z);
  }
  
  void MPlane::set(const MVector3 &Origin, const MVector3 &Normal)
  {
    A = Normal.x;
    B = Normal.y;
    C = Normal.z;
    
    D = -(A*Origin.x + B*Origin.y + C*Origin.z);
  }
  
  MVector3 MPlane::normal() const {
		MVector3 N(A, B, C);
		return N;
	}

  //----------------------------------------------------------------------------------------
  //  MVector4
  //----------------------------------------------------------------------------------------
  MVector4::MVector4()
  {
  }
  
  MVector4::MVector4(const MVector4 &Src)
  {
    x = Src.x;
    y = Src.y;
    z = Src.z;
    w = Src.w;
  }
  
  MVector4::MVector4(const MVector3 &Src)
  {
    x = Src.x;
    y = Src.y;
    z = Src.z;
    w = 1.0f;
  }
  
  MVector4::MVector4(float nx, float ny, float nz, float nw) 
    : x(nx), y(ny), z(nz), w(nw) 
  {
  }
  
  // Memeber functions.
  void MVector4::set(float nx, float ny, float nz, float nw)
  {
    x = nx;
    y = ny;
    z = nz;
    w = nw;
  }
  
  
  float& MVector4::operator[](int i)
  {
    static float      DummyVal;
    switch(i)
    {
    case 0:
      return x;
    case 1:
      return y;
    case 2:
      return z;
    case 3:
      return w;
    default:
      return DummyVal;
    }
  }
  
  float MVector4::operator[](int i) const
  {
    static float      DummyVal;
    switch(i)
    {
    case 0:
      return x;
    case 1:
      return y;
    case 2:
      return z;
    case 3:
      return w;
    default:
      return DummyVal;
    }
  }
  
  //----------------------------------------------------------------------------------------
  //  MMatrix4
  //----------------------------------------------------------------------------------------
  
  MMatrix4::MMatrix4()
  {
  }
  
  MMatrix4::MMatrix4(const MMatrix4 &Src)
  {
    *this = Src;
  }
  
  MMatrix4::MMatrix4(const float Src[4][4])
  {
    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
        m[i][j] = Src[i][j];
    }
  }
  
  MMatrix4::MMatrix4(const float Src[16])
  {
    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
        m[i][j] = Src[j*4+i];
    }
  }
  
  MMatrix4::MMatrix4(const double Src[16])
  {
    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
        m[i][j] = (float)Src[j*4+i];
    }
  }
  
  MMatrix4::MMatrix4(const MVector3 &srcAxis, const MVector3 &axisToMatch) {
    createRotationMatrix(srcAxis, axisToMatch);
  }

  MMatrix4& MMatrix4::operator=(const MMatrix4 &RHS)
  {
    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
        m[i][j] = RHS.m[i][j];
    }
    
    return *this;
  }

  MMatrix4& MMatrix4::operator=(const Matrix3 &RHS)
  {
    for (int i=0; i<3; i++) {
      m[0][i] = RHS.m1[i];
      m[1][i] = RHS.m2[i];
      m[2][i] = RHS.m3[i];
    }
    
    m[3][0] = 0;
    m[3][1] = 0;
    m[3][2] = 0;
    m[3][3] = 1;
    m[2][3] = 0;
    m[1][3] = 0;
    m[0][3] = 0;

    return *this;
  }
  
  MMatrix4& MMatrix4::operator+=(const MMatrix4 &RHS)
  {
    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
        m[i][j] += RHS.m[i][j];
    }
    
    return *this;
  }
  
  MMatrix4& MMatrix4::operator*=(const MMatrix4 &RHS)
  {
    MMatrix4    Mat;
    
    /*   int   i,j,k;
    float Sum;
    
      for (i=0; i<4; i++)
      {
      for (j=0; j<4; j++)
      {
      Sum = 0.0f;
      for (k=0; k<4; k++)
      {
      Sum += m[k][i]*RHS.m[j][k];
      }
      Mat.m[j][i] = Sum;
      }
      }
    *this = Mat;*/
    
    Mat = *this * RHS;
    *this = Mat;
    
    return *this;
  }
  
  
  MMatrix4 MMatrix4::operator+(const MMatrix4 &RHS) const
  {
    MMatrix4 Mat;
    
    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
        Mat.m[i][j] = m[i][j] + RHS.m[i][j];
    }
    
    return Mat;
  }
  
  MMatrix4 MMatrix4::operator-(const MMatrix4 &RHS) const
  {
    MMatrix4 Mat;
    
    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
        Mat.m[i][j] = m[i][j] - RHS.m[i][j];
    }
    
    return Mat;
  }
  
  
  MMatrix4 MMatrix4::operator*(const MMatrix4 &RHS) const
  {
    MMatrix4    Mat;
    
    int   i,j,k;
    float Sum;
    
    for (i=0; i<4; i++)
    {
      for (j=0; j<4; j++)
      {
        Sum = 0.0f;
        for (k=0; k<4; k++)
        {
          Sum += m[j][k]*RHS.m[k][i];
        }
        Mat.m[j][i] = Sum;
      }
    }
    return Mat;
  }
  
  MVector4 MMatrix4::operator*(const MVector4 &RHS) const
  {
    MVector4    Vec;
    
    int   i,j;
    float Sum;
    
    for (i=0; i<4; i++)
    {
      Sum = 0.0f;
      for (j=0; j<4; j++)
      {
        Sum += m[j][i]*RHS[j];
      }
      Vec[i] = Sum;
    }
    
    return Vec;
  }
  
  MVector3 MMatrix4::operator*(const MVector3 &RHS) const
  {
    MVector3    Res;
    
    Res.x = m[0][0]*RHS.x + m[1][0]*RHS.y + m[2][0]*RHS.z + m[3][0];
    Res.y = m[0][1]*RHS.x + m[1][1]*RHS.y + m[2][1]*RHS.z + m[3][1];
    Res.z = m[0][2]*RHS.x + m[1][2]*RHS.y + m[2][2]*RHS.z + m[3][2];
    
    /*   Res.x = m[0][0]*RHS.x + m[0][1]*RHS.y + m[0][2]*RHS.z + m[0][3];
    Res.y = m[1][0]*RHS.x + m[1][1]*RHS.y + m[1][2]*RHS.z + m[1][3];
    Res.z = m[2][0]*RHS.x + m[2][1]*RHS.y + m[2][2]*RHS.z + m[2][3];*/
    
    return Res;
  }

  MRay MMatrix4::operator*(const MRay &rhs) const {
    MRay result;

    result.Org = (*this)*rhs.Org;

    // the direction is only effected by the rotation part of the matrix
    result.Dir.x = m[0][0]*rhs.Dir.x + m[1][0]*rhs.Dir.y + m[2][0]*rhs.Dir.z;
    result.Dir.y = m[0][1]*rhs.Dir.x + m[1][1]*rhs.Dir.y + m[2][1]*rhs.Dir.z;
    result.Dir.z = m[0][2]*rhs.Dir.x + m[1][2]*rhs.Dir.y + m[2][2]*rhs.Dir.z;

    return result;
  }

  
  MMatrix4& MMatrix4::transpose()
  {
    for (int i=0; i<4; i++)
    {
      for (int j=i+1; j<4; j++)
      {
        float    Temp;
        
        Temp = m[i][j];
        m[i][j] = m[j][i];
        m[j][i] = Temp;
      }
    }
    return *this;
  }
  
  MMatrix4& MMatrix4::inverse()
  {
    
    MMatrix4    Mat;
    
    Mat = *this;
    
    int indxc[4], indxr[4], ipiv[4];
    int i, icol, irow, j, k, l, ll;
    float big, dum, pivinv;
    
    icol = irow = 0;
    ipiv[0] = ipiv[1] = ipiv[2] = ipiv[3] = 0;
    
    for (i = 0; i < 4; i++) // compile with loop unrolling
    {
      big = 0;
      for (j = 0; j < 4; j++)
      {
        if (ipiv[j] != 1)
        {
          for (k = 0; k < 4; k++)
          {
            if (!ipiv[k])
            {
              if ((dum = (float)fabs(m[j][k])) >= big)
              {
                big = dum;
                irow = j;
                icol = k;
              }
            }
            else if (ipiv[k] > 1)
              return *this;
          }
        }
      }
      ++ipiv[icol];
      if (irow != icol)
      {
        for (l = 0; l < 4; l++)
        {
          float tmp = m[irow][l];
          m[irow][l] = m[icol][l];
          m[icol][l] = tmp;
        }
      }
      indxr[i] = irow;
      indxc[i] = icol;
      if ((dum = m[icol][icol])==0)
        return *this;
      
      pivinv = 1 / dum;
      m[icol][icol] = 1;
      for (l = 0; l < 4; l++)
        m[icol][l] *= pivinv;
      for (ll = 0; ll < 4; ll++)
      {
        if (ll != icol)
        {
          dum = m[ll][icol];
          m[ll][icol] = 0;
          for (l = 0; l < 4; l++)
            m[ll][l] -= m[icol][l]*dum;
        }
      }
    }
    for (l = 3; l >= 0; l--)
    {
      if (indxr[l] != indxc[l])
      {
        for (k = 0; k < 4; k++)
        {
          float tmp = m[k][indxr[l]];
          m[k][indxr[l]] = m[k][indxc[l]];
          m[k][indxc[l]] = tmp;
        }
      }
    }
    
    MMatrix4    Result;
    
    Result = (*this)*Mat;
    
    
    
    return *this;
  }
  
  MMatrix4& MMatrix4::identity()
  {
    for (int i=0; i<4; i++)
    {
      for (int j=0; j<4; j++)
        m[i][j] = 0.0f;
      
      m[i][i] = 1.0f;
    }
    
    return *this;
  }
  

  void MMatrix4::convertFromEuler(MVector3 src, int Order) {
    src.x = -src.x;
    src.y = -src.y;
    src.z = -src.z;
    float ti, tj, th, ci, cj, ch, si, sj, sh, cc, cs, sc, ss;
    int i,j,k,h,n,s,f;
    EulGetOrd(Order,i,j,k,h,n,s,f);
    if (f==EulFrmR) {float t = src.x; src.x = src.z; src.z = t;}
    if (n==EulParOdd) {src.x = -src.x; src.y = -src.y; src.z = -src.z;}
    ti = src.x;	  tj = src.y;	th = src.z;
    ci = (float)cos(ti); cj = (float)cos(tj); ch = (float)cos(th);
    si = (float)sin(ti); sj = (float)sin(tj); sh = (float)sin(th);
    cc = ci*ch; cs = ci*sh; sc = si*ch; ss = si*sh;
    if (s==EulRepYes) {
      m[i][i] = cj;	  m[i][j] =  sj*si;    m[i][k] =  sj*ci;
      m[j][i] = sj*sh;  m[j][j] = -cj*ss+cc; m[j][k] = -cj*cs-sc;
      m[k][i] = -sj*ch; m[k][j] =  cj*sc+cs; m[k][k] =  cj*cc-ss;
    } else {
      m[i][i] = cj*ch; m[i][j] = sj*sc-cs; m[i][k] = sj*cc+ss;
      m[j][i] = cj*sh; m[j][j] = sj*ss+cc; m[j][k] = sj*cs-sc;
      m[k][i] = -sj;	 m[k][j] = cj*si;    m[k][k] = cj*ci;
    }
    m[Quat_W][Quat_X] = m[Quat_W][Quat_Y] = m[Quat_W][Quat_Z] = m[Quat_X][Quat_W] = m[Quat_Y][Quat_W] = m[Quat_Z][Quat_W]=0.0; 
    m[Quat_W][Quat_W]=1.0;
  }
  
  
  void MMatrix4::convertToEuler(MVector3 &Dest, int Order)
  {
    EulerAngles ea;
    int i,j,k,h,n,s,f;
    EulGetOrd(Order,i,j,k,h,n,s,f);
    if (s==EulRepYes) 
    {
      double sy = sqrt(m[i][j]*m[i][j] + m[i][k]*m[i][k]);
      if (sy > 16*FLT_EPSILON)
      {
        ea.x = (float)atan2(m[i][j], m[i][k]);
        ea.y = (float)atan2((double)sy, (double)m[i][i]);
        ea.z = (float)atan2(m[j][i], -m[k][i]);
      }
      else
      {
        ea.x = (float)atan2(-m[j][k], m[j][j]);
        ea.y = (float)atan2((double)sy, (double)m[i][i]);
        ea.z = 0;
      }
    } else {
      double cy = sqrt(m[i][i]*m[i][i] + m[j][i]*m[j][i]);
      if (cy > 16*FLT_EPSILON) {
        ea.x = (float)atan2(m[k][j], m[k][k]);
        ea.y = (float)atan2((double)-m[k][i], cy);
        ea.z = (float)atan2(m[j][i], m[i][i]);
      } else  {
        ea.x = (float)atan2((double)-m[j][k], (double)m[j][j]);
        ea.y = (float)atan2((double)-m[k][i], (double)cy);
        ea.z = 0;
      }
      
    }

    if (n==EulParOdd){
      ea.x = -ea.x; 
      ea.y = -ea.y; 
      ea.z = -ea.z;
    }
    if (f==EulFrmR) {
      float t = ea.x; 
      ea.x = ea.z; 
      ea.z = t;
    }
    ea.w = (float)Order;
    
    Dest.x = -ea.x;
    Dest.y = -ea.y;
    Dest.z = -ea.z;
  }
 
  MMatrix4& MMatrix4::rotateWithVector(const MVector3 axis, float ang) {
    MVector3 in, out;

    in.set(m[0][0], m[0][1], m[0][2]);
    Aztec::rotateWithVector(in, axis, ang, out);
    m[0][0] = out.x; m[0][1] = out.y; m[0][2] = out.z;

    in.set(m[1][0], m[1][1], m[1][2]);
    Aztec::rotateWithVector(in, axis, ang, out);
    m[1][0] = out.x; m[1][1] = out.y; m[1][2] = out.z;

    in.set(m[2][0], m[2][1], m[2][2]);
    Aztec::rotateWithVector(in, axis, ang, out);
    m[2][0] = out.x; m[2][1] = out.y; m[2][2] = out.z;

    in.set(m[3][0], m[3][1], m[3][2]);
    Aztec::rotateWithVector(in, axis, ang, out);
    m[3][0] = out.x; m[3][1] = out.y; m[3][2] = out.z;

    return *this;
  }

  void MMatrix4::normaliseRotation() {
    MVector3 temp;

    temp.set(m[0][0], m[0][1], m[0][2]);
    temp.normalize();
    m[0][0] = temp.x; m[0][1] = temp.y; m[0][2] = temp.z; 

    temp.set(m[1][0], m[1][1], m[1][2]);
    temp.normalize();
    m[1][0] = temp.x; m[1][1] = temp.y; m[1][2] = temp.z; 

    temp.set(m[2][0], m[2][1], m[2][2]);
    temp.normalize();
    m[2][0] = temp.x; m[2][1] = temp.y; m[2][2] = temp.z; 
    
  }

  void MMatrix4::extractFrustum(MPlane frustum[]) {
    // code adapted from Mark Morley's page, 
    // http://www.markmorley.com/opengl/frustumculling.html
    float   t;
    
    /* Extract the numbers for the RIGHT plane */
    frustum[0].A = m[0][3] - m[0][0];
    frustum[0].B = m[1][3] - m[1][0];
    frustum[0].C = m[2][3] - m[2][0];
    frustum[0].D = m[3][3] - m[3][0];
    
    /* Normalize the result */
    t = (float)sqrt( frustum[0].A * frustum[0].A + frustum[0].B * frustum[0].B + frustum[0].C * frustum[0].C );
    frustum[0].A /= t;
    frustum[0].B /= t;
    frustum[0].C /= t;
    frustum[0].D /= t;
    
    /* Extract the numbers for the LEFT plane */
    frustum[1].A = m[0][3] + m[0][0];
    frustum[1].B = m[1][3] + m[1][0];
    frustum[1].C = m[2][3] + m[2][0];
    frustum[1].D = m[3][3] + m[3][0];
    
    /* Normalize the result */
    t = (float)sqrt( frustum[1].A * frustum[1].A + frustum[1].B * frustum[1].B + frustum[1].C * frustum[1].C );
    frustum[1].A /= t;
    frustum[1].B /= t;
    frustum[1].C /= t;
    frustum[1].D /= t;
    
    /* Extract the BOTTOM plane */
    frustum[2].A = m[0][3] + m[0][1];
    frustum[2].B = m[1][3] + m[1][1];
    frustum[2].C = m[2][3] + m[2][1];
    frustum[2].D = m[3][3] + m[3][1];
    
    /* Normalize the result */
    t = (float)sqrt( frustum[2].A * frustum[2].A + frustum[2].B * frustum[2].B + frustum[2].C * frustum[2].C );
    frustum[2].A /= t;
    frustum[2].B /= t;
    frustum[2].C /= t;
    frustum[2].D /= t;
    
    /* Extract the TOP plane */
    frustum[3].A = m[0][3] - m[0][1];
    frustum[3].B = m[1][3] - m[1][1];
    frustum[3].C = m[2][3] - m[2][1];
    frustum[3].D = m[3][3] - m[3][1];
    
    /* Normalize the result */
    t = (float)sqrt( frustum[3].A * frustum[3].A + frustum[3].B * frustum[3].B + frustum[3].C * frustum[3].C );
    frustum[3].A /= t;
    frustum[3].B /= t;
    frustum[3].C /= t;
    frustum[3].D /= t;
    
    /* Extract the FAR plane */
    frustum[4].A = m[0][3] - m[0][2];
    frustum[4].B = m[1][3] - m[1][2];
    frustum[4].C = m[2][3] - m[2][2];
    frustum[4].D = m[3][3] - m[3][2];
    
    /* Normalize the result */
    t = (float)sqrt( frustum[4].A * frustum[4].A + frustum[4].B * frustum[4].B + frustum[4].C * frustum[4].C );
    frustum[4].A /= t;
    frustum[4].B /= t;
    frustum[4].C /= t;
    frustum[4].D /= t;
    
    /* Extract the NEAR plane */
    frustum[5].A = m[0][3] + m[0][2];
    frustum[5].B = m[1][3] + m[1][2];
    frustum[5].C = m[2][3] + m[2][2];
    frustum[5].D = m[3][3] + m[3][2];
    
    /* Normalize the result */
    t = (float)sqrt( frustum[5].A * frustum[5].A + frustum[5].B * frustum[5].B + frustum[5].C * frustum[5].C );
    frustum[5].A /= t;
    frustum[5].B /= t;
    frustum[5].C /= t;
    frustum[5].D /= t;

  }

  void MMatrix4::createRotationMatrix(const MVector3 &fromNormal, const MVector3 &toNormal) {
    MVector3 axis;
    MMatrix4 result;
    
    identity();

    float angle((float)acos(fromNormal * toNormal));

    if (fabs(angle) > 0.01) {
      axis = fromNormal / toNormal;
      axis.normalize();

      // rotate with vector takes radians
      rotateWithVector(axis, angle);
    }
  }


  //----------------
  // MQuaternion
  //----------------

  MQuaternion::MQuaternion() 
    : W(0.0), X(0.0), Y(0.0), Z(0.0) {
  }

  MQuaternion::MQuaternion(float angle, float x, float y, float z) 
    : W(angle), X(x), Y(y), Z(z) {
  }

  MQuaternion operator *(const MQuaternion &lhs, const MQuaternion &rhs) {
    float w,x,y,z;
    
    w = lhs.W*rhs.W - (lhs.X*rhs.X + lhs.Y*rhs.Y + lhs.Z*rhs.Z);
    
    x = lhs.W*rhs.X + rhs.W*lhs.X + lhs.Y*rhs.Z - lhs.Z*rhs.Y;
    y = lhs.W*rhs.Y + rhs.W*lhs.Y + lhs.Z*rhs.X - lhs.X*rhs.Z;
    z = lhs.W*rhs.Z + rhs.W*lhs.Z + lhs.X*rhs.Y - lhs.Y*rhs.X;
    
    return MQuaternion(w,x,y,z);
  }

  const MQuaternion& MQuaternion::operator *=(const MQuaternion &rhs) {
    float w = W*rhs.W - (X*rhs.X + Y*rhs.Y + Z*rhs.Z);

    float x = W*rhs.X + rhs.W*X + Y*rhs.Z - Z*rhs.Y;
    float y = W*rhs.Y + rhs.W*Y + Z*rhs.X - X*rhs.Z;
    float z = W*rhs.Z + rhs.W*Z + X*rhs.Y - Y*rhs.X;

    W = w;
    X = x;
    Y = y;
    Z = z;
    return *this;
  }

  const MQuaternion& MQuaternion::operator ~() {
    X = -X;
    Y = -Y;
    Z = -Z;
    return *this;
  }

  const MQuaternion& MQuaternion::operator -() {
    float norme = (float)sqrt(W*W + X*X + Y*Y + Z*Z);
    if (norme == 0.0f)
      norme = 1.0f;

    float recip = 1.0f / norme;

    W =  W * recip;
    X = -X * recip;
    Y = -Y * recip;
    Z = -Z * recip;

    return *this;  
  }

  const MQuaternion& MQuaternion::normalize() {
    float norme = (float)sqrt(W*W + X*X + Y*Y + Z*Z);
    if (norme == 0.0f) {
      W = 1.0f;
      X = Y = Z = 0.0f;

    } else {

      float recip = 1.0f / norme;
      
      W *= recip;
      X *= recip;
      Y *= recip;
      Z *= recip;
    }
    return *this;
  }

  void MQuaternion::slerp(const MQuaternion &from, const MQuaternion &to, const float t) {
    float omega, cosom, sinom, sclp, sclq;
    
    cosom = from.X*to.X + from.Y*to.Y + from.Z*to.Z + from.W*to.W;
    
    if ((1.0f+cosom) > FLT_EPSILON) {
      if ((1.0f-cosom) > FLT_EPSILON) {
        omega = (float)acos(cosom);
        sinom = (float)sin(omega);
        sclp = (float)sin((1.0f-t)*omega) / sinom;
        sclq = (float)sin(t*omega) / sinom;
      } else {
        sclp = 1.0f - t;
        sclq = t;
      }
      
      X = sclp*from.X + sclq*to.X;
      Y = sclp*from.Y + sclq*to.Y;
      Z = sclp*from.Z + sclq*to.Z;
      W = sclp*from.W + sclq*to.W;
      
    } else {
      X =-from.Y;
      Y = from.X;
      Z =-from.W;
      W = from.Z;
      
      sclp = (float)sin((1.0f-t) * PI * 0.5);
      sclq = (float)sin(t * PI * 0.5);
      
      X = sclp*from.X + sclq*to.X;
      Y = sclp*from.Y + sclq*to.Y;
      Z = sclp*from.Z + sclq*to.Z;
      
    }
  }

  const MQuaternion& MQuaternion::exp() {
    float Mul;
    float Length = (float)sqrt(X*X + Y*Y + Z*Z);
    
    if (Length > 1.0e-4) {
      Mul = (float)sin(Length)/Length;
    } else {
      Mul = 1.0f;
    }
    
    W = (float)cos(Length);
    
    X *= Mul;
    Y *= Mul;
    Z *= Mul;
    
    return *this;
  }

  const MQuaternion& MQuaternion::log() {
    float Length;
    
    Length = (float)sqrt(X*X + Y*Y + Z*Z);
    Length = (float)atan(Length/W);
    
    W = 0.0;
    
    X *= Length;
    Y *= Length;
    Z *= Length;
    
    return *this;  
  }
    
  const MQuaternion& MQuaternion::fromAxis(const float angle, float x, float y, float z) {
    float omega, s, c;
    
    s = (float)sqrt(x*x + y*y + z*z);
    
    if (fabs(s) > FLT_EPSILON) {
      c = 1.0f / s;
      
      x *= c;
      y *= c;
      z *= c;
      
      omega = -0.5f * angle;
      s = (float)sin(omega);
      
      X = s*x;
      Y = s*y;
      Z = s*z;
      W = (float)cos(omega);
    } else {
      X = Y = 0.0f;
      Z = 0.0f;
      W = 1.0f;
    }
    normalize();
    return *this;
  }

  void MQuaternion::toMatrix(float matrix[3][3]) const {
    matrix[0][0] = 1.0f - 2*Y*Y - 2*Z*Z;
    matrix[1][0] = 2*X*Y + 2*W*Z;
    matrix[2][0] = 2*X*Z - 2*W*Y;
    
    matrix[0][1] = 2*X*Y - 2*W*Z;
    matrix[1][1] = 1.0f - 2*X*X - 2*Z*Z;
    matrix[2][1] = 2*Y*Z + 2*W*X;
    
    matrix[0][2] = 2*X*Z + 2*W*Y;
    matrix[1][2] = 2*Y*Z - 2*W*X;
    matrix[2][2] = 1.0f - 2*X*X - 2*Y*Y;
  }

  void MQuaternion::toMatrix(MMatrix4 &matrix) const {

    matrix.m[0][0] = 1.0f - 2*Y*Y - 2*Z*Z;
    matrix.m[1][0] = 2*X*Y + 2*W*Z;
    matrix.m[2][0] = 2*X*Z - 2*W*Y;
    matrix.m[3][0] = 0.0;

    matrix.m[0][1] = 2*X*Y - 2*W*Z;
    matrix.m[1][1] = 1.0f - 2*X*X - 2*Z*Z;
    matrix.m[2][1] = 2*Y*Z + 2*W*X;
    matrix.m[3][1] = 0.0;

    matrix.m[0][2] = 2*X*Z + 2*W*Y;
    matrix.m[1][2] = 2*Y*Z - 2*W*X;
    matrix.m[2][2] = 1.0f - 2*X*X - 2*Y*Y;
    matrix.m[3][2] = 0.0f;

    matrix.m[0][3] = 0.0;
    matrix.m[1][3] = 0.0;
    matrix.m[2][3] = 0.0;
    matrix.m[3][3] = 1.0;

  }

  std::string MQuaternion::toString() const {
    char buf[100];

    sprintf(buf, "%.3f + %.3fi + %.3fj + %.3fk", W, X, Y, Z);

    return std::string(buf);
  }

  MOutputStream &operator <<(MOutputStream &out, const MVector3 &vec) {
    out << vec.x << vec.y << vec.z;
    return out;
  }

  MInputStream &operator >>(MInputStream&in, MVector3 &vec) {
    in >> vec.x >> vec.y >> vec.z;
    return in;
  }

  MOutputStream &operator <<(MOutputStream &out, const MMatrix4 &mat) {
    for (int i = 0; i < 4; ++i) {
      for (int j = 0; j < 4; ++j) {
        out << mat.m[i][j];
      }
    }
    return out;
  }

  MInputStream &operator >>(MInputStream &in, MMatrix4 &mat) {
    for (int i = 0; i < 4; ++i) {
      for (int j = 0; j < 4; ++j) {
        in >> mat.m[i][j];
      }
    }
    return in;
  }


}