#include "StdAfx.h"

#include <GL/gl.h>
#include <GL/glu.h>

#include "MImages.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {
  
  //---------------
  //  MImage
  //---------------

  MImage::MImage() {
	m_FType = miRGB;
	m_Stride = 0;
	m_BitsPerPixel = 0;
	m_ChannelsPerPixel = 0;

	m_BitsPerMapPixel = 0;
	m_ChannelsPerMapPixel = 0;
	m_MapLen = 0;
	m_ColorMap = NULL;

    m_Pixels = NULL;
    m_Alpha = NULL;
    
    m_RequireAlpha = false;
    
    m_GLImage = NULL;
  }
  
  MImage::~MImage() {
    if (m_Pixels) {
      delete[] m_Pixels;
    }
    
    if (m_ColorMap) {
      delete[] m_ColorMap;
    }
        
    if (m_Alpha) {
      delete[] m_Alpha;
    }
    
    if (m_GLImage != NULL) {
      deleteGLImage();
      m_GLImage = NULL;
    }
  }

  int MImage::getWidth() {
    return m_Width;
  }

  int MImage::getHeight() {
    return m_Height;
  }
  

  MBaseObjectPtr MImage::createNew() {
    MImagePtr Image;
    
    Image = new MImage;
    
    return Image;
  }
  
  
  void MImage::setSize(int Width, int Height, MImageSizeOp Op) {
    BYTE  *NewPixels, *NewAlpha;
    
    if (Width <= 0 || Height <= 0) {
      m_Width = 0;
      m_Height = 0;
      if (m_Pixels) {
        delete[] m_Pixels;
      }
      
      if (m_Alpha)
        delete[] m_Alpha;
      m_Pixels = NULL;
      m_Alpha = NULL;
      
      deleteGLImage();
      
      return;
    }
    
    NewPixels = new BYTE[Width*Height*3];
    
    if (m_RequireAlpha)
      NewAlpha = new BYTE[Width*Height];
    else
      NewAlpha = NULL;
    
    if (Op == miDiscard) {
      if (m_Pixels) {
        delete[] m_Pixels;
      }
      
      if (m_Alpha) {
        delete[] m_Alpha;
      }
      
      m_Pixels = NewPixels;
      m_Alpha = NewAlpha;
      
      m_Width = Width;
      m_Height = Height;
      
      clear(0,0,0);
    }

    if (Op == miStretch) {
      if (m_Pixels) {
        gluScaleImage(GL_RGB, m_Width, m_Height, GL_UNSIGNED_BYTE, m_Pixels, Width, Height, GL_UNSIGNED_BYTE, NewPixels);
        delete[] m_Pixels;
      }
      if (m_Alpha && m_RequireAlpha) {
        gluScaleImage(GL_ALPHA, m_Width, m_Height, GL_UNSIGNED_BYTE, m_Alpha, Width, Height, GL_UNSIGNED_BYTE, NewAlpha);
        delete[] m_Alpha;
      }
      
      m_Pixels = NewPixels;
      m_Alpha = NewAlpha;
      
      m_Width = Width;
      m_Height = Height;
      
    }
    if (Op == miCrop) {
      m_Width = Width;
      m_Height = Height;
    }
    
    deleteGLImage();
  }
  
  void MImage::clear(BYTE r, BYTE g, BYTE b, BYTE a) {
    BYTE     *Pixel, *Alpha;
    int      n;
    
    Pixel = m_Pixels;
    Alpha = m_Alpha;
    
    if (m_Pixels) {
      for (n=m_Width*m_Height-1;n>=0;n--) {
        *Pixel = r;    Pixel++;
        *Pixel = g;    Pixel++;
        *Pixel = b;    Pixel++;
      }
    }
    if (m_Alpha && m_RequireAlpha) {
      for (n = m_Width*m_Height-1 ; n >= 0 ; n--) {
        *Alpha = a;    Alpha++;
      }
    }
    
    deleteGLImage();
  }

  BYTE* MImage::getPixelData() {
    return m_Pixels;
  }

  BYTE* MImage::getAlphaData() {
    return m_Alpha;
  }
  
  MImage& MImage::operator=(const MImage &Src) {
    if (m_Pixels) {
      delete[] m_Pixels;
    }

    m_Pixels = NULL;
    
    if (m_Alpha) {
      delete[] m_Alpha;
    }
    m_Alpha = NULL;
    
    m_RequireAlpha = Src.m_RequireAlpha;
    setSize(Src.m_Width, Src.m_Height);
    
    // Copy the pixel data
    if (m_Width > 0 && m_Height > 0) {
      memcpy(m_Pixels, Src.m_Pixels, m_Width*m_Height*3);
      if (m_RequireAlpha) {
        memcpy(m_Alpha, Src.m_Alpha, m_Width*m_Height);
      }
    }
    
    deleteGLImage();
    
    return *this;   
  }
  
  void MImage::setFrom8Bit(BYTE *SrcPixels, BYTE **SrcPal) {
    int   n;
    BYTE  *Pixel, *Src;
    
    Pixel = m_Pixels;
    Src = SrcPixels;
    for (n = m_Width*m_Height-1 ; n>=0 ; n--) {
      *Pixel = SrcPal[*Src][0];  Pixel++;
      *Pixel = SrcPal[*Src][1];  Pixel++;
      *Pixel = SrcPal[*Src][2];  Pixel++;
      
      Src++;
    }
  }
  
  void MImage::setSizeGLFriendly(bool RoundDown, int Max, MImageSizeOp Op) {
    // Take the current image and adjusts it so the dimensions are 2^n by 2^m
    
    int PowerX, PowerY;
    
    for (PowerX = 0; 1 << PowerX < m_Width && 1 << PowerX < Max; PowerX++) {
    }

    if (RoundDown) {
      PowerX -- ;
    }
    
    for (PowerY = 0; 1 << PowerY < m_Height && 1 << PowerY < Max; PowerY++) {
    }

    if (RoundDown) {
      PowerY -- ;
    }
    
    setSize(1 << PowerX, 1 << PowerY, Op);
  }
  
  // Returns the MImage object that is useable for OpenGL Textures.
  MImagePtr MImage::getGLImage() {
    return m_GLImage;
  }
  
  // Returns the OpenGL Name that represents the texture.
  UINT MImage::getGLImageName() {
    if (m_GLImage != NULL) {
      return m_GLImage->m_TextureName;
    }
    
    return 0;
  }
  
  // Creates a MImage object that is useable for GL Textues and names the texture
  void MImage::createGLImage(bool DontCreateIfExists) {
	  if (m_GLImage != NULL && DontCreateIfExists) {
		  return;
	  }
	  
	  deleteGLImage();
	  
	  m_GLImage = new MImage;
	  
	  *m_GLImage = *this;
	  
	  m_GLImage->setSizeGLFriendly(false, 256, miStretch);
	  
	  glGenTextures(1,&m_GLImage->m_TextureName);
	  
	  
	  UINT     OldTextureBind;
	  
	  glGetIntegerv(GL_TEXTURE_BINDING_2D, (GLint*)&OldTextureBind);
	  
	  glBindTexture(GL_TEXTURE_2D, m_GLImage->m_TextureName);
	  
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	  
	  glPixelTransferf(GL_RED_SCALE, 1.0);
	  glPixelTransferf(GL_GREEN_SCALE, 1.0);
	  glPixelTransferf(GL_BLUE_SCALE, 1.0);
	  
	  if (m_RequireAlpha) {
		  // Make an rgba thingy
		  BYTE  *NewData;
		  
		  NewData = new BYTE[m_GLImage->m_Width*m_GLImage->m_Height*4];
		  for (int n=m_GLImage->m_Width*m_GLImage->m_Height-1; n>=0; n--) {
			  NewData[n*4] = m_GLImage->m_Pixels[n*3];
			  NewData[n*4+1] = m_GLImage->m_Pixels[n*3+1];
			  NewData[n*4+2] = m_GLImage->m_Pixels[n*3+2];
			  NewData[n*4+3] = m_GLImage->m_Alpha[n];
		  }
		  
		  glTexImage2D(GL_TEXTURE_2D, 0, 4, m_GLImage->m_Width,m_GLImage->m_Height,0,GL_RGBA,GL_UNSIGNED_BYTE , NewData);
		  
		  delete[] NewData;
	  } else {
		  glTexImage2D(GL_TEXTURE_2D, 0, 3, m_GLImage->m_Width,m_GLImage->m_Height,0,GL_RGB,GL_UNSIGNED_BYTE , m_GLImage->m_Pixels);
	  }
  
  glBindTexture(GL_TEXTURE_2D, OldTextureBind);
  }
  
  // Deletes the MImage objects used for GL Textures and deletes the name for the texture.
  void MImage::deleteGLImage() {
    if (m_GLImage == NULL) {
      return;
    }
    
    glDeleteTextures(1, &m_GLImage->m_TextureName);
    
    m_GLImage = NULL;
  }
  
  void MImage::setGLImageAsCurrent(GLenum Target) {
    
    if (m_GLImage == NULL) {
      createGLImage();
    }
    
    glTexParameteri(Target, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(Target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    //   glTexParameteri(Target, GL_TEXTURE_WRAP_S, GL_REPEAT);
    //   glTexParameteri(Target, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glBindTexture(Target, m_GLImage->m_TextureName);
    // glTexImage2D(GL_TEXTURE_2D, 0, 3, m_GLImage->m_Width,m_GLImage->m_Height,0,GL_RGB,GL_UNSIGNED_BYTE , m_GLImage->m_Pixels);
    
  }
  
  void MImage::flipVertical() {
    if (!m_Pixels) {
      return;
    }
    
    if (m_Pixels) {
      BYTE     *Buf;
      
      Buf = new BYTE[m_Height*m_Width*3];
      
      for (int v=0; v < m_Height; v++) {
        memcpy(&Buf[(m_Height-v-1)*m_Width*3], &m_Pixels[v*m_Width*3], m_Width*3);
      }

      delete[] m_Pixels;
      m_Pixels = Buf;
    }
    
    if (m_Alpha) {
      BYTE     *Buf;
      
      Buf = new BYTE[m_Height*m_Width];
      
      for (int v=0; v < m_Height; v++) {
        memcpy(&Buf[(m_Height-v-1)*m_Width], &m_Alpha[v*m_Width], m_Width);
        
      }
      delete[] m_Alpha;
      m_Alpha = Buf;
    }
  }
  
}
