#ifndef MVector3KeyableValue_Header 
#define MVector3KeyableValue_Header

#include "ModelGeneric.h"

namespace Aztec {

  class MVector3KeyableValue;
  typedef MRefCountedPtr<MVector3KeyableValue> MVector3KeyableValuePtr;

}

#include <MFloatKeyableValue.h>
#include "MVector3Value.h"
#include "MKeyableValue.h"

namespace Aztec {

  /**
   * This is an interface class for a list of Keys. Each type
   * of key list implements this interface.
   */
  class MGENEXPORT MVector3KeyableValue : public MVector3Value,
                                          public MKeyableValue {
  public:
    /**
     * This sets the initial value of the vector that is used when
     * there are no keys set.
     *
     * @param value The value to use as the initial value.
     */
    virtual void setInitialValue(const MVector3 &value) = 0;

    /**
     * This sets a key at the given time, overwriting any key that is 
     * already present at that time.
     *
     * @param value The Value to be set at the given time. 
     * @param time The time to set the key at
     */
    virtual void setKey(const MVector3 &value, long time) = 0;

    /**
     * This sets a key for the X component at the given time, overwriting
     * any key that may already be present at that time. This method may
     * also set keys for all three (X, Y and Z) component depending on
     * the implementation.
     *
     * @param value The new value for the key
     * @param time The time for the key to set.
     */
    virtual void setXKey(float value, long time) = 0;

    /**
     * This sets a key for the Y component at the given time, overwriting
     * any key that may already be present at that time. This method may
     * also set keys for all three (X, Y and Z) component depending on
     * the implementation.
     *
     * @param value The new value for the key
     * @param time The time for the key to set.
     */
    virtual void setYKey(float value, long time) = 0;

    /**
     * This sets a key for the Z component at the given time, overwriting
     * any key that may already be present at that time. This method may
     * also set keys for all three (X, Y and Z) component depending on
     * the implementation.
     *
     * @param value The new value for the key
     * @param time The time for the key to set.
     */
    virtual void setZKey(float value, long time) = 0;
  
    /**
     * This gets the Float Animated value for the X coordinte. 
     *
     * @return The Float Value for the X coordinate.
     */
    virtual MFloatKeyableValuePtr getXValue() = 0;

    /**
     * This gets the Float Animated value for the Y coordinte. 
     *
     * @return The Float Value for the Y coordinate.
     */
    virtual MFloatKeyableValuePtr getYValue() = 0;

    /**
     * This gets the Float Animated value for the Z coordinte. 
     *
     * @return The Float Value for the Z coordinate.
     */
    virtual MFloatKeyableValuePtr getZValue() = 0;
  };
  

}

#endif 
