#ifndef MKeyList_Header 
#define MKeyList_Header

#include "ModelGeneric.h"

namespace Aztec {

  class MKeyList;
  typedef MRefCountedPtr<MKeyList> MKeyListPtr;

}

#include "MDAGNode.h"
#include "MKey.h"

namespace Aztec {

  /**
   * This is an interface class for a list of Keys. Each type
   * of key list implements this interface.
   */
  class MGENEXPORT MKeyList : public virtual MDAGNode {
  public:
    typedef enum { METHOD_REPLACE, METHOD_INSERT } MethodType;

    /**
     * This gets the number of keys in the key list.
     */
    virtual int getKeyCount() = 0;

    /**
     * This gets the key at the given index
     *
     * @param index The index of the key to get
     * @return The key at the given index, NULL if the index was 
     *         out of range.
     */
    virtual MKeyPtr getKeyAtIndex(int index) = 0;

    /**
     * This gets the Key that is closest to the given time. If it is 
     * evenly between two keys, it will return the earlier one. 
     *
     * @param time The time of the key to retrieve.
     * @return The key that is closest to the time.
     */
    virtual MKeyPtr getKeyAtTime(long time) = 0;

    /**
     * This sets a key in the key list at the given time, using the method
     * given. If the method is an insert operation, all keys on or after the 
     * given time are shifted by the granularity of the list. If the method
     * is a replace operation, then a key will be placed at the time, overwriting
     * any other key that is there.
     */
    virtual void setKey(MKeyPtr key, long time) = 0;

    /**
     * This copies the keys form the given key list into this one. It will also 
     * change the granularity of the key list.
     * 
     * @param source The key list to copy from.
     */
    virtual void copyFromKeyList(MKeyListPtr source) = 0;

    /**
     * This gets the granularity of the key list. The granularity
     * is the minimum amount of time that each key can be from another
     * key.
     *
     * @return The granularity.
     */
    virtual int getGranularity() = 0;

    /**
     * This sets the granularity of the key list. The granularity
     * is the minimum amount of time that each key can be from another
     * key.
     *
     * @param granularity The granularity.
     */
    virtual void setGranularity(int granularity) = 0;
  };
  

}

#endif 
